// DeviceTreeView.cpp : implementation file
//

#include "stdafx.h"
#include "windowsx.h"
#include "Writeblocker.h"
#include "MainFrm.h"
#include "ToggleMedia.h"
#include "DeviceTreeView.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// setting this flag enables write block toggling from the device tree
#define ENABLE_DEVICETREE_WB_TOGGLING	0

#define DEVICE_TREE_DUMMY_DATA

#define BITMAP_HEIGHT					16
#define BITMAP_WIDTH					16

#define LOCATION_NAME_LOCAL				"Local Devices"
#define LOCATION_NAME_NETWORK			"Network Drives"

#define MENU_ENABLE_WRITEBLOCKING		"&Enable Write Blocking"
#define MENU_DISABLE_WRITEBLOCKING		"&Disable Write Blocking"

extern CStorageDevices LocalDevices;
extern CNetworkDrives NetworkDrives;

/////////////////////////////////////////////////////////////////////////////
// CDeviceTreeView

IMPLEMENT_DYNCREATE(CDeviceTreeView, CTreeView)

CDeviceTreeView::CDeviceTreeView()
{
	// initialize image list
	m_ImageList.Create(BITMAP_WIDTH, BITMAP_HEIGHT, ILC_COLOR4, 2, 1);
	m_ImageMap.InitHashTable(29);

	// store pointer to tree control
	m_pTreeCtrl = &GetTreeCtrl();

	m_hLocalDevices = NULL;
	m_hNetworkDrives = NULL;
}

CDeviceTreeView::~CDeviceTreeView()
{
	m_ImageList.DeleteImageList();
	m_ImageMap.RemoveAll();
}


BEGIN_MESSAGE_MAP(CDeviceTreeView, CTreeView)
	//{{AFX_MSG_MAP(CDeviceTreeView)
	ON_WM_CREATE()
	ON_NOTIFY_REFLECT(NM_RCLICK, OnRclick)
	ON_COMMAND(IDM_DEVICE_WRITEBLOCK, OnDeviceWriteblock)
	ON_WM_SETFOCUS()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CDeviceTreeView drawing

void CDeviceTreeView::OnDraw(CDC* pDC)
{
	CDocument* pDoc = GetDocument();
	// TODO: add draw code here
}

/////////////////////////////////////////////////////////////////////////////
// CDeviceTreeView diagnostics

#ifdef _DEBUG
void CDeviceTreeView::AssertValid() const
{
	CTreeView::AssertValid();
}

void CDeviceTreeView::Dump(CDumpContext& dc) const
{
	CTreeView::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CDeviceTreeView message handlers

int CDeviceTreeView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CTreeView::OnCreate(lpCreateStruct) == -1)
		return -1;

	// modify tree control style
	m_pTreeCtrl->ModifyStyle(0, TVS_HASLINES | TVS_LINESATROOT | TVS_HASBUTTONS);

	// load the tree control's image list
	LoadImageList();
		
	return 0;
}

void CDeviceTreeView::LoadImageList()
{
	CBitmap Bitmap;

	// insert unknown device bitmap into image list
	Bitmap.LoadBitmap(IDB_UNKNOWN_DEVICE);
	m_ImageMap[IDB_UNKNOWN_DEVICE] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert unknown device write blocked bitmap into image list
	Bitmap.LoadBitmap(IDB_UNKNOWN_DEVICE_WRITEBLOCKED);
	m_ImageMap[IDB_UNKNOWN_DEVICE_WRITEBLOCKED] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert local devices bitmap into image list
	Bitmap.LoadBitmap(IDB_LOCAL_DEVICES);
	m_ImageMap[IDB_LOCAL_DEVICES] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert network devices bitmap into image list
	Bitmap.LoadBitmap(IDB_NETWORK_DRIVES);
	m_ImageMap[IDB_NETWORK_DRIVES] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert CD-ROM drive bitmap into image list
	Bitmap.LoadBitmap(IDB_CDROM_DRIVE);
	m_ImageMap[IDB_CDROM_DRIVE] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert CD-ROM drive write blocked bitmap into image list
	Bitmap.LoadBitmap(IDB_CDROM_DRIVE_WRITEBLOCKED);
	m_ImageMap[IDB_CDROM_DRIVE_WRITEBLOCKED] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert disk drive bitmap into image list
	Bitmap.LoadBitmap(IDB_DISK_DRIVE);
	m_ImageMap[IDB_DISK_DRIVE] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert disk drive write blocked bitmap into image list
	Bitmap.LoadBitmap(IDB_DISK_DRIVE_WRITEBLOCKED);
	m_ImageMap[IDB_DISK_DRIVE_WRITEBLOCKED] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert floppy drive bitmap into image list
	Bitmap.LoadBitmap(IDB_FLOPPY_DRIVE);
	m_ImageMap[IDB_FLOPPY_DRIVE] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert floppy drive write blocked bitmap into image list
	Bitmap.LoadBitmap(IDB_FLOPPY_DRIVE_WRITEBLOCKED);
	m_ImageMap[IDB_FLOPPY_DRIVE_WRITEBLOCKED] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert firewire controller bitmap into image list
	Bitmap.LoadBitmap(IDB_FIREWIRE_CONTROLLER);
	m_ImageMap[IDB_FIREWIRE_CONTROLLER] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert IDE controller bitmap into image list
	Bitmap.LoadBitmap(IDB_IDE_CONTROLLER);
	m_ImageMap[IDB_IDE_CONTROLLER] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert PCMCIA controller bitmap into image list
	Bitmap.LoadBitmap(IDB_PCMCIA_CONTROLLER);
	m_ImageMap[IDB_PCMCIA_CONTROLLER] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert SCSI controller bitmap into image list
	Bitmap.LoadBitmap(IDB_SCSI_CONTROLLER);
	m_ImageMap[IDB_SCSI_CONTROLLER] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert USB controller bitmap into image list
	Bitmap.LoadBitmap(IDB_USB_CONTROLLER);
	m_ImageMap[IDB_USB_CONTROLLER] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert network drive bitmap into image list
	Bitmap.LoadBitmap(IDB_NETWORK_DRIVE);
	m_ImageMap[IDB_NETWORK_DRIVE] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// insert network drive write blocked bitmap into image list
	Bitmap.LoadBitmap(IDB_NETWORK_DRIVE_WRITEBLOCKED);
	m_ImageMap[IDB_NETWORK_DRIVE_WRITEBLOCKED] = m_ImageList.Add(&Bitmap, RGB(0, 0, 0));
	Bitmap.DeleteObject();

	// update tree control's image list
	m_pTreeCtrl->SetImageList(&m_ImageList, TVSIL_NORMAL);
}

void CDeviceTreeView::DeleteChildren(HTREEITEM hTreeItem)
{
	HTREEITEM hChildItem;

	// delete each child of the tree item
	hChildItem = m_pTreeCtrl->GetNextItem(hTreeItem, TVGN_CHILD);
	while (hChildItem) {

		// if the child could not be deleted then abort or else we'll be stuck
		// in an infinite loop
		if (!m_pTreeCtrl->DeleteItem(hChildItem))
			break;

		hChildItem = m_pTreeCtrl->GetNextItem(hTreeItem, TVGN_CHILD);
	}
}

void CDeviceTreeView::PopulateLocalDevicesTree()
{
	CONTROLLER_DEVICE Controller;
	STORAGE_DEVICE StorageDevice;
	HTREEITEM hControllerItem;

	DeleteChildren(m_hLocalDevices);	// clear local devices tree

	// loop through all controllers
	if (LocalDevices.GetFirstController(&Controller)) {

		do {
			// add controller to tree control
			hControllerItem = AddLocalController(&Controller);
			if (hControllerItem != NULL) {

				// loop through the storage devices for this controller
				if (LocalDevices.GetFirstStorageDevice(Controller.dnDevInst,
					&StorageDevice)) {

					do {
						// add storage device to tree control
						AddLocalStorageDevice(&StorageDevice, hControllerItem);

					} while (LocalDevices.GetNextStorageDevice(&StorageDevice));
				}

				// expand the controller's tree
				m_pTreeCtrl->Expand(hControllerItem, TVE_EXPAND);
			}

		} while (LocalDevices.GetNextController(&Controller));
	}

	// expand the local devices tree
	m_pTreeCtrl->Expand(m_hLocalDevices, TVE_EXPAND);
}

void CDeviceTreeView::PopulateNetworkDrivesTree()
{
	NETWORK_DRIVE NetworkDrive;

	DeleteChildren(m_hNetworkDrives);	// clear network drives tree

	// loop through all possible drive letters
	for (char c='A'; c <= 'Z'; c++) {

		// check if this drive letter is mapped to a network drive
		if (NetworkDrives.GetNetworkDriveContainingDriveLetter(c,
			&NetworkDrive)) {

			// add network drive to tree control
			AddNetworkDrive(&NetworkDrive, c);
		}
	}

	// expand the network drives tree
	m_pTreeCtrl->Expand(m_hNetworkDrives, TVE_EXPAND);
}

void CDeviceTreeView::OnInitialUpdate() 
{
	int dImageIndex;

	CTreeView::OnInitialUpdate();
	
	// get the index of the local devices bitmap in the tree control's image 
	// list
	if (!m_ImageMap.Lookup(IDB_LOCAL_DEVICES, dImageIndex))
		dImageIndex = 0;

	// add the local devices location to the device tree
	m_hLocalDevices = m_pTreeCtrl->InsertItem(LOCATION_NAME_LOCAL, 
		dImageIndex, dImageIndex);

	// populate the local devices tree
	PopulateLocalDevicesTree();

	// get the index of the network drives bitmap in the tree control's image 
	// list
	if (!m_ImageMap.Lookup(IDB_NETWORK_DRIVES, dImageIndex))
		dImageIndex = 0;

	// add the network drives location to the device tree
	m_hNetworkDrives = m_pTreeCtrl->InsertItem(LOCATION_NAME_NETWORK, 
		dImageIndex, dImageIndex);

	// populate the network drives tree
	PopulateNetworkDrivesTree();
}

void CDeviceTreeView::OnRclick(NMHDR* pNMHDR, LRESULT* pResult) 
{
#if ENABLE_DEVICETREE_WB_TOGGLING

	HTREEITEM hDeviceItem;
	POINT ScreenPos, ClientPos;
	DWORD dwPos;
	UINT uiFlags;

	// get cursor location in screen coordinates
	dwPos = GetMessagePos();
	ScreenPos.x = GET_X_LPARAM(dwPos);
	ScreenPos.y = GET_Y_LPARAM(dwPos);

	// convert screen coordinates to client coordinates
	ClientPos = ScreenPos;
	m_pTreeCtrl->ScreenToClient(&ClientPos);
	
	// get handle to device tree item corresponding to the client coordinates
	hDeviceItem = m_pTreeCtrl->HitTest(ClientPos, &uiFlags);
	if (hDeviceItem) {

		// only display popup menu for storage devices
		if ((hDeviceItem != m_hLocalDevices) && 
			(hDeviceItem != m_hNetworkDrives)) {
			CMenu Menu;
			CMenu* pPopupMenu;

			m_pTreeCtrl->Select(hDeviceItem, TVGN_CARET);

			// load popup menu
			Menu.LoadMenu(IDM_DEVICE_CONTEXTMENU);
			pPopupMenu = Menu.GetSubMenu(0);

			// if network drive
			if (IsNetworkDrive(hDeviceItem)) {
				NETWORK_DRIVE NetworkDrive;
				char cDriveLetter;

				// get drive letter associated with this network drive
				cDriveLetter = (char) m_pTreeCtrl->GetItemData(hDeviceItem);

				// get info about the network drive
				if (NetworkDrives.GetNetworkDriveContainingDriveLetter(cDriveLetter,
					&NetworkDrive)) {

					// modify the menu item to display the appropriate string
					// to allow enabling or disabling of write blocking
					if (NetworkDrive.bWriteBlocked) {
						pPopupMenu->ModifyMenu(IDM_DEVICE_WRITEBLOCK, MF_BYCOMMAND |
							MF_STRING, IDM_DEVICE_WRITEBLOCK, MENU_DISABLE_WRITEBLOCKING);
					}
					else {
						pPopupMenu->ModifyMenu(IDM_DEVICE_WRITEBLOCK, MF_BYCOMMAND |
							MF_STRING, IDM_DEVICE_WRITEBLOCK, MENU_ENABLE_WRITEBLOCKING);
					}
				}
			}
			// else physical media
			else {
				DWORD dwDevInst;

				// get device instance handle for this device
				dwDevInst = m_pTreeCtrl->GetItemData(hDeviceItem);

				// if this is a controller, disable the "Enable Write Blocking" 
				// menu item
				if (LocalDevices.IsController(dwDevInst)) {
					pPopupMenu->EnableMenuItem(IDM_DEVICE_WRITEBLOCK, MF_GRAYED);
				}
				// else modify the menu item to display the appropriate string 
				// to allow enabling or disabling of write blocking
				else {
					if (LocalDevices.DeviceIsWriteBlocked(dwDevInst)) {
						pPopupMenu->ModifyMenu(IDM_DEVICE_WRITEBLOCK, MF_BYCOMMAND |
							MF_STRING, IDM_DEVICE_WRITEBLOCK, MENU_DISABLE_WRITEBLOCKING);
					}
					else {
						pPopupMenu->ModifyMenu(IDM_DEVICE_WRITEBLOCK, MF_BYCOMMAND |
							MF_STRING, IDM_DEVICE_WRITEBLOCK, MENU_ENABLE_WRITEBLOCKING);
					}
				}
			}

			// display and track popup menu
			pPopupMenu->TrackPopupMenu(TPM_LEFTALIGN, ScreenPos.x, ScreenPos.y,
				this);
		}
	}

#endif

	*pResult = 0;
}

void CDeviceTreeView::OnDeviceWriteblock() 
{
	HTREEITEM hDeviceItem;

	// get handle to selected item
	hDeviceItem = m_pTreeCtrl->GetSelectedItem();
	if (hDeviceItem) {
		MEDIA_EVENT MediaEvent;
		CWriteblockerDoc* pDoc;
		CString csText;
		int dImageIndex, dStatus;
		BOOL bIsWriteblocked;

		// if selected item is a network drive...
		if (IsNetworkDrive(hDeviceItem)) {
			NETWORK_DRIVE NetworkDrive;
			char cDriveLetter;

			// get drive letter associated with this network drive
			cDriveLetter = (char) m_pTreeCtrl->GetItemData(hDeviceItem);

			// get info this network drive
			if (NetworkDrives.GetNetworkDriveContainingDriveLetter(cDriveLetter,
				&NetworkDrive)) {

				CToggleMedia ToggleMediaDlg(NetworkDrive.szNetworkPath);

				// display dialog to toggle write block status
				dStatus = ToggleMediaDlg.DoModal();

				// if user didn't cancel request to toggle write block status, send
				// request to driver to toggle write block status
				if (dStatus == IDOK) {

					// if write block status of network drive toggled successfully, 
					// update display
					bIsWriteblocked = NetworkDrive.bWriteBlocked;
					if (NetworkDrives.SetNetworkDriveWriteBlockStatus(NetworkDrive.szNetworkPath, 
						!bIsWriteblocked)) {

						// this is only a copy of what's stored in NetworkDrives
						// so we need to update its write blocked value
						NetworkDrive.bWriteBlocked = !bIsWriteblocked;

						// update the device's bitmap to show its write block 
						// status
						dImageIndex = GetImageIndex(&NetworkDrive);
						m_pTreeCtrl->SetItemImage(hDeviceItem, dImageIndex,
							dImageIndex);

						if (NetworkDrive.bWriteBlocked)
							MediaEvent.dwEventType = EVENT_NETWORK_TOGGLE_WBON;
						else
							MediaEvent.dwEventType = EVENT_NETWORK_TOGGLE_WBOFF;
						strcpy(MediaEvent.Media.szNetworkPath, NetworkDrive.szNetworkPath);

						// get pointer to main view's document
						pDoc = (CWriteblockerDoc*) GetDocument();

						// add media event
						pDoc->AddMediaEvent(MediaEvent);
						pDoc->UpdateAllViews(this);
					}
					else {
						csText.Format("%s might not be installed or running properly.  Cannot toggle write block status.", WRITEBLOCKER_APP_NAME);
						MessageBox(csText, WRITEBLOCKER_APP_NAME, MB_OK | MB_ICONERROR);
					}
				}
				else if (dStatus == IDCANCEL) {
					MessageBox("User canceled the attempt to toggle the media.",
						WRITEBLOCKER_APP_NAME, MB_ICONERROR | MB_OK);
				}
			}
		}
		// else selected item is a local device
		else {
			DWORD dwDevInst;

			// get device instance handle for this device
			dwDevInst = m_pTreeCtrl->GetItemData(hDeviceItem);
		
			CToggleMedia ToggleMediaDlg(dwDevInst);

			// display dialog to toggle write block status
			dStatus = ToggleMediaDlg.DoModal();

			// if user didn't cancel request to toggle write block status, send
			// request to driver to toggle write block status
			if (dStatus == IDOK) {

				// if write block status of device toggled successfully, update
				// display
				bIsWriteblocked = LocalDevices.DeviceIsWriteBlocked(dwDevInst);
				if (LocalDevices.SetDeviceWriteBlockStatus(dwDevInst, 
					!bIsWriteblocked)) {
					STORAGE_DEVICE StorageDevice;

					// update the device's bitmap to show its write block status
					if (LocalDevices.GetStorageDeviceInfo(dwDevInst, 
						&StorageDevice)) {

						dImageIndex = GetImageIndex(&StorageDevice);
						m_pTreeCtrl->SetItemImage(hDeviceItem, dImageIndex,
							dImageIndex);
					}

					if (!bIsWriteblocked)
						MediaEvent.dwEventType = EVENT_PHYSICAL_TOGGLE_WBON;
					else
						MediaEvent.dwEventType = EVENT_PHYSICAL_TOGGLE_WBOFF;
					MediaEvent.Media.dnDevInst = dwDevInst;

					// get pointer to device tree view's document
					pDoc = (CWriteblockerDoc*) GetDocument();

					// add media event
					pDoc->AddMediaEvent(MediaEvent);
					pDoc->UpdateAllViews(this);
				}
				else {
					csText.Format("%s might not be installed or running properly.  Cannot toggle write block status.", WRITEBLOCKER_APP_NAME);
					MessageBox(csText, WRITEBLOCKER_APP_NAME, MB_OK | MB_ICONERROR);
				}
			}
			else if (dStatus == IDCANCEL) {
				MessageBox("User canceled the attempt to toggle the media.",
					WRITEBLOCKER_APP_NAME, MB_OK | MB_ICONERROR);
			}
		}
	}
}

void CDeviceTreeView::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint) 
{
	MEDIA_EVENT MediaEvent;

	// get pointer to the document
	CWriteblockerDoc* pDoc = (CWriteblockerDoc*) GetDocument();

	// get the last event that requested this update
	if (pDoc->GetLastMediaEvent(&MediaEvent)) {

		// if this is not a write attempt event, process it
		if (!WRITE_ATTEMPT_EVENT(MediaEvent.dwEventType)) {

			// process network drive event
			if (NETWORK_EVENT(MediaEvent.dwEventType))
				ProcessNetworkDriveEvent(&MediaEvent);
			// process physical device event
			else
				ProcessLocalDeviceEvent(&MediaEvent);
		}
	}	
}

HTREEITEM CDeviceTreeView::AddLocalController(PCONTROLLER_DEVICE pController)
{
	HTREEITEM hTreeItem, hInsertAfter;
	int dImageIndex;

	// get the index of the bitmap in the tree control's image list
	dImageIndex = GetImageIndex(pController);

	// determine where to insert the controller
	hInsertAfter = GetLocalControllerInsertionPos(pController);

	// add the controller to the device tree
	hTreeItem = m_pTreeCtrl->InsertItem(pController->csDeviceName, dImageIndex, 
		dImageIndex, m_hLocalDevices, hInsertAfter);

	// associate the controller's device instance with the new tree item -- the
	// device instance uniquely identifies a controller
	if (hTreeItem != NULL)
		m_pTreeCtrl->SetItemData(hTreeItem, pController->dnDevInst);

	return hTreeItem;
}

int CDeviceTreeView::GetImageIndex(PCONTROLLER_DEVICE pController)
{
	UINT uiBitmapID;
	int dImageIndex;

	// get the resource ID of the bitmap for this controller
	switch (pController->eType) {
	case FLOPPY_CONTROLLER :
	case IDE_CONTROLLER :
		uiBitmapID = IDB_IDE_CONTROLLER;
		break;
	case SCSI_CONTROLLER :
		uiBitmapID = IDB_SCSI_CONTROLLER;
		break;
	case USB_CONTROLLER :
		uiBitmapID = IDB_USB_CONTROLLER;
		break;
	case PCMCIA_CONTROLLER :
		uiBitmapID = IDB_PCMCIA_CONTROLLER;
		break;
	case FIREWIRE_CONTROLLER:
		uiBitmapID = IDB_FIREWIRE_CONTROLLER;
		break;
	default :
		uiBitmapID = IDB_UNKNOWN_DEVICE;
	}

	// get the index of the bitmap in the tree control's image list
	if (!m_ImageMap.Lookup(uiBitmapID, dImageIndex))
		dImageIndex = 0;

	return dImageIndex;
}

HTREEITEM CDeviceTreeView::GetLocalControllerInsertionPos(PCONTROLLER_DEVICE pController)
{
	CONTROLLER_DEVICE CurrController;
	HTREEITEM hCurrController;
	HTREEITEM hPrevController = TVI_FIRST;
	HTREEITEM hInsertAfter = TVI_LAST;
	DWORD dwDevInst;

	// loop through all controllers in the device tree
	hCurrController = m_pTreeCtrl->GetChildItem(m_hLocalDevices);
	while (hCurrController != NULL) {

		// get device instance handle for this controller
		dwDevInst = m_pTreeCtrl->GetItemData(hCurrController);

		// retrieve info about the controller
		if (LocalDevices.GetControllerInfo(dwDevInst, &CurrController)) {

			// if this is a floppy drive controller, insert it before any other
			// types of controllers but after any floppy controllers that are
			// currently present in the device tree
			if (pController->eType == FLOPPY_CONTROLLER) {
				if (CurrController.eType != FLOPPY_CONTROLLER) {
					hInsertAfter = hPrevController;
					break;
				}
			}
			// if this is an IDE controller, insert it after floppy drive
			// controllers but before any other types of controllers
			else if (pController->eType == IDE_CONTROLLER) {
				if (CurrController.eType == IDE_CONTROLLER) {

					// Insert primary IDE controllers before any other IDE
					// controllers.  Checking for the existence of "Primary"
					// should do for now.  If we ever come across machines
					// that have more than two IDE controllers, we may have
					// to store and compare the port number of the controllers.
					if (pController->csDeviceName.Find("Primary") != -1)
						hInsertAfter = hPrevController;
					else
						hInsertAfter = hCurrController;
					break;
				}
				// if we get to this point then we know this is not a floppy
				// drive or IDE controller so add it immediately before this
				// controller
				else if (CurrController.eType != FLOPPY_CONTROLLER) {
					hInsertAfter = hPrevController;
					break;
				}
			}
		}

		hPrevController = hCurrController;
		hCurrController = m_pTreeCtrl->GetNextSiblingItem(hCurrController);
	}

	return hInsertAfter;
}

HTREEITEM CDeviceTreeView::AddLocalStorageDevice(PSTORAGE_DEVICE pStorageDevice, 
												 HTREEITEM hController)
{
	HTREEITEM hTreeItem;
	int dImageIndex;

	// get the index of the bitmap for this device in the tree control's image 
	// list
	dImageIndex = GetImageIndex(pStorageDevice);

	// add the storage device to the device tree
	hTreeItem = m_pTreeCtrl->InsertItem(pStorageDevice->csDeviceName, 
		dImageIndex, dImageIndex, hController);

	// associate the storage device's device instance with the new tree 
	// item -- the device instance uniquely identifies a storage device
	if (hTreeItem != NULL)
		m_pTreeCtrl->SetItemData(hTreeItem, pStorageDevice->dnDevInst);

	return hTreeItem;
}

int CDeviceTreeView::GetImageIndex(PSTORAGE_DEVICE pStorageDevice)
{
	UINT uiBitmapID;
	int dImageIndex;

	// get the resource ID of the bitmap for this storage device
	switch (pStorageDevice->eType) {
	case FLOPPY_DISK :
		if (pStorageDevice->bWriteBlocked)
			uiBitmapID = IDB_FLOPPY_DRIVE_WRITEBLOCKED;
		else
			uiBitmapID = IDB_FLOPPY_DRIVE;
		break;
	case CDROM_DISK :
		if (pStorageDevice->bWriteBlocked)
			uiBitmapID = IDB_CDROM_DRIVE_WRITEBLOCKED;
		else
			uiBitmapID = IDB_CDROM_DRIVE;
		break;
	case IDE_DISK :
	case SCSI_DISK :
	case USB_DISK :
		if (pStorageDevice->bWriteBlocked)
			uiBitmapID = IDB_DISK_DRIVE_WRITEBLOCKED;
		else
			uiBitmapID = IDB_DISK_DRIVE;
		break;
	default :
		if (pStorageDevice->bWriteBlocked)
			uiBitmapID = IDB_UNKNOWN_DEVICE_WRITEBLOCKED;
		else
			uiBitmapID = IDB_UNKNOWN_DEVICE;
	}

	// get the index of the bitmap in the tree control's image list
	if (!m_ImageMap.Lookup(uiBitmapID, dImageIndex))
		dImageIndex = 0;

	return dImageIndex;
}

void CDeviceTreeView::RemoveLocalDevice(DEVINST dnDevInst)
{
	HTREEITEM hItem, hParentItem;

	// get handle to device's tree item
	hItem = FindLocalDevice(dnDevInst);
	if (hItem != NULL) {

		// if the device's parent tree item is a controller, check if the
		// controller has more than 1 storage device attached to it
		hParentItem = m_pTreeCtrl->GetParentItem(hItem);
		if ((hParentItem != NULL) && (hParentItem != m_hLocalDevices)) {

			// if the controller has more than 1 storage device attached to it,
			// just delete the specified storage device
			if (GetChildrenCount(hParentItem) > 1)
				m_pTreeCtrl->DeleteItem(hItem);
			// else delete the controller
			else
				m_pTreeCtrl->DeleteItem(hParentItem);
		}
		// else this is a controller so just delete it
		else
			m_pTreeCtrl->DeleteItem(hItem);
	}
}

HTREEITEM CDeviceTreeView::FindLocalDevice(DEVINST dnDevInst)
{
	HTREEITEM hController, hStorageDevice;
	DWORD dwDevInst;

	// step through each controller in the device tree
	hController = m_pTreeCtrl->GetChildItem(m_hLocalDevices);
	while (hController != NULL) {

		// get device instance handle of this controller
		dwDevInst = m_pTreeCtrl->GetItemData(hController);

		// if this is the device we're looking for, return handle to it
		if (dwDevInst == dnDevInst)
			return hController;

		// step through the controller's storage devices
		hStorageDevice = m_pTreeCtrl->GetChildItem(hController);
		while (hStorageDevice != NULL) {
			
			// get device instance handle of this storage device
			dwDevInst = m_pTreeCtrl->GetItemData(hStorageDevice);

			// if this is the device we're looking for, return handle to it
			if (dwDevInst == dnDevInst)
				return hStorageDevice;

			hStorageDevice = m_pTreeCtrl->GetNextSiblingItem(hStorageDevice);
		}

		hController = m_pTreeCtrl->GetNextSiblingItem(hController);
	}

	return NULL;
}

HTREEITEM CDeviceTreeView::AddNetworkDrive(PNETWORK_DRIVE pNetworkDrive, 
										   char cDriveLetterToDisplay)
{
	HTREEITEM hTreeItem;
	char szHost[MAX_NETWORK_PATH_LENGTH] = "";
	char szPath[MAX_NETWORK_PATH_LENGTH] = "";
	CString csDisplayText;
	char* pszNetworkPath, *pChar;
	int dImageIndex;

	// make sure network drive is mapped to specified drive letter
	if (strchr(pNetworkDrive->szDriveLetters, cDriveLetterToDisplay) == NULL)
		return NULL;

	// get the index of the network drive bitmap in the tree control's image 
	// list
	dImageIndex = GetImageIndex(pNetworkDrive);

	pszNetworkPath = pNetworkDrive->szNetworkPath + 1;

	// get name of computer that the shared directory resides on
	strcpy(szHost, pszNetworkPath);
	pChar = strchr(szHost, '\\');
	if (pChar)
		*pChar = '\0';

	// get name of shared directory
	pChar = strchr(pszNetworkPath, '\\');
	if (pChar) {
		pChar++;
		strcpy(szPath, pChar);
	}

	// create network drive string that will be displayed
	csDisplayText.Format("%s on '%s' (%c:)", szPath, szHost,
		cDriveLetterToDisplay);

	// add the network drive to the device tree
	hTreeItem = m_pTreeCtrl->InsertItem(csDisplayText, dImageIndex, 
		dImageIndex, m_hNetworkDrives);

	// associate the drive letter that is being displayed with the new tree 
	// item
	if (hTreeItem != NULL)
		m_pTreeCtrl->SetItemData(hTreeItem, cDriveLetterToDisplay);

	return hTreeItem;
}

int CDeviceTreeView::GetImageIndex(PNETWORK_DRIVE pNetworkDrive)
{
	UINT uiBitmapID;
	int dImageIndex;

	// get the resource ID of the bitmap for this network drive
	if (pNetworkDrive->bWriteBlocked)
		uiBitmapID = IDB_NETWORK_DRIVE_WRITEBLOCKED;
	else
		uiBitmapID = IDB_NETWORK_DRIVE;

	// get the index of the bitmap in the tree control's image list
	if (!m_ImageMap.Lookup(uiBitmapID, dImageIndex))
		dImageIndex = 0;

	return dImageIndex;
}

void CDeviceTreeView::RemoveNetworkDrive(char cDriveLetter)
{
	HTREEITEM hItem;

	// get handle to network drive's tree item and delete it
	hItem = FindNetworkDrive(cDriveLetter);

	if (hItem != NULL)
		m_pTreeCtrl->DeleteItem(hItem);
}

HTREEITEM CDeviceTreeView::FindNetworkDrive(char cDriveLetter)
{
	HTREEITEM hNetworkDrive;
	char cDrvLetter;

	// step through each network drive in the network drives tree
	hNetworkDrive = m_pTreeCtrl->GetChildItem(m_hNetworkDrives);
	while (hNetworkDrive != NULL) {

		// get drive letter assigned to this network drive
		cDrvLetter = (char) m_pTreeCtrl->GetItemData(hNetworkDrive);

		// if this is the network drive we're looking for, return handle to it
		if (cDrvLetter == cDriveLetter)
			return hNetworkDrive;

		hNetworkDrive = m_pTreeCtrl->GetNextSiblingItem(hNetworkDrive);
	}

	return NULL;
}

void CDeviceTreeView::ProcessLocalDeviceEvent(PMEDIA_EVENT pMediaEvent)
{
	// if physical device removal event, remove device from tree control
	if (pMediaEvent->dwEventType == EVENT_PHYSICAL_DEVICE_REMOVAL) {
		RemoveLocalDevice(pMediaEvent->Media.dnDevInst);
	}
	else {
		STORAGE_DEVICE StorageDevice;

		// get info about the device described in the event
		if (LocalDevices.GetStorageDeviceInfo(pMediaEvent->Media.dnDevInst, 
			&StorageDevice)) {

			// if device arrival event, add device to tree control
			if (pMediaEvent->dwEventType == EVENT_PHYSICAL_DEVICE_ARRIVAL) {
				CONTROLLER_DEVICE Controller;

				// get info about controller for storage device
				if (LocalDevices.GetParentController(StorageDevice.dnDevInst,
					&Controller)) {
					HTREEITEM hController;

					// find controller in tree control
					hController = FindLocalDevice(Controller.dnDevInst);
					
					// if controller isn't in tree control yet, add it
					if (hController == NULL)
						hController = AddLocalController(&Controller);

					// add storage device to tree control
					if (hController != NULL) {
						AddLocalStorageDevice(&StorageDevice, hController);

						// expand the controller's tree
						m_pTreeCtrl->Expand(hController, TVE_EXPAND);
					}
				}
			}
			// if device toggle write block status event, update the device's 
			// bitmap to show its write block status
			else if ((pMediaEvent->dwEventType == EVENT_PHYSICAL_TOGGLE_WBON) ||
				(pMediaEvent->dwEventType == EVENT_PHYSICAL_TOGGLE_WBOFF)) {
				HTREEITEM hStorageDevice;

				// get handle to storage device's tree control item
				hStorageDevice = FindLocalDevice(StorageDevice.dnDevInst);
				if (hStorageDevice != NULL) {
					int dImageIndex;

					// update the device's bitmap
					dImageIndex = GetImageIndex(&StorageDevice);
					m_pTreeCtrl->SetItemImage(hStorageDevice,
						dImageIndex, dImageIndex);
				}
			}
		}
	}
}

void CDeviceTreeView::ProcessNetworkDriveEvent(PMEDIA_EVENT pMediaEvent)
{
	// if network drive removal event, remove network drive from tree control
	if (pMediaEvent->dwEventType == EVENT_NETWORK_DRIVE_REMOVAL) {
		RemoveNetworkDrive(pMediaEvent->Media.cNetworkDrive);
	}
	else {
		NETWORK_DRIVE NetworkDrive;

		// if network drive arrival event, add network drive to tree control
		if (pMediaEvent->dwEventType == EVENT_NETWORK_DRIVE_ARRIVAL) {

			// get info about the network drive described in the event
			if (NetworkDrives.GetNetworkDriveContainingDriveLetter(pMediaEvent->Media.cNetworkDrive,
				&NetworkDrive)) {

				AddNetworkDrive(&NetworkDrive, pMediaEvent->Media.cNetworkDrive);

				// expand the network drives tree -- we need to do this because
				// there may not be any network drives mapped initially
				m_pTreeCtrl->Expand(m_hNetworkDrives, TVE_EXPAND);
			}
		}
		// else process network drive toggle write block status event
		else if ((pMediaEvent->dwEventType == EVENT_NETWORK_TOGGLE_WBON) ||
			(pMediaEvent->dwEventType == EVENT_NETWORK_TOGGLE_WBOFF)) {

			// get info about the network drive described in the event
			if (NetworkDrives.GetNetworkDriveInfo(pMediaEvent->Media.szNetworkPath,
				&NetworkDrive)) {
				HTREEITEM hNetworkDrive;
				WORD wLength;
				int dImageIndex;

				wLength = strlen(NetworkDrive.szDriveLetters);
				
				// for each drive letter mapped to this network drive, update
				// the drive's bitmap to show its write block status
				for (WORD x=0; x < wLength; x++) {
					if ((NetworkDrive.szDriveLetters[x] >= 'A') &&
						(NetworkDrive.szDriveLetters[x] <= 'Z')) {

						// get handle to the drive's tree control item
						hNetworkDrive = FindNetworkDrive(NetworkDrive.szDriveLetters[x]);
						if (hNetworkDrive != NULL) {

							// update the network drive's bitmap
							dImageIndex = GetImageIndex(&NetworkDrive);
							m_pTreeCtrl->SetItemImage(hNetworkDrive,
								dImageIndex, dImageIndex);
						}
					}
				}
			}
		}
	}
}

int CDeviceTreeView::GetChildrenCount(HTREEITEM hTreeItem)
{
	HTREEITEM hChildItem;
	int dCount = 0;

	// return count of children that the specified tree item has
	hChildItem = m_pTreeCtrl->GetChildItem(hTreeItem);
	while (hChildItem != NULL) {
		dCount++;

		hChildItem = m_pTreeCtrl->GetNextSiblingItem(hChildItem);
	}

	return dCount;
}

BOOL CDeviceTreeView::IsLocalDevice(HTREEITEM hTreeItem)
{
	// return TRUE if the device belongs to the 'Local Devices' tree
	if (hTreeItem != m_hLocalDevices) {
		HTREEITEM hRootItem;

		hRootItem = m_pTreeCtrl->GetNextItem(hTreeItem, TVGN_ROOT);
		if (hRootItem == m_hLocalDevices)
			return TRUE;
	}

	return FALSE;
}

BOOL CDeviceTreeView::IsNetworkDrive(HTREEITEM hTreeItem)
{
	// return TRUE if the device belongs to the 'Network Drives' tree
	if (hTreeItem != m_hNetworkDrives) {
		HTREEITEM hRootItem;

		hRootItem = m_pTreeCtrl->GetParentItem(hTreeItem);
		if (hRootItem == m_hNetworkDrives)
			return TRUE;
	}

	return FALSE;
}

void CDeviceTreeView::OnSetFocus(CWnd* pOldWnd) 
{
	CTreeView::OnSetFocus(pOldWnd);
	
	// get pointer to main frame and disable the toolbar button to toggle write
	// blocking
	CMainFrame* pMainFrm = (CMainFrame*) GetParentFrame();

	if (pMainFrm)
		pMainFrm->EnableToggleButton(FALSE);
}
