# define ATARAW_H_ID  "@(#) ataraw.h Linux Version 1.2 created 12/14/11 at 16:09:55"

/*
 * @(#) ataraw.h Linux Version 1.2 created 12/14/11 at 16:09:55
 *
 * The software provided here is released by the National
 * Institute of Standards and Technology (NIST), an agency of
 * the U.S. Department of Commerce, Gaithersburg MD 20899,
 * USA.  The software bears no warranty, either expressed or
 * implied. NIST does not assume legal liability nor
 * responsibility for a User's use of the software or the
 * results of such use.
 *
 * Please note that within the United States, copyright
 * protection, under Section 105 of the United States Code,
 * Title 17, is not available for any work of the United
 * States Government and/or for any works created by United
 * States Government employees. User acknowledges that this
 * software contains work which was created by NIST employees
 * and is therefore in the public domain and not subject to
 * copyright.  The User may use, distribute, or incorporate
 * this software provided the User acknowledges this via an
 * explicit acknowledgment of NIST-related contributions to
 * the User's work. User also agrees to acknowledge, via an
 * explicit acknowledgment, that any modifications or
 * alterations have been made to this software before
 * redistribution.
 * --------------------------------------------------------------------
 *
 * Revision History:
 * 2011 May Ben Livelsberger: forked file from the version in 
 *    the NPS ataraw-0.2.1 package.  Extended
 * 2009 Jan Kyle Sanders & Simson Garfinkel: Extended
 * 2008 Nov Kyle Sanders - Created
 */



/*
 * ata.h:
 * Stuff about the ATA spec.
 */

#ifndef _ATARAW_H_
#define _ATARAW_H_

#include "ataraw_config.h"

#include <sys/types.h>
#include <stdint.h>
#include <time.h>

#ifdef HAVE_LINUX_HDREG_H
#include <linux/hdreg.h>
#endif

#ifdef HAVE_SCSI_SG_H
#include <scsi/sg.h>
#endif

#define ATA_RAW_BAD_FILE_FD	 -99
#define ATA_RAW_BAD_COMMAND_CODE -98	// this command not supported
#define ATA_RAW_NO_MEM		 -97
#define ATA_RAW_IOCTL_FAIL       -96
#define ATA_RAW_BAD_INPUT        -95
#define ATA_RAW_NO_PASSTHROUGH   -94   // no way to pass raw ATA commands to kernel
#define ATA_RAW_SGIO_AUX_INFO    -93   // SG_IO structure reports auxilary information available
#define ATA_RAW_ATA_ERR_STATUS   -92   // the ERR bit in the status register is set indicating that the command completed with an error
#define ATA_RAW_ERR              -1
#define ATA_RAW_OK              0

#define SECTOR_SIZE 512
#define PIO_OUT_STATIC 1
#define PIO_OUT_DYNAMIC 2

#define ATA_RAW_SCSI_DEVICE    2
#define ATA_RAW_ATA_DEVICE     1
#define ATA_RAW_INVALID_DEVICE 0

#define TRUE 0x01
#define FALSE 0x00

/** Magic number for PIO in transfer mode. */
#define PIO_IN		0
/** Magic number for PIO out transfer mode.*/ 
#define PIO_OUT		1
/**Magic number for NON DATA transfer mode.*/
#define NON_DATA	2
/** Magic number for PIO in extended transfer mode.*/
#define PIO_IN_EXT	3
/**Magic number for PIO out extended transfer mode.*/
#define PIO_OUT_EXT	4
/**Magic number for NON DATA extended mode.*/
#define NON_DATA_EXT	5
/**Magic number for DMA transfer mode.*/
#define ATA_DMA_IN	6
/**Magic number for DMA out transfer mode. */
#define ATA_DMA_OUT     7
/**Magic number for DMA extended transfer mode.*/
#define ATA_DMA_IN_EXT	8
/**Magic number for DMA extended transfer mode.*/
#define ATA_DMA_OUT_EXT 9


typedef uint16_t word;
typedef struct security_erase_pword
{
	/**
	 * The control word specifies which password to compare.
	 * if bit 0 is equal to zero then it will compare the 
	 * user password.  If bit 0 is equal to one then the
	 * master password will be compared.  All other bits are
	 * reserved.
	 */
	word control_word;
	/**
	 * This is the password for the device.*/
	word password[16];
	/**
	 * When using security disable password this field is reserve
	 */
	word master_rev; 
	/**
	 * When using security disable password this field is reserve
	 */
	word reserved[238];
} security_password;
/** Structure that represents the Device Configuration Overlay.*/
typedef struct device_configuration_overlay_struct
{
	/**
	 * 
	 */
	word revision;
	word m_dma;
	word u_dma;
	word max_lba[4];
	word command_feature;
	word reserved[247];
	word integrity;
}dco_struct;

typedef struct device_smart_data_structure
{
    u_char vendor_specific[362];
    u_char off_line_data_status;
    u_char self_execution_status;
    u_char time_complete_collection[2]; //in seconds
    u_char vendor_specific2;
    u_char off_line_capability;
    u_char smart_capability[2];
    u_char err_log_capability;
    u_char vendor_specific3;
    u_char sself_test_poll_time; //in minutes
    u_char eself_test_poll_time; //in minutes
    u_char reservers[12];
    u_char vendor_specific4[25];
    u_char checksum;
}smart_struc;

extern int ataraw_debug;
extern char *ATARAW_C_ID;
extern char *WRAPPER_C_ID;

// Helper functions
word compute_dco_checksum( dco_struct* target );
void set_sector_number( u_char* buff, u_long sector );
void set_sector_number_ext( u_char* registers, u_char* e_registers, uint64_t sector );
void set_scsi_sector_number( u_char* cmdblk, uint64_t sector, int cmdblk_size );
u_long get_sector_number(u_char *registers);
uint64_t get_sector_number_ext(u_char *registers,u_char *e_registers);
int chk_48bit_feature_support(int fd);
int chk_device_type(int fd);
void log_close(time_t from);

//ata commands
/** Hex value for READ SECTOR(S) defined in the ATA 6 spec. Pg 199 */
#define READ_SECTORS 0x20
int ata_read_sector( int fd, u_long sector_number, u_char number, u_char* buffer );
#define READ_SECTORS_NO_RETRY 0x21
int ata_read_sector_no_retry( int fd, u_long sector_number, u_char number, u_char* buffer );
#define READ_LONG 0x22
int ata_read_long( int fd, u_long sector_number, u_char* buffer );
#define READ_LONG_NO_RETRY 0x23
int ata_read_long_no_retry( int fd, u_long sector_number, u_char* buffer );
/** Hex value for IDENTIFY DEVICE defined in the ATA 6 spec. Pg 113 */
#define IDENTIFY_DEVICE 0xec
int ata_identify_device( int fd, u_char* buffer );
/** Hex value for DEVICE CONFIGURATION IDENTIFY defined in the ATA 6 spec. Pg 93 */
#define DEVICE_CONFIGURATION_IDENTIFY 0xc2
int ata_device_configuration_identify( int fd, u_char* buffer );
/** Hex value for CFA TRANSLATE SECTOR defined in the ATA 6 spec. Pg 80 */
#define CFA_TRANSLATE_SECTOR 0x87
int ata_cfa_translate_sector( int fd, u_long sector, u_char* buffer );
#define IDENTIFY_PACKET_DEVICE 0xa1
int ata_identify_packet_device( int fd, u_char* buffer );
/** Hex value for READ BUFFER defined in the ATA 6 spec. Pg 165 */
#define READ_BUFFER 0xe4
int ata_read_buffer( int fd, u_char buffer[SECTOR_SIZE] );
/** Hex value for READ MULTIPLE defined in the ATA 6 spec.  Pg. 188 */
#define READ_MULTIPLE 0xc4
int ata_read_multiple( int fd, u_long sector, u_char count, u_char* buffer );
/** Hex Value for SMART READ DATA defined in the ATA 6 spec.  Pg. 257 */
#define SMART_READ_DATA 0xd0
int ata_smart_read_data( int fd, smart_struc* buff );
#define SMART_READ_LOG 0xd5
int ata_smart_read_log( int fd, u_char count, u_char addr, u_char* buffer );
//
//pio_out_cmds
//
/**Hex value for WRITE SECTOR defined in the ATA 6 spec.  Pg. 303 */
#define WRITE_SECTORS 0x30
int ata_write_sector( int fd, u_long sector, u_char count, u_char* buffer );
#define WRITE_SECTORS_NO_RETRY 0x31
int ata_write_sector_no_retry( int fd, u_long sector, u_char count, u_char* buffer );
/**Hex value for WRITE LONG defined in the ATA 3 spec.  Pg. 101 */
#define WRITE_LONG 0x32
int ata_write_long( int fd, u_long sector, u_char* buffer );
#define WRITE_MULTIPLE 0xC5
int ata_write_multiple( int fd, u_long sector, u_char count, u_char* buffer );
#define CFA_WRITE_MULTIPLE_WITHOUT_ERASE 0xcd
int ata_cfa_write_mult_wo_erase( int fd, u_long sector, u_char sector_count, u_char* buffer );
#define CFA_WRITE_SECTORS_WITHOUT_ERASE 0x38
int ata_cfa_write_sector_wo_erase( int fd, u_long sector, u_char sector_count, u_char* buffer );
#define DEVICE_CONFIGURATION_SET 0xc3
int ata_dco_set( int fd, dco_struct* overlay );
/**Hex value for SECURITY DISABLE PASSWORD defined in the ATA 6 spec.  Pg. 209*/
#define SECURITY_DISABLE_PASSWORD_ATA 0xf6
int ata_security_disable_password( int fd, u_char password[32], u_char password_select );
/**Hex value for SECURITY ERASE UNIT defined in the ATA 6 spec.  Pg. 212*/
#define SECURITY_ERASE_UNIT_ATA 0xf4
int ata_security_erase_unit( int fd, u_char password[32], u_char control_info );
/**hex value for SECURITY SET PASSWORD defined in the ATA 6 spec.  Pg. 216*/
#define SECURITY_SET_PASSWORD_ATA 0xf1
int ata_security_set_password( int fd, u_char password[32], u_char password_select, word revision_code );
/**hex value for SECURITY UNLOCK defined in the ATA 6 spec.  Pg. 219*/
#define SECURITY_UNLOCK_ATA 0xf2
int ata_security_unlock( int fd, u_char password[32], u_char password_select );
#define WRITE_VERIFY 0x3C
int ata_write_verify( int fd, u_long sector, u_char count, u_char* buffer );
//
//non-data cmds
//
#define READ_NATIVE_MAX_ADDRESS 0xf8
int ata_read_native_max_address( int fd, u_long *address );
#define SET_MAX_ADDRESS 0xf9
int ata_set_max_address( int fd, int new_max_addr, u_char volatility );
#define SEEK 0x70
int ata_seek( int fd, u_long sector_to_seek );
#define CFA_ERASE_SECTOR 0xc0
int ata_cfa_erase_sector( int fd, u_long sector_to_erase, u_char count );
#define CFA_REQUEST_EXTENDED_ERROR_CODE 0x03
int ata_cfa_extended_err( int fd, int* err_code );
#define CHECK_MEDIA_CARD_TYPE 0xd1
int ata_check_media_card_type( int fd, u_char enable, u_char* card_type );
#define CHECK_POWER_MODE 0xe5
int ata_check_power_mode( int fd, u_char* power_mode );
#define DEVICE_CONFIGURATION_RESTORE 0xc0
int ata_dco_restore( int fd );
#define DEVICE_CONFIGURATION_FREEZE_LOCK 0xc1
int ata_dco_freeze_lock( int fd );
#define FLUSH_CACHE 0xe7
int ata_flush_cache( int fd );
#define GET_MEDIA_STATUS 0xda
int ata_get_media_status( int fd, u_char* status_register );
#define IDLE 0xe3
int ata_idle( int fd );
#define IDLE_IMMEDIATE 0xe1
int ata_idle_immediate( int fd );
#define MEDIA_EJECT 0xed
int ata_media_eject( int fd );
#define MEDIA_LOCK 0xde
int ata_media_lock( int fd );
#define MEDIA_UNLOCK 0xdf
int ata_media_unlock( int fd );
#define NOP 0x00
int ata_nop( int fd );
#define SLEEP 0xe6
int ata_sleep( int fd );
#define STANDBY 0xe2
int ata_standby( int fd );
#define STANDBY_IMMEDIATE 0xe0
int ata_standby_immediate( int fd );
/**hex value for SECURITY FREEZE LOCK defined in the ATA 6 spec.  Pg. 215 */
#define SECURITY_FREEZE_LOCK_ATA 0xf5
int ata_security_freeze_lock( int fd );
/** hex value for SECURITY ERASE PREAPRE defined in the ATA 6 spec.  Pg. 211 */
#define SECURITY_ERASE_PREPARE_ATA 0xf3
int ata_security_erase_prepare( int fd );
/** Hex value for READ VERIFY SECTOR defined in the ATA 6 spec.  Pg. 204 */
#define READ_VERIFY_SECTOR 0x40
int ata_read_verify_sector( int fd, u_long sector, u_char count );
/** Hex value for Smart Disable Operations defined in the ATA 6 spec.  Pg. 248 */
#define SMART_DISABLE_OPERATIONS 0xd9
int ata_smart_disable_operations( int fd );
/** Hex value for Smart Disable Operations defined in the ATA 6 spec.  Pg. 248 */
#define SMART_ENABLE_DISABLE_ATTRIBUTE_AUTOSAVE 0xd2
int ata_smart_attribute_autosave( int fd, int enable );
/** Hex value for Smart Enable Operations defined in the ATA 6 spec.  Pg. 248 */
#define SMART_ENABLE_OPERATIONS 0xd8
int ata_smart_enable_operations( int fd );
/** Hex value for Smart Execute Off-Line Immediate defined in the ATA 6 spec.  Pg. 253 */
#define SMART_EXECUTE_OFF_LINE_IMMEDIATE 0xd4
/**
 * Sub command code for SMART_EXECUTE_OFFLINE_IMMEDIATE.  This command code will perform
 * a SMART off-line routine immediately in off-line mode.
 */
#define EXEC_SMART_OFFLINE_ROUTINE 0x00
/**
 * Sub command code for SMART_EXECUTE_OFFLINE_IMMEDIATE.  This command code will perform 
 * a SMART short self-test routine immediately in off-line mode.
 */
#define EXEC_SMART_SHORT_SELF_TEST_OFFLINE 0x01
/**
 * Sub command code for SMART_EXECUTE_OFFLINE_IMMEDIATE.  This command code will perform 
 * a SMART extended self-test routine immediately in off-line mode.
 */
#define EXEC_SMART_EXTENDED_SELF_TEST_OFFLINE 0x02
/**
 * Sub command code for SMART_EXECUTE_OFFLINE_IMMEDIATE.  This command code will abort
 * off-line mode self-test routine.
 */
#define ABORT_OFFLINE_SELF_TEST 0x7f
/**
 * Sub command code for SMART_EXECUTE_OFFLINE_IMMEDIATE.  This command code will execute
 * a SMART short self-test routine immediately in captive mode.
 */
#define EXEC_SMART_SHORT_TEST_CAPTIVE 0x81
/**
 * Sub command code for SMART_EXECUTE_OFFLINE_IMMEDIATE.  This command code will execute
 * a SMART extended self-test routine immediately in captive mode.
 */
#define EXEC_SMART_EXTENDED_SELF_TEST_CAPTIVE 0x82
int ata_smart_execute_off_line_immediate( int fd, u_char subcommand );

#define DEVICE_RESET 0x08
int ata_device_reset( int fd );

//pio_in_ext
#define READ_SECTOR_EXT 0x24
int ata_read_sector_ext( int fd, uint64_t sector_to_read, u_char* buffer );
/** Hex value for READ MULTIPLE EXT defined in the ATA 6 spec.  Pg. 191 */
#define READ_MULTIPLE_EXT 0x29
int ata_read_multiple_ext( int fd, uint64_t starting_sector, u_short count, u_char* buffer );
#define READ_STREAM_EXT 0x2b
int ata_read_stream_ext( int fd, uint64_t sector_to_read, u_char* buffer );

//pio_out_ext
/**Hex value for WRITE SECTOR EXT defined in the ATA 6 spec.  Pg. 305 */
#define WRITE_SECTOR_EXT 0x34
int ata_write_sector_ext( int fd, uint64_t sector, u_short count, u_char* buffer );
#define WRITE_STREAM_EXT 0x3B
int ata_write_stream_ext( int fd, uint64_t sector, u_short count, u_char* buffer );
#define WRITE_MULTIPLE_EXT 0x39
int ata_write_multiple_ext( int fd, uint64_t sector, u_short count, u_char* buffer );
#define WRITE_MULTIPLE_FUA_EXT 0xCE
int ata_write_multiple_fua_ext( int fd, uint64_t sector, u_short count, u_char* buffer );

//non-data_ext
#define READ_NATIVE_MAX_ADDRESS_EXT 0x27
int ata_read_native_max_ext( int fd, uint64_t *address );
#define FLUSH_CACHE_EXT 0xea
int ata_flush_cache_ext( int fd );
/**Hex value for Read Verify Sector Ext defined in the ATA 6 spec.  Pg. 208 */
#define READ_VERIFY_SECTOR_EXT 0x42
int ata_read_verify_sector_ext( int fd, u_short count, uint64_t sector );
/**Hex value for Set Max Address Ext defined in the ATA 6 Spec.  Pg. 240 */
#define SET_MAX_ADDRESS_EXT 0x37
int ata_set_max_address_ext( int fd, uint64_t address, u_char volatility );

//DMA Commands
#define READ_DMA 0xc8
int ata_read_dma( int fd, u_long address, u_char* buffer );
#define READ_DMA_NO_RETRY 0xc9
int ata_read_dma_no_retry( int fd, u_long address, u_char* buffer );
/**Hex value for Read DMA Ext defined in the ATA 6 spec.  Pg. 168 */
#define READ_DMA_EXT 0x25
int ata_read_dma_ext( int fd, uint64_t address, u_char* buffer );
#define READ_DMA_QUEUED 0xc7
int ata_read_dma_queued( int fd, u_long address, u_char* buffer );
#define READ_DMA_QUEUED_EXT 0x26
int ata_read_dma_queued_ext( int fd, uint64_t address, u_char* buffer );
#define READ_FPDMA_QUEUED 0x60
int ata_read_fpdma_queued( int fd, uint64_t address, u_char* buffer );
#define READ_STREAM_DMA_EXT 0x2A
int ata_read_stream_dma_ext( int fd, uint64_t address, u_char* buffer );

#define WRITE_DMA 0xca
int ata_write_dma( int fd, u_long sector, u_char count, u_char* buffer );
#define WRITE_DMA_NO_RETRY 0xcb
int ata_write_dma_no_retry(int fd,u_long sector,u_char count,u_char* buffer );
#define WRITE_DMA_EXT 0x35
int ata_write_dma_ext( int fd, uint64_t sector, u_short count, u_char* buffer );
#define WRITE_DMA_FUA_EXT 0x3D
int ata_write_dma_fua_ext( int fd, uint64_t sector, u_short count, u_char* buffer );
#define WRITE_DMA_QUEUED 0xcc
int ata_write_dma_queued( int fd, u_long sector, u_char* buffer );
#define WRITE_DMA_QUEUED_EXT 0x36
int ata_write_dma_queued_ext( int fd, uint64_t sector, u_char* buffer );
#define WRITE_FPDMA_QUEUED 0x61
int ata_write_fpdma_queued( int fd, uint64_t sector, u_char* buffer );
#define WRITE_DMA_QUEUED_FUA_EXT 0x3e
int ata_write_dma_queued_fua_ext( int fd, uint64_t address, u_char* buffer );
#define WRITE_STREAM_DMA_EXT 0x3a
int ata_write_stream_dma_ext( int fd, uint64_t sector, u_short count, u_char* buffer );

//special case for DCO
#define DEVICE_CONFIGURATION 0xB1
//special case for SMART
#define SMART 0xb0
//atapi commands

/** Hex value for DEVICE RESET defined in the ATA 6 spec. Pg 101 */
#define DEVICE_RESET 0x08
int atapi_device_reset( int fd );

int ata_dco_set_max_lba( int fd, uint64_t sector );

//scsi commands (see T10 SCSI Block Commands - 2, sbc-2).
#define READ_6 0x08
int scsi_read_6( int fd, uint64_t sector_number, u_char* buffer );
#define READ_10 0x28
int scsi_read_10( int fd, uint64_t sector_number, u_char* buffer );
#define READ_12 0xA8
int scsi_read_12( int fd, uint64_t sector_number, u_char* buffer );
#define READ_16 0x88
int scsi_read_16( int fd, uint64_t sector_number, u_char* buffer );
#define READ_32 0x7F
#define READ_32_SA 0x09
int scsi_read_32( int fd, uint64_t sector_number, u_char* buffer );

#define WRITE_6 0x0A
int scsi_write_6( int fd, uint64_t sector_number, u_char* buffer );
#define WRITE_10 0x2A
int scsi_write_10( int fd, uint64_t sector_number, u_char* buffer );
#define WRITE_12 0xAA
int scsi_write_12( int fd, uint64_t sector_number, u_char* buffer );
#define WRITE_16 0x8A
int scsi_write_16( int fd, uint64_t sector_number, u_char* buffer );
#define WRITE_32 0x7F
#define WRITE_32_SA 0x0B
int scsi_write_32( int fd, uint64_t sector_number, u_char* buffer );

#define INQUIRY 0x12
int scsi_inquiry( int fd, u_char* buffer);

//main api for ataraw
 int ata_raw( const char*  ataDevice, u_char* registers, u_char* extended_registers, u_char* buffer );
 int ata_raw_fd( const int fd, unsigned char* registers, unsigned char* extended_registers, unsigned char* buffer );
 int ata_pio( const int fd, unsigned char* registers, unsigned char* buffer, const int xfer_mode,
	      unsigned char* extended_registers );
 int ata_dma( const int fd, unsigned char* registers, unsigned char* buffer, const int xfer_mode,
	      unsigned char* extended_registers );
 int scsi_raw_fd( const int fd, unsigned char* cmdblk, int cmdblk_size, unsigned char* buffer, int xfer_len );
 void get_supported_commands( u_char sup[256] );
 //end main api

void ata_print_sector(unsigned char *buffer);
void ata_print_long_sector(unsigned char *buffer);
void ata_print_xxd(unsigned char *buffer, int bytes);

#endif
