static char *SCCS_ID[] = {"@(#) try_read.c Linux Version 1.5 created 02/17/12 at 13:45:57",__DATE__,__TIME__};
static char *test_version = "*** DEV VERSION ";

/*
 *
 * The software provided here is released by the National
 * Institute of Standards and Technology (NIST), an agency of
 * the U.S. Department of Commerce, Gaithersburg MD 20899,
 * USA.  The software bears no warranty, either expressed or
 * implied. NIST does not assume legal liability nor
 * responsibility for a User's use of the software or the
 * results of such use.
 *
 * Please note that within the United States, copyright
 * protection, under Section 105 of the United States Code,
 * Title 17, is not available for any work of the United
 * States Government and/or for any works created by United
 * States Government employees. User acknowledges that this
 * software contains work which was created by NIST employees
 * and is therefore in the public domain and not subject to
 * copyright.  The User may use, distribute, or incorporate
 * this software provided the User acknowledges this via an
 * explicit acknowledgment of NIST-related contributions to
 * the User's work. User also agrees to acknowledge, via an
 * explicit acknowledgment, that any modifications or
 * alterations have been made to this software before
 * redistribution.
 * --------------------------------------------------------------------
 *
 * Revision History:
 * 2011 May Ben Livelsberger - Created
 *
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <ctype.h>
#include "ataraw.h"
#ifdef HAVE_ERR_H
#include <err.h>
#endif

#ifndef HAVE_ERR
#include <stdarg.h>
void err(int eval,const char *fmt,...)
{
  va_list ap;
  va_start(ap,fmt);
  vfprintf(stderr,fmt,ap);
  va_end(ap);
  fprintf(stderr,": %s\n",strerror(errno));
  exit(eval);
}
#endif

const char *progname="main"; 

void usage()
{
  fprintf(stderr,"Usage: %s [-options] <device>     send all SCSI or ATA read\n",progname);
  fprintf(stderr,"\t\tcommands to device. try_read will try to guess whether\n\t\tdevice is SCSI or ATA.\n");
  fprintf(stderr,"Options:\n -a\tforce ATA read commands to be sent\n");
  fprintf(stderr," -s\tforce SCSI read commands to be sent\n");
  fprintf(stderr," -d\tprint extra debugging information\n\n");
  exit(1);
}

/**
 * print the buffer using ata_print_sector if debugging is enabled.
 *
 */
void print_buffer(const char* text, u_char* buffer){
  if (ataraw_debug){
    printf("%s:\n",text);
    ata_print_sector(buffer);
  }
}

/**
 * print the long (1024 bytes vs 512) buffer using ata_print_long_sector 
 * if debugging is enabled.
 *
 */
void print_long_buffer(const char* text, u_char* buffer){
  if (ataraw_debug){
    printf("%s:\n",text);
    ata_print_long_sector(buffer);
  }
}

/**
 * With the way that we issue some commands (e.g., the READ QUEUED commands...
 * not following up QUEUED commands w/ the SERVICE command), some subsequent 
 * commands can be subject to terminating w/ errors when they would have 
 * otherwise completed w/o error.  We cycle (close and then reopen) the file 
 * descriptor as a way of trying to get around this.
 * 
 */
void cycle_fd(int fd, const char* device){
  close(fd);
  fd = open(device,O_RDONLY);
  if(fd<0){
    perror(device);
    exit(1);
  }  
}

/**
 * validate_read does the meat of the work.  Read the same sector using the 
 * ata READ DMA or READ DMA EXT, or the scsi READ 16 as appropriate and then
 * measure whether the original ATA read command executed was successful.  
 * Then check if the sector read has been changed.
 *
 */
int validate_read (int fd, const char* cmd_name, u_char* buffer, int result, uint64_t sector_number, int* unchanged, int device_type, int ext_support){
  int v_result, lba=0, success = 0;
  u_char v_buffer[SECTOR_SIZE];
  memset( &v_buffer, 0, sizeof(v_buffer) );

  if ( device_type == ATA_RAW_SCSI_DEVICE ) { // for scsi devices issue READ 16
    print_buffer("READ 10 verify buffer (BEFORE)", v_buffer);
    //v_result = scsi_read_16(fd, sector_number, v_buffer);
    v_result = scsi_read_10(fd, sector_number, v_buffer);
    print_buffer("READ 10 verify buffer (AFTER)", v_buffer);
    /* if our READ 16 failed then exit. */
    if( v_result != ATA_RAW_OK ){
      err(1,"READ 10 verification read (%s) FAILED, RETURN CODE: %d\n", cmd_name, v_result );
    }
  } else { // ATA device
    if ( ext_support == 0 ) { /* Does the drive support the 48-bit 
    feature set? use READ DMA EXT or READ SECTOR(S) accordingly, to issue 
    a verifying read to the same lba. */
      print_buffer("READ DMA EXT verify buffer (BEFORE)", v_buffer);
      v_result = ata_read_dma_ext(fd, sector_number, v_buffer);
      print_buffer("READ DMA EXT verify buffer (AFTER)", v_buffer);
      /* if our READ DMA EXT failed then try again using READ SECTORS EXT. */
      if( v_result != ATA_RAW_OK ){
	// make sure that we're working w/ a clean buffer
	memset( &v_buffer, 0, sizeof(v_buffer) );
	print_buffer("READ SECTORS EXT verify buffer (BEFORE)", v_buffer);
	v_result = ata_read_sector_ext(fd, sector_number, v_buffer);
	print_buffer("READ SECTORS EXT verify buffer (BEFORE)", v_buffer);
	/* if our READ SECTORS EXT failed then exit. */
	if( v_result != ATA_RAW_OK ){
	  err(1,"READ SECTORS EXT verification read (%s) FAILED, RETURN CODE: %d\n", cmd_name, v_result );
	}
      }
    } else {
      print_buffer("READ DMA verify buffer (BEFORE)", v_buffer);
      v_result = ata_read_dma(fd,(u_long) sector_number, v_buffer);
      print_buffer("READ DMA verify buffer (AFTER)", v_buffer);

      /* if our READ DMA failed then try the read again using READ SECTORS. */
      if( v_result != ATA_RAW_OK ){
	// make sure that we're working w/ a clean buffer
	memset( &v_buffer, 0, sizeof(v_buffer) );
	print_buffer("READ SECTORS verify buffer (BEFORE)", v_buffer);
	v_result = ata_read_sector(fd,(u_long) sector_number, 1, v_buffer);
	print_buffer("READ SECTORS verify buffer (BEFORE)", v_buffer);
	/* if our READ SECTORS failed then exit. */
	if( v_result != ATA_RAW_OK ){
	  err(1,"READ SECTORS verification read (%s) FAILED, RETURN CODE: %d\n", cmd_name, v_result );
	}
      }
    }
  }

  /* Check if the ATA read command we're verifying executed sucessfully.  
     It was successful if it completed with a good status return code
     and if what it read back matches what was read back by 
     READ DMA EXT. */
  //if ((result == ATA_RAW_OK) && (! memcmp(buffer, v_buffer, SECTOR_SIZE))){
  if (! memcmp(buffer, v_buffer, SECTOR_SIZE)){
    printf("%s\tSUCCESS\treturn code: %d  \t",cmd_name,result);
    success++;
  } else {
    printf("%s\tFAIL\treturn code: %d  \t",cmd_name,result);
  }

  /* Look at the sector contents (from the READ DMA EXT read). 
     If it was initialized correctly and hasn't been written to since, 
     then it should contain the FS-TST diskwipe wipe pattern. Extract
     the lba and compare to the sector address read. */
  //ata_print_sector(v_buffer);
  lba = atoi(strndup((char *) &v_buffer[13], 12));
  //printf("<unchanged: %d>",*unchanged);
  if (lba == (int) sector_number) {
    printf("unchanged");
    if(ataraw_debug){ 
      printf(" (%d v. %d)",(int) sector_number,lba);
    }
    printf("\n");
    *unchanged += 1;
  } else {
    printf("changed");
    if(ataraw_debug){ 
      printf(" (%d v. %d)",(int) sector_number,lba);
    }
    printf("\n");
  }

/*   if(ataraw_debug){ */
/*     printf("buffer contents from the READ cmd under test:\n"); */
/*     ata_print_sector(buffer); */
/*     printf("buffer contents from the reading same sector with the READ DMA EXT cmd:\n"); */
/*     ata_print_sector(v_buffer); */
/*   } */
  return success;
}


int main(int argc,char **argv)
{
  int opt, i;
  int force_ata = 0;
  int force_scsi = 0;
  int run_ata = 0;
  int run_scsi = 0;
  static time_t from;

  if (SCCS_ID[0][0] == '%') SCCS_ID[0] = test_version;
  progname = argv[0];
  time(&from);

  printf ("%s %s%s\n",progname,ctime(&from),SCCS_ID[0]);
  //printf ("try_read %s\n",SCCS_ID[0]);
  printf ("compiled %s %s with gcc Version %s\n",__DATE__,
	  __TIME__,__VERSION__);
  printf ("%s%s%s\n",WRAPPER_C_ID,ATARAW_C_ID,ATARAW_H_ID); /* log support lib and header file version */

  /* print the command line */
  printf("cmd: ");
  for(i=0; i<argc; i++) {
    printf(" %s", argv[i]);
  }
  printf("\n");

  while((opt=getopt(argc,argv,"das"))!=-1){
    switch(opt){
      // -d option for debug mode  
    case 'd':
      ataraw_debug = 1;
      break;
    case 'a':
      force_ata = 1;
      break;
    case 's':
      force_scsi = 1;
      break;
    default:
      usage();
    }
  }

  argc -= optind;
  argv += optind;

  if(argc<1) usage();

  const char *device = argv[0];

  argv++;
  argc--;

  int fd = open(device,O_RDONLY);
  if(fd<0){
    perror(device);
    exit(1);
  }

  int read_cnt = 0;
  int success_cnt = 0;
  int unchanged_cnt = 0;
  int ext_cmd_support = 1; //default to "unsupported"
  int result;
  uint64_t sector_number;
  unsigned char buffer[SECTOR_SIZE];
  unsigned char long_buffer[2*SECTOR_SIZE];

  /* Does fd refer to an ATA device or to a SCSI device? */
  int device_type = chk_device_type( fd );
  if ( device_type == ATA_RAW_ATA_DEVICE ){
    /* extended (48-bit) commands supported? */
    ext_cmd_support = chk_48bit_feature_support(fd);
    if( ext_cmd_support == ATA_RAW_ERR )
      err(1,"chk_48bit_feature_support() FAILED, RETURN CODE: %d\n",ext_cmd_support );
    printf("%s device type:\tATA\n", device);
    if( ext_cmd_support == 0 )
      printf("48-bit Address Feature Set:     SUPPORTED\n"); 
    else
      printf("48-bit Address Feature Set:     NOT SUPPORTED\n"); 
  } else if ( device_type == ATA_RAW_SCSI_DEVICE ){
    printf("%s device type:\tSCSI\n", device);
  } else {
      err(1,"%s:\tinvalid device -- SCSI, ATA command sets not supported\n", device );
  }
  
  /*   /\* the -a and -s options should be mutually exclusive. enforce this... *\/ */
  /*   if ( force_ata && force_scsi ){ */
  /*     fprintf(stderr,"error: both '-a' and '-s' selected. Please select one or the other, but not both\n"); */
  /*     usage(); */
  /*   } */ 
  
  
  /* select which read commands (ATA and/or SCSI) will be executed */
  if ( force_ata ){
    printf("*** forcing ata... ***\n\n");
    run_ata = 1;
  }
  if ( force_scsi ){
    printf("*** forcing scsi... ***\n\n");
    run_scsi = 1;
  }
  if ( !( force_ata || force_scsi ) ){ // if we're not forcing ata or forcing scsi, then let device type determine which reads get executed
    if ( device_type == ATA_RAW_ATA_DEVICE )
      run_ata = 1;    
    else
      run_scsi = 1;
  }

  /* print column headers */
  printf("Opcode\tCommand Name\t\tStatus\tReturn Code\t\tSector Fill\n");    

  if ( run_ata ){ 
    /* if ATA, Goal: issue each defined ATA read command */

    /**
     * 28-bit commands 
     */

    /* execute READ DMA - read sector 0x0000 C800 */
    sector_number = 51200;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("READ DMA cmd buffer (BEFORE)", buffer);
    result = ata_read_dma(fd, (u_long) sector_number, buffer);
    print_buffer("READ DMA cmd buffer (AFTER)", buffer);
    read_cnt++;
    /* call validate_read- validate the read results and check the
       sector contents for change. */
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"C8h\tREAD DMA\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

    /* execute READ LONG - read sector 0x0000 2200 */
    cycle_fd(fd, device);
    sector_number = 8704;
    memset( &long_buffer, 0, sizeof(long_buffer) );
    print_long_buffer("READ LONG cmd buffer (BEFORE)", long_buffer);
    result = ata_read_long(fd, (u_long) sector_number, long_buffer);
    print_long_buffer("READ LONG cmd buffer (AFTER)", long_buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"22h\tREAD LONG\t",(u_char *) long_buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

    /* Scenario: we ran into a problem, where we'd execute the READ DMA QUEUED
       command and the subsequent verifying READ DMA EXT command would fail.
       The sense data that would come back for the READ DMA EXT would be for
       the previous READ DMA QUEUED.  To get around this problem we cycle the
       file descriptor.  A more elegant solution would be to execute the READ
       DMA QUEUED command properly (in conjunction w/ the SERVICE command, I
       think).  Come back to w/ time. */
    /* execute READ DMA QUEUED - read sector 0x0000 C700 */
    cycle_fd(fd, device);
    sector_number = 50944;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("READ DMA QUEUED cmd buffer (BEFORE)", buffer);
    result = ata_read_dma_queued(fd, (u_long) sector_number, (u_char *) buffer);
    print_buffer("READ DMA QUEUED cmd buffer (AFTER)", buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"C7h\tREAD DMA QUEUED\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

    /* execute READ SECTORS - read sector 0x0000 2000 */
    cycle_fd(fd, device);
    sector_number = 8192;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("READ SECTOR cmd buffer (BEFORE)", buffer);
    result = ata_read_sector(fd,(u_long) sector_number, 1, buffer);
    print_buffer("READ SECTOR cmd buffer (AFTER)", buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"20h\tREAD SECTOR\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

    /* execute READ MULTIPLE - read sector 0x0000 C400 */
    cycle_fd(fd, device);
    sector_number = 50176;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("READ MULTIPLE cmd buffer (BEFORE)", buffer);
    result = ata_read_multiple(fd,(u_long) sector_number, 1, buffer);
    print_buffer("READ MULTIPLE cmd buffer (AFTER)", buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"C4h\tREAD MULTIPLE\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

    /*
     * ATA-3 commands 28-bit commands
     */
    /* execute READ DMA w/o retries - read sector 0x0000 C900 */
    sector_number = 51456;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("READ DMA w/o retries cmd buffer (BEFORE)", buffer);
    cycle_fd(fd, device);
    result = ata_read_dma_no_retry(fd, (u_long) sector_number, buffer);
    print_buffer("READ DMA w/o retries cmd buffer (AFTER)", buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"C9h\tREAD DMA w/o retries",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

    /* execute READ LONG w/o retries - read sector 0x0000 2300 */
    cycle_fd(fd, device);
    sector_number = 8960;
    memset( &long_buffer, 0, sizeof(long_buffer) );
    print_buffer("READ LONG w/o retries cmd buffer (BEFORE)", long_buffer);
    result = ata_read_long_no_retry(fd, (u_long) sector_number, long_buffer);
    print_buffer("READ LONG w/o retries cmd buffer (AFTER)", long_buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"23h\tREAD LONG w/o retries",(u_char *) long_buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

    /* execute READ SECTORS w/o retries - read sector 0x0000 2100 */
    cycle_fd(fd, device);
    sector_number = 8448;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("READ SECTOR w/o retries cmd buffer (BEFORE)", buffer);
    result = ata_read_sector_no_retry(fd,(u_long) sector_number, 1, buffer);
    print_buffer("READ SECTOR w/o retries cmd buffer (AFTER)", buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"21h\tREAD SECTOR w/o retries",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

    /**
     * 48-bit (EXT) commands (can only verify if drive supports EXT cmds)
     */
    if (!ext_cmd_support){
      /* execute READ DMA EXT - read sector 0x1025 0000 */
      cycle_fd(fd, device);
      sector_number = 270860288;
      memset( &buffer, 0, sizeof(buffer) );
      print_buffer("READ DMA EXT cmd buffer (BEFORE)", buffer);
      result = ata_read_dma_ext(fd, sector_number, buffer);
      print_buffer("READ DMA EXT cmd buffer (AFTER)", buffer);
      read_cnt++;
      cycle_fd(fd, device);
      success_cnt += validate_read(fd,"25h\tREAD DMA EXT\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

      /* execute READ DMA QUEUED EXT - read sector 0x1026 0000 */
      cycle_fd(fd, device);
      sector_number = 270925824;
      memset( &buffer, 0, sizeof(buffer) );
      print_buffer("READ DMA QUEUED EXT cmd buffer (BEFORE)", buffer);
      result = ata_read_dma_queued_ext(fd, sector_number, buffer);
      print_buffer("READ DMA QUEUED EXT cmd buffer (AFTER)", buffer);
      read_cnt++;
      cycle_fd(fd, device);
      success_cnt += validate_read(fd,"26h\tREAD DMA QUEUED EXT",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

      /* when testing the eSATA interface of the Forensic UDv4, the WRITE FPDMA QUEUED
       * command was breaking the connection to the drive.  Just to be safe, commented
       * out READ FPDMA QUEUED as well */
      /* /\* execute READ FPDMA QUEUED - read sector 0x1060 0000 *\/ */
      /* cycle_fd(fd, device); */
      /* sector_number = 274726912; */
      /* memset( &buffer, 0, sizeof(buffer) ); */
      /* print_buffer("READ FPDMA QUEUED cmd buffer (BEFORE)", buffer); */
      /* result = ata_read_fpdma_queued(fd, sector_number, buffer); */
      /* print_buffer("READ FPDMA QUEUED cmd buffer (AFTER)", buffer); */
      /* read_cnt++; */
      /* cycle_fd(fd, device); */
      /* success_cnt += validate_read(fd,"60h\tREAD FPDMA QUEUED",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support); */

      /* execute READ MULTIPLE EXT - read sector 0x1029 0000 */
      cycle_fd(fd, device);
      sector_number = 271122432;
      memset( &buffer, 0, sizeof(buffer) );
      print_buffer("READ MULTIPLE EXT cmd buffer (BEFORE)", buffer);
      result = ata_read_multiple_ext(fd,sector_number,(u_short) 1, buffer);
      print_buffer("READ MULTIPLE EXT cmd buffer (AFTER)", buffer);
      //ata_print_sector(buffer);
      read_cnt++;
      cycle_fd(fd, device);
      success_cnt += validate_read(fd,"29h\tREAD MULTIPLE EXT",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

      /* execute READ SECTOR(S) EXT - read sector 0x1024 0000 */
      cycle_fd(fd, device);
      sector_number = 270794752;
      memset( &buffer, 0, sizeof(buffer) );
      print_buffer("READ SECTORS EXT cmd buffer (BEFORE)", buffer);
      result = ata_read_sector_ext(fd, sector_number, buffer);
      print_buffer("READ SECTORS EXT cmd buffer (AFTER)", buffer);
      //ata_print_sector(buffer);
      read_cnt++;
      cycle_fd(fd, device);
      success_cnt += validate_read(fd,"24h\tREAD SECTOR EXT\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

      /* execute READ STREAM EXT - read sector 0x102B 0000 */
      cycle_fd(fd, device);
      sector_number = 271253504;
      memset( &buffer, 0, sizeof(buffer) );
      print_buffer("READ STREAM EXT cmd buffer (BEFORE)", buffer);
      result = ata_read_stream_ext(fd, sector_number, buffer);
      print_buffer("READ STREAM EXT cmd buffer (AFTER)", buffer);
      //ata_print_sector(buffer);
      read_cnt++;
      cycle_fd(fd, device);
      success_cnt += validate_read(fd,"2Bh\tREAD STREAM EXT\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);

      /* execute READ STREAM DMA EXT - read sector 0x102A 0000 */
      cycle_fd(fd, device);
      sector_number = 271187968;
      memset( &buffer, 0, sizeof(buffer) );
      print_buffer("READ STREAM DMA EXT cmd buffer (BEFORE)", buffer);
      result = ata_read_stream_dma_ext(fd, sector_number, buffer);
      print_buffer("READ STREAM DMA EXT cmd buffer (AFTER)", buffer);
      //ata_print_sector(buffer);
      read_cnt++;
      cycle_fd(fd, device);
      success_cnt += validate_read(fd,"2Ah\tREAD STREAM DMA EXT",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_ATA_DEVICE, ext_cmd_support);
     }
  } 
  if ( run_scsi ){
    /* if SCSI, Goal: issue each defined SCSI read command */

    /**
     * SCSI read commands 
     */

    /* execute READ 6 - read sector 0x0810 */
    sector_number = 2064;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("(scsi) READ 6 cmd buffer (BEFORE)", buffer);
    result = scsi_read_6(fd, sector_number, buffer);
    print_buffer("(scsi) READ 6 cmd buffer (AFTER)", buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"08h\t(scsi) READ 6\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_SCSI_DEVICE, ext_cmd_support);

    /* execute READ 10 - read sector 0x1028 0010 */
    sector_number = 271056912;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("(scsi) READ 10 cmd buffer (BEFORE)", buffer);
    result = scsi_read_10(fd, sector_number, buffer);
    print_buffer("(scsi) READ 10 cmd buffer (AFTER)", buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"28h\t(scsi) READ 10\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_SCSI_DEVICE, ext_cmd_support);

    /* execute READ 12 - read sector 0x10A8 0010 */
    sector_number = 279445520;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("(scsi) READ 12 cmd buffer (BEFORE)", buffer);
    result = scsi_read_12(fd, sector_number, buffer);
    print_buffer("(scsi) READ 12 cmd buffer (AFTER)", buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"A8h\t(scsi) READ 12\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_SCSI_DEVICE, ext_cmd_support);

    /* execute READ 16 - read sector 0x1088 0010 */
    sector_number = 277348368;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("(scsi) READ 16 cmd buffer (BEFORE)", buffer);
    result = scsi_read_16(fd, sector_number, buffer);
    print_buffer("(scsi) READ 16 cmd buffer (AFTER)", buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"88h\t(scsi) READ 16\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_SCSI_DEVICE, ext_cmd_support);

    /* execute READ 32 - read sector 0x107F 0910 */
    sector_number = 276760848;
    memset( &buffer, 0, sizeof(buffer) );
    print_buffer("(scsi) READ 32 cmd buffer (BEFORE)", buffer);
    result = scsi_read_32(fd, sector_number, buffer);
    print_buffer("(scsi) READ 32 cmd buffer (AFTER)", buffer);
    read_cnt++;
    cycle_fd(fd, device);
    success_cnt += validate_read(fd,"7Fh\t(scsi) READ 32\t",(u_char *) buffer,result,sector_number,&unchanged_cnt, ATA_RAW_SCSI_DEVICE, ext_cmd_support);
  }

  printf("%d sector(s) examined, %d unchanged, %d changed.\n", read_cnt, unchanged_cnt, read_cnt-unchanged_cnt);    
  printf("%d read command(s) issued, %d succeeded, %d failed.\n", read_cnt, success_cnt, read_cnt-success_cnt);
  log_close(from);

  close(fd);
  return 0;
}
