static char *SCCS_ID[] = {"@(#) write_verify.c Linux Version 1.6 created 02/17/12 at 13:45:31",__DATE__,__TIME__};
static char *test_version = "*** DEV VERSION ";

/*
 *
 * The software provided here is released by the National
 * Institute of Standards and Technology (NIST), an agency of
 * the U.S. Department of Commerce, Gaithersburg MD 20899,
 * USA.  The software bears no warranty, either expressed or
 * implied. NIST does not assume legal liability nor
 * responsibility for a User's use of the software or the
 * results of such use.
 *
 * Please note that within the United States, copyright
 * protection, under Section 105 of the United States Code,
 * Title 17, is not available for any work of the United
 * States Government and/or for any works created by United
 * States Government employees. User acknowledges that this
 * software contains work which was created by NIST employees
 * and is therefore in the public domain and not subject to
 * copyright.  The User may use, distribute, or incorporate
 * this software provided the User acknowledges this via an
 * explicit acknowledgment of NIST-related contributions to
 * the User's work. User also agrees to acknowledge, via an
 * explicit acknowledgment, that any modifications or
 * alterations have been made to this software before
 * redistribution.
 * --------------------------------------------------------------------
 *
 * Revision History:
 * 2011 May Ben Livelsberger - Created
 *
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <ctype.h>
#include "ataraw.h"
#ifdef HAVE_ERR_H
#include <err.h>
#endif

#ifndef HAVE_ERR
#include <stdarg.h>
void err(int eval,const char *fmt,...)
{
  va_list ap;
  va_start(ap,fmt);
  vfprintf(stderr,fmt,ap);
  va_end(ap);
  fprintf(stderr,": %s\n",strerror(errno));
  exit(eval);
}
#endif

const char *progname="main"; 

void usage()
{
  fprintf(stderr,"Usage: %s [-options] <device>     read back select sectors to\n",progname);
  fprintf(stderr,"\t\tmeasure which (if any) write commands successfully wrote\n\t\tto device.\n");
  fprintf(stderr,"Options:\n -d\tprint extra debugging information\n\n");
  exit(1);
}

/**
 * Read sector_number and determine if it contains the valid diskwipe wipe
 * pattern.  If it does, the sector is unchanged, if it does not, it's been
 * written to.
 *
 */
void validate_write(int fd, const char* cmd_name, uint64_t sector_number, int* unchanged, int device_type, int ext_support){
  int lba, v_result;
  u_char v_buffer[SECTOR_SIZE];
  memset( &v_buffer, 0, sizeof(v_buffer) );

  /* is the drive scsi or ata?  does the drive support the 48-bit feature 
     set? use READ DMA EXT, READ SECTOR(S), or READ 16 accordingly. */
  if ( device_type == ATA_RAW_SCSI_DEVICE ) { // for scsi devices issue READ 16
    v_result = scsi_read_10(fd, sector_number, v_buffer);
    /* if our READ 16 failed then exit. */
    if( v_result != ATA_RAW_OK ){
      err(1,"READ 10 verification read (%s) FAILED, RETURN CODE: %d\n", cmd_name, v_result );
    }
  } else { // ATA device
    if ( ext_support == 0 ) { /* Does the drive support the 48-bit
      feature set? use an EXT or non-EXT read command accordingly, to issue 
      a verifying read to the same lba. */
      v_result = ata_read_dma_ext(fd, sector_number, v_buffer);
      /* if our READ DMA EXT failed then try again using READ SECTORS EXT. */
      if( v_result != ATA_RAW_OK ){
	// make sure that we're working w/ a clean buffer
	memset( &v_buffer, 0, sizeof(v_buffer) );
	v_result = ata_read_sector_ext(fd, sector_number, v_buffer);
	/* if our READ SECTORS EXT failed then exit. */
	if( v_result != ATA_RAW_OK ){
	  err(1,"READ SECTORS EXT verification read (%s) FAILED, RETURN CODE: %d\n", cmd_name, v_result );
	}
      }
    } else {
      /* try reading the sector using READ DMA first */
      v_result = ata_read_dma(fd,(u_long) sector_number, v_buffer);
      /* if our READ DMA failed, then try the read again using READ SECTORS. */
      if( v_result != ATA_RAW_OK ){
	// make sure that we're working w/ a clean buffer
	memset( &v_buffer, 0, sizeof(v_buffer) );
	v_result = ata_read_sector(fd,(u_long) sector_number, 1, v_buffer);
	/* if our READ SECTORS failed then exit. */
	if( v_result != ATA_RAW_OK ){
	  err(1,"READ SECTORS verification read (%s) FAILED, RETURN CODE: %d\n", cmd_name, v_result );
	}
      }    
    }
  }
  /* Look at the sector contents (from the READ DMA EXT read). If 
     try_write was able to write to it successfully, then the diskwipe 
     wipe pattern won't be there any more. If it was initialized 
     correctly and hasn't been written to since, then it should 
     contain the FS-TST diskwipe wipe pattern. Extract the lba and 
     compare to the sector address read. */
  //ata_print_sector((unsigned char *) &v_buffer);
  lba = atoi(strndup((char *) &v_buffer[13], 12));
  printf("%s\t",cmd_name);
  if (lba == sector_number) {
    printf("unchanged");
    *unchanged += 1;
  } else {
    printf("changed");
  }

  if (ataraw_debug){
    printf(" (lba %d)\n",(int) sector_number);
    ata_print_sector((u_char *) v_buffer);
  } else {
    printf ("\n");
  }

  return;
}


int main(int argc,char **argv)
{
  int opt, i;
  static time_t from;

  if (SCCS_ID[0][0] == '%') SCCS_ID[0] = test_version;
  progname = argv[0];
  time(&from);

  /* print all the file version and compilation info */
  printf ("%s %s%s\n",progname,ctime(&from),SCCS_ID[0]);
  //printf ("write_verify %s\n",SCCS_ID[0]);
  printf ("compiled %s %s with gcc Version %s\n",__DATE__,
	  __TIME__,__VERSION__);
  printf ("%s%s%s\n",WRAPPER_C_ID,ATARAW_C_ID,ATARAW_H_ID);

  /* print the command line */
  printf("cmd: ");
  for(i=0; i<argc; i++) {
    printf(" %s", argv[i]);
  }
  printf("\n");

    while((opt=getopt(argc,argv,"d"))!=-1){
	switch(opt){
	  // -d option for debug mode  
	case 'd':
	    ataraw_debug = 1;
	    break;
	default:
	    usage();
	}
    }

    argc -= optind;
    argv += optind;

    if(argc<1) usage();

    const char *device = argv[0];

    argv++;
    argc--;

    int fd = open(device,O_RDONLY);
    if(fd<0){
	perror(device);
	exit(1);
    }

    /* Expectation: prior to running write_verify, the user has 1) wiped the 
       target drive using the diskwipe program and 2) attempted to send all
       defined ATA write commands to the drive using try_write.  
       Goal: the goal is to read back the sector contents from the target 
       drive to see if any sectors have been changed.  We want to verify 
       and log which (if any) write commands successfully wrote to the drive.
       If a given sector's contents no longer holds the diskwipe pattern, 
       then the associated write command was able to successfully write to 
       the drive. */

    int unchanged_cnt = 0; 
    int validate_cnt = 0;
    uint64_t sector_number;
    int ext_cmd_support = 1; //default to "unsupported"

  /* Does fd refer to an ATA device or to a SCSI device? */
  int device_type = chk_device_type( fd );
  if ( device_type == ATA_RAW_ATA_DEVICE ){
    /* extended (48-bit) commands supported? */
    ext_cmd_support = chk_48bit_feature_support(fd);
    if( ext_cmd_support == ATA_RAW_ERR )
      err(1,"chk_48bit_feature_support() FAILED, RETURN CODE: %d\n",ext_cmd_support );
    printf("%s device type:\tATA\n", device);
    if( ext_cmd_support == 0 )
      printf("48-bit Address Feature Set:     SUPPORTED\n\n"); 
    else
      printf("48-bit Address Feature Set:     NOT SUPPORTED\n\n"); 
  } else if ( device_type == ATA_RAW_SCSI_DEVICE ){
    printf("%s device type:\tSCSI\n\n", device);
  } else {
      err(1,"%s:\tinvalid device -- SCSI, ATA command sets not supported\n", device );
  }

    /* For each defined ATA write command, check to see if the associated 
       sector (the sector that try_write would have written to) has changed.
       (for the regular ATA write commands, check sector address 0xXX00; 
       for extended ATA write commands, check 0x10XX 0000 where XX is 
       the command opcode. */

  /* print column headers */
  printf("Opcode\tCommand Name\t\tSector Fill\n");    

    /** 
     * non-EXT ATA commands 
     */
    //if ( ( ext_cmd_support == 0 ) || ( ext_cmd_support == 1 ) ){
    /* CFA WRITE MULTIPLE W/O ERASE CDh */ 
    sector_number = 52480; // 0xCD00
    validate_write(fd,"CDh\tCFA WRITE MULTIPLE W/O ERASE",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
    validate_cnt++;

    /* CFA WRITE SECTORS W/O ERASE 38h */ 
    sector_number = 14336; // 0x3800
    validate_write(fd,"38h\tCFA WRITE SECTORS W/O ERASE",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
    validate_cnt++;

    /* WRITE SECTOR(S) 30h */
    sector_number = 12288; // 0x3000
    validate_write(fd,"30h\tWRITE SECTOR(S)\t",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
    validate_cnt++;

    /* WRITE DMA CAh */
    sector_number = 51712; // 0xCA00
    validate_write(fd,"CAh\tWRITE DMA\t",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
    validate_cnt++;

    /* WRITE DMA QUEUED CCh */
    sector_number = 52224; // 0xCC00
    validate_write(fd,"CCh\tWRITE DMA QUEUED",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
    validate_cnt++;

    /* WRITE MULTIPLE C5h */
    sector_number = 50432; // 0xC500
    validate_write(fd,"C5h\tWRITE MULTIPLE\t",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
    validate_cnt++;

    /*
     * write commands from ATA-3 spec.
     */

    /* WRITE LONG 32h -- leave for now... need to figure out how to write a 
     * proper ECC to the drive*/
/*     sector_number = 12800; // 0x3200 */
/*     validate_write(fd,"32h\tWRITE LONG",sector_number,&unchanged_cnt,device_type,ext_cmd_support); */
/*     validate_cnt++; */

    /* WRITE SECTOR(S) w/o retries 31h */
    sector_number = 12544; // 0x3100
    validate_write(fd,"31h\tWRITE SECTOR(S) w/o retries",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
    validate_cnt++;

    /* WRITE DMA w/o retries CBh */
    sector_number = 51968; // 0xCB00
    validate_write(fd,"CBh\tWRITE DMA w/o retries",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
    validate_cnt++;

    /* WRITE VERIFY 3Ch */
    sector_number = 15360; // 0x3C00
    validate_write(fd,"3Ch\tWRITE VERIFY\t",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
    validate_cnt++;
    //}

    /* 
     * EXT commands (can only verify if drive supports EXT cmds.  NOTE: we require the test drive to be 200+ GB so EXT support should, in theory, be guaranteed)
     */
    //if ( ext_cmd_support == 0 ){
      /* WRITE SECTOR(S) EXT 34h */
      sector_number = 271843328; // 0x1034 0000
      validate_write(fd,"34h\tWRITE SECTOR(S) EXT",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;    

      /* WRITE MULTIPLE EXT 39h */
      sector_number = 272171008; // 0x1039 0000
      validate_write(fd,"39h\tWRITE MULTIPLE EXT",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE MULTIPLE FUA EXT CEh */
      sector_number = 281935872; // 0x10CE 0000
      validate_write(fd,"CEh\tWRITE MULTIPLE FUA EXT",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE STREAM EXT 3Bh */
      sector_number = 272302080; // 0x103B 0000
      validate_write(fd,"3Bh\tWRITE STREAM EXT",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE DMA EXT 35h */
      sector_number = 271908864; // 0x1035 0000
      validate_write(fd,"35h\tWRITE DMA EXT\t",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE DMA FUA EXT 3Dh */
      sector_number = 272433152; // 0x103D 0000
      validate_write(fd,"3Dh\tWRITE DMA FUA EXT",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE DMA QUEUED EXT 36h */
      sector_number = 271974400; // 0x1036 0000
      validate_write(fd,"36h\tWRITE DMA QUEUED EXT",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE DMA QUEUED FUA EXT 3Eh */
      sector_number = 272498688; // 0x103E 0000
      validate_write(fd,"3Eh\tWRITE DMA QUEUED FUA EXT",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE FPDMA QUEUED 61h */
      sector_number = 274792448; // 0x1061 0000
      validate_write(fd,"61h\tWRITE FPDMA QUEUED",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE STREAM DMA EXT 3Ah */
      sector_number = 272236544; // 0x103A 0000
      validate_write(fd,"3Ah\tWRITE STREAM DMA EXT",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;
      //}
    /** 
     * SCSI commands 
     */
    //    if ( ext_cmd_support == 2 ){    
      /* WRITE 6 0Ah */ 
      sector_number = 2576; // 0x0A10
      validate_write(fd,"0Ah\tWRITE 6\t\t",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE 10 2Ah */ 
      sector_number = 271187984; // 0x102A 0010
      validate_write(fd,"2Ah\tWRITE 10\t",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE 12 AAh */ 
      sector_number = 279576592; // 0x10AA 0010
      validate_write(fd,"AAh\tWRITE 12\t",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE 16 8Ah */ 
      sector_number = 277479440; // 0x108A 0010
      validate_write(fd,"8Ah\tWRITE 16\t",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;

      /* WRITE 32 7Fh */ 
      sector_number = 276761360; // 0x107F 0B10
      validate_write(fd,"7Fh\tWRITE 32\t",sector_number,&unchanged_cnt,device_type,ext_cmd_support);
      validate_cnt++;
      //    }

    printf("\n%d sector(s) examined, %d unchanged, %d changed.\n", validate_cnt, unchanged_cnt, validate_cnt-unchanged_cnt);    
    log_close(from);

    close(fd);
    return 0;
}
