char *WRAPPER_C_ID = "@(#) wrapper.c Linux Version 1.4 support lib created 02/17/12 at 13:45:48\n";

/* 
 * @(#) wrapper.c Linux Version 1.4 created 02/17/12 at 13:45:48
 *
 * These are a collection of all ATA functions.  All of these commands use the general 
 * purpose ata_raw call.  These are built for convenience and as an example for 
 * using the general purpose call.
 *
 * Revision History:
 * 2011 May Ben Livelsberger: forked file from the version in 
 *    the NPS ataraw-0.2.1 package.  Extended
 * 2009 Jan Kyle Sanders & Simson Garfinkel: Extended
 * 2008 Nov Kyle Sanders - Created
 */

#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "ataraw.h"


/****************************************************************
 *** HELPER FUNCTIONS 
 ****************************************************************/
/**
 * Sets the password in the security passwor structure.
 * abstracted to function to reduce retyping.
 */
static void set_password( security_password* sp, u_char* password )
{
	u_char* password_dest = (u_char*)sp->password;
	memcpy(password_dest, password, 16*2 );
}
/** 
 * Checksum is calculated as specified in the ATA standard.
 * @param dco_overlay [dco_struct] The DCO structure to compute
 * the checksum.  This will place the checksum into this struct.
 */
word compute_dco_checksum( dco_struct* dco_overlay )
{
        int i = 0;
        unsigned char acc = 0;
        unsigned char* dco = (unsigned char* )dco_overlay;
        unsigned char high = 0xa5;;
        unsigned char low = 0x00;
        word returnval = 0;
        for( i = 0; i < 510; i++ ) {
                acc = acc + dco[i];
        }
        low = acc + high;
        low = (~low) + 1;
        returnval = ( high << 8 ) | low;
        dco[510] = high;
	dco[511] = low; 
        return returnval;
}

/**
 * Given a sector number, set the specified registers.
 * @param registers [u_char*] The registers buffer to
 * copy the sector number into.
 * @param sector [u_long] The sector number to copy
 * into the registers buffer.
 */
void set_sector_number( u_char* registers, u_long sector ){
    registers[3] = 0xff & sector;
    registers[4] = 0xff & ( sector >> 8 );
    registers[5] = 0xff & ( sector >> 16 );
    registers[6] = 0x40 | ( 0x0f & ( sector >> 24) ); /* 0x40 means this is an LBA */
}

void set_sector_number_ext( u_char* registers, u_char* e_registers, uint64_t sector ){
    set_sector_number(registers,(u_long)sector & 0x0fffffff); /* set the bottom 28 bits */
    registers[6] = 0x40;		/* we are using LBA; get rid of top nibble */
    e_registers[3] =     0xff & (sector >> 24 );
    e_registers[4] =     0xff & (sector >> 32 );
    e_registers[5] =     0xff & (sector >> 40 );
}

/**
 * Given a sector number, command block, and command block size, set the 
 * command block LBA field to the address to be read/written.
 * @param cmdblk [u_char*] The command block buffer.
 * @param sector [uint64_t] The sector address to be read/write.
 * @param cmdblk_size [int] The scsi command block size.
 *
 */
void set_scsi_sector_number( u_char* cmdblk, uint64_t sector, int cmdblk_size ){

  int i,j=0;

  switch (cmdblk_size){

  case 6:
    for ( i = 3; i >= 2; i--){
      cmdblk[i] = 0xff & ( sector >> (j*8) );
      j++;
    }
    return;
  case 10:
    for ( i = 5; i >= 2; i--){
      cmdblk[i] = 0xff & ( sector >> (j*8) );
      j++;
    }
    return;
  case 12:
    for ( i = 5; i >= 2; i--){
      cmdblk[i] = 0xff & ( sector >> (j*8) );
      j++;
    }
    return;
  case 16:
    for ( i = 9; i >= 2; i--){
      cmdblk[i] = 0xff & ( sector >> (j*8) );
      j++;
    }
    return;
  case 32:
    for ( i = 19; i >= 12; i--){
      cmdblk[i] = 0xff & ( sector >> (j*8) );
      j++;
    }
    return;
  }
}

/** 
 * Get the sector number out of the register buffer
 * @param registers [u_char*] registers buffer to copy the sector number out of.
 * @return u_long The sector number contained in the registers parameter(28 bit).
 */
u_long get_sector_number(u_char *registers)
{
    return registers[3]
	|          ( registers[4] << 8 )
	|          ( registers[5] << 16 )
	| (( 0x0f & registers[6]) << 24 ) ;    
}
/**
 * Gets the sector number out of the registers and extended registers buffer.
 * This is used on EXT commands that hold sector numbers in both the registers
 * and extended registers.
 * @param registers [u_char*] Registers buffer to copy the sector number from.
 * @param e_registers [u_char*] Extended registers buffer to copy the sector
 * number from.
 * @return uint64_t 48 bit sector number contained in the register(s) buffer.
 */
uint64_t get_sector_number_ext(u_char *registers,u_char *e_registers)
{
    return registers[3]
	| (registers[4]<<8)
	| (registers[5]<<16)
	| (e_registers[3]<<24)
	| (((uint64_t)e_registers[4])<<32)
	| (((uint64_t)e_registers[5])<<40);
}

/**
 * Read a sector(s) from an ata device.  This is a 28 bit command so only sector numbers up 
 * to 0x0fffffff can be addressed.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The starting sector address to read from.  Values over 0x0fffffff 
 * will be ignored as this is only a 28 bit command.
 * @param buffer [u_char*] Pointer to the buffer to be read into.  This must be of size count * sector 
 * size.  Where count is the passed in parameter and sector size is 512 bytes.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_long( int fd, u_long sector_number, unsigned char* buffer )
{
    u_char registers[8];
    int result = ATA_RAW_ERR;
    
    if (sector_number > 0x0FFFFFFF) return result; /* check to see if larger than 28 bits */ 
    memset( registers, 0, 8 );
    registers[2] = 0x01;
    set_sector_number( registers, sector_number );
    registers[7] = READ_LONG;
    result = ata_raw_fd( fd, registers, NULL, buffer );
    return result;
}

/**
 * Read a sector(s) from an ata device.  This is a 28 bit command so only sector numbers up 
 * to 0x0fffffff can be addressed.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The starting sector address to read from.  Values over 0x0fffffff 
 * will be ignored as this is only a 28 bit command.
 * @param buffer [u_char*] Pointer to the buffer to be read into.  This must be of size count * sector 
 * size.  Where count is the passed in parameter and sector size is 512 bytes.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_long_no_retry( int fd, u_long sector_number, u_char* buffer )
{
    u_char registers[8];
    int result = ATA_RAW_ERR;
    
    if (sector_number > 0x0FFFFFFF) return result; /* check to see if larger than 28 bits */ 
    memset( registers, 0, 8 );
    registers[2] = 0x01;
    set_sector_number( registers, sector_number );
    registers[7] = READ_LONG_NO_RETRY;
    result = ata_raw_fd( fd, registers, NULL, buffer );
    return result;
}

/**
 * Read a sector(s) from an ata device.  This is a 28 bit command so only sector numbers up 
 * to 0x0fffffff can be addressed.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The starting sector address to read from.  Values over 0x0fffffff 
 * will be ignored as this is only a 28 bit command.
 * @param count [u_char] The number of sectors to read.  
 * @param buffer [u_char*] Pointer to the buffer to be read into.  This must be of size count * sector 
 * size.  Where count is the passed in parameter and sector size is 512 bytes.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_sector( int fd, u_long sector_number, const u_char count, u_char* buffer )
{
    u_char registers[8];
    int result = ATA_RAW_ERR;
    
    if (sector_number > 0x0FFFFFFF) return result; /* check to see if larger than 28 bits */ 
    memset( registers, 0, 8 );
    registers[2] = count;
    set_sector_number( registers, sector_number );
    registers[7] = READ_SECTORS;
    result = ata_raw_fd( fd, registers, NULL, buffer );
    return result;
}

/**
 * Read a sector(s) w/o retries from an ata device.  This is a 28 bit command so only sector numbers up 
 * to 0x0fffffff can be addressed.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The starting sector address to read from.  Values over 0x0fffffff 
 * will be ignored as this is only a 28 bit command.
 * @param count [u_char] The number of sectors to read.  
 * @param buffer [u_char*] Pointer to the buffer to be read into.  This must be of size count * sector 
 * size.  Where count is the passed in parameter and sector size is 512 bytes.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_sector_no_retry( int fd, u_long sector_number, const u_char count, u_char* buffer )
{
    u_char registers[8];
    int result = ATA_RAW_ERR;
    
    if (sector_number > 0x0FFFFFFF) return result; /* check to see if larger than 28 bits */ 
    memset( registers, 0, 8 );
    registers[2] = count;
    set_sector_number( registers, sector_number );
    registers[7] = READ_SECTORS_NO_RETRY;
    result = ata_raw_fd( fd, registers, NULL, buffer );
    return result;
}

/**
 * Perform a device identify command.  
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param buffer [u_char*] Buffer where the identify command information will be placed.  
 * This buffer should be 512 bytes in length.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_identify_device( int fd, u_char* buffer )
{
	u_char registers[8];
	memset( registers, 0, 8 );
	registers[7] = IDENTIFY_DEVICE;
	return ata_raw_fd( fd, registers, 0, buffer );
}
/**
 * Performs a Device Configuration Overlay identify command on the device specified by the 
 * file descriptor.  
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param buffer [u_char*] Buffer where the identify command information will be placed.  
 * This buffer should be 513 bytes in length.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_device_configuration_identify( int fd, u_char* buffer )
{
	u_char registers[8];
	u_char e_registers[8];
	memset( registers, 0 , 8);
	registers[1] = DEVICE_CONFIGURATION_IDENTIFY;
	registers[6] = 0x40;
	registers[7] = DEVICE_CONFIGURATION;
	return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * Performs a seek command.  This will move the read head of the device to the specified sector.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_to_seek [int] The sector number to seek to.  This is a 28 bit ATA command so the 
 * maximum sector number is 0x0fffffff.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_seek( int fd, u_long sector_to_seek )
{
	int result = ATA_RAW_ERR;
	u_char registers[8];
	memset( registers, 0, 8 );
	set_sector_number(registers,sector_to_seek);
	registers[7] = SEEK;
	result = ata_raw_fd( fd, registers, 0, 0 );
	return result;
}
/**
 * Reads the native max address of the device.  This is the maximum addressable sector for the host.  
 * This is used as part of the HPA feature set.  This performs a Read Native Max command 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param addr [int*] pointer to an integer.  This will be populated with the maximum 
 * addressable sector value.  Only the lower 28 bits will be filled as this is a 28 bit command
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_native_max_address( int fd, u_long* addr )
{
	int result = ATA_RAW_ERR;
	u_char registers[8];
	memset( registers, 0, 8 );
	registers[6] = 0x40;
	registers[7] = READ_NATIVE_MAX_ADDRESS;
	result = ata_raw_fd( fd, registers, 0, 0 );
	if( result == ATA_RAW_OK ) {
	    *addr = get_sector_number(registers);
	}
	return  result;
}
/**
 * Performs a Set Max Address command.  This will set the maximum addressable sector on the device.  
 * In fact, this is how an HPA can be set or cleared.  This command is part of the HPA feature set.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param addr [int] Integer value for the maximum addressable sector.  Only the first 
 * 28 bits of the integer will be used as this is a 28 bit command.
 * @param volatility [u_char] Set 0x01 or 0x00.  If set to 1 the native max address will be set 
 * permanently (i.e. after reset and power down).  If set to 0 the native max address will reset 
 * after device restart or power down.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_set_max_address( int fd, const int new_max_addr, const u_char volatility )
{
	int result = ATA_RAW_ERR;
	u_char registers[8];
	memset( & registers, 0, 8 );
	registers[2] = 0x01 & volatility;
	registers[3] = 0xff & new_max_addr;
	registers[4] = 0xff & (new_max_addr >> 8 );
	registers[5] = 0xff & (new_max_addr >> 16 );
	registers[6] = 0x40 | ( 0x0f & (new_max_addr >> 24 ));
	registers[7] = SET_MAX_ADDRESS;
	result = ata_raw_fd( fd, registers, 0, 0 );
	return result;
}
/**
 * Performs a Read Native Max Ext command.  This will read the maximum addressable sector 
 * from the device.  This is part of the HPA feature set.  This command differs from the 
 * Read Native Max address command because it is the EXT version (48 bit).
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param address [long*] Pointer to long value where the maximum addressable sector will be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_native_max_ext( int fd, uint64_t *address )
{
	int result = ATA_RAW_ERR;
	u_char registers[8];
	u_char e_registers[8];
	memset( &registers, 0, 8);
	memset( &e_registers, 0, 8 );
	registers[6] = 0x40;
	registers[7] = READ_NATIVE_MAX_ADDRESS_EXT;
	result = ata_raw_fd( fd, registers, e_registers, 0 );	
if( result == ATA_RAW_OK ) {
	    *address = get_sector_number_ext(registers,e_registers);
	}
	return result;
}
/**
 * Performs a Read Sector Et command.  This reads a sector of the callers choosing from the device.  
 * This command differs from Read Sector by being the Ext version (48 bit).
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [long] The sector to read on the device.  Only the first 48 bits of this value will 
 * be used as this is a 48 bit command.
 * @param buffer [u_char*] Pointer to the buffer to be read into.  This must be of size count * sector 
 * size.  Where count is the passed in parameter and sector size is 512 bytes.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_sector_ext( int fd, uint64_t sector, u_char* buffer )
{
	u_char registers[8];
	u_char e_registers[8];
	//u_short count = 1;
	memset( &registers, 0, 8 );
	memset( &e_registers, 0, 8 );
	//registers[2] = 0xff & count;
	//e_registers[2] = 0xff & (count>>8);
	registers[2] = 0x01;
	set_sector_number_ext( registers, e_registers, sector );
	//registers[6] = 0x40;
	registers[7] = READ_SECTOR_EXT;
	return ata_raw_fd( fd, registers, e_registers, buffer );
}

/**
 * Performs a Read Stream Ext command.  This reads a sector of the callers choosing from the device.  
 * This command differs from Read Sector by being the Ext version (48 bit).
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [long] The sector to read on the device.  Only the first 48 bits of this value will 
 * be used as this is a 48 bit command.
 * @param buffer [u_char*] Pointer to the buffer to be read into.  This must be of size count * sector 
 * size.  Where count is the passed in parameter and sector size is 512 bytes.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_stream_ext( int fd, uint64_t sector, u_char* buffer )
{
	u_char registers[8];
	u_char e_registers[8];
	memset( &registers, 0, 8 );
	memset( &e_registers, 0, 8 );
	//registers[2] = 0x01;
	registers[6] = 0x40;
	registers[7] = READ_STREAM_EXT;
	set_sector_number_ext( registers, e_registers, sector );
	return ata_raw_fd( fd, registers, e_registers, buffer );
}

/**
 * The command will read a sector from the device using DMA.  Only one sector will be read.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param address [int] Address of the sector to read.  This is a 28 bit command so only the first 28
 * bits of this value will be used.
 * @param buff [u_char*] Pointer to the buffer to be read into.  Only one sector is being read so 
 * the buffer must be of size SECTOR_SIZE (currently 512 bytes).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_dma( int fd, u_long sector, u_char* buff )
{
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[2] = 1;
	registers[6] = 0x40;
	registers[7] = READ_DMA;
	set_sector_number( registers, sector );
	return ata_raw_fd( fd, registers, 0, buff );
}

/**
 * The command will read a sector from the device using DMA (w/o retries).  Only one sector will be read.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param address [int] Address of the sector to read.  This is a 28 bit command so only the first 28
 * bits of this value will be used.
 * @param buff [u_char*] Pointer to the buffer to be read into.  Only one sector is being read so 
 * the buffer must be of size SECTOR_SIZE (currently 512 bytes).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_dma_no_retry( int fd, u_long sector, u_char* buff )
{
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[2] = 1;
	registers[6] = 0x40;
	registers[7] = READ_DMA_NO_RETRY;
	set_sector_number( registers, sector );
	return ata_raw_fd( fd, registers, 0, buff );
}

/**
 * The command will read a sector from the device using DMA.  Only one sector will be read.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param address [int] Address of the sector to read.  This is a 28 bit command so only the first 28
 * bits of this value will be used.
 * @param buff [u_char*] Pointer to the buffer to be read into.  Only one sector is being read so 
 * the buffer must be of size SECTOR_SIZE (currently 512 bytes).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_dma_queued( int fd, u_long sector, u_char* buff )
{
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[1] = 1;  // sector count
	//registers[2] = 0x08;  // Tag value (0-31)
       	registers[2] = 0;  // Tag value (0-31)
	registers[6] = 0x40;
	registers[7] = READ_DMA_QUEUED;
	set_sector_number( registers, sector );
	return ata_raw_fd( fd, registers, 0, buff );
}
/**
 * This will perform the CFA Erase Sector command.  This command is part of the CFA 
 * feature set and is not mandatory for devices.  
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_to_erase [int] Address of the starting sector to erase.  This is a 
 * 28 bit command so only the first 28 bits will be used to determine address.
 * @param count [u_char] The number of sectors to erase.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_cfa_erase_sector( int fd, u_long sector_to_erase, u_char count ){
	u_char registers[8];
	memset( &registers, 0, sizeof( registers ) );
	registers[2] = count;
	registers[6] = 0x40;
	registers[7] = CFA_ERASE_SECTOR;
	set_sector_number( registers, sector_to_erase );
	return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This will perform the CFA Request Extended Error Code command.  As stated in the ATA
 * spec "This command provides an extended error code which identifies the cause of an error 
 * condition in more detail than is available with Status and Error register values."  All 
 * err values are available in the ATA 6 spec (Page 79).  This command is part of the CFA feature set.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param err_code [int*] Pointer to integer value where the error code will be stored.  
 * The error code is only 8 bits wide so only the first 8 bits of the int will be the err value.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_cfa_extended_err(int fd, int* err_code ){
	int result;
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[7] = CFA_REQUEST_EXTENDED_ERROR_CODE;
	result = ata_raw_fd( fd, registers, 0, 0 );
	*err_code = registers[1];
	return result;
}
/**
 * This will perform the CFA Translate sector command. This command will provide 
 * information about an individual sector(number of erases and number of writes).  
 * A table describing the data structure that is returned is available on page 83 
 * of the ATA spec.  This command is part of the CFA feature set.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [int] The address of the sector to gather information about.
 * @param buffer [u_char*] Pointer to the buffer where information will be read into.
 * The should be of size SECTOR_SIZE (currently 512 bytes).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_cfa_translate_sector( int fd, u_long sector, u_char* buffer ){
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[6] = 0x40;
	set_sector_number( registers, sector );
	registers[7] = CFA_TRANSLATE_SECTOR;
	return ata_raw_fd( fd, registers, 0, buffer );
}
/**
 * Internal helper function.  This will set up some of the values and make the
 * ata_raw call.  
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [int] The sector address to write to.  This is a 28 bit call
 * so only the first 28 bits of the address will be used.
 * @param sector_count [u_char] The number of sectors to write. It can only be up to 255.
 * @param registers [u_char*] The registers array that was initialized in the calling
 * function.
 * @param buffer [u_char*] the buffer to write to the device.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
static int wrapper_pio_out( int fd, u_long sector, u_char sector_count, u_char* registers, u_char* buffer ){
	registers[2] = sector_count;
	registers[6] = 0x40;
	set_sector_number( registers, sector );
	return ata_raw_fd( fd, registers, 0, buffer );
}
/**
 * This performs a CFA Write Multiple Without Erase command.  This function writes the provided
 * data to the device without erasing the sector beforehand.  Furthermore, this is a write
 * multiple so interrupts will only be generated after all sectors have been transfered.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [int] Address of the sector to write to.  This is a 28 bit command so
 * only the first 28 bits of address will be used.
 * @param sector_count [u_char] The number of sectors to write.
 * @param buffer [u_char*] The buffer to be written to the device.  This should be of size 
 * sector_count * SECTOR_SIZE.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_cfa_write_mult_wo_erase( int fd, u_long sector, u_char sector_count, u_char* buffer ){
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[7] = CFA_WRITE_MULTIPLE_WITHOUT_ERASE;
	return wrapper_pio_out( fd, sector, sector_count, registers, buffer );
}
/**
 * This performs an CFA Write Sector Without Erase.  This function writes the provided data
 * to the device without erasing beforehand.  This command differs from write multiple because
 * each transfered sector will generate an interrupt.  This command is part of the CFA feature
 * set.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [int] Address of the sector to write to.  This is a 28 bit command so
 * only the first 28 bits of address will be used.
 * @param sector_count [u_char] The number of sectors to write.
 * @param buffer [u_char*] The buffer to be written to the device.  This should be of size 
 * sector_count * SECTOR_SIZE. 
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_cfa_write_sector_wo_erase( int fd, u_long sector, u_char sector_count, u_char* buffer ){
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[7] = CFA_WRITE_SECTORS_WITHOUT_ERASE;
	return wrapper_pio_out( fd , sector, sector_count, registers, buffer );
}
/**
 * This performs Check Media Card Type. This function allows the host to determine
 * if the device supports the Media Card passthrough set.  This command also lets the 
 * host enable the media card pass through feature set.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param enable [u_char] This will enable the media card pass through feature set.  The 
 * value 0x01 enables the pass through.  All other values will disable it.
 * @param card_type [u_char*] Pointer to a char that will be set with the card type.  To 
 * interpret card type values consult the table on page 87 of the ATA 6 spec.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_check_media_card_type( int fd, u_char enable, u_char* card_type ){
	int result;
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[1] = 0x01 & enable;
	registers[7] = CHECK_MEDIA_CARD_TYPE;
	result = ata_raw_fd( fd, registers, 0, 0 );
	*card_type = registers[6];
	return result;
}
/**
 * This function will perform a Check Power Mode command.  This command allows the host
 * determine the power mode of the device.  This command is part of the power management 
 * feature set.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param power_mode [u_char*] Pointer to u_char that will hold the power mode after
 * function completion.  The value is only 1 byte wide.  To interpret hex values for
 * power mode consult page 89 of the ATA 6 spec.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_check_power_mode( int fd, u_char* power_mode ){
	int result;
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[7] = CHECK_POWER_MODE;
	result = ata_raw_fd( fd, registers, 0, 0 );
	*power_mode = registers[2];
	return result;
}
/**
 * This function will perform a DCO Restore command.  This command returns the device
 * to the default factory state before any DCO Set commands were issued.  This command
 * is part of the Device Configuration Overlay feature set.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_dco_restore( int fd ){
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[1] = DEVICE_CONFIGURATION_RESTORE;
	registers[7] = DEVICE_CONFIGURATION;
	return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function will perform a DCO Freeze Lock command.  This command puts the device in a state 
 * such that all subsequent DCO commands will fail.  The intent of this command
 * is to prevent the device configuration overlay from being accidentally set.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */ 
int ata_dco_freeze_lock( int fd ){
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[1] = DEVICE_CONFIGURATION_FREEZE_LOCK;
	registers[7] = DEVICE_CONFIGURATION;
	return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * The command performs a DCO Set command.  This allows the host to change the Device Configuration
 * Overlay of the device.  This command is part of the Device Configuration
 * Overlay feature set.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param overlay [dco_struct*] Pointer to a dco_struct.  This will be the overlay for the device.
 * For a more indepth description of struct fields please consult ataraw.h.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_dco_set( int fd, dco_struct* overlay ){
	u_char registers[8];
	memset( &registers, 0, 8 );
	registers[1] = DEVICE_CONFIGURATION_SET;
	registers[7] = DEVICE_CONFIGURATION;
	compute_dco_checksum( overlay );
	return ata_raw_fd( fd, registers, 0, (unsigned char*)overlay );
}
/**
 * This function performs a Flush Cache command.  This command will request the device flush the
 * write cache to disk.  
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_flush_cache( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = FLUSH_CACHE;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs a Flush Cache Ext command.  This command will request the device flush
 * the write cache to the device.  This is identical to the Flush Cache command except it is the
 * 48 bit version.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_flush_cache_ext( int fd ){
    u_char registers[8];
    u_char e_registers[8];
    memset( &registers, 0, 8 );memset( &e_registers, 0, 8 );
    registers[7] = FLUSH_CACHE_EXT;
    return ata_raw_fd( fd, registers, e_registers, 0 );
}
/**
 * This function performs a Get Media Status command.  This will return the media status for the
 * device, if supported.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param status_register [u_char*] Pointer to a char that will hold the status register after
 * command completion.  See page 112 of the ATA 6 spec to interpret value.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_get_media_status( int fd, u_char* status_register ){
    u_char registers[8];
    int result;
    memset( & registers, 0, 8 );
    registers[7] = GET_MEDIA_STATUS;
    result = ata_raw_fd( fd, registers, 0, 0 );
    *status_register = registers[7];
    return result;
}
/**
 * This function performs a Identify Packet Device command.  This command is identical to the
 * IDENTIFY DEVICE command except that it is specific to a device that implements ATAPI.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda). 
 * @param buffer [u_char*] Buffer where the identify command information will be placed.  
 * This buffer should be 512 bytes in length. 
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_identify_packet_device( int fd, u_char* buffer ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = IDENTIFY_PACKET_DEVICE;
    return ata_raw_fd( fd, registers, 0, buffer );
}
/**
 * This function performs a IDLE command.  This enables the host to place the drive
 * into idle mode.  This command is part of the Power Management feature set.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda). 
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_idle( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = IDLE;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs an IDLE IMMEDIATE command.  This is identical to IDLE except
 * the device is forced to idle once the command is executed.  This is part of the 
 * Power Management feature set. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda). 
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_idle_immediate( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = IDLE_IMMEDIATE;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs a MEDIA EJECT command.  This command will finish any pending commands,
 * spin down the ATA device and unlock the media before subsequently ejecting.  
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda). 
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_media_eject( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = MEDIA_EJECT;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs a MEDIA LOCK command.  This will place the device in a LOCKED status.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda). 
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_media_lock( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = MEDIA_LOCK;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs a MEDIA UNLOCK command.  This will place the device in a 
 * UNLOCKED status.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda). 
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_media_unlock( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = MEDIA_UNLOCK;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs the NOP command.  This command will do nothing as the
 * namesake suggests.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_nop( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = NOP;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs the SLEEP command.  When this command is executed the device will be given
 * a chance to finish any remaining commands before enttering sleep mode.  Once in sleep mode
 * only a software reset, hardware reset or DEVICE RESET command.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_sleep( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = SLEEP;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs the STANDBY command.  This enables the device to enter stand-by
 * mode.  This allows the device to finish any pending requests beforehand.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_standby( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = STANDBY;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs the STANDBY IMMEDIATE command.  This is identical to STANDBY
 * except this will not wait for impending requests.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_standby_immediate( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = STANDBY_IMMEDIATE;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * Executes a Read buffer command.  This command is optional for devices to implement.
 * To know if a device supports this command check the results of the identify device
 * command.  This will read the ATA device's sector buffer.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param buffer [u_char[SECTOR_SIZE]] The buffer the data will be read into.  
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_buffer( int fd, u_char buffer[SECTOR_SIZE] ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[7] = READ_BUFFER;
    return ata_raw_fd( fd, registers, 0, buffer );
}
/**
 * Executes a Read Multiple command.  This command is almost identical to Read Sector 
 * except interrupts are not generated after each sector is read.  Rather only one is generated
 * after all sectors have been read.  This is a 28 bit command.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The starting sector address to read from.  Values over 0x0fffffff 
 * will be ignored as this is only a 28 bit command.
 * @param count [u_char] The number of sectors to read.  
 * @param buffer [u_char*] Pointer to the buffer to be read into.  This must be of size count * sector 
 * size.  Where count is the passed in parameter and sector size is 512 bytes.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_multiple( int fd, u_long sector, u_char count, u_char* buffer ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[6] = 0x40;
    set_sector_number( registers, sector );
    registers[2] = count;
    registers[7] = READ_MULTIPLE;
    return ata_raw_fd( fd, registers, 0, buffer );
}
/**
 * Executes a Read Multiple Ext command.  This command is almost identical to Ream Sector Ext
 * except interrupts are not generated after each sector is read.  Rather only one is generated 
 * after all sectors have been read.  This is a 48 bit command.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param count [u_short] the number of sectors to read.
 * @param starting_sector [uint64_t] The sector to start reading from.  Since this is a 48 bit command 
 * only the first 48 bits will be used.  
 * @param buffer [u_char*] The buffer to read into.  This must be size count * sector size.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_multiple_ext( int fd, uint64_t starting_sector, u_short count, u_char* buffer ){
    u_char registers[8];
    u_char e_registers[8];
    memset( & registers, 0, 8 );
    memset( &e_registers, 0, 8 );

    //registers[2] = 0x00ff & count;
    //e_registers[2] = 0x00ff & ( count >> 8 );

    registers[6] = 0x40;
    registers[7] = READ_MULTIPLE_EXT;

    set_sector_number_ext( registers, e_registers, starting_sector );
    return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * Executes a Read Verify Sector command.  This command is identical to Read Sector except no
 * data is transfered.  This ensures that the sector can in fact be read.  This is a 28 bit command.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The starting sector address to read from.  Values over 0x0fffffff 
 * will be ignored as this is only a 28 bit command.
 * @param count [u_char] The number of sectors to read.  
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_verify_sector( int fd, u_long sector, u_char count ){
    u_char registers[8];
    memset(&registers, 0, 8 );
    registers[2] = count;
    set_sector_number( registers, sector );
    registers[7] = READ_VERIFY_SECTOR;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * Executes a Read Verify Sector Ext command.  This command is identical to Read Sector Ext
 * except no data is transfered.  This ensures that the sector can in fact be read.  This is a 48
 * bit command.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param count [u_short] the number of sectors to read.
 * @param sector [long] The sector to start reading from.  Since this is a 48 bit command 
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_read_verify_sector_ext( int fd, u_short count, uint64_t sector ){
    u_char registers[8];
    u_char e_registers[8];
    memset( registers, 0, 8 );
    memset( e_registers, 0, 8 );
    registers[2] = 0x00ff & count;
    e_registers[2] = 0x00ff & ( count >> 8 );
    set_sector_number_ext( registers, e_registers, sector );
    registers[7] = READ_VERIFY_SECTOR;
    return ata_raw_fd( fd, registers, e_registers, 0 );
}


/**
 * This executes a SMART DISABLE OPERATIONS command.  This command disables all smart capabilities
 * except SMART ENABLE OPERATIONS.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_smart_disable_operations( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[1] = SMART_DISABLE_OPERATIONS;
    registers[4] = 0x4f;
    registers[5] = 0xc2;
    registers[7] = SMART;
    return ata_raw_fd( fd, registers, 0, 0 );
}

/**
 * This function performs a SMART ENABLE OPERATIONS command.  This command enables all
 * SMART capabilities.  Until this command is run the device will ignore SMART data.  The
 * state od SMART is enabled across power cycles.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_smart_enable_operations( int fd ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[1] = SMART_ENABLE_OPERATIONS;
    registers[4] = 0x4f;
    registers[5] = 0xc2;
    registers[7] = SMART;
    return ata_raw_fd( fd, registers, 0, 0 );
}


/**
 * This function performs a SMART ATTRIBUTE AUTOSAVE command.  This enables or disables the
 * optional the optional aatribute  autosave feature.  
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param enable [int] If the value is 0x01 then this feature is enabled.  All other values
 * are assumed to be disabled.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_smart_attribute_autosave( int fd, int enable ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[1] = SMART_ENABLE_DISABLE_ATTRIBUTE_AUTOSAVE;
    if( enable ) registers[2] = 0xf1;
    else registers[2] = 0x00;
    registers[4] = 0x4f;
    registers[5] = 0xc2;
    registers[7] = SMART;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs a SMART EXECUTE OFFLINE IMMEDIATE.  As stated in
 * the ATA 6 Spec, " This command causes the device to immediately initiate 
 * the optional set of activities that collect SMART data in an off-line 
 * mode and then save this data to the device's non-volatile memory, or 
 * execute a self-diagnostic test routine in either captive or off-line mode."
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param subcommand [u_char] Hex value of sub command to execute.  See
 * Table 51 on Page 256 of the ATA 6 specification.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_smart_execute_off_line_immediate( int fd, u_char subcommand ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[1] = SMART_EXECUTE_OFF_LINE_IMMEDIATE;
    registers[3] = subcommand;
    registers[4] = 0x4f;
    registers[5] = 0xc2;
    registers[7] = SMART;
    return ata_raw_fd( fd, registers, 0, 0 );
}
/**
 * This function performs a SMART READ DATA command.  This will read the SMART
 * data structure from device and return it to the host.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param buff [smart_struc*] Structure to read the SMART data into.  For full
 * description of the buffer see ataraw.h or Page 259 of the ATA 6 spec.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_smart_read_data( int fd, smart_struc* buff ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[1] = SMART_READ_DATA;
    registers[4] = 0x4f;
    registers[5] = 0xc2;
    registers[7] = SMART;
    return ata_raw_fd( fd, registers, 0, (u_char*) buff );
}
/**
 * This function performs a SMART READ LOG.  
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param count [u_char] The number of sectors to read.
 * @param addr [u_char] Specifies which log to be read from the device.
 * @param buffer [u_char] Buffer to read the log into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_smart_read_log( int fd, u_char count, u_char addr, u_char* buffer ){
    u_char registers[8];
    memset( &registers, 0, 8 );
    registers[1] = SMART_READ_DATA;
    registers[2] = count;
    registers[3] = addr;
    registers[4] = 0x4f;
    registers[5] = 0xc2;
    registers[7] = SMART;
    return ata_raw_fd( fd, registers, 0, buffer );
}
int ata_set_max_address_ext( int fd, uint64_t address, u_char volatility ){
    u_char registers[8];
    u_char e_registers[8];
    memset( registers, 0, 8 );
    memset( e_registers, 0, 8 );
    set_sector_number_ext( registers, e_registers, address );
    registers[2] = 0x01 & volatility;
    registers[7] = SET_MAX_ADDRESS_EXT;
    return ata_raw_fd( fd, registers, e_registers, 0 );
}
/**
 * This function performs a WRITE SECTOR(S) command.  This writes the 
 * contents of buffer to a sector of your choosing. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param count [u_char] The number of sectors to write.
 * @param sector [u_long] The 28 bit sector number to start writing to.
 * @param buffer [u_char*] Pointer to the buffer to write to the disk.  This
 * buffer should be of size 512 * count.  Where count is the parameter.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_write_sector( int fd, u_long sector, u_char count, u_char* buffer ){
    u_char registers[8];
    memset( registers, 0, 8 );
    set_sector_number( registers, sector );
    registers[2] = count;
    registers[7] = WRITE_SECTORS;
    return ata_raw_fd( fd, registers, NULL, buffer );
}
/**
 * This function performs a WRITE SECTOR(S) w/o retries (obsolete) command.  This writes the 
 * contents of buffer to a sector of your choosing. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param count [u_char] The number of sectors to write.
 * @param sector [u_long] The 28 bit sector number to start writing to.
 * @param buffer [u_char*] Pointer to the buffer to write to the disk.  This
 * buffer should be of size 512 * count.  Where count is the parameter.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_write_sector_no_retry( int fd, u_long sector, u_char count, u_char* buffer ){
    u_char registers[8];
    memset( registers, 0, 8 );
    set_sector_number( registers, sector );
    registers[2] = count;
    registers[7] = WRITE_SECTORS_NO_RETRY;
    return ata_raw_fd( fd, registers, NULL, buffer );
}
/**
 * This function performs a WRITE LONG command.  This writes the 
 * contents of buffer to a sector of your choosing. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [u_long] The 28 bit sector number to start writing to.
 * @param buffer [u_char*] Pointer to the buffer to write to the disk.  This
 * buffer should be of size 512 * count.  Where count is the parameter.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_write_long( int fd, u_long sector, u_char* buffer ){
    u_char registers[8];
    memset( registers, 0, 8 );
    set_sector_number( registers, sector );
    registers[2] = 0x01;
    registers[6] = 0xE0 | registers[6];
    registers[7] = WRITE_LONG;
    return ata_raw_fd( fd, registers, NULL, buffer );
}
/**
 * This function performs a WRITE SECTOR(S) EXT command.  This writes the contents
 * of buffer to a sector of your choosing.  This command is identical to
 * ata_write_sector except this is the ext version.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [uint64_t] The 48 bit sector number to start writing to.
 * @param count [u_short] The number of sectors to write.
 * @param buffer [u_char*] Pointer to the buffer to write to the disk.  This
 * buffer should be of size 512 * count.  Where count is the parameter.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_write_sector_ext( int fd, uint64_t sector, u_short count, u_char* buffer ){
    u_char registers[8];
    u_char e_registers[8];
    memset( registers, 0, 8 );
    memset( e_registers, 0, 8 );
    registers[2] = 0xff & count;
    e_registers[2] = 0xff & (count>>8);
    set_sector_number_ext( registers, e_registers, sector );
    registers[7] = WRITE_SECTOR_EXT;
    return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * This function performs a WRITE STREAM EXT command.  This writes the contents
 * of buffer to a sector of your choosing.  This command is identical to
 * ata_write_sector except this is the ext version.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [uint64_t] The 48 bit sector number to start writing to.
 * @param count [u_short] The number of sectors to write.
 * @param buffer [u_char*] Pointer to the buffer to write to the disk.  This
 * buffer should be of size 512 * count.  Where count is the parameter.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_write_stream_ext( int fd, uint64_t sector, u_short count, u_char* buffer ){
    u_char registers[8];
    u_char e_registers[8];
    memset( registers, 0, 8 );
    memset( e_registers, 0, 8 );
    registers[2] = 0xff & count;
    e_registers[2] = 0xff & (count>>8);
    set_sector_number_ext( registers, e_registers, sector );
    registers[7] = WRITE_STREAM_EXT;
    return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * This function performs a WRITE MULTIPLE command.  This writes the 
 * contents of buffer (potentially multiple sectors) to a sector(s) of 
 * your choosing. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param count [u_char] The number of sectors to write.
 * @param sector [u_long] The 28 bit sector number to start writing to.
 * @param buffer [u_char*] Pointer to the buffer to write to the disk.  This
 * buffer should be of size 512 * count.  Where count is the parameter.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_write_multiple( int fd, u_long sector, u_char count, u_char* buffer ){
    u_char registers[8];
    memset( registers, 0, 8 );
    set_sector_number( registers, sector );
    registers[2] = count;
    registers[7] = WRITE_MULTIPLE;
    return ata_raw_fd( fd, registers, NULL, buffer );
}
/**
 * This function performs a WRITE MULTIPLE EXT command.  This writes the 
 * contents of buffer (potentially multiple sectors) to a sector(s) of 
 * your choosing.  This command is identical to ata_write_multiple except 
 * this is the ext version.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [uint64_t] The 48 bit sector number to start writing to.
 * @param count [u_short] The number of sectors to write.
 * @param buffer [u_char*] Pointer to the buffer to write to the disk.  This
 * buffer should be of size 512 * count.  Where count is the parameter.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_write_multiple_ext( int fd, uint64_t sector, u_short count, u_char* buffer ){
    u_char registers[8];
    u_char e_registers[8];
    memset( registers, 0, 8 );
    memset( e_registers, 0, 8 );
    registers[2] = 0xff & count;
    e_registers[2] = 0xff & (count>>8);
    set_sector_number_ext( registers, e_registers, sector );
    registers[7] = WRITE_MULTIPLE_EXT;
    return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * This function performs a WRITE MULTIPLE FUA EXT command.  This writes the 
 * contents of buffer (potentially multiple sectors) to a sector(s) of 
 * your choosing.  This command is identical to ata_write_multiple except 
 * this is the ext version.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [uint64_t] The 48 bit sector number to start writing to.
 * @param count [u_short] The number of sectors to write.
 * @param buffer [u_char*] Pointer to the buffer to write to the disk.  This
 * buffer should be of size 512 * count.  Where count is the parameter.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_write_multiple_fua_ext( int fd, uint64_t sector, u_short count, u_char* buffer ){
    u_char registers[8];
    u_char e_registers[8];
    memset( registers, 0, 8 );
    memset( e_registers, 0, 8 );
    registers[2] = 0xff & count;
    e_registers[2] = 0xff & (count>>8);
    set_sector_number_ext( registers, e_registers, sector );
    registers[7] = WRITE_MULTIPLE_FUA_EXT;
    return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * This function is not a proper ATA command.  However, it will change the
 * maximum LBA of a device using the device configuration overlay.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector [uint64_t] The maximum user adressable sector to set on
 * the device.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_dco_set_max_lba( int fd, uint64_t sector ){
    dco_struct dco;
    int result;
    int i;

    //get the existing dco struct from the drive
    //and then simply change the lba.  this way no erroneous
    //side effects occur.

    result = ata_device_configuration_identify( fd, (u_char*)&dco );
    if( result != ATA_RAW_OK ){
        return result;			/* identify failed, so return */
    }
    for( i = 0; i < 4; i++) dco.max_lba[i] = 0xffff & (sector >> (16*i) );

    return ata_dco_set( fd, &dco );	/* ata_dco_set will compute checksum */
}
/**
 * Ths function performs a READ DMA EXT command.  This will use DMA
 * to transfer back one sector from the disk.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 *
 */
int ata_read_dma_ext( int fd, uint64_t address, u_char* buffer ){
	u_char registers[8];
	u_char e_registers[8];
	memset( registers, 0, 8 );
	memset( e_registers, 0, 8 );
	registers[2] = 0x01;
	set_sector_number_ext( registers, e_registers, address );
	registers[7] = READ_DMA_EXT;
	return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * This function performs a READ DMA QUEUED EXT command.  This will use DMA
 * to transfer back one sector from the disk.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 *
 */
int ata_read_dma_queued_ext( int fd, uint64_t address, u_char* buffer ){
	u_char registers[8];
	u_char e_registers[8];
	memset( registers, 0, 8 );
	memset( e_registers, 0, 8 );
	registers[1] = 0x01; // sector count
	registers[2] = 0x00; // Tag value
	//registers[2] = 0x08; // Tag value
	set_sector_number_ext( registers, e_registers, address );
	registers[7] = READ_DMA_QUEUED_EXT;
	return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * This function performs a READ FPDMA QUEUED command.  This will use DMA
 * to transfer back one sector from the disk.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 *
 */
int ata_read_fpdma_queued( int fd, uint64_t address, u_char* buffer ){
	u_char registers[8];
	u_char e_registers[8];
	memset( registers, 0, 8 );
	memset( e_registers, 0, 8 );
	registers[1] = 0x01; // sector count
	registers[2] = 0x00; // Tag value
	//registers[2] = 0x08; // Tag value
	set_sector_number_ext( registers, e_registers, address );
	registers[6] = 0x40;
	registers[7] = READ_FPDMA_QUEUED;
	return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * This function performs a READ STREAM DMA EXT command.  This will use DMA
 * to transfer back one sector from the disk.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 *
 */
int ata_read_stream_dma_ext( int fd, uint64_t address, u_char* buffer ){
	u_char registers[8];
	u_char e_registers[8];
	memset( registers, 0, 8 );
	memset( e_registers, 0, 8 );
	registers[2] = 0x01;
	set_sector_number_ext( registers, e_registers, address );
	registers[7] = READ_STREAM_DMA_EXT;
	return ata_raw_fd( fd, registers, e_registers, buffer );
}

/**
 * Writes a sector using DMA.  This is the function wrapper for
 * for a WRITE DMA command.
 */
int ata_write_dma( int fd, u_long sector, u_char count, u_char* buffer ){
    u_char registers[8];
    memset( registers, 0, 8 );
    set_sector_number( registers, sector );
    registers[2] = count;
    registers[7] = WRITE_DMA;
    return ata_raw_fd( fd, registers, NULL, buffer );
}
int ata_write_dma_no_retry( int fd, u_long sector, u_char count, u_char* buffer ){
    u_char registers[8];
    memset( registers, 0, 8 );
    set_sector_number( registers, sector );
    registers[2] = count;
    registers[7] = WRITE_DMA_NO_RETRY;
    return ata_raw_fd( fd, registers, NULL, buffer );
}



/**
 * This function performs a WRITE DMA QUEUED command.  This will use DMA
 * to write one sector to disk.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 *
 */
int ata_write_dma_queued( int fd, u_long sector, u_char* buffer ){
    u_char registers[8];
    memset( registers, 0, 8 );
    registers[1] = 0x01; // sector count
    registers[2] = 0;  // Tag value (0-31)
    registers[6] = 0x40;
    registers[7] = WRITE_DMA_QUEUED;
    set_sector_number( registers, sector );
    return ata_raw_fd( fd, registers, NULL, buffer );
}

/**
 * This function performs a WRITE DMA QUEUED EXT command.  This will use DMA
 * to write one sector to disk.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 *
 */
int ata_write_dma_queued_ext( int fd, uint64_t address, u_char* buffer ){
	u_char registers[8];
	u_char e_registers[8];
	memset( registers, 0, 8 );
	memset( e_registers, 0, 8 );
	registers[1] = 0x01; // sector count
	registers[2] = 0x00; // Tag value
	//registers[2] = 0x08; // Tag value
	set_sector_number_ext( registers, e_registers, address );
	registers[7] = WRITE_DMA_QUEUED_EXT;
	return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * This function performs a WRITE FPDMA QUEUED command.  This will use DMA
 * to write one sector to disk.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 *
 */
int ata_write_fpdma_queued( int fd, uint64_t address, u_char* buffer ){
	u_char registers[8];
	u_char e_registers[8];
	memset( registers, 0, 8 );
	memset( e_registers, 0, 8 );
	registers[1] = 0x01; // sector count
	registers[2] = 0x00; // Tag value
	//registers[2] = 0x08; // Tag value
	set_sector_number_ext( registers, e_registers, address );
	registers[6] = 0x40;
	registers[7] = WRITE_FPDMA_QUEUED;
	return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * This function performs a WRITE DMA QUEUED FUA EXT command.  This will use DMA
 * to write one sector to disk.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 *
 */
int ata_write_dma_queued_fua_ext( int fd, uint64_t address, u_char* buffer ){
	u_char registers[8];
	u_char e_registers[8];
	memset( registers, 0, 8 );
	memset( e_registers, 0, 8 );
	registers[1] = 0x01; // sector count
	registers[2] = 0x00; // Tag value
	//registers[2] = 0x08; // Tag value
	set_sector_number_ext( registers, e_registers, address );
	registers[7] = WRITE_DMA_QUEUED_FUA_EXT;
	return ata_raw_fd( fd, registers, e_registers, buffer );
}

int ata_write_dma_ext( int fd, uint64_t sector, u_short count, u_char* buffer ){
    u_char registers[8];
    u_char e_registers[8];
    memset( registers, 0, 8 );
    memset( e_registers, 0, 8 );
    registers[2] = 0xff & count;
    e_registers[2] = 0xff & (count>>8);
    set_sector_number_ext( registers, e_registers, sector );
    registers[7] = WRITE_DMA_EXT;
    return ata_raw_fd( fd, registers, e_registers, buffer );
}
int ata_write_stream_dma_ext( int fd, uint64_t sector, u_short count, u_char* buffer ){
    u_char registers[8];
    u_char e_registers[8];
    memset( registers, 0, 8 );
    memset( e_registers, 0, 8 );
    registers[2] = 0xff & count;
    e_registers[2] = 0xff & (count>>8);
    set_sector_number_ext( registers, e_registers, sector );
    registers[7] = WRITE_STREAM_DMA_EXT;
    return ata_raw_fd( fd, registers, e_registers, buffer );
}
int ata_write_dma_fua_ext( int fd, uint64_t sector, u_short count, u_char* buffer ){
    u_char registers[8];
    u_char e_registers[8];
    memset( registers, 0, 8 );
    memset( e_registers, 0, 8 );
    registers[2] = 0xff & count;
    e_registers[2] = 0xff & (count>>8);
    set_sector_number_ext( registers, e_registers, sector );
    registers[7] = WRITE_DMA_FUA_EXT;
    return ata_raw_fd( fd, registers, e_registers, buffer );
}
/**
 * This function performs a WRITE VERIFY command.  This is similar to a 
 * WRITE SECTOR(S) command, except that each sector is verified before the
 * command completes.
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param count [u_char] The number of sectors to write.
 * @param sector [u_long] The 28 bit sector number to start writing to.
 * @param buffer [u_char*] Pointer to the buffer to write to the disk.  This
 * buffer should be of size 512 * count.  Where count is the parameter.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */
int ata_write_verify( int fd, u_long sector, u_char count, u_char* buffer ){
    u_char registers[8];
    memset( registers, 0, 8 );
    set_sector_number( registers, sector );
    registers[2] = count;
    registers[7] = WRITE_VERIFY;
    return ata_raw_fd( fd, registers, NULL, buffer );
}
int ata_security_disable_password( int fd, u_char password[32], u_char password_select ){
    u_char registers[8];
    security_password sp;
    memset( registers, 0, sizeof( registers ) );
    memset( &sp, 0, sizeof( security_password) );
    sp.control_word = password_select;
    set_password( &sp, password );
    registers[7] = SECURITY_DISABLE_PASSWORD_ATA;
    return ata_raw_fd( fd, registers, NULL, (u_char*)&sp );
	
}
int ata_security_erase_prepare( int fd ){
    u_char registers[8];
    memset( registers, 0, 8 );
    registers[7] = SECURITY_ERASE_PREPARE_ATA;
    return ata_raw_fd( fd, registers, NULL, NULL );
}

int ata_security_freeze_lock( int fd ){
    u_char registers[8];
    memset( registers, 0, 8 );
    registers[7] = SECURITY_FREEZE_LOCK_ATA;
    return ata_raw_fd( fd, registers, NULL, NULL );
}



//contol info = bit0 (1 Master Password, 0 User Password) bit 1( 0 Normal Erase, 1 Enhanced Erase )
int ata_security_erase_unit( int fd, u_char password[32], u_char control_info){
    u_char registers[8];
    security_password sp;
    memset( registers, 0, sizeof( registers ) );
    memset( &sp, 0, sizeof( security_password) );
    set_password( &sp, password );
    sp.control_word = control_info;
    registers[7] = SECURITY_ERASE_UNIT_ATA;
    //call the prepare here to help out
    ata_security_erase_prepare( fd );
    return ata_raw_fd( fd, registers, NULL, (u_char*)&sp );
}


int ata_security_set_password( int fd, u_char password[32], u_char password_select, word revision_code){
    u_char registers[8];
    security_password sp;
    memset( registers, 0, sizeof( registers ) );
    memset( &sp, 0, sizeof( security_password) );
    set_password( &sp, password );
    sp.control_word = password_select;
    sp.master_rev = revision_code;
    registers[7] = SECURITY_SET_PASSWORD_ATA;
    return ata_raw_fd( fd, registers, NULL, (u_char*)&sp );
}


int ata_security_unlock( int fd, u_char password[32], u_char password_select ){
    u_char registers[8];
    security_password sp;
    memset( registers, 0, sizeof( registers ) );
    memset( &sp, 0, sizeof( security_password) );
    set_password( &sp, password );
    sp.control_word = password_select;
    registers[7] = SECURITY_UNLOCK_ATA;
    return ata_raw_fd( fd, registers, NULL, (u_char*)&sp );
}



int ata_device_reset( int fd ){
    u_char registers[8];
    memset( registers, 0, 8 );
    registers[7] = DEVICE_RESET;
    return ata_raw_fd( fd, registers, NULL, NULL );
}

/**
 * Request that a specified (1 sector) logical block be transfered from 
 * a SCSI device using the READ 6 command defined in the T10 SBC-2. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The sector address to read from.
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_read_6( int fd, uint64_t sector_number, u_char* buffer )
{
    u_char cmdblk[6];
    int result = ATA_RAW_ERR;
    
    memset( cmdblk, 0, 6 );
    cmdblk[0] = READ_6;
    set_scsi_sector_number( cmdblk, sector_number, 6);
    cmdblk[4] = 0x1; // number of sectors to read.
    result = scsi_raw_fd( fd, cmdblk, 6, buffer, SECTOR_SIZE );
    return result;
}

/**
 * Request that a specified (1 sector) logical block be transfered from 
 * a SCSI device using the READ 10 command defined in the T10 SBC-2. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The sector address to read from.
 * will be ignored as this is only a 28 bit command.
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_read_10( int fd, uint64_t sector_number, u_char* buffer )
{
    u_char cmdblk[10];
    int result = ATA_RAW_ERR;
    
    memset( cmdblk, 0, 10 );
    cmdblk[0] = READ_10;
    set_scsi_sector_number( cmdblk, sector_number, 10);
    cmdblk[8] = 0x1; // number of sectors to read.
    result = scsi_raw_fd( fd, cmdblk, 10, buffer, SECTOR_SIZE );
    return result;
}

/**
 * Request that a specified (1 sector) logical block be transfered from 
 * a SCSI device using the READ 12 command defined in the T10 SBC-2. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The sector address to read from.
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_read_12( int fd, uint64_t sector_number, u_char* buffer )
{
    u_char cmdblk[12];
    int result = ATA_RAW_ERR;
    
    memset( cmdblk, 0, 12 );
    cmdblk[0] = READ_12;
    set_scsi_sector_number( cmdblk, sector_number, 12);
    cmdblk[9] = 0x1; // number of sectors to read.
    result = scsi_raw_fd( fd, cmdblk, 12, buffer, SECTOR_SIZE );
    return result;
}

/**
 * Request that a specified (1 sector) logical block be transfered from 
 * a SCSI device using the READ 16 command defined in the T10 SBC-2. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The sector address to read from.
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_read_16( int fd, uint64_t sector_number, u_char* buffer )
{
    u_char cmdblk[16];
    int result = ATA_RAW_ERR;
    
    memset( cmdblk, 0, 16 );
    cmdblk[0] = READ_16;
    set_scsi_sector_number( cmdblk, sector_number, 16);
    cmdblk[13] = 0x1; // number of sectors to read.
    result = scsi_raw_fd( fd, cmdblk, 16, buffer, SECTOR_SIZE );
    return result;
}

/**
 * Request that a specified (1 sector) logical block be transfered from 
 * a SCSI device using the READ 32 command defined in the T10 SBC-2. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The sector address to read from.
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_read_32( int fd, uint64_t sector_number, u_char* buffer )
{
    u_char cmdblk[32];
    int result = ATA_RAW_ERR;
    
    memset( cmdblk, 0, 32 );
    cmdblk[0] = READ_32; 
    cmdblk[7] = 0x18;    // additional cdb length
    cmdblk[9] = 0x09;    // service action
    set_scsi_sector_number( cmdblk, sector_number, 32);
    cmdblk[31] = 0x01; // number of sectors to read.
    result = scsi_raw_fd( fd, cmdblk, 32, buffer, SECTOR_SIZE );
    return result;
}

/**
 * Request that the buffer contents be written to the specified lba by use
 * of the scsi WRITE 6 command as defined in the T10 SBC-2. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The sector address to read from.
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_write_6( int fd, uint64_t sector_number, u_char* buffer )
{
    u_char cmdblk[6];
    int result = ATA_RAW_ERR;
    
    memset( cmdblk, 0, 6 );
    cmdblk[0] = WRITE_6;
    set_scsi_sector_number( cmdblk, sector_number, 6);
    cmdblk[4] = 0x1; // number of sectors to be written.
    result = scsi_raw_fd( fd, cmdblk, 6, buffer, SECTOR_SIZE );
    return result;
}

/**
 * Request that the buffer contents be written to the specified lba by use
 * of the scsi WRITE 10 command as defined in the T10 SBC-2. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The sector address to read from.
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_write_10( int fd, uint64_t sector_number, u_char* buffer )
{
    u_char cmdblk[10];
    int result = ATA_RAW_ERR;
    memset( cmdblk, 0, 10 );
    cmdblk[0] = WRITE_10;
    set_scsi_sector_number( cmdblk, sector_number, 10);
    cmdblk[8] = 0x1; // number of sectors to be written.
    result = scsi_raw_fd( fd, cmdblk, 10, buffer, SECTOR_SIZE );
    return result;
}

/**
 * Request that the buffer contents be written to the specified lba by use
 * of the scsi WRITE 12 command as defined in the T10 SBC-2. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The sector address to read from.
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_write_12( int fd, uint64_t sector_number, u_char* buffer )
{
    u_char cmdblk[12];
    int result = ATA_RAW_ERR;
    memset( cmdblk, 0, 12 );
    cmdblk[0] = WRITE_12;
    set_scsi_sector_number( cmdblk, sector_number, 12);
    cmdblk[9] = 0x1; // number of sectors to be written.
    result = scsi_raw_fd( fd, cmdblk, 12, buffer, SECTOR_SIZE );
    return result;
}

/**
 * Request that the buffer contents be written to the specified lba by use
 * of the scsi WRITE 16 command as defined in the T10 SBC-2. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The sector address to read from.
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_write_16( int fd, uint64_t sector_number, u_char* buffer )
{
    u_char cmdblk[16];
    int result = ATA_RAW_ERR;
    memset( cmdblk, 0, 16 );
    cmdblk[0] = WRITE_16;
    set_scsi_sector_number( cmdblk, sector_number, 16);
    cmdblk[13] = 0x1; // number of sectors to be written.
    result = scsi_raw_fd( fd, cmdblk, 16, buffer, SECTOR_SIZE );
    return result;
}

/**
 * Request that the buffer contents be written to the specified lba by use
 * of the scsi WRITE 32 command as defined in the T10 SBC-2. 
 * @param fd [int] File descriptor for the open device (i.e. /dev/hda).
 * @param sector_number [u_int] The sector address to read from.
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_write_32( int fd, uint64_t sector_number, u_char* buffer )
{
    u_char cmdblk[32];
    int result = ATA_RAW_ERR;
    memset( cmdblk, 0, 32 );
    cmdblk[0] = WRITE_32;
    cmdblk[7] = 0x18; // additional CDB length
    cmdblk[9] = 0x0b; // service action
    set_scsi_sector_number( cmdblk, sector_number, 32);
    cmdblk[31] = 0x01; // number of sectors to be written.
    result = scsi_raw_fd( fd, cmdblk, 32, buffer, SECTOR_SIZE );
    return result;
}

/**
 * Issue the scsi INQUIRY command (see the T10 SPC-3), reading the 1st 
 * 36 bytes of INQUIRY data into the provided data buffer.
 * @param fd [int] File descriptor for the open device (i.e. /dev/sda).
 * @param buffer [u_char*] Pointer to the buffer to be read into.
 * @return int returns ATA_RAW_SCS on success.  A negative number is returned on failure.
 */

int scsi_inquiry( int fd, u_char* buffer )
{
    u_char cmdblk[6];
    int result = ATA_RAW_ERR;
    memset( cmdblk, 0, 6 );
    cmdblk[0] = INQUIRY;
    cmdblk[4] = 0x24; // set ALLOCATION LENGTH to 36 bytes
    result = scsi_raw_fd( fd, cmdblk, 6, buffer, 36 );
    return result;
}

/** Print a sector in XXD format */
void ata_print_sector( unsigned char * buffer )
{
    int offset,i;
    for(offset=0;offset<512;offset+=16){
	printf("%04x: ",offset);
	for( i = 0; i<16; i+=2){
	    printf("%02x",buffer[offset+i+1]);
	    printf("%02x",buffer[offset+i]);
	    putchar(' ');
	}
	putchar(' ');
	putchar(' ');

	for( i = 0; i<16; i++){
	    if(isprint(buffer[offset+i])) putchar(buffer[offset+i]);
	    else putchar('.');
	}
	putchar('\n');
    }
}

/** Print two sectors in XXD format */
void ata_print_long_sector( unsigned char * buffer )
{
    int offset,i;
    for(offset=0;offset<1024;offset+=16){
	printf("%04x: ",offset);
	for( i = 0; i<16; i+=2){
	    printf("%02x",buffer[offset+i+1]);
	    printf("%02x",buffer[offset+i]);
	    putchar(' ');
	}
	putchar(' ');
	putchar(' ');

	for( i = 0; i<16; i++){
	    if(isprint(buffer[offset+i])) putchar(buffer[offset+i]);
	    else putchar('.');
	}
	putchar('\n');
    }
}

/** Print buffer content in XXD format. Note: if the buffer size is not 
 *  divisible by 16, the "remainder" (mod 16) will not be printed.  */
void ata_print_xxd( unsigned char * buffer, int bytes )
{
    int offset,i;
    int nbytes = bytes / 16 * 16;

    for(offset=0;offset<nbytes;offset+=16){
	printf("%04x: ",offset);
	for( i = 0; i<16; i+=2){
	    printf("%02x",buffer[offset+i+1]);
	    printf("%02x",buffer[offset+i]);
	    putchar(' ');
	}
	putchar(' ');
	putchar(' ');

	for( i = 0; i<16; i++){
	    if(isprint(buffer[offset+i])) putchar(buffer[offset+i]);
	    else putchar('.');
	}
	putchar('\n');
    }
}

/**
 * Check if the 48-bit feature set (e.g., EXT commands) is supported.
 * return 0 if 48-bit commands are supported, 1 if they're not, and -1 on
 * error.
 */
int chk_48bit_feature_support(int fd){
  unsigned char* buffer = calloc( 512,1 );

  /* pull back the IDENTIFY DEVICE info. */ 
  if( ata_identify_device( fd, buffer ) == ATA_RAW_OK ){
#define words(x) ((buffer[x*2])+(buffer[(x*2)+1]<<8))
    if ( (words(83)>>8 )& 0x04 ){
      //printf("48-bit Address Feature Set:     SUPPORTED\n"); 
      return 0;
    } else {
      //printf("48-bit Address Feature Set:     NOT SUPPORTED\n"); 
      return 1;
    }
  }
  return ATA_RAW_ERR;
}

/**
 * Issue the ATA IDENTIFY DEVICE command to the open fd.  If the command
 * succeeds, assume the device is an ATA device.  If the command fails, 
 * assume it's not (e.g., it's a SCSI device).
 * @return [int] returns ATA_RAW_ATA_DEVICE if fd refers to an ATA device, 
 * ATA_RAW_SCSI_DEVICE if fd refers to a SCSI device, and 
 * ATA_RAW_INVALID_DEVICE otherwise (note: currently, ATA_RAW_INVALID_DEVICE
 * will only be returned if the ioctl call in scsi_in() fails.  this is a 
 * problem b/c it doesn't take into account the sg_io_hdr_t.info field or 
 * any returned sense data.  TODO: update scsi_in() to check the info field
 * and sense data.
 */
int chk_device_type(int fd){
  unsigned char* buffer = calloc( 512,1 );

  int result = ata_identify_device( fd, buffer );
  /* purge the buffer so we can reuse it */
  memset( buffer, 0, 512 );  
  //if( ata_identify_device( fd, buffer ) == ATA_RAW_OK )
  //printf("result: %d\n", result);
  if( result == ATA_RAW_OK ){
    return ATA_RAW_ATA_DEVICE;
  } else if ( scsi_inquiry( fd, buffer ) == ATA_RAW_OK ){
    return ATA_RAW_SCSI_DEVICE;
  } else {
    return ATA_RAW_INVALID_DEVICE;
  }
}

/*****************************************************************
Compute elapsed time and close a log file
*****************************************************************/
void log_close (time_t from /* time program started running */)
{
  time_t till; /* time program finished (actually time log_close called) */
  unsigned long	et, /* elapsed time (program run time: seconds of wall clock) */
    tmin, /* total minutes running */
    min, /* minutes for elapsed time after whole hours deducted */
    sec, /* seconds of elapsed run time after hours & minutes */
    hours; /* hours of elapsed time */

  time(&till); /* get current time */
  et = till - from; /* elapsed time in seconds */
  tmin = et/60; /* elapsed time in minutes */
  sec = et%60; /* fraction of last minute in seconds */
  hours = tmin/60; /* hours elapsed time */
  min = tmin%60; /* fraction of last hour in whole minutes */
   
  printf ("run start %s",ctime(&from));
  printf ("run finish %s",ctime(&till));

  printf ("elapsed time %lu:%lu:%lu\n",hours,min,sec);
  printf ("Normal exit\n\n"); 
}
