char *ATARAW_C_ID = "@(#) ataraw.c Linux Version 1.2 support lib created 12/13/11 at 08:42:56\n";

/*
 * The software provided here is released by the National
 * Institute of Standards and Technology (NIST), an agency of
 * the U.S. Department of Commerce, Gaithersburg MD 20899,
 * USA.  The software bears no warranty, either expressed or
 * implied. NIST does not assume legal liability nor
 * responsibility for a User's use of the software or the
 * results of such use.
 *
 * Please note that within the United States, copyright
 * protection, under Section 105 of the United States Code,
 * Title 17, is not available for any work of the United
 * States Government and/or for any works created by United
 * States Government employees. User acknowledges that this
 * software contains work which was created by NIST employees
 * and is therefore in the public domain and not subject to
 * copyright.  The User may use, distribute, or incorporate
 * this software provided the User acknowledges this via an
 * explicit acknowledgment of NIST-related contributions to
 * the User's work. User also agrees to acknowledge, via an
 * explicit acknowledgment, that any modifications or
 * alterations have been made to this software before
 * redistribution.
 * --------------------------------------------------------------------
 *
 * Revision History:
 * 2011 May Ben Livelsberger: Extended -- forked file from the 
 *    version in the NPS ataraw-0.2.1 package.
 * 2009 Jan Kyle Sanders & Simson Garfinkel: Extended
 * 2008 Nov Kyle Sanders - Created
 */
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include "ataraw.h"

int ataraw_debug = 0;

/** We use FF termination for the command strings because 0x00 is a valid ATA command. */
#define FULL_TERM 0xff
/** Array representing all supported PIO in commands.*/
static u_char pio_in_cmds[] = { READ_SECTORS, IDENTIFY_DEVICE, CFA_TRANSLATE_SECTOR,
				IDENTIFY_PACKET_DEVICE, READ_BUFFER, READ_MULTIPLE, 
				SMART_READ_DATA, SMART_READ_LOG, READ_LONG, READ_LONG_NO_RETRY, READ_SECTORS_NO_RETRY, FULL_TERM };
/**Array representing all supported PIO out commands.*/ 
static u_char pio_out_cmds[] = { WRITE_SECTORS, WRITE_SECTORS_NO_RETRY, CFA_WRITE_MULTIPLE_WITHOUT_ERASE, 
				 CFA_WRITE_SECTORS_WITHOUT_ERASE,
				 DEVICE_CONFIGURATION_SET, SECURITY_DISABLE_PASSWORD_ATA, 
				 SECURITY_ERASE_UNIT_ATA, SECURITY_SET_PASSWORD_ATA, 
				 SECURITY_UNLOCK_ATA, WRITE_MULTIPLE, WRITE_LONG, WRITE_VERIFY, FULL_TERM};
/**Array representing all supported NON DATA commands.*/
static unsigned char non_data_cmds[] = { CFA_ERASE_SECTOR, CFA_REQUEST_EXTENDED_ERROR_CODE, 
					READ_NATIVE_MAX_ADDRESS, SEEK, SET_MAX_ADDRESS, 
					CHECK_MEDIA_CARD_TYPE, FLUSH_CACHE, GET_MEDIA_STATUS, 
					IDLE, IDLE_IMMEDIATE, MEDIA_EJECT, MEDIA_LOCK, 
					MEDIA_UNLOCK, NOP, SLEEP, STANDBY, STANDBY_IMMEDIATE, 
					READ_VERIFY_SECTOR, SMART_DISABLE_OPERATIONS, 
					SMART_ENABLE_DISABLE_ATTRIBUTE_AUTOSAVE, SMART_ENABLE_OPERATIONS, 
					SMART_EXECUTE_OFF_LINE_IMMEDIATE, SECURITY_ERASE_PREPARE_ATA, 
					SECURITY_FREEZE_LOCK_ATA, FULL_TERM };
/**Array representing all supported PIO in extended commands.*/
static unsigned char pio_in_ext_cmds[] = { READ_SECTOR_EXT, READ_MULTIPLE_EXT, READ_STREAM_EXT, FULL_TERM };
/**Array representing all supported PIO out extended commands.*/
static unsigned char pio_out_ext_cmds[] = {WRITE_SECTOR_EXT, WRITE_MULTIPLE_EXT, WRITE_STREAM_EXT, WRITE_MULTIPLE_FUA_EXT, FULL_TERM };
/** Array representing all supported NON DATA extended commands. */
static unsigned char non_data_ext_cmds[] = {READ_NATIVE_MAX_ADDRESS_EXT, FLUSH_CACHE_EXT, 
					    READ_VERIFY_SECTOR_EXT, SET_MAX_ADDRESS_EXT, 
					    FULL_TERM };
/** Array representing all supported DMA commands. */
static unsigned char ata_dma_in_cmds[] = { READ_DMA, READ_DMA_QUEUED, READ_DMA_NO_RETRY, FULL_TERM };
static unsigned char ata_dma_out_cmds[]={WRITE_DMA, WRITE_DMA_NO_RETRY, WRITE_DMA_QUEUED, FULL_TERM };
/** Array representing all supported DMA EXT commands. */
static u_char ata_dma_in_ext_cmds[] = {READ_DMA_EXT, READ_DMA_QUEUED_EXT, READ_STREAM_DMA_EXT,READ_FPDMA_QUEUED, FULL_TERM };
static u_char ata_dma_out_ext_cmds[] = {WRITE_DMA_EXT, WRITE_DMA_FUA_EXT, WRITE_DMA_QUEUED_EXT, WRITE_DMA_QUEUED_FUA_EXT, WRITE_STREAM_DMA_EXT, WRITE_FPDMA_QUEUED, FULL_TERM };
/* Arrays representing all supported scsi commands. */
static u_char scsi_in_cmds[] = {READ_6, READ_10, READ_12, READ_16, READ_32, INQUIRY, FULL_TERM };
static u_char scsi_out_cmds[] = {WRITE_6, WRITE_10, WRITE_12, WRITE_16, WRITE_32, FULL_TERM };
/** Arrays representing the service action codes for 32 byte cmdblk commands */
static u_char scsi_in_32byte_serviceaction_codes[] = {READ_32_SA, FULL_TERM};
static u_char scsi_out_32byte_serviceaction_codes[] = {WRITE_32_SA, FULL_TERM};
/**
 * Internal function to help with counting and adding the supported commands to
 * a central array.
 * @param arr ( u_char* ) Array to add supported commands.  This is the destination
 * for the function.
 * @param cmds ( u_char* ) Array of supported commands.  This is the source for the
 * function.
 * @param place ( int* )  Pointer to integer value representing current index into
 * arr.  This is updated as each supported command is added to the arr list.
 */
static void add_cmds( unsigned char* arr, const unsigned char* cmds, int* place )
{
	int i = 0;
	for( i = 0;cmds[i] != FULL_TERM; (*place)++, i++) {
		arr[(*place)] = cmds[i];
	}
}

/**
 * Returns a string of commands that are "supported" by the ata_raw 
 * function call.  Supported commands are commands that have been tested
 * and have implemented wrapper functions.  This list of commands 
 * will be terminated with FULL_TERM (0xff).
 * @param ( u_char[256] ) Array to be filled with supported commands.  
 * This will be terminated with 0xff.  It is assumed that this buffer
 * is of size 256.
 */
void get_supported_commands( u_char sup[256])
{
    int size = 0;
    add_cmds( sup, pio_in_cmds, &size );
    add_cmds( sup, pio_in_ext_cmds, &size );
    add_cmds( sup, pio_out_cmds, &size );
    add_cmds( sup, pio_out_ext_cmds, &size );
    add_cmds( sup, non_data_cmds, &size );
    add_cmds( sup, non_data_ext_cmds, &size);
    add_cmds( sup, ata_dma_in_cmds, &size ); 
    add_cmds( sup, ata_dma_out_cmds, &size );
    add_cmds( sup, ata_dma_in_ext_cmds, &size );
    add_cmds( sup, ata_dma_out_ext_cmds, &size );
    sup[size] = FULL_TERM;
}

static unsigned char contains( unsigned char* array, unsigned char val )
{
	int i = 0;
	unsigned char current = 0x00;
	while( current != FULL_TERM )	{
		current = array[i];
		if( current == val ) return TRUE;
		++i;
	}
	return FALSE;
}
/**
 * Internal function that handles all smart commands.  Since smart
 * have two register values these must be processed differently than
 * normal ATA commands.
 * @param fd [int] File description which to work.
 * @param registers [u_char*] pointer to the register values to pass
 * to the device.  Once the function has executed the registers from
 * the device will be reflected in the registers array.  *THIS
 * PARAMETER CHANGES*
 * @param buffer [char*] pointer to the general purpose buffer.  This
 * is used for commands that require data to be sent to the device or
 * retrevied from it.  If a device retrieves data from the device the
 * pace must be allocated before this function is called.
 * @return [int] Returns ATA_RAW_ERR on err or ATA_RAW_OK on success.
 * Other error values may occur base on input parameters.  All error
 * values are negative.
*/
static int smart_handler( const int fd, u_char* registers, u_char* buffer ){
    if( registers[1] == SMART_DISABLE_OPERATIONS ) return ata_pio( fd, registers, FALSE, NON_DATA, FALSE );
    if( registers[1] == SMART_ENABLE_DISABLE_ATTRIBUTE_AUTOSAVE ) return ata_pio( fd, registers, FALSE, NON_DATA, FALSE);
    if( registers[1] == SMART_ENABLE_OPERATIONS ) return ata_pio( fd, registers, FALSE, NON_DATA, FALSE );
    if( registers[1] == SMART_EXECUTE_OFF_LINE_IMMEDIATE ) return ata_pio( fd, registers, FALSE, NON_DATA, FALSE );
    if( registers[1] == SMART_READ_DATA ) return ata_pio( fd, registers, buffer, PIO_IN, FALSE );
    return ATA_RAW_ERR;
}
/**
 * Internal helper function for handling DCO commands.  DCO commands use two
 * registers for commands.  This dffers from normal commands that only
 * use one.
 * @param fd [int] File description which to work.
 * @param registers [u_char*] pointer to the register values to pass
 * to the device.  Once the function has executed the registers from
 * the device will be reflected in the registers array.  *THIS
 * PARAMETER CHANGES*
 * @param buffer [char*] pointer to the general purpose buffer.  This
 * is used for commands that require data to be sent to the device or
 * retrevied from it.  If a device retrieves data from the device the
 * pace must be allocated before this function is called.
 * @return [int] Returns ATA_RAW_ERR on err or ATA_RAW_OK on success.
 * Other error values may occur base on input parameters.  All error 
 * values are negative.
 */
static int dco_handler( const int fd, unsigned char* registers, unsigned char* buffer )
{
    /* dco_handler only works with a DEVICE_CONFIGURATION_IDENTIFY command... */
    if( registers[1] == DEVICE_CONFIGURATION_IDENTIFY) return ata_pio( fd, registers, buffer, PIO_IN, FALSE );
    if( registers[1] == DEVICE_CONFIGURATION_RESTORE) return ata_pio( fd, registers, NULL, NON_DATA, FALSE );
    if( registers[1] == DEVICE_CONFIGURATION_FREEZE_LOCK) return ata_pio( fd, registers, NULL, NON_DATA, FALSE );
    if( registers[1] == DEVICE_CONFIGURATION_SET ) return ata_pio( fd, registers, buffer, PIO_OUT, FALSE );
    return ATA_RAW_BAD_COMMAND_CODE;
}


void debug_print_registers(const char *where,unsigned char *registers,unsigned char *extended_registers)
{
    if(ataraw_debug){
	int i;
	printf("====== %s ======\n",where);
	printf("ata registers:\n");
	for(i=0;i<8;i++){
	    printf("%d: 0x%02x   ",i,registers[i]);
	    if(extended_registers){
		printf("0x%02x",extended_registers[i]);
	    }
	    printf("\n");
	}
	fflush(stdout);
    }
}

void debug_printf(const char *str)
{
  if(ataraw_debug){
    printf("%s\n",str);
  }
}

#ifdef HAVE_SCSI_SG_H
/**
 * Take the values from the ATA register file and put them in the appropriate
 * places of the sg_io IO CTL (which has a pass-through for ATA commands.)  
 * This function simply munges the data into the appropriate places.
 * @param hdr [sg_io_hdr_t*] Pointer to the sg_io_hdr structure to fill.
 * @param registers [u_char*] pointer to the register values to pass
 * to the device.
 * @param xfer [unsigned char] The direction of transfer for the ATA command.
 * This value gets placed directly into the sg_io_hdr.
 * @param sbp [unsigned char*]
 */
static void set_up_sg( sg_io_hdr_t* hdr, unsigned char* registers,
		       unsigned char xfer, unsigned char* cmdp, unsigned char* sbp,
		       int xfer_dev, int size, unsigned char* data_buf, int ext, unsigned char* e_registers )
{
    cmdp[0] = 0x85;
    cmdp[1] = xfer;			//( 4 << 1 );
    if( xfer_dev == SG_DXFER_NONE ) cmdp[2] = 0x20;
    if( xfer_dev == SG_DXFER_TO_DEV ) cmdp[2] = 0x26;
    cmdp[4] = registers[1];
    cmdp[6] = registers[2];
    cmdp[8] = registers[3];
    cmdp[10] = registers[4];
    cmdp[12] = registers[5];
    cmdp[13] = registers[6];
    cmdp[14] = registers[7];
    if( ext ) {
	cmdp[1] |= 1;
	cmdp[3] = e_registers[1];
	cmdp[5] = e_registers[2];
	cmdp[7] = e_registers[3];
	cmdp[9] = e_registers[4];
	cmdp[11] = e_registers[5];
    }
    hdr->interface_id = 'S';
    hdr->cmd_len = 16;
    hdr->mx_sb_len = 32;
    hdr->dxfer_direction = xfer_dev;//SG_DXFER_FROM_DEV;
    hdr->dxfer_len = size;
    hdr->dxferp = data_buf;
    hdr->cmdp = (void*)cmdp;
    hdr->sbp = (void*)sbp;
    hdr->timeout = 12000;
}

/** Gets the results back from the sg_io IOCTL and put them where the register file code expects it.
 */

static void tear_down_sg( sg_io_hdr_t* hdr, unsigned char* registers, int ext, unsigned char* e_registers )
{
    registers[1] = hdr->sbp[11];
    registers[2] = hdr->sbp[13];
    registers[3] = hdr->sbp[15];
    registers[4] = hdr->sbp[17];
    registers[5] = hdr->sbp[19];
    registers[6] = hdr->sbp[20];
    registers[7] = hdr->sbp[21];
    
    if( ext ) {
	e_registers[2] = hdr->sbp[12];
	e_registers[3] = hdr->sbp[14];
	e_registers[4] = hdr->sbp[16];
	e_registers[5] = hdr->sbp[18];
    }
}

static int dma_sg( const int fd, unsigned char* registers, unsigned char* buffer, int ext, unsigned char* e_registers, int xfer )
{

    int i, result = ATA_RAW_ERR;
    u_char cmdp[16];
    u_char sbp[32];
    sg_io_hdr_t io_hdr;
    u_char* data;
    int size = SECTOR_SIZE;

    if(ext &&( registers[2] != 0x00 || e_registers[2] != 0x00) ) {
	size = SECTOR_SIZE * ( registers[2] + (e_registers[2] << 8 ) );
    }
    else {
	if( registers[2] != 0x00 ) size = registers[2] * SECTOR_SIZE;
    }
    data = malloc( size );		/* this has to be malloc'ed, as it depends on request */
    if( data == NULL ) {
	return ATA_RAW_NO_MEM;
    }
    memset( &cmdp, 0, 16 );
    memset( &sbp, 0, 32 );
    memset( &io_hdr, 0, sizeof( sg_io_hdr_t ) );
    memset( data, 0, size );
    if( xfer == ATA_DMA_IN || xfer == ATA_DMA_IN_EXT ){
        set_up_sg( &io_hdr, registers, (6 << 1), cmdp, sbp, SG_DXFER_FROM_DEV, size, data, ext, e_registers );
    }
    else{//doing out
    	memcpy( data, buffer, size );
        set_up_sg( &io_hdr, registers, (6 << 1), cmdp, sbp, SG_DXFER_TO_DEV, size, data, ext, e_registers );
    }
    result = ioctl( fd, SG_IO, &io_hdr );
    if (ataraw_debug){
      printf("io_hdr->status = %x\n", io_hdr.status);
      printf("io_hdr->sb_len_wr = %x\n", io_hdr.sb_len_wr);
      printf("io_hdr->host_status = %x\n", io_hdr.host_status);
      printf("io_hdr->driver_status = %x\n", io_hdr.driver_status);
      printf("io_hdr->resid = %d\n", io_hdr.resid);
      printf("io_hdr->info = %x\n", io_hdr.info);
      printf("io_hdr->sbp = \n");
      //for (i = 0; i < (int) io_hdr.sb_len_wr; i++) printf("%x  ", sbp[i]);
      for (i = 0; i < 8; i++){
	printf("cmd[%d]: %2x\tcmd[%d]: %2x\tcmd[%d]: %2x\tcmd[%d]: %2x\n", i,sbp[i],i+8,sbp[i+8],i+16,sbp[i+16],i+24,sbp[i+24]);
      }
      printf("\n");
      // if this is a read command, print out the data buffer contents
      if( xfer == ATA_DMA_IN || xfer == ATA_DMA_IN_EXT ){ 
	//printf("dma_sg(): \"data\" buffer contents:\n");
	//ata_print_xxd(data, size);
      } 
    }

    /* check the ioctl return status value for a non-zero value */  
    if( result != 0 ){
	result = ATA_RAW_IOCTL_FAIL;
    } else { 
      if( xfer == ATA_DMA_IN || xfer == ATA_DMA_IN_EXT ){ // copy the data buffer if the command was a read
	memcpy( buffer, data, size );
      }	
      tear_down_sg( &io_hdr, registers, ext, e_registers );
      //if ( (io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK ){ // check the info field value to see if the issued command completed abnormally
      //result = ATA_RAW_SGIO_AUX_INFO;
      if ( ( registers[7] & 0x01 ) == 0x01 ){ // check status register's ERR bit
	result = ATA_RAW_ATA_ERR_STATUS;
      }
      else {
	result = ATA_RAW_OK;
      }
    }

    free( data );
    return result;
}

static int out_sg( const int fd, unsigned char* registers, unsigned char* buffer, int ext, unsigned char* extended_registers )
{
	
        int i, result = ATA_RAW_ERR;
	unsigned char cmdp[16];
	unsigned char sbp[32];
	sg_io_hdr_t io_hdr;
	unsigned char* data;
	int size = SECTOR_SIZE;
	if(ext &&( registers[2] != 0x00 || extended_registers[2] != 0x00) ) {
		size = SECTOR_SIZE * ( registers[2] + (extended_registers[2] << 8 ) );
	}
	else {
		if( registers[2] != 0x00 ) size = registers[2] * SECTOR_SIZE;

	}
	data = malloc( size );
	if( data == NULL ) {
		free( data );
		return ATA_RAW_NO_MEM;
	}
	memset( &cmdp, 0, 16 );
	memset( &sbp, 0, 32 );
	memset( &io_hdr, 0, sizeof( sg_io_hdr_t ) );
	memset( data, 0, size );
	memcpy( data, buffer, size );
	set_up_sg( &io_hdr, registers, (5 << 1), cmdp, sbp, SG_DXFER_TO_DEV, size, data, ext, extended_registers );
	result = ioctl( fd, SG_IO, &io_hdr );
	if (ataraw_debug){
	  printf("io_hdr->status = %x\n", io_hdr.status);
	  printf("io_hdr->sb_len_wr = %x\n", io_hdr.sb_len_wr);
	  printf("io_hdr->host_status = %x\n", io_hdr.host_status);
	  printf("io_hdr->driver_status = %x\n", io_hdr.driver_status);
	  printf("io_hdr->resid = %d\n", io_hdr.resid);
	  printf("io_hdr->info = %x\n", io_hdr.info);
	  printf("io_hdr->sbp = ");
	  for (i = 0; i < (int) io_hdr.sb_len_wr; i++) printf("%x  ", sbp[i]);
	  debug_print_registers("out_sg() post execute", registers, extended_registers);
	}
	//if( result == 0  && ((io_hdr.info & SG_INFO_OK_MASK) == SG_INFO_OK)) {
	if( result == 0 ) {
	  //if( result == 0 ) {
	  tear_down_sg( &io_hdr, registers, ext, extended_registers );	
	  if ( ( registers[7] & 0x01 ) == 0x01 ){ // check status register's ERR bit
	    result = ATA_RAW_ATA_ERR_STATUS;
	  }	  
	  else {
	    result = ATA_RAW_OK;
	  }
	}
	else {
	  result = ATA_RAW_IOCTL_FAIL;
	}
	free( data );
	return result;
}

static int pio_non_sg( const int fd, unsigned char* registers, int ext, unsigned char* e_registers )
{

	int result = ATA_RAW_ERR;
	unsigned char cmdp[16];
	unsigned char sbp[32];
	sg_io_hdr_t io_hdr;
	memset( cmdp, 0, 16 );
	memset( sbp, 0, 32 );
	memset( &io_hdr, 0, sizeof( sg_io_hdr_t ) );
	set_up_sg( &io_hdr, registers, (3 << 1), cmdp, sbp, SG_DXFER_NONE, 0x00, 0x00, ext, e_registers );
	result = ioctl( fd, SG_IO, &io_hdr );
	//if( result == 0  && ((io_hdr.info & SG_INFO_OK_MASK) == SG_INFO_OK)) {
	if( result == 0 ){
	    tear_down_sg( &io_hdr, registers, ext, e_registers );	
	    if ( ( registers[7] & 0x01 ) == 0x01 ){ // check status register's ERR bit
	      result = ATA_RAW_ATA_ERR_STATUS;
	    }	  
	    else {
	      result = ATA_RAW_OK;
	    }
	    debug_print_registers("pio_non_sg success",registers,e_registers);
	} else result = ATA_RAW_IOCTL_FAIL;
	return result;
}

static int pio_in_sg( const int fd, unsigned char* registers, unsigned char* buffer, int ext, unsigned char* e_registers )
{
  int i, result = ATA_RAW_ERR;
  unsigned char cmdp[16];
  unsigned char sbp[32];
  sg_io_hdr_t io_hdr;
  unsigned char* data;
  int size = SECTOR_SIZE;
  int cmd_code = registers[7];

  if( ext && (registers[2] != 0x00 || e_registers[2] != 0x00 ) ) {
    size = SECTOR_SIZE * ( registers[2] + (e_registers[2] <<8 ) );
  }
  else {
    if( registers[2] != 0x00 ) size = registers[2] * SECTOR_SIZE;
  }
  data = malloc( size );
  if( data == NULL )	{
    return ATA_RAW_NO_MEM;
  }
  memset( &cmdp, 0, 16 );
  memset( &sbp, 0, 32 );
  memset( &io_hdr, 0, sizeof( sg_io_hdr_t ) );
  memset( data, 0, size );

  set_up_sg( &io_hdr, registers, (4 << 1), cmdp, sbp, SG_DXFER_FROM_DEV, size, data, ext, e_registers );

  if (ataraw_debug){
    //printf("data b4:\n");
    //ata_print_xxd( data, size );
  }
  result = ioctl( fd, SG_IO, &io_hdr );
  if (ataraw_debug){
    printf("io_hdr->status = %x\n", io_hdr.status);
    printf("io_hdr->sb_len_wr = %x\n", io_hdr.sb_len_wr);
    printf("io_hdr->host_status = %x\n", io_hdr.host_status);
    printf("io_hdr->driver_status = %x\n", io_hdr.driver_status);
    printf("io_hdr->resid = %d\n", io_hdr.resid);
    printf("io_hdr->info = %x\n", io_hdr.info);
    printf("io_hdr->sbp = \n");
    //for (i = 0; i < (int) io_hdr.sb_len_wr; i++) printf("%x  ", sbp[i]);
    for (i = 0; i < 8; i++){
      printf("cmd[%d]: %2x\tcmd[%d]: %2x\tcmd[%d]: %2x\tcmd[%d]: %2x\n", i,sbp[i],i+8,sbp[i+8],i+16,sbp[i+16],i+24,sbp[i+24]);
    }
    printf("\n");
    printf("pio_in_sg() \"data\" buffer contents:\n");

    ata_print_xxd(data, size);
    //printf("\ndata after:\n");
    //ata_print_xxd( data, size );
  }

  /* check the ioctl return status value for a non-zero value */  
  if( result != 0 ){
    result = ATA_RAW_IOCTL_FAIL;
  } else { 
    memcpy( buffer, data, size );
    tear_down_sg( &io_hdr, registers, ext, e_registers );
/*     if ( (io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK ){ // check the info field value to see if the issued command completed abnormally */
/*       result = ATA_RAW_SGIO_AUX_INFO; */
/*     } else { */
/*       result = ATA_RAW_OK; */
/*     } */
    if ( ( registers[7] & 0x01 ) == 0x01 ){ // check status register's ERR bit
      result = ATA_RAW_ATA_ERR_STATUS;
    }	  
    else {
      /* The ATA IDENTIFY DEVICE command fails when we issue it to a SCSI drive, 
       * but the status register's ERR bit doesn't always get set.  So use the 
       * sgio_hdr_t.info field to check for errors on the IDENTIFY DEVICE command.
       */
      if ( ( cmd_code == IDENTIFY_DEVICE ) && ( (io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK ) )
	result = ATA_RAW_SGIO_AUX_INFO; 
      else
	result = ATA_RAW_OK;
    }
  }

  free( data );
  return result;
}

#endif


#ifdef HAVE_LINUX_HDREG_H
static int dma_tf( const int fd, unsigned char* registers, unsigned char* buffer, int ext, unsigned char* e_registers)
{

    ide_task_request_t request;
    unsigned char* buff;
    int result = ATA_RAW_ERR;
    int size = 0;
    int i = 0;

    memset( &request, 0, sizeof( ide_task_request_t ) );
    if( ext && (registers[2] != 0x00 || e_registers[2] != 0 ) )	{
	size = SECTOR_SIZE * (registers[2] + (e_registers[2] << 8 ) ); 
    }
    else {
	if( registers[2] != 0x00 ) size = registers[2] * SECTOR_SIZE;
    }
    for( i=0;i<8;i++) request.io_ports[i] = registers[i];
    if( ext ) {
	for( i=0;i<8;i++) request.hob_ports[i] = e_registers[i];
    }
    buff = malloc( sizeof( ide_task_request_t ) + size );
    if( buff == NULL )	{
	return ATA_RAW_NO_MEM;
    }
    request.data_phase = TASKFILE_IN_DMA;
    request.req_cmd = IDE_DRIVE_TASK_IN;
    request.out_size = size;

    memset( buff, 0, sizeof( ide_task_request_t ) + size );
    memcpy( buff, &request, sizeof(ide_task_request_t) );
    result = ioctl( fd, HDIO_DRIVE_TASKFILE, buff );
    if( result == 0 )	{
	memcpy( &request, buff, sizeof( ide_task_request_t ) );
	memcpy( buffer, buff+sizeof( ide_task_request_t ), size );
	for(i=0;i<8;i++) registers[i] = request.io_ports[i];
	if( ext )		{
	    for( i=0;i<8;i++) e_registers[i] = request.hob_ports[i];
	}
    }
    free( buff );
    return result;
}

/** PIO output with the task file.
 */

static int out_tf( const int fd, unsigned char* registers, unsigned char* buffer, int ext, unsigned char* extended_registers )
{
    ide_task_request_t request;
    unsigned char* buff;
    int result = ATA_RAW_ERR;
    int size = 0;
    int i = 0;
    memset( &request, 0, sizeof( ide_task_request_t ) );
    if( ext && (registers[2] != 0x00 || extended_registers[2] != 0 ) ) 	{
	size = SECTOR_SIZE * (registers[2] + (extended_registers[2] << 8 ) ); 
    }
    else	{
	if( registers[2] != 0x00 ) size = registers[2] * SECTOR_SIZE;
    }
    for( i=0;i<8;i++) request.io_ports[i] = registers[i];
    if( ext )	{
	for( i=0;i<8;i++) request.hob_ports[i] = extended_registers[i];
    }
    buff = malloc( sizeof( ide_task_request_t ) + size );
    if( buff == NULL )	{
	return ATA_RAW_NO_MEM;
    }
    //debug_printf("out_tf() b");
    request.data_phase = TASKFILE_OUT;
    request.req_cmd = IDE_DRIVE_TASK_OUT;
    request.out_size = size;
    //debug_printf("out_tf() b.5");
    memset( buff, 0, sizeof( ide_task_request_t ) + size );
    //debug_printf("out_tf() b.6");
    memcpy( buff, &request, sizeof(ide_task_request_t) );
    //debug_printf("out_tf() b.7");
    //printf("size: %d\nsizeof(ide_task_request_t): %d\n",size, sizeof(ide_task_request_t));
    memcpy( buff+sizeof(ide_task_request_t), buffer, size );

    //debug_printf("out_tf() c");
    result = ioctl( fd, HDIO_DRIVE_TASKFILE, buff );
    //debug_printf("out_tf() d");
    if( result == 0 )	{
	memcpy( &request, buff, sizeof( ide_task_request_t ) );
	for(i=0;i<8;i++) registers[i] = request.io_ports[i];
	if( ext )		{
	    for( i=0;i<8;i++) extended_registers[i] = request.hob_ports[i];
	}
    }
    free( buff );
    return result;
}
static int pio_non_tf( const int fd, unsigned char* registers, int ext, unsigned char* e_registers )
{
	int i;
	ide_task_request_t request;
	int result = ATA_RAW_ERR;
	memset( &request, 0, sizeof( ide_task_request_t ) );
	request.data_phase = TASKFILE_NO_DATA;
	request.req_cmd = IDE_DRIVE_TASK_NO_DATA;
	for( i=0; i<8; i++) request.io_ports[i] = registers[i];
	result = ioctl( fd, HDIO_DRIVE_TASKFILE, &request );
	if( ext )
	{
		for( i=0; i<8;i++) request.hob_ports[i] = e_registers[i];
	}
	if( result == 0 )
	{
		for( i=0; i<8; i++) registers[i] = request.io_ports[i];
		if( ext )
		{
			for( i=0;i<8; i++) e_registers[i] = request.hob_ports[i];
		}
		result = ATA_RAW_OK;
	}else result = ATA_RAW_IOCTL_FAIL;
	return result;
}

/**
 * This method makes a copy of the request so that it can't change in another thread.
 */

static int pio_in_tf( const int fd, unsigned char* registers, unsigned char* buffer, int ext, unsigned char* e_registers )
{
	int i;
	unsigned char* buff;
        ide_task_request_t request;
        int result = ATA_RAW_ERR;
	int size = SECTOR_SIZE;
	if( ext && (registers[2] != 0x00 || e_registers[2] != 0x00 ) )
	{
		size = ( registers[2] + (e_registers[2] << 8 ) ) * SECTOR_SIZE;
	}
	else
	{
	  //size = registers[2] * SECTOR_SIZE * 2;
	  if( registers[2] != 0x00 ) size = registers[2] * SECTOR_SIZE;
	}
	memset( &request, 0, sizeof(request));
	for( i = 0; i<8; i++ ) request.io_ports[i] = registers[i];

	/* check for extended commands */
	if( ext ) {			
		for(i=0; i<8; i++) request.hob_ports[i] = e_registers[i];
	}
        request.data_phase = TASKFILE_IN;
        request.req_cmd = IDE_DRIVE_TASK_IN;
        request.out_size = 0;
        request.in_size = size;
	//request.in_size = 644;

        buff = malloc( sizeof( ide_task_request_t ) + size );
	if( buff == NULL )	{
		return ATA_RAW_NO_MEM;
	}
	memset( buff, 0, sizeof( ide_task_request_t ) + size );
        memcpy( buff, &request, sizeof( request ) ); /* copy over the request */
	if (ataraw_debug){
	  // printf("buff b4:\n");
	  //ata_print_xxd(buff, sizeof( ide_task_request_t ) + size );
	}
        result = ioctl( fd, HDIO_DRIVE_TASKFILE, buff );
	if (ataraw_debug){
	  //printf("buff after:\n");
	  //ata_print_xxd(buff, sizeof( ide_task_request_t ) + size );
	}
        memcpy( &request, buff, sizeof( request ) ); /* copy back the result */
        memcpy( buffer, buff+sizeof( ide_task_request_t ), size ); /* and copy out the buffer */

	for( i = 0; i < 8; i++ ) registers[i] = request.io_ports[i];
        if( ext ) {
		for( i=0; i<8; i++) e_registers[i] = request.io_ports[i];
	}
	free( buff );
        if( result  == 0 ) return ATA_RAW_OK;
        return ATA_RAW_IOCTL_FAIL;
}
#endif

/** Executes a raw ATA command that involves DMA.
 *
 * First tries the task file commands. This works if the "CONFIG_IDE_TASK_IOCTL" option
 * is compiled into the kernel. That's the cleaner way to do it.
 * If this option is not compiled in, the code attempts to use the sg_io taskfile
 * passthrough ioctl.
 *
 * @param fd File description which to work.
 * @param registers register file
 * @param buffer where the data comes from or goes; must be large enough.
 * @param xfer_mode Transfer mode.
 */
int ata_dma( const int fd, unsigned char* registers, unsigned char* buffer, const int xfer_mode,
		       unsigned char* extended_registers )
{
    int result;
    /* cache the original register values locally */
    u_char local_reg[8];
    u_char local_e_reg[8];
    memcpy( local_reg, registers, 8 ); 
    if ((xfer_mode == ATA_DMA_IN_EXT) || (xfer_mode == ATA_DMA_OUT_EXT))
      memcpy( local_e_reg, extended_registers, 8 ); 

    switch( xfer_mode ) {
    case ATA_DMA_IN:
#ifdef HAVE_LINUX_HDREG_H
	result = dma_tf( fd, registers, buffer, FALSE, 0);
	if( result == ATA_RAW_OK) return result;
	//if (ataraw_debug) printf("taskfile-based dma FAILED... re-trying via sg_io ioctl\n");
#endif
#ifdef HAVE_SCSI_SG_H			/* see if we can do scsi_sg ATA passthrough */
	// they might have been changed by a tf call, so copy the orig. values back into the registers
	memcpy( registers, local_reg, 8 ); 
	return dma_sg( fd, registers, buffer, FALSE, 0, ATA_DMA_IN );
#else
	return result;			/* return the original error */
#endif	

    case ATA_DMA_OUT:
#ifdef HAVE_LINUX_HDREG_H
	result = out_tf( fd, registers, buffer, FALSE, 0 );		/* not implemented yet */
	if( result == ATA_RAW_OK) return result;
	//if (ataraw_debug) printf("taskfile-based dma FAILED... re-trying via sg_io ioctl\n");
#endif
#ifdef HAVE_SCSI_SG_H			/* see if we can do scsi_sg ATA passthrough */
	memcpy( registers, local_reg, 8 ); 
	return dma_sg( fd, registers, buffer, FALSE, 0, ATA_DMA_OUT );
#endif
    case ATA_DMA_IN_EXT:
#ifdef HAVE_LINUX_HDREG_H
	result = dma_tf( fd, registers, buffer, TRUE, extended_registers );		/* not implemented yet */
	if( result == ATA_RAW_OK) return result;
	//if (ataraw_debug) printf("taskfile-based dma FAILED... re-trying via sg_io ioctl\n");
#endif
#ifdef HAVE_SCSI_SG_H			/* see if we can do scsi_sg ATA passthrough */
	// they might have been changed by a tf call, so copy the orig. values back into the registers
	memcpy( registers, local_reg, 8 ); 
	memcpy( extended_registers, local_e_reg, 8 ); 
	return dma_sg( fd, registers, buffer, TRUE, extended_registers, ATA_DMA_IN_EXT );
#endif

    case ATA_DMA_OUT_EXT:
#ifdef HAVE_LINUX_HDREG_H
	result = out_tf( fd, registers, buffer, TRUE, extended_registers );		/* not implemented yet */
	debug_print_registers("ata_dma post execute",registers,extended_registers);
	if( result == ATA_RAW_OK) return result;
	//if (ataraw_debug) printf("taskfile-based dma FAILED... re-trying via sg_io ioctl\n");
#endif
#ifdef HAVE_SCSI_SG_H			/* see if we can do scsi_sg ATA passthrough */
	memcpy( registers, local_reg, 8 ); 
	memcpy( extended_registers, local_e_reg, 8 ); 
	return dma_sg( fd, registers, buffer, TRUE, extended_registers, ATA_DMA_OUT_EXT );
#endif
	return ATA_RAW_NO_PASSTHROUGH;	/*  */
    }
    return ATA_RAW_BAD_COMMAND_CODE;			/*  */
}

/** Execute a raw ATA command that uses programmed io (PIO).
 * @param fd FIle descriptor with which to work.
 * @param registers register file
 * @param buffer where the data gets written
 * @param xfer_mode mode for the command; should be PIO_IN, PIO_IN_EXT, PIO_OUT, PIO_OUT_EXT, NON_DATA, or NON_DATA_EXT
 */
int ata_pio( const int fd, unsigned char* registers, unsigned char* buffer, const int xfer_mode, unsigned char* extended_registers )
{
    int result = ATA_RAW_ERR;
    /* cache the original register values locally */
    u_char local_reg[8];
    u_char local_e_reg[8];
    memcpy( local_reg, registers, 8 ); 
    if ((xfer_mode == PIO_IN_EXT) || (xfer_mode == PIO_OUT_EXT) || (xfer_mode == NON_DATA_EXT))
    memcpy( local_e_reg, extended_registers, 8 ); 

    switch( xfer_mode ) {
    case PIO_IN: 
#ifdef HAVE_LINUX_HDREG_H
      //if (ataraw_debug) printf("attempting PIO using pio_in_tf()\n");
      result = pio_in_tf( fd, registers, buffer, FALSE, 0 );
      if( result==ATA_RAW_OK ) return result;
      //if (ataraw_debug) printf("PIO using pio_in_tf() FAILED\n");
#endif
#ifdef HAVE_SCSI_SG_H			/* see if we can do scsi_sg ATA passthrough */
      //if (ataraw_debug) printf("attempting PIO using pio_in_sg()\n");
      // they might have been changed by a tf call, so copy the orig. values back into the registers
      memcpy( registers, local_reg, 8 ); 
      return pio_in_sg( fd, registers, buffer, FALSE, 0 );
#else
 	return result;
#endif

    case PIO_IN_EXT:
#ifdef HAVE_LINUX_HDREG_H
      //if (ataraw_debug) printf("attempting PIO using pio_in_tf()\n");
	result = pio_in_tf( fd, registers, buffer, TRUE, extended_registers );
	if( result == ATA_RAW_OK ) return result;
	//if (ataraw_debug) printf("PIO using pio_in_tf() FAILED\n");
#endif
#ifdef HAVE_SCSI_SG_H			/* see if we can do scsi_sg ATA passthrough */
	//if (ataraw_debug) printf("attempting PIO using pio_in_sg()\n");
	// they might have been changed by a tf call, so copy the orig. values back into the registers
	memcpy( registers, local_reg, 8 ); 
	memcpy( extended_registers, local_e_reg, 8 ); 
	return pio_in_sg( fd, registers, buffer, TRUE, extended_registers );
#else
	return result;
#endif

    case PIO_OUT:
#ifdef HAVE_LINUX_HDREG_H
      result = out_tf(fd, registers, buffer, FALSE, extended_registers );
      debug_print_registers("ata_pio post execute",registers,extended_registers);
      if(result==ATA_RAW_OK) return result;
      //if (ataraw_debug) printf("PIO using out_tf() FAILED\n");
#endif
#ifdef HAVE_SCSI_SG_H			/* see if we can do scsi_sg ATA passthrough */
      // in case they got changed by a tf call, freshen the registers back to their orig. values
      memcpy( registers, local_reg, 8 ); 

	return out_sg(fd, registers, buffer, FALSE, NULL );
#else
	return result;
#endif	

    case PIO_OUT_EXT:
#ifdef HAVE_LINUX_HDREG_H
	result = out_tf(fd, registers, buffer, TRUE, extended_registers );
	if(result==ATA_RAW_OK) return result;
	//if (ataraw_debug) printf("PIO using out_tf() FAILED\n");
#endif
#ifdef HAVE_SCSI_SG_H			/* see if we can do scsi_sg ATA passthrough */
	// they might have been changed by a tf call, so copy the orig. values back into the registers
	memcpy( registers, local_reg, 8 ); 
	memcpy( extended_registers, local_e_reg, 8 ); 
	return out_sg(fd, registers, buffer, TRUE, extended_registers );
#else
	return result;
#endif

    case NON_DATA: 
#ifdef HAVE_LINUX_HDREG_H
	result = pio_non_tf( fd, registers, FALSE, 0 );
	if( result == ATA_RAW_OK ) return result;
	//if (ataraw_debug) printf("PIO using pio_non_tf() FAILED\n");
#endif
#ifdef HAVE_SCSI_SG_H			/* see if we can do scsi_sg ATA passthrough */
      // in case they got changed by a tf call, freshen the registers back to their orig. values
        memcpy( registers, local_reg, 8 ); 
	return pio_non_sg( fd, registers, FALSE, 0 );
#else
	return result;
#endif
    case NON_DATA_EXT:
#ifdef HAVE_LINUX_HDREG_H
	result = pio_non_tf( fd, registers, TRUE, extended_registers );
	if( result==ATA_RAW_OK) return result;
	//if (ataraw_debug) printf("PIO using pio_non_tf() FAILED\n");
#endif
#ifdef HAVE_SCSI_SG_H			/* see if we can do scsi_sg ATA passthrough */
	// they might have been changed by a tf call, so copy the orig. values back into the registers
	memcpy( registers, local_reg, 8 ); 
	memcpy( extended_registers, local_e_reg, 8 ); 
	return pio_non_sg( fd, registers, TRUE, extended_registers );
#else
	return result;
#endif	
    }
    return result;			/* return the error condition */
}

/**
 * Top level call for userspace to send lowlevel commands to ata
 * devices.  If you are using this code it is assumed that you 
 * have an understanding of the ATA spec.
 *
 * NOTE: the ata_raw commands only work for recognized ATA6 commands.
 * If you have a command that you want to execute that isn't recognized,
 * YOU NEED TO KNOW IF IT IS A DMA or PIO command and in which direction
 * the data is being moved.  If you know that, you can use the ata_dma or ata_pio
 * interface, rather than ata_raw.
 *
 * This command is identical to ata_raw except it takes a file descriptor instead
 * of a character array.
 *
 * @param fd ( int ) The file descriptor of the open device.
 *
 * @param registers [u_char*] pointer to the register values to pass
 * to the device.  Once the function has executed the registers from
 * the device will be reflected in the registers array.  *THIS
 * PARAMETER CHANGES*
 *
 * @param extended_registers [u_char*] pointer to extended registers.
 * This is used only when executing EXT commands (48 bit commands).
 * Once this function has finished this will reflect the current state
 * of device register values after the command.  *THIS PARAMETER
 * CHANGES*
 *
 * @param buffer [char*] pointer to the general purpose buffer.  This
 * is used for commands that require data to be sent to the device or
 * retrevied from it.  If a device retrieves data from the device the
 * pace must be allocated before this function is called.
 *
 * @return [int] Returns ATA_RAW_ERR on err or ATA_RAW_OK on success.
 * Other error values may occur base on input parameters.  All error
 * values are negative.
 */
int ata_raw_fd( const int fd, unsigned char* registers, unsigned char* extended_registers, unsigned char* buffer )
{
  //debug_print_registers("ata_raw_fd setup",registers,extended_registers);
    //dco case
    if( registers[7] == DEVICE_CONFIGURATION ) return dco_handler( fd, registers, buffer );
    //SMART case
    if( registers[7] == SMART ) return smart_handler( fd, registers, buffer );
    //is pio-in
    if( contains( pio_in_cmds, registers[7] ) == TRUE ) {
      return ata_pio( fd, registers, buffer, PIO_IN, NULL );
    }
    if( contains( pio_in_ext_cmds, registers[7]) == TRUE ) return ata_pio( fd, registers, buffer, PIO_IN_EXT, extended_registers);
    //is pio-out
    if( contains( pio_out_cmds, registers[7] ) == TRUE ) {
      return ata_pio( fd, registers, buffer, PIO_OUT,NULL );
    }
    if( contains( pio_out_ext_cmds, registers[7] ) == TRUE ) return ata_pio( fd, registers, buffer, PIO_OUT_EXT, extended_registers );
    //is non-data
    if( contains( non_data_cmds, registers[7] ) == TRUE ) return ata_pio( fd, registers, buffer, NON_DATA, NULL );
    if( contains( non_data_ext_cmds, registers[7] ) == TRUE ) return ata_pio( fd, registers, buffer, NON_DATA_EXT, extended_registers );
    //dma
    if( contains( ata_dma_in_cmds, registers[7] ) == TRUE ) return ata_dma( fd, registers, buffer, ATA_DMA_IN, NULL ); 
    if( contains( ata_dma_out_cmds, registers[7] ) == TRUE ) return ata_dma( fd, registers, buffer, ATA_DMA_OUT, NULL );
    //dma ext
    if( contains( ata_dma_in_ext_cmds, registers[7] ) == TRUE ) return ata_dma( fd, registers, buffer, ATA_DMA_IN_EXT, extended_registers ); 
    if( contains( ata_dma_out_ext_cmds, registers[7] ) == TRUE ) return ata_dma( fd, registers, buffer, ATA_DMA_OUT_EXT, extended_registers ); 

    return ATA_RAW_BAD_COMMAND_CODE;
}

/**
 * Top level call for userspace to send lowlevel commands to ata
 * devices.  If you are using this code it is assumed that you 
 * have an understanding of the ATA spec.
 *
 * NOTE: the ata_raw commands only work for recognized ATA6 commands.
 * If you have a command that you want to execute that isn't recognized,
 * YOU NEED TO KNOW IF IT IS A DMA or PIO command and in which direction
 * the data is being moved.  If you know that, you can use the ata_dma or ata_pio
 * interface, rather than ata_raw.
 *
 * @param ataDevice [char*] Name of the device to send the command to (i.e. /dev/hda). 
 *
 * @param registers [u_char*] pointer to the register values to pass
 * to the device.  Once the function has executed the registers from
 * the device will be reflected in the registers array.  *THIS
 * PARAMETER CHANGES*
 *
 * @param extended_registers [u_char*] pointer to extended registers.
 * This is used only when executing EXT commands (48 bit commands).
 * Once this function has finished this will reflect the current state
 * of device register values after the command.  *THIS PARAMETER
 * CHANGES*
 *
 * @param buffer [char*] pointer to the general purpose buffer.  This
 * is used for commands that require data to be sent to the device or
 * retrevied from it.  If a device retrieves data from the device the
 * pace must be allocated before this function is called.
 *
 * @return [int] Returns ATA_RAW_ERR on err or ATA_RAW_OK on success.
 * Other error values may occur base on input parameters.  All error
 * values are negative.
 */
int ata_raw( const char*  ataDevice, unsigned char* registers, unsigned char* extended_registers, unsigned char* buffer )
{
	int fd = open( ataDevice, O_RDWR );				//file descriptor for device
	if( fd <= 0 ) return ATA_RAW_BAD_FILE_FD;
	int result = ata_raw_fd( fd, registers, extended_registers, buffer );		
	close( fd );
	return result;	
}

static int scsi_in( const int fd, unsigned char* cmdblk, int cmdblk_size, unsigned char* buffer, int xfer_len ){
  int i, result = ATA_RAW_ERR;
  unsigned char sbp[32];
  sg_io_hdr_t io_hdr;
  unsigned char* data;

  data = malloc( xfer_len );
  if( data == NULL )	{
    return ATA_RAW_NO_MEM;
  }

  /* zero out all our data structures */
  memset( &sbp, 0, 32 );
  memset( &io_hdr, 0, sizeof( sg_io_hdr_t ) );
  memset( data, 0, xfer_len );

  io_hdr.interface_id = 'S';
  io_hdr.cmd_len = cmdblk_size;
  io_hdr.mx_sb_len = 32;
  io_hdr.dxfer_direction = SG_DXFER_FROM_DEV;
  io_hdr.dxfer_len = xfer_len;
  io_hdr.dxferp = data;
  io_hdr.cmdp = cmdblk;
  io_hdr.sbp = sbp;
  io_hdr.timeout = 12000;

  result = ioctl( fd, SG_IO, &io_hdr );

  if (ataraw_debug){
    printf("io_hdr->status = %x\n", io_hdr.status);
    printf("io_hdr->sb_len_wr = %x\n", io_hdr.sb_len_wr);
    printf("io_hdr->host_status = %x\n", io_hdr.host_status);
    printf("io_hdr->driver_status = %x\n", io_hdr.driver_status);
    printf("io_hdr->resid = %d\n", io_hdr.resid);
    printf("io_hdr->info = %x\n", io_hdr.info);
    printf("io_hdr->sbp = \n");
    //for (i = 0; i < (int) io_hdr.sb_len_wr; i++) printf("%x  ", sbp[i]);
    for (i = 0; i < 8; i++){
      printf("cmd[%d]: %2x\tcmd[%d]: %2x\tcmd[%d]: %2x\tcmd[%d]: %2x\n", i,sbp[i],i+8,sbp[i+8],i+16,sbp[i+16],i+24,sbp[i+24]);
    }
    printf("\n");
    //printf("scsi_in() \"data\" buffer contents:\n");

    //ata_print_xxd(data, xfer_len);
    //printf("\ndata after:\n");
    //ata_print_xxd( data, xfer_len );
  }

  /* check the ioctl return status value for a non-zero value */  
  if( result != 0 ){
    result = ATA_RAW_IOCTL_FAIL;
  } else { 
    memcpy( buffer, data, xfer_len );
    //tear_down_sg( &io_hdr, registers, ext, e_registers );
    /*     if ( (io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK ){ // check the info field value to see if the issued command completed abnormally */
    /*       result = ATA_RAW_SGIO_AUX_INFO; */
    /*     } else { */
    /*       result = ATA_RAW_OK; */
    /*     } */
    //if ( ( registers[7] & 0x01 ) == 0x01 ){ // check status register's ERR bit
    //result = ATA_RAW_ATA_ERR_STATUS;
    //}	  
    //else {
    result = ATA_RAW_OK;
    //}
  }

  free( data );

  return result;
}

static int scsi_out( const int fd, unsigned char* cmdblk, int cmdblk_size, unsigned char* buffer, int xfer_len ){
  int i, result = ATA_RAW_ERR;
  unsigned char sbp[32];
  sg_io_hdr_t io_hdr;
  unsigned char* data;

  data = malloc( xfer_len );
  if( data == NULL )	{
    return ATA_RAW_NO_MEM;
  }

  /* zero out all our data structures */
  memset( &sbp, 0, 32 );
  memset( &io_hdr, 0, sizeof( sg_io_hdr_t ) );
  memset( data, 0, xfer_len );

  /* copy the data we want to write into a local buffer */
  memcpy( data, buffer, xfer_len );

  io_hdr.interface_id = 'S';
  io_hdr.cmd_len = cmdblk_size;
  io_hdr.mx_sb_len = 32;
  io_hdr.dxfer_direction = SG_DXFER_TO_DEV;
  io_hdr.dxfer_len = xfer_len;
  io_hdr.dxferp = data;
  io_hdr.cmdp = cmdblk;
  io_hdr.sbp = sbp;
  io_hdr.timeout = 12000;

  result = ioctl( fd, SG_IO, &io_hdr );

  if (ataraw_debug){
    printf("io_hdr->status = %x\n", io_hdr.status);
    printf("io_hdr->sb_len_wr = %x\n", io_hdr.sb_len_wr);
    printf("io_hdr->host_status = %x\n", io_hdr.host_status);
    printf("io_hdr->driver_status = %x\n", io_hdr.driver_status);
    printf("io_hdr->resid = %d\n", io_hdr.resid);
    printf("io_hdr->info = %x\n", io_hdr.info);
    printf("io_hdr->sbp = \n");
    //for (i = 0; i < (int) io_hdr.sb_len_wr; i++) printf("%x  ", sbp[i]);
    for (i = 0; i < 8; i++){
      printf("cmd[%d]: %2x\tcmd[%d]: %2x\tcmd[%d]: %2x\tcmd[%d]: %2x\n", i,sbp[i],i+8,sbp[i+8],i+16,sbp[i+16],i+24,sbp[i+24]);
    }
    printf("\n");
    printf("scsi_out() \"data\" buffer contents:\n");

    ata_print_xxd(data, xfer_len);
    //printf("\ndata after:\n");
    //ata_print_xxd( data, xfer_len );
  }

  /* check the ioctl return status value for a non-zero value */  
  if( result != 0 ){
    //printf("result: %d\n",result);
    result = ATA_RAW_IOCTL_FAIL;
  } else { 
    //tear_down_sg( &io_hdr, registers, ext, e_registers );
    /*     if ( (io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK ){ // check the info field value to see if the issued command completed abnormally */
    /*       result = ATA_RAW_SGIO_AUX_INFO; */
    /*     } else { */
    /*       result = ATA_RAW_OK; */
    /*     } */
    //if ( ( registers[7] & 0x01 ) == 0x01 ){ // check status register's ERR bit
    //result = ATA_RAW_ATA_ERR_STATUS;
    //}	  
    //else {
    result = ATA_RAW_OK;
    //}
  }

  free( data );

  return result;  
}

/**
 * Top level call for userspace to send lowlevel commands to ata
 * devices.  If you are using this code it is assumed that you 
 * have an understanding of the ATA spec.
 *
 * NOTE: the ata_raw commands only work for recognized ATA6 commands.
 * If you have a command that you want to execute that isn't recognized,
 * YOU NEED TO KNOW IF IT IS A DMA or PIO command and in which direction
 * the data is being moved.  If you know that, you can use the ata_dma or ata_pio
 * interface, rather than ata_raw.
 *
 * This command is identical to ata_raw except it takes a file descriptor instead
 * of a character array.
 *
 * @param fd ( int ) The file descriptor of the open device.
 *
 * @param registers [u_char*] pointer to the register values to pass
 * to the device.  Once the function has executed the registers from
 * the device will be reflected in the registers array.  *THIS
 * PARAMETER CHANGES*
 *
 * @param extended_registers [u_char*] pointer to extended registers.
 * This is used only when executing EXT commands (48 bit commands).
 * Once this function has finished this will reflect the current state
 * of device register values after the command.  *THIS PARAMETER
 * CHANGES*
 *
 * @param buffer [char*] pointer to the general purpose buffer.  This
 * is used for commands that require data to be sent to the device or
 * retrevied from it.  If a device retrieves data from the device the
 * pace must be allocated before this function is called.
 *
 * @param xfer_len [int] number of bytes to be transfered.
 *
 * @return [int] Returns ATA_RAW_ERR on err or ATA_RAW_OK on success.
 * Other error values may occur base on input parameters.  All error
 * values are negative.
 */
int scsi_raw_fd( const int fd, unsigned char* cmdblk, int cmdblk_size, unsigned char* buffer, int xfer_len )
{
  // consider building scsi_in_32byte_serice-action_codes[] array?
  if( cmdblk[0] == 0x7f ){
    if ( contains( scsi_in_32byte_serviceaction_codes, cmdblk[9] ) == TRUE){
      return scsi_in( fd, cmdblk, cmdblk_size, buffer, xfer_len );      
    }
    else 
      return scsi_out( fd, cmdblk, cmdblk_size, buffer, xfer_len );
  }
  if( contains( scsi_in_cmds, cmdblk[0] ) == TRUE ) return scsi_in( fd, cmdblk, cmdblk_size, buffer, xfer_len );
  if( contains( scsi_out_cmds, cmdblk[0] ) == TRUE ) return scsi_out( fd, cmdblk, cmdblk_size, buffer, xfer_len );
  
  return ATA_RAW_BAD_COMMAND_CODE;
}
