;****************************************************
; The software provided here is released by the National
; Institute of Standards and Technology (NIST), an agency of
; the U.S. Department of Commerce, Gaithersburg MD 20899,
; USA.  The software bears no warranty, either expressed or
; implied. NIST does not assume legal liability nor
; responsibility for a User's use of the software or the
; results of such use.
; 
; Please note that within the United States, copyright
; protection, under Section 105 of the United States Code,
; Title 17, is not available for any work of the United
; States Government and/or for any works created by United
; States Government employees. User acknowledges that this
; software contains work which was created by NIST employees
; and is therefore in the public domain and not subject to
; copyright.  The User may use, distribute, or incorporate
; this software provided the User acknowledges this via an
; explicit acknowledgment of NIST-related contributions to
; the User's work. User also agrees to acknowledge, via an
; explicit acknowledgment, that any modifications or
; alterations have been made to this software before
; redistribution.
;****************************************************
;	baddisk -- program to simulate disk I/O errors on
;	disks attached to an Intel PC via the BIOS Int 13
;	Author: Dr. James R. Lyle
;	Usage: baddisk drive cyl head sec command error_code
;		drive indicates the hard drive,
;			Drive numbers start at 80
;		cyl head sec is the disk address of a target sector
;		command is the command code of a target command
;			(2 is read, 3 is write, 10 is long read)
;		error_code is an "error code" to return if "command"
;			is requested for sector "cyl/head/sec" on "drive"
;****************************************************
			.model	tiny
			version	m510
.code
cr			equ		0ah
lf			equ		0dh
doscall		equ		21h
read_cmd 	equ		2h
write_cmd	equ		3h
is_active	equ		0d0h
report_cmd	equ		0d1h
report_cnt	equ		0d2h
report_max	equ		0d3h
dos_tsr		equ		3100h
get_i13		equ		3513h
set_i13		equ		2513h
print_cmd	equ		9h
cr			equ		0ah
lf			equ		0Dh
;****************************************************
; Print a message
;****************************************************
print 		MACRO	message
			lea		dx,message
			mov		ah,9
			int		21h
			ENDM
;****************************************************
; Format the number in value as 5 ASCII char in field
; value is a word
;****************************************************
format_word MACRO value,field
			push	ax bx cx dx
			mov		ax,value
			lea		bx,field
			call	dectochar
			pop		dx cx bx ax
			ENDM
;****************************************************
; Format the number in value as 5 ASCII char in field
; value is a byte
;****************************************************
format_byte MACRO value,field
			push	ax bx cx dx
			mov		al,value
			mov		ah,0
			lea		bx,field
			call	dectochar
			pop		dx cx bx ax
			ENDM

;****************************************************
decode	MACRO	from ; from is a byte register with the digit pair
	mov	AH,from	; move digits to AH
	shr	AX,4	; shift left digit into low bits & right digit into AL
	shr	AL,4	; move right digit into low bits
	or	AX,3030H ; convert to ASCII, leave in AX
	ENDM
get_date MACRO	string	; get a date, put in string
	mov		AH,04H	; setup for date BIOS service
	int		1AH	; get the date mm in DH
	decode		DH	; decode the month
	mov		string,AH	; save month in string
	mov		string+1,AL
	decode		DL		; day is in DL
	mov		string+3,AH	; save day of month
	mov		string+4,AL
	decode		CL		; year is in CL
	mov		string+6,AH	; save year digits
	mov		string+7,AL
	ENDM

get_time MACRO	string			; get current time from BIOS
	mov		AH,02H		; setup for time service
	int		1AH		; get the time
	decode		CH		; hours is in CH
	mov		string,AH	; save hours
	mov		string+1,AL
	decode		CL		; minutes is in CL
	mov		string+3,AH	; save minutes
	mov		string+4,AL
	decode		DH		; seconds is in DH
	mov		string+6,AH	; save seconds
	mov		string+7,AL
	ENDM
;****************************************************
;****************************************************
			org		100h
start:
			jmp		install
;****************************************************
;	DISK REQUEST
;		AL	# of sectors to read/write
;		AH	Command, 2 = read, 3 = write
;		BX	Data area offset
;		CL	Sector (6 bits) cylinder (2 high bits)
;		CH	Cylinder (low 8 bits)
;		DL	Drive ID
;		DH	Head
;		ES  Data area segment
;****************************************************
do_operation:
			mov		ax,axx
			mov		bx,bxx
			mov		cx,cxx
			mov		dx,dxx
			pop		ds
			jmp		CS:bios_old
new_service:
			push	ds				; set ds <= cs
			push	cs
			pop		ds
			mov		axx,ax			; save registers
			mov		bxx,bx
			mov		cxx,cx
			mov		dxx,dx
;****************************************************
;			TEST: is this the target I/O request
;****************************************************
;****************************************************
;		Test if disk matches
			cmp		dl,disk
			jne		do_operation		; no match, op not to target
;		Test if command matches
			cmp		ah,command			; is this the target command?
			jne		do_operation		; nope, do the op
;****************************************************
;	disk and operation match, now check if disk addr in range
	call	decode_disk_request	
	mov		bx,cylinder
	mov		cx,sector
	mov		dx,head
	call	is_fit
	cmp		ah,0
	je		do_operation
;****************************************************
; The disk, operation and address all fit, need to compute
;	adjustment to sector!
	mov		bx,cylinder
	mov		cx,sector
	mov		dx,head
	call	get_n_sectors
;	CL has the offset of the bad sector which is also
;	the number of sectors to read/write to simulate
;	the sector being bad
	mov		sector_offset,cl

			; NEED TO FIX FLAGS ON STACK
			pop		ax		;ds
			pop		bx		; IP
			pop		cx		; CS
			pop		dx		; flags
			or		dx,01	; set carry (error flag)
			push	dx cx bx ax ; put it back

			mov		ax,axx	
			cmp		ah,2
			je		catch_rd
			cmp		ah,3
			je		catch_wt
;			print	got_other
			jmp		restore
catch_rd:
;			print	got_read
			jmp		restore
catch_wt:
;			print	got_write
			jmp		restore

restore:
			mov		ax,axx	; restore registers with
			mov		bx,bxx	; I/O request parameters
			mov		cx,cxx
			mov		dx,dxx
			mov		al,sector_offset	; adjust # sectors
			cmp		al,0
			je		no_need_to_call		; skip zero length op




			pushf
			call	CS:bios_old
			mov		ah,ecode
			pop		ds
			iret
no_need_to_call:
			mov		ah,ecode
			pop		ds
			iret
;***************************.data
;****************************************************
; Resident data area
;****************************************************
got_read	db		'RD: make bad',cr,lf,'$'
got_write	db		'WT: make bad',cr,lf,'$'
got_other	db		'OT: make bad',cr,lf,'$'
bios_old 	label	dword
biosoff 	dw		0
biosseg 	dw		0
disk		db		?		; target disk
command		db 		?		; target command
ecode		db		?		; error code to return
cylinder 	dw		?		; target disk address: cylinder
head		dw		?		; target disk address: head
sector		dw		?		; target disk address: sector
n_sectors	db		?	; # of sectors to read/write
sector_offset db	?	; # of sectors upto the bad one
;			starting disk I/O address
low_cylinder dw		?	; starting cylinder
low_head	db		?	; starting head
low_sector	db		?	; starting sector
;			Maximum heads per cylinder
max_hd		dw		15
;			Disk I/O goes up to (not including) high
high_head	db		?
high_cylinder dw 	?
high_sector	db		?
axx			dw		?	; save register area
bxx			dw		?
cxx			dw		?
dxx			dw		?

;****************************************************
; Procedure to compute the difference between two
; disk addresses
;  low_[sector|head|cylinder] contains base
;  registers contain high value, assumed to be
;  less than 256 sectors higher (max # sectors that
;  can be read by BIOS int 13 2/3 commands
;		bx = cyl, dl = head, cl = sector
;  ax is used as a work area
;
;  RETURN value is CL byte
;
;    Want to compute offset in sectors from base address
;    delta_x is x - x_low, x is [cyl|hd|sec]
;    delta_cyl = cyl - cyl_low ==> diff in cyl
;    if delta_cyl > 0 modify head by adding HPC*delta_cyl
;    to head (HPC is number of heads per cylinder)
;    delta_head = head - hd_low
;    now convert delta_head to sectors by multiplying by 63
;    (the number of sectors per head) and adding the
;    result to sector:
;    sector = sector + 63*delta_head
;    delta_sector = sector - sector_low
;
;  RETURN value is CL byte
;****************************************************
get_n_sectors proc
	;	get offset from low[c/h/s]
	sub		bx,low_cylinder	; cylinders different
	jz		do_head			; no
	mov		ax,0			; set up ax to compute
	mov		ax,max_hd		; delta cylinder*HD per CYL
	inc		ax	; ax is # of heads per cylinder
mod_hd:			; OK I know I could multiply here
	add		dx,ax
	sub		bx,1
	jnz		mod_hd
do_head:
	mov		ax,0
	mov		al,low_head
	sub		dx,ax		; dx is delta head
	mov		al,63		; convert to sectors
	mul		dl
	add		cx,ax		; add sectors from cyl & hd
	mov		ax,0
	mov		al,low_sector
	sub		cx,ax		; compute delta_sector
	ret
	endp	get_n_sectors
;****************************************************
; Procedure to test if low <= addr < high
; Where low, addr and high are disk addresses in
; the form c/h/s
;
; The condition is:
;	(
;		(c > c_low)
;		or
;		((c = c_low)  and  (h > h_low))
;		or
;		((c = c_low) and (h = h_low) and (s >= s_low))
;	)
;	and
;	(
;		(c < c_high)
;		or
;		((c = c_high)  and  (h < h_high))
;		or
;		((c = c_high) and (h = h_high) and (s < s_high))
;	)
;
;
;   RETURN AH = 1 ==> FIT,  AH=0 ==> Outside range
;****************************************************

is_fit proc
;		bx = cyl, dl = head, cl = sector
	cmp	bx,low_cylinder
	jl	no_fit				; c > c_low failed
	cmp	bx,high_cylinder
	jg	no_fit				; c < c_high failed
	; c >= c_low and c <= c_high
	cmp	bx,low_cylinder
	jg	high_test			; c > c_low
	; c == c_low
	cmp	dl,low_head
	ja	high_test			; c = c_low and h > h_low
	jb	no_fit				; c = c_low but h < h_low
							; i.e., cyl ok but hd below
	; h == h_low
	cmp	cl,low_sector		; c = c_low & h = h_low
	jb	no_fit				; but sector below
	; sector is at or above low[c/h/s]
	; now check if sector below high[c/h/s]
high_test:
	; c <= c_high
	cmp	bx,high_cylinder
	jl	fit					; clear fit: c < c_high
	cmp	dl,high_head
	ja	no_fit				; c=c_high & h>h_high
	jb	fit					; c=c_high & h<h_high
	cmp	cl,high_sector		; c=c_high & h=h_high
	jnb	no_fit				; s >= s_high => fail
fit:			; it's a fit!!!
	mov	ah,1
	ret
no_fit:			; sorry, outside the range
	mov	ah,0
	ret
	endp is_fit
;****************************************************
;****************************************************
compute_limit proc
	mov		bx,low_cylinder
	mov		dx,0
	mov		ch,0
	mov		dl,low_head
	mov		ah,0
	add		cx,ax
@@:
	cmp		cx,63	; while (sector > 63)
	jle		@F
	inc		dx
	sub		cx,63
	jmp		@B
@@:
	cmp		dx,max_hd	; while (head > heads per cylinder)
	jle		@F
	inc		bx
	sub		dx,max_hd	; subtract (max_hd + 1)
	dec		dx
	jmp		@B
@@:
	mov		high_cylinder,bx
	mov		high_head,dl
	mov		high_sector,cl
	ret
	endp compute_limit
;****************************************************
;	Compute LOW_{cyl,head,sector} and
;	call compute_limit to set HIGH_{cyl,head,sector} and
;****************************************************
decode_disk_request proc
	mov		low_head,dh	; save the head value
	mov		dx,0		; get cylinder
	mov		dh,cl		; get high order bits
	shr		dh,6		; shift out sector value
	mov		dl,ch		; get low order bits
	mov		low_cylinder,dx
	and		cl,3Fh
	mov		low_sector,cl
	call	compute_limit	; set HIGH_{cyl,head,sector}
	ret
	endp	decode_disk_request
;****************************************************
;.code

install:								; install TSR
;****************************************************
;	Look at command line
;****************************************************
			cmp		byte ptr es:[80h],1
			jle		nopram
			jmp		F
;****************************************************
;	If nothing on command line, print message and exit
;****************************************************
nopram:					;
	print	usage		;
	mov		ah,4Ch		;
	mov		al,0		;
	int		21h			; exit without going TSR
;****************************************************
;	Print error messages, then exit
sector_too_big:
			print	big_sector
			jmp		nopram
sector_zero:
			print	zero_sector
			jmp		nopram
head_too_big:
			print	big_head
			jmp		nopram
;****************************************************
;	Loop to move command line to buffer
;****************************************************
F:
			call		extract_cmd_line
			print	signon

;	Check for valid disk addr: sector [1..max_sector],
;		head [0..max_head]
;	exit if head or sector not valid
;	Warn but allow if cylinder too big

;	Check Sector
			mov		ax,sector			; get requested sector
			cmp		ax,0				; test for zero
			jz		sector_zero
			cmp		ax,max_sector		; test for big sector
			ja		sector_too_big
;	check Head
			mov		ax,max_hd			; get max head value
			cmp		ax,head				; compare requested head
			jb		head_too_big		; too bad
;	check Cylinder
			mov		ax,max_cyl			; get max cyl value
			cmp		ax,cylinder			; compare requested cyl
			jae		@F					; OK

			print	cyl_beyond_bios
@@:

			get_date date		; add date to signoff line
			get_time time		; add time to signoff line
			mov		ax,get_i13			; get DISK BIOS interrupt vector
			int		doscall

			mov		biosoff,bx			; save offset of DISK handler
			mov		biosseg,es			; and segment
		
			lea		dx,new_service		; set address of new disk service
			mov		ax,set_i13			;
			int		doscall

			print	signoff			; print signoff message

			lea		dx,install			; calculate resident part size
			add		dx,110h				; don't forget PSP
			mov		cx,4
			shr		dx,cl				; size is in paragraphs!!!
			mov		ax,dos_tsr			; get ready to TSR
			int		doscall				; Now TSR

dectochar	proc		; Convert value in ax to ASCII
			mov		si,5		; Digit count
			mov		di,4		; offset in ASCII string (ans)
l1:
			mov		dx,0		; clear dx for divide
			mov		cx,10		; set for base 10
			div		cx			; extract rightmost digit to dl
			add		dx,'0'		; convert to ASCII
			mov		[bx+di],dl	; move digit to answer (BX=>output string)
			dec		di			; adjust digit position
			dec		si			; dec digit count
			jnz		l1			; done?
			ret
			endp	dectochar
			
chartodec	proc
			mov		si,0
			mov		ax,0
@@:
			cmp		byte ptr [bx+si],' '
			jne		l2
			inc		si
			jmp		@B
l2:
			mov		cl,[bx+si]
			mov		ch,0
			add		ax,cx
			sub		ax,030h
			inc		si
			cmp		byte ptr [bx+si],' '
			je		l3
			mov		cx,10
			mul		cx
			jmp		l2
l3:
			add		bx,si
			ret
			endp	chartodec
extract_cmd_line proc
			mov		ah,0
			mov		al,byte ptr es:[80h]
			mov		si,81h
			mov		di,0
l4:
			mov		bl,es:[si]
			mov		buff[di],bl
			inc		di
			inc		si
			dec		al
			jnz		l4
			mov		buff[di],' '
			print	buff
;****************************************************
;	scan command line for target
;   return error_code on command that includes
;	cylinder/head/sector in the range of command
;****************************************************
			lea		bx,buff		; get disk id
			inc		bx
			call	chartodec
			sub		ax,80		; Convert decimal 80 to hex 80
			add		ax,80h		; adjust for hard disk
			mov		disk,al
;			call	chartodec	; scanf max head value
			mov		max_hd,ax
			call	chartodec	; scanf cylinder
			mov		cylinder,ax
			call	chartodec	; scanf head
			mov		head,ax
			call	chartodec
			mov		sector,ax	; scanf sector
			call	chartodec
			mov		command,al	; scanf command code
			call	chartodec
			mov		ecode,al	; scanf error code
;****************************************************
;	Print feed back line
;****************************************************
;			mov		ax,max_hd
;			lea		bx,hd_per
;			call	dectochar
			mov		ax,cylinder		; format cylinder
			lea		bx,cyl
			call	dectochar
			mov		ax,head		; format head
			lea		bx,hd
			call	dectochar
			mov		ax,sector		; format sector
			lea		bx,sec
			call	dectochar
			mov		ah,0
			mov		al,disk		; format disk id
			sub		al,80H		; convert HEX to Dec
			add		al,80
			lea		bx,dk
			call	dectochar
			mov		ah,0
			mov		al,command		; format command
			lea		bx,iocmd
			call	dectochar
			mov		al,ecode		; format error code
			mov		ah,0
			lea		bx,code
			call	dectochar

			print	spec
;****************************************************
;	Get Drive parameters	
;****************************************************
			mov		ah,08		; get parameters cmd code
			mov		dh,0		; clear DH
			mov 	dl,disk		; set the drive ID (80|81|...)
			int		13H			; Disk BIOS call
			jnc		@F			; carry flag ==> no such disk

			print	no_such_disk ; print error message
			mov		ah,4CH		; exit without TSR
			mov		al,0
			int		21h			; bye

		@@:	; disk drive OK
; save max heads
			mov		ah,0		; zero high bits
			mov		al,dh		; get max head value
			mov		max_hd,ax	; save max head value
; save max sector
			mov		ah,0		; clear high bits
			mov		al,cl		; get max sector value
			and		al,03Fh		; clear two cyl bits
			mov		max_sector,ax
; save max Cylinder
			mov		al,ch		; get low byte
			mov		ah,cl		; get two high bits
			shr		ah,6		; shift out sector bits
			mov		max_cyl,ax	; save max cyl value
;****************************************************
;	print BIOS geometry
;****************************************************

			mov		ax,max_cyl	; format BIOS cylinder
			lea		bx,cyl_per
			call	dectochar

			mov		ax,max_hd	; format BIOS head
			lea		bx,hd_per
			call	dectochar

			mov		ax,max_sector	; format BIOS sector
			lea		bx,sec_per
			call	dectochar

			print	bios_val

			ret
			endp	extract_cmd_line
;****************************************************
; non-resident data area
;****************************************************

no_such_disk db		'Could not access disk drive (NO TSR SET)',cr,lf,'$'
signon 		db		'Monitor BIOS interrupt 13h (disk service)',cr,lf
			db		??filename,' compiled on '
			db		??date,' at ',??time,cr,lf
 			db		'@(#) Version 3.2 Created 06/05/02 at 13:50:39',cr,lf,'$'
signoff		db		'Now ('
date		db		'99/99/99 at '
time		db		'99:99:99) Going . . .  TSR',cr,lf,'$'
buff		db		'                                      ',cr,lf,'$'
spec		db		'return code '
code		db		'xxxxx on command '
iocmd		db		'xxxxx from disk '
dk			db		'xxxxx',cr,lf,'at address '
cyl			db		'xxxxx/'
hd			db		'xxxxx/'
sec			db		'xxxxx',cr,lf,'$'
usage		db		'Usage: baddisk drive cyl head sec'
			db		' command error_code',cr,lf,'$'
bios_val	db		'Bios disk geometry: '
cyl_per		db		'XXXXX/'
hd_per		db		'HHHHH/'
sec_per		db		'SSSSS',cr,lf,'$'
max_cyl		dw		0
max_sector	dw		0
cyl_beyond_bios db	'NOTE: requested cylinder beyond end of BIOS',cr,lf,'$'
zero_sector		db	'Sector 0 specified, must be at least 1',cr,lf,'$'
big_sector		db	'Sector value too big',cr,lf,'$'
big_head		db	'Head value too big',cr,lf,'$'
			end		start
