/******************************************************************************
The software provided here is released by the National
Institute of Standards and Technology (NIST), an agency of
the U.S. Department of Commerce, Gaithersburg MD 20899,
USA.  The software bears no warranty, either expressed or
implied. NIST does not assume legal liability nor
responsibility for a User's use of the software or the
results of such use.

Please note that within the United States, copyright
protection, under Section 105 of the United States Code,
Title 17, is not available for any work of the United
States Government and/or for any works created by United
States Government employees. User acknowledges that this
software contains work which was created by NIST employees
and is therefore in the public domain and not subject to
copyright.  The User may use, distribute, or incorporate
this software provided the User acknowledges this via an
explicit acknowledgment of NIST-related contributions to
the User's work. User also agrees to acknowledge, via an
explicit acknowledgment, that any modifications or
alterations have been made to this software before
redistribution.
******************************************************************************/ 
static char *SCCS_ID[] = {"@(#) seccopy.cpp Version 3.1 Created 10/11/01 at 12:40:27",
			__DATE__,__TIME__};
/***** Author: Dr. James R. Lyle, NIST/SDCT/SQG ****/
# include <stdio.h>
# include "zbios.h"
# include <time.h>
# include <string.h>
# include <malloc.h>
# include <fcntl.h>
# include <io.h>
/*****************************************************************
Copy a block of sectors from one disk to another disk

*****************************************************************/
/*****************************************************************
Disk Read/write routine: read (or write) the sector at address <a>
if <op> is 0 , then read; if <op> is not zero, then write
put the data in (or get the data from) <buffer>
*****************************************************************/
int disk_io (disk_control_ptr d, chs_addr *a, int op, unsigned char *buffer)
{
	static int	error_code,drive,rd_err,wt_err;
	static unsigned long lba;
	static unsigned char disk_addr_packet [16];
	static unsigned int *dap_count = (unsigned int *)disk_addr_packet,
		*dap_n = (unsigned int *)&disk_addr_packet[2],
		*dap_buffer_offset = (unsigned int *)&disk_addr_packet[4],
		*dap_buffer_segment = (unsigned int *)&disk_addr_packet[6];
	static unsigned long *dap_lba = (unsigned long *)&disk_addr_packet[8],
		*dap_fill = (unsigned long *)&disk_addr_packet[12];
	static unsigned int dap_segment = FP_SEG(disk_addr_packet),
				dap_offset = FP_OFF(disk_addr_packet);


	if (d->drive == 0) return 1;
	if (d->use_bios_x){
/*****************************************************************
Read/write one sector from/to disk d at address a
Using the ATA BIOS extensions
*****************************************************************/
			/* convert a to LBA */
		lba = (a->cylinder*d->disk_max.head + a->head)*63 + a->sector - 1;
		drive = d->drive; /* disk drive to use */
		*dap_count = (int) 16; /* size of <disk address packet> */
		 *dap_n = (int) 1; /* one sector */
		*dap_buffer_offset = FP_OFF(buffer); /* address of data buffer */
		*dap_buffer_segment = FP_SEG(buffer);
		*dap_lba = lba; /* disk address for operation */
		*dap_fill = (long) 0; /* MBZ */
		error_code = 0; rd_err = 0; wt_err = 0;
		if (op == 0){ /* read a sector into buffer */
				asm {
			mov ah,0x42          /* set read command */
			mov dl,BYTE PTR drive  /* set BIOS drive number */
			mov ds,dap_segment    /* set address of command block */
			mov si,dap_offset
			int 0x13             /* read */
			mov BYTE PTR error_code,ah /* save error code */
			jnc	read_ok
		}
			rd_err = 1; read_ok: ;
		}
		else { /* write a sector from buffer */
				asm {
			mov ah,0x43          /* set write command */
			mov	al,0x0
			mov dl,BYTE PTR drive  /* set BIOS drive number */
			mov ds,dap_segment    /* set address of command block */
			mov si,dap_offset
			int 0x13             /* do the write thing */
			mov BYTE PTR error_code,ah
			jnc	write_ok
		}
			wt_err = 1; write_ok:;
		}
	}
	else {
/*****************************************************************
Read or write a sector from or to disk d at address a
using the ATA Legacy BIOS
*****************************************************************/
			error_code = biosdisk (op?_DISK_WRITE:_DISK_READ,d->drive,(int)a->head,
				(int)a->cylinder,(int)a->sector,(int)1,(char *)buffer);
	}
	if (rd_err || wt_err) printf ("ERROR: Carry flag set on disk I/O op\n");
	return error_code;
}

/*****************************************************************
Decode a text string to both an LBA value and a Cylinder/Head/Sector value.
If the text string is one integer then it is assumbed to be an LBA address
If the text string is three integers separated by slashes then it is
assumed to be an address specified as C/H/S.
Otherwise is an error (return 1)
The disk address is converted to the other style so that both styles of
disk address are returned to the caller.
The parameter heads is used to in the conversion.
*****************************************************************/

int decode_disk_addr (char *addr, /* disk address as a string */
								int heads, /* number of heads per cylinder */
								unsigned long *lba, /* lba disk address */
								chs_addr *chs) /* C/H/S disk address */
{
	int	n; /* number of items decoded from addr: 1=>LBA, 3=>CHS */
	unsigned long track;

	chs->head = 0;
	chs -> sector = 0;
	*lba = 0;
	n = sscanf (addr,"%lu/%lu/%lu",&chs->cylinder,&chs->head,&chs->sector);
	if (n == 1){ /* one value ==> LBA address */
		 *lba = chs->cylinder;
		 chs->sector = (*lba)%63 + 1;
		 track = (*lba)/63;
		 chs->head = track%heads;
		 chs->cylinder = track/heads;
	}
	else if (n == 3){ /* three values separated by slashes ==> C/H/S */
		*lba = (chs->cylinder*heads + chs->head)*63 + chs->sector - 1;
	}
	else return 1; /* failed to get a valid disk address */
	return 0; /* OK */
}


disk_control_ptr setup_disk (char *drive_no,char *caption, FILE *log)
{
	int	status;
	disk_control_ptr	disk;
	int	drive;

	status = sscanf (drive_no,"%x",&drive);
	if (status != 1){
		printf ("%s drive number (%s) not valid\n",caption,drive_no); 
		fprintf (log,"%s drive number (%s) not valid\n",caption,drive_no);
		return NULL;
	}
	disk = open_disk (drive,&status);
	if (status){
		printf ("Could not access %s disk (%s) error code %d\n",
			caption,drive_no,status); 
		fprintf (log,"Could not access %s disk (%s) error code %d\n",
			caption,drive_no,status);
	}
	log_disk (log,caption,disk);
	return disk;
}


int check_addr (char *caption,char *addr_string, FILE *log, disk_control_ptr disk,
	unsigned long length, unsigned long *lba)
{
	static int status;
	static chs_addr	addr;

	status = decode_disk_addr (addr_string, /* disk address as a string */
								(int)n_heads(disk), /* number of heads per cylinder */
								lba, /* lba disk address */
								&addr); /* C/H/S disk address */
	if (status) {
			fprintf (log,"%s disk sector addr (%s) is not valid\n",caption,addr_string);
			printf ("%s disk sector addr (%s) is not valid\n",caption,addr_string);
			return 1;
	}
	if (*lba+length >= n_sectors(disk)){
		fprintf (log,"Copy length (%lu) exceeds (%lu) %s disk size (%lu)\n",
			length,*lba+length,caption,*lba);
		printf ("Copy length (%lu) exceeds (%lu) %s disk size (%lu)\n",
			length,*lba+length,caption,*lba);
		return 1;
	}
	return 0;
}

int copy_sector (FILE *log,disk_control_ptr src_disk, unsigned long src_lba,
					  disk_control_ptr dst_disk, unsigned long dst_lba)
{
	static unsigned char sector[512];
	static int	status;
	static chs_addr	src_chs,dst_chs;

	lba_to_chs (src_disk,src_lba,&src_chs);
	lba_to_chs (dst_disk,dst_lba,&dst_chs);

	status = disk_io (src_disk, &src_chs, 0, sector);
	if (status){
		printf ("Read error (%d) on src sector (%lu [%lu/%lu/%lu])\n",
			status,src_lba,src_chs.cylinder,src_chs.head,src_chs.sector);
		fprintf (log,"Read error (%d) on src sector (%lu [%lu/%lu/%lu])\n",
			status,src_lba,src_chs.cylinder,src_chs.head,src_chs.sector);
		return 1;
	}
	status = disk_io (dst_disk, &dst_chs, 1, sector);
	if (status){
		printf ("Read error (%d) on dst sector (%lu [%lu/%lu/%lu])\n",
			status,dst_lba,dst_chs.cylinder,dst_chs.head,dst_chs.sector);
		fprintf (log,"Read error (%d) on dst sector (%lu [%lu/%lu/%lu])\n",
			status,dst_lba,dst_chs.cylinder,dst_chs.head,dst_chs.sector);
		return 1;
	}

	return 0;
}

/*****************************************************************
Print instructions about running cmd and options
*****************************************************************/
void print_help(char *p)
{
	static int been_here = 0;
	if (been_here) return;
	been_here = 1;

	printf ("Usage: %s test-case host src_drive src_addr dst_drive dst_addr length [/opts]\n",p);
	printf ("/comment \" ... \"\tGive comment on command line\n");
	printf ("\t<addr> can be specified as either an LBA address (an integer)\n");
	printf ("\tor as cylinder/head/sector (three slash separated integers)\n");
	printf ("/new_log\tStart a new log file (default is append to old log file)\n");
	printf ("/?\tPrint this option list\n");
}

main (int np, char **p)
{
	int	help = 0,i /* loop index */;
	int	status; /* return code for read/write operations */
	static time_t from /* start time */;
	static FILE *log /* the log file */;
	static char comment[200] = "";
	unsigned long	lba; /* file offset (location) to make change */
	char	*access = "a";
	static unsigned long length,src_lba,dst_lba;
	static disk_control_ptr src_disk,dst_disk;

	time(&from);
	printf ("\n%s compiled at %s on %s\n", p[0],
		__TIME__,__DATE__);

/*****************************************************************
Decode the command line
*****************************************************************/
		/* get the command line */
	if (np < 8) help = 1;
	for (i = 8; i < np; i++){
		if (strcmp (p[i],"/?") == 0) {help = 1; break;}

		else if (strcmp (p[i],"/new_log")== 0) access = "w";
		else if (strcmp (p[i],"/comment")== 0){
			i++;
			if (i >= np){
				printf ("%s: /comment option requires a comment\n",p[0]);
				help = 1;
			} else strcpy (comment,p[i]);
		}
		else help = 1;
	}


	if (help){
		print_help(p[0]);
		return 0;
	}
   
/*****************************************************************
Check and log the disk drive
*****************************************************************/
	log = log_open("A:\\COPYLOG.TXT",access,comment,SCCS_ID,np,p);

	src_disk = setup_disk (p[3],"Source",log);
	dst_disk = setup_disk (p[5],"Destination",log);
	if ((src_disk == NULL) || (dst_disk == NULL)) return 1;

	status = sscanf(p[7],"%lu",&length);
	if (status != 1){
		printf ("Length (%s) is not valid\n",p[7]); 
		fprintf (log,"Length (%s) is not valid\n",p[7]);
		return 1;
	}
	if (check_addr ("Source",p[4],log,src_disk,length,&src_lba)) return 1;
	if (check_addr ("Destination",p[6],log,dst_disk,length,&dst_lba)) return 1;
	printf ("Copy %lu sectors from %s %lu to %s %lu\n",
		length,p[3],src_lba,p[5],dst_lba);
	fprintf (log,"Copy %lu sectors from %s %lu to %s %lu\n",
		length,p[3],src_lba,p[5],dst_lba);
	fflush(log);
	for (lba = 0; lba < length; lba++){
		if (copy_sector (log,src_disk,src_lba++,dst_disk,dst_lba++)){
			return 1;
		}
		if (length > 100) feedback (from,0L,lba,length);
	}
	fflush(log);
	fprintf (log,"%lu sectors copied from %s to %s\n",length,p[3],p[5]);
	printf ("%lu sectors copied from %s to %s\n",length,p[3],p[5]);
	fflush(log);



	 /* ran OK, log results */
	log_close (log,from); 
	fflush(log);
	return 0;
}
