/*****************************************************************************
 *  sha.c                                                     VERSION 3.1    *
 *---------------------------------------------------------------------------*
 *  S E C U R E   H A S H   S T A N D A R D  (SHA)                           *
 *  FEDERAL INFORMATION PROCESSING STANDARDS PUBLICATION 180                 *
 *  APRIL 1993                                                               *
 *---------------------------------------------------------------------------*
 *  This software was produced at the National Institute of Standards and    *
 *  Technology (NIST) as part of research efforts and is for demonstration   *
 *  purposes only. Our primary goals in its design did not include widespread*
 *  use outside of our own laboratories. Acceptance of this software implies *
 *  that you agree to use it for non-commercial purposes only and that you   *
 *  agree to accept it as nonproprietary and unlicensed, not supported by    *
 *  NIST and not carrying any warranty, either expressed or implied, as to   *
 *  its performance or fitness for any particular purpose.                   *
 *---------------------------------------------------------------------------*
 *  Produced by the National Institute of Standards and Technology (NIST),   *
 *  Computer Systems Laboratory (CSL) Security Technology Group.             *
 *---------------------------------------------------------------------------*
 *  History:                                                                 *
 *  VERSION     DATE         AUTHOR               CHANGES                    *
 *   2.00b       ?          NIST CSL             Original                    *
 *   2.01     24 Oct, 1995  Isadore Schoen      FIPS 180-1, SHATest mods, DLL*
 *   3.0      28 June,1996  Larry Bassham, NIST  Change to bit oriented code *  
 *   3.1      24 Mar, 1998  Larry Bassham, NIST  Fixed Update for leftovers  *
 ****************************************************************************/

#include "config.h"
#include <stdio.h>
#include <time.h>
#include <memory.h>
#include <string.h>
#include "sha.h"

/* The circular shifts. */
#define CS1(x) ((((ULONG)x)<<1)|(((ULONG)x)>>31))
#define CS5(x)  ((((ULONG)x)<<5)|(((ULONG)x)>>27))
#define CS30(x)  ((((ULONG)x)<<30)|(((ULONG)x)>>2))

/* K constants */

#define K0  0x5a827999L
#define K1  0x6ed9eba1L
#define K2  0x8f1bbcdcL
#define K3  0xca62c1d6L

#define f1(x,y,z)   ( (x & (y ^ z)) ^ z )

#define f3(x,y,z)   ( (x & ( y ^ z )) ^ (z & y) )

#define f2(x,y,z)   ( x ^ y ^ z )                           /* Rounds 20-39 */

#define  expand(x)  Wbuff[x%16] = CS1(Wbuff[(x - 3)%16 ] ^ Wbuff[(x - 8)%16 ] ^ Wbuff[(x - 14)%16] ^ Wbuff[x%16])

#define sub1Round1(count)      { \
	 temp = CS5(A) + f1(B, C, D) + E + Wbuff[count] + K0; \
	 E = D; \
	 D = C; \
	 C = CS30( B ); \
	 B = A; \
	 A = temp; \
	 } \

#define sub2Round1(count)   \
	 { \
	 expand(count); \
	 temp = CS5(A) + f1(B, C, D) + E + Wbuff[count%16] + K0; \
	 E = D; \
	 D = C; \
	 C = CS30( B ); \
	 B = A; \
	 A = temp; \
	} \

#define Round2(count)     \
	 { \
	 expand(count); \
	 temp = CS5( A ) + f2( B, C, D ) + E + Wbuff[count%16] + K1;  \
	 E = D; \
	 D = C; \
	 C = CS30( B ); \
	 B = A; \
	 A = temp;  \
	 } \

#define Round3(count)    \
	 { \
	 expand(count); \
	 temp = CS5( A ) + f3( B, C, D ) + E + Wbuff[count%16] + K2; \
	 E = D; \
	 D = C; \
	 C = CS30( B ); \
	 B = A; \
	 A = temp; \
	 }

#define Round4(count)    \
	 { \
	 expand(count); \
	 temp = CS5( A ) + f2( B, C, D ) + E + Wbuff[count%16] + K3; \
	 E = D; \
	 D = C; \
	 C = CS30( B ); \
	 B = A; \
	 A = temp; \
	 }

ULONG Wbuff[16];

/***********************************************************************

	This is the routine that implements the SHA.

***********************************************************************/
void ProcessBlock(shaContext)
SHA_CTX *shaContext;
{

	ULONG  A, B, C, D, E, temp;
	register int    j;

	A = shaContext->H[0];
	B = shaContext->H[1];
	C = shaContext->H[2];
	D = shaContext->H[3];
	E = shaContext->H[4];

/*                          
	for (j=0; j<16; j++ )
		printf("MBlock[%d] is %08lx\n", j, shaContext->Mblock[j]);
*/

	for (j=0; j<16; j++ )
		Wbuff[j]=shaContext->Mblock[j];

	sub1Round1( 0 );  sub1Round1( 1 );  sub1Round1( 2 );  sub1Round1( 3 );
	sub1Round1( 4 );  sub1Round1( 5 );  sub1Round1( 6 );  sub1Round1( 7 );
	sub1Round1( 8 );  sub1Round1( 9 );  sub1Round1( 10 ); sub1Round1( 11 );
	sub1Round1( 12 ); sub1Round1( 13 ); sub1Round1( 14 ); sub1Round1( 15 );
	sub2Round1( 16 ); sub2Round1( 17 ); sub2Round1( 18 ); sub2Round1( 19 );
	Round2( 20 ); Round2( 21 ); Round2( 22 ); Round2( 23 );
	Round2( 24 ); Round2( 25 ); Round2( 26 ); Round2( 27 );
	Round2( 28 ); Round2( 29 ); Round2( 30 ); Round2( 31 );
	Round2( 32 ); Round2( 33 ); Round2( 34 ); Round2( 35 );
	Round2( 36 ); Round2( 37 ); Round2( 38 ); Round2( 39 );
	Round3( 40 ); Round3( 41 ); Round3( 42 ); Round3( 43 );
	Round3( 44 ); Round3( 45 ); Round3( 46 ); Round3( 47 );
	Round3( 48 ); Round3( 49 ); Round3( 50 ); Round3( 51 );
	Round3( 52 ); Round3( 53 ); Round3( 54 ); Round3( 55 );
	Round3( 56 ); Round3( 57 ); Round3( 58 ); Round3( 59 );
	Round4( 60 ); Round4( 61 ); Round4( 62 ); Round4( 63 );
	Round4( 64 ); Round4( 65 ); Round4( 66 ); Round4( 67 );
	Round4( 68 ); Round4( 69 ); Round4( 70 ); Round4( 71 );
	Round4( 72 ); Round4( 73 ); Round4( 74 ); Round4( 75 );
	Round4( 76 ); Round4( 77 ); Round4( 78 ); Round4( 79 );
	shaContext->H[0] += A;
	shaContext->H[1] += B ;
	shaContext->H[2] += C;
	shaContext->H[3] += D;
	shaContext->H[4] += E;
	}/* End ProcessBlock */

/**********************************************************************/
/*  Performs byte reverse for PC based implementation (little endian) */
void byteReverse( buffer, byteCount )
ULONG *buffer;
int byteCount;
{
	ULONG value;
	int count;

	byteCount /= sizeof( ULONG );
	for( count = 0; count < byteCount; count++ ) {
		value = ( buffer[ count ] << 16 ) | ( buffer[ count ] >> 16 );
		buffer[ count ] = ( ( value & 0xFF00FF00L ) >> 8 ) | ( ( value & 0x00FF00FFL ) << 8 );
		}
	}

/**********************************************************************
	 SHA initialization routine.
		Clears all fields in the SHA Context structure and primes the
		hash with the initialization vector.
**********************************************************************/

int SHAInit(shaContext)
SHA_CTX *shaContext;
{
	int i;
	ULONG IH[5] = {0x67452301L, 0xefcdab89L, 0x98badcfeL,
				 0x10325476L, 0xc3d2e1f0L};

	shaContext->Numblocks[0] = 0;
	shaContext->Numblocks[1] = 0;
	shaContext->Numbits = 0;
	for (i=0; i<16; i++)
		shaContext->Mblock[i] = (ULONG) 0L;
	for (i=0; i<5 ;i++)
		shaContext->H[i] = IH[i];
	return 0;
	}

/**********************************************************************
	 SHA initialization routine.
		Clears all fields in the SHA Context structure and primes the
		hash with the user supplied initialization vector.
**********************************************************************/

int SHAInit2(shaContext, IH)
SHA_CTX *shaContext;
ULONG   *IH;
{
	int i;

	shaContext->Numblocks[0] = 0;
	shaContext->Numblocks[1] = 0;
	shaContext->Numbits = 0;
	for (i=0; i<16; i++)
		shaContext->Mblock[i] = (ULONG) 0L;
	for (i=0; i<5 ;i++)
		shaContext->H[i] = IH[i];
	return 0;
	}

/**********************************************************************
	 SHAUpdate hashes full 512 bit blocks. Assumes that
	 bitcount is a multiple of SHABLOCKLEN until the final
	 buffer is being processed. In this case, the data has
	 less than 512 bits then it saves the data for a
	 subsequent call to SHAFinal.
**********************************************************************/
int SHAUpdate(shaContext, buffer, bitcount)
SHA_CTX *shaContext;
unsigned char *buffer;
int bitcount;
{
	int	offsetBytes, offsetBits;
  	BYTE	tbuf[SHABYTEBLOCKLEN+1];

	while ( (bitcount+shaContext->Numbits) >= SHABLOCKLEN ) {
   	offsetBytes = shaContext->Numbits / 8;
   	offsetBits = shaContext->Numbits % 8;

		/* increment block counter  */
		if (shaContext->Numblocks[1] == 0xffffffffL) {
			shaContext->Numblocks[0]++;
			shaContext->Numblocks[1] = 0L;
			}
		else
			shaContext->Numblocks[1]++;

		/* Process full block now */
   	if ( offsetBits == 0 ) {
			memcpy((BYTE *) shaContext->Mblock+offsetBytes, buffer, SHABYTEBLOCKLEN-offsetBytes);
			}
      else {
      	BYTE	tbuf[SHABYTEBLOCKLEN+1];
         int	i;

         memset(tbuf, '\0', SHABYTEBLOCKLEN+1);
         memcpy(tbuf+offsetBytes, buffer, SHABYTEBLOCKLEN-offsetBytes);
         for ( i=0; i<offsetBits; i++ )
	         bshr(tbuf, SHABYTEBLOCKLEN+1);
         xor(shaContext->Mblock, shaContext->Mblock, tbuf, SHABYTEBLOCKLEN);
         shaContext->Numbits += bitcount;
      	}
/*		memcpy(shaContext->Mblock, buffer, SHABYTEBLOCKLEN); */

#ifdef LITTLE_ENDIAN
byteReverse(shaContext->Mblock, SHABYTEBLOCKLEN);
#endif

		/* Process full block  */
		ProcessBlock(shaContext);
      if ( offsetBits != 0 ) {
      	shaContext->Numbits = offsetBits;
			memcpy((BYTE *) shaContext->Mblock, tbuf+SHABYTEBLOCKLEN, 1);
         }
      else
      	shaContext->Numbits = 0; 
      buffer += (SHABYTEBLOCKLEN-offsetBytes);
		bitcount -= ((SHABYTEBLOCKLEN-offsetBytes)*8);
		}

	/* Save partial block for subsequent invocation of SHAFinal */
	if ( bitcount )
   	if ( (shaContext->Numbits%8) == 0 ) {
			memcpy((BYTE *) shaContext->Mblock+(shaContext->Numbits/8), buffer, (bitcount+7)/8);
			shaContext->Numbits += bitcount;
			}
      else {
      	BYTE	tbuf[SHABYTEBLOCKLEN];
         int	offset, i;

         memset(tbuf, '\0', SHABYTEBLOCKLEN);
      	offset = shaContext->Numbits/8;
         memcpy(tbuf+offset, buffer, (bitcount+7)/8);
         for ( i=0; i<shaContext->Numbits%8; i++ )
	         bshr(tbuf, SHABYTEBLOCKLEN);
         xor(shaContext->Mblock, shaContext->Mblock, tbuf, SHABYTEBLOCKLEN);
         shaContext->Numbits += bitcount;
      	}

	return 0;
	}

/**********************************************************************
	 SHAFinal does the hashing of the last block of the message.
	 It is this routine that does the necessary padding of zeros
	 and sets the length of the data at the end.
**********************************************************************/
int SHAFinal(shaContext)
SHA_CTX *shaContext;
{
	int     i,k,numsub;
	ULONG   numbits;
	ULONG   padbits=0x80000000L, padbits2=0xFFFFFFFFL;

#ifdef LITTLE_ENDIAN
byteReverse(shaContext->Mblock, SHABYTEBLOCKLEN);
#endif

	numbits=shaContext->Numbits;
	numsub=(int)numbits/32;

	/* put in the "1" bit  */
	padbits >>= numbits % 32L;
	padbits2 <<= (31L - (numbits % 32L));
	shaContext->Mblock[numsub] |= padbits;
	shaContext->Mblock[numsub] &= padbits2;

	/* put in the zero bits  */
	for (k=numsub+1; k<16L; k++)
		shaContext->Mblock[k] = (ULONG)0L;
	/* If more than 447 data bits in last block, there isn't enough room for
		the 1 bit and size field.  Fill this block out with zeros and Process it.
		Then fill another block with zeros and be ready to insert the length
		field.  */
	if ( numsub > 13 ) {
		ProcessBlock(shaContext);
		for (i=0; i<14; i++) shaContext->Mblock[i] = (ULONG)0L;
		}
	/*  Put in the length field of the data hashed.  */
	shaContext->Mblock[14] = (ULONG)((shaContext->Numblocks[0] << 9)
					+  (shaContext->Numblocks[1] >> 23));
	shaContext->Mblock[15] = (ULONG)(shaContext->Numblocks[1] << 9) + numbits;

	ProcessBlock(shaContext);

	return 0;
	}

