/******************************************************************************
The software provided here is released by the National
Institute of Standards and Technology (NIST), an agency of
the U.S. Department of Commerce, Gaithersburg MD 20899,
USA.  The software bears no warranty, either expressed or
implied. NIST does not assume legal liability nor
responsibility for a User's use of the software or the
results of such use.

Please note that within the United States, copyright
protection, under Section 105 of the United States Code,
Title 17, is not available for any work of the United
States Government and/or for any works created by United
States Government employees. User acknowledges that this
software contains work which was created by NIST employees
and is therefore in the public domain and not subject to
copyright.  The User may use, distribute, or incorporate
this software provided the User acknowledges this via an
explicit acknowledgment of NIST-related contributions to
the User's work. User also agrees to acknowledge, via an
explicit acknowledgment, that any modifications or
alterations have been made to this software before
redistribution.
******************************************************************************/ 
# define Z_H_ID "@(#) zbios.h Version 2.6 Created 07/10/01 at 11:19:56"
/***** Author: Dr. James R. Lyle, NIST/SDCT/SQG ****/
# include <dos.h>
# include <bios.h>
# include <time.h>
# define GET_DISK_PARMS 8
# define N_RANGE 20

/******************************************************************************
A disk address in cylinder/head/sector format
******************************************************************************/
typedef struct {
	unsigned long cylinder,head,sector; } chs_addr; /* C/H/S disk address */

/******************************************************************************
Basic information (model, serial number, max user addressable sectors)
obtained by IDENTIFY_DEVICE command to a disk controler
******************************************************************************/

typedef struct {
	unsigned long max_user_sectors;
	char	serial_no[21];
	char	model_no[41];
} ide_rec, *ide_ptr;

/******************************************************************************
The disk_control_block contains all information about a disk drive
Disk geometry as seen by legacy BIOS (interrupt 13/command 0x08): logical disk
Disk geometry as seen by XBIOS (int 13/cmd 0x48): disk_max
number of sectors reported by BIOS: n_sectors
A buffer holding 63 sectors: buffer
Drive number: drive
Flag indicating XBIOS active: use_bios_x
IDE Drive information: ide_info
******************************************************************************/

typedef unsigned char physical_sector[512]; /* a sector of 512 bytes */
typedef physical_sector physical_track[63]; /* a track is an array of 63 sectors */
typedef struct disk_struct disk_control_block, *disk_control_ptr;

struct disk_struct {
	unsigned long 	n_sectors;
	chs_addr			disk_max;     /* number of cyl, number of head */
	chs_addr			logical_disk; /* max cyl, max head */
	chs_addr 		this_track;   /* cyl and head of track in buffer */
	int				drive; 			/* drive number (starts at 0x80) */
	int				use_bios_x;    /* is disk > 8.4GB */
	int				valid_sectors;  /* track buffer has data */
   ide_ptr			ide_info;		/* Model#, Serial# & max sectors */
	physical_track buffer;      /* 63 sectors (1 track) buffer[sector][byte]*/
};

/******************************************************************************
Partition table entry layout on disk
******************************************************************************/

typedef struct pts { /* partition table layout */
	unsigned char bootid,
					  start_head,
					  start_sector,
					  start_cylinder,
					  type_code,
					  end_head,
					  end_sector,
					  end_cylinder;
	unsigned long starting_lba_sector,
					  n_sectors;
} partition_table_rec, *partition_table_pointer;
/******************************************************************************
Layout of a partition table in a boot sector
******************************************************************************/

typedef struct { /* partition table in Master Boot Record */
	char						fill[446]; /* MBR boot code */
	partition_table_rec 	pe[4];	  /* master partition table */
	unsigned int 			sig;		  /* partition table signature word 0xAA55 */
}mbr_sector,*mbr_ptr;

/******************************************************************************
Data structure to keep partition table information
******************************************************************************/

typedef struct pte_struct pte_rec, *pte_ptr;
struct pte_struct {
	pte_ptr				next;
	unsigned char		is_boot,type;
	chs_addr				start,end;
	unsigned long		lba_start,lba_length;
};


/******************************************************************************
Data structure to track ranges of integers. The compare programs examine each
disk sector in LBA address sequence and assign each sector to a catagory, e.g.,
dst sector that differs from corresponding src sector, zero filled, src filled,
dst filled, etc. This data structure is used to track blocks of
disk sector LBA addresses for sectors that are classified in the same catagory.
******************************************************************************/

typedef struct {unsigned long from,to;} lba_range;
typedef struct { /* structure to keep a list of ranges */
	int			n;
	long 			is_more; /* more than N_RANGE ranges present (i.e., some not recorded) */
	lba_range 	r[N_RANGE];
	} range_list,*range_ptr;


/******************************************************************************
Function decls for zbios.cpp
******************************************************************************/

int 					read_lba (disk_control_ptr ,unsigned long , unsigned char **);
int 					disk_write (disk_control_ptr, chs_addr *);
int 					disk_read (disk_control_ptr, chs_addr *);
disk_control_ptr 	open_disk (int, int *);
void 					lba_to_chs (disk_control_block *,unsigned long , chs_addr *);

FILE 					*log_open (char *,char *,char *,char **, int, char **);
void 					log_close (FILE *,time_t);
void 					log_disk(FILE *, char *, disk_control_ptr);


void 			print_dcb(disk_control_ptr);
int 			get_partition_table(disk_control_block *,pte_ptr );
void 			print_partition_table(FILE *,pte_rec *, int, int);

void 			feedback (time_t, unsigned long, unsigned long, unsigned long);

range_ptr 	create_range_list(void);
void 			add_to_range (range_ptr, unsigned long );
void 			print_range_list(FILE *, char *,range_ptr);

# define is_big(d) (d)->use_bios_x
# define n_sectors(d) ((d)->ide_info?(d)->ide_info->max_user_sectors:(d)->n_sectors)
# define n_cylinders(d) (d)->disk_max.cylinder
# define n_heads(d) (d)->disk_max.head
# define n_tracks(d) ((d)->disk_max.head*(d)->disk_max.cylinder)

