/******************************************************************************
The software provided here is released by the National
Institute of Standards and Technology (NIST), an agency of
the U.S. Department of Commerce, Gaithersburg MD 20899,
USA.  The software bears no warranty, either expressed or
implied. NIST does not assume legal liability nor
responsibility for a User's use of the software or the
results of such use.

Please note that within the United States, copyright
protection, under Section 105 of the United States Code,
Title 17, is not available for any work of the United
States Government and/or for any works created by United
States Government employees. User acknowledges that this
software contains work which was created by NIST employees
and is therefore in the public domain and not subject to
copyright.  The User may use, distribute, or incorporate
this software provided the User acknowledges this via an
explicit acknowledgment of NIST-related contributions to
the User's work. User also agrees to acknowledge, via an
explicit acknowledgment, that any modifications or
alterations have been made to this software before
redistribution.
******************************************************************************/ 
/***** Author: Dr. James R. Lyle, NIST/SDCT/SQG ****/
# include <stdio.h>
# include "zbios.h"
# include <string.h> 
# include <malloc.h>
# include <time.h>
extern char *SCCS_Z = "@(#) support lib zbios.cpp Version 3.1 created 10/11/01 at 12:40:23"\
"\nsupport lib compiled "__DATE__" at "__TIME__"\n"Z_H_ID;
# define GET_DISK_PARMS 8


/*****************************************************************
Support Library
	Probe an IDE disk for Mfg & Serial #
	Log file: open, log disk, close
	Give user progress and completion time feedback
	Disk utilities: open, read, write, convert LBA <=> C/H/S
	Partition table: get and print
*****************************************************************/

/*****************************************************************
Remove any leading or trailing blanks from s
*****************************************************************/

void trim(char *s /* s is the string to trim */)
{
	int	at = strlen(s),k;

/*****************************************************************
Scan s from the end toward the beginning to find last non-blank
*****************************************************************/
	while (at){
		at--;
		if (s[at] != ' '){
			s[at+1] = '\0'; /* snip trailing blanks */
			break;
		}
	}
	if (s[0] != ' ')return; /* done if no leading blanks */
	at = 0;
	while (s[at] == ' ') at++; /* find first non-blank */
	k = 0;
	while (s[at]){ /* move string on top of leading blanks */
		s[k++] = s[at++];
	}
	s[k] = '\0'; /* terminate string */
	return;
}

/*****************************************************************
Macros for accessing IDE disk controler
adapter(0) is the primary IDE channel
adapter (1) is the secondary IDE channel
drive_no(0) is the master drive for the channel
drive_no(1) is the slave drive for the channel
dev_port and cmd_port are device registers to issue an I/O command
id_dev is the command code for IDENTIFY_DEVICE
*****************************************************************/
# define adapter(x) ((x)?0x0170:0x01F0)
# define drive_no(x) ((x)?0x00B0:0x00A0)
# define dev_port(x) (adapter(x)+6)
# define cmd_port(x) (adapter(x)+7)
# define id_dev 0x0EC
/*****************************************************************
Issue an IDENTIFY DEVICE command to an IDE disk controler to get
the disk model number and serial number.
See the ATA standard for details but the basic idea is
	use outportb to set the command registers and
	use inport to get the results.
	The command generates an interrupt, but just sleep through it.
*****************************************************************/
ide_ptr probe_ide_adapter (int adapter_no, /* specify IDE channel */
									int unit_no) /* identify master or slave */
{
	ide_ptr	ide = NULL;
	char	c; /* valid results return */
	static unsigned int w[64]; /* input buffer for responce to IDENTIFY_DEVICE cmd */
	int	k; /* loop index */

	/* issue the command */
	outportb(dev_port(adapter_no),drive_no(unit_no));
	outportb(cmd_port(adapter_no),id_dev);
	sleep(1); /* let somebody else answer the interrupt */
	c = inp(cmd_port(adapter_no));
	if (c != 0x58) return NULL; /* no data found */
	/* get the first 64 words, discard the remainder */
	for (k = 0; k < 64; k++)
		w[k] = inport(adapter(adapter_no));
	ide = (ide_ptr) malloc (sizeof(ide_rec));
	/* get the disk model number */
	for (k = 0; k < 20; k++){
		ide->model_no[2*k + 1] = w[k+27] & 0x00FF;
		ide->model_no[2*k] = (w[k+27]&0xFF00) >> 8;
	}
	ide->model_no[40] = '\0';
	/* get the disk serial number */
	for (k = 0; k < 10; k++){
		ide->serial_no[2*k + 1] = w[k+10] & 0x00FF;
		ide->serial_no[2*k] = (w[k+10]&0xFF00) >> 8;
	}
	ide->serial_no[20] = '\0';
	trim (ide->serial_no);
	trim (ide->model_no);
	/* get the max user addressable sectors. Note: this is only
	valid for IDE disks in LBA mode. i.e., really old disks that
	do not support LBA return ZERO */
	ide->max_user_sectors = (((unsigned long)w[61])<<16) +
									 ((unsigned long)w[60]);
	return ide;

}

/*****************************************************************
Map IDE drive number to Controler adapter/unit number.
This mapping is NOT correct in general, but works in the usual
hardware setup (80 -- 0x1F0,0; 81 -- 0x1F0,1; 82 -- 0x170,0 ; etc)
*****************************************************************/
ide_ptr probe_ide (int drive)
{
	int	adapter_no = 0,unit_no = 0,d = 0;
	static int first_time = 1;
	static ide_ptr drive_info[4] = {NULL,NULL,NULL,NULL};
	ide_ptr	info;
	/* first time probe for four disks: IDE primary & secondary for
	master and slave, i.e., disks 80-83, and save results. After first
	time just return the result for the given drive. */
	if (first_time){
		first_time = 0;
		/* try each unit of each adapter */
		for (adapter_no = 0; adapter_no < 2; adapter_no++)
			for (unit_no = 0; unit_no < 2; unit_no++){
				info = probe_ide_adapter (adapter_no,unit_no);
				if (info) drive_info[d++] = info;
			}
	}
	/* store info on drive 0x80 is position 0 of drive_info, 0x81 at 1 etc */
	drive = drive - 0x80;
	if ((drive < 0) || (drive > 3)) return NULL;
	return drive_info[drive];
}

/*****************************************************************
Create an empty list of ranges
*****************************************************************/
range_ptr create_range_list(void)
{
	range_ptr	p;
	/* allocate space for a list */
	p = (range_ptr) malloc (sizeof(range_list));
	p->n = 0; /* list starts out empty */
	p->is_more = 0;
	return p;
}

/*****************************************************************
Add x to the list of ranges (r)
	(1) look for a range that can be expanded to include x
		 i.e., range of form a--b where b+1=x, a--b can then be
		 expanded to a--x
	(2) If an expandable range can't be found create a range of
		 the form x--x
*****************************************************************/
void add_to_range (range_ptr r, unsigned long x)
{
	int k = r->n - 1;
	if (r->n){ /* there is already at least one range */
		/* if x is one past edge of last range, expand range */
		if (r->r[k].to + 1 == x) r->r[k].to = x;
		else { /* either add new range of count total in category */
			k++;
			if (k >= N_RANGE){ /* too many ranges, just count */
				r->is_more++;
				return;
			}
			/* add new range to list: x--x */
			r->n++;
			r->r[k].from = r->r[k].to = x;
		}
	}
	else { /* first time: set number of ranges in range list to 1 */
		r->n = 1;
		/* set range: x--x */
		r->r[0].from = r->r[0].to = x;
	}
}

void print_range_list(FILE *log, /* log file */
							 char *caption, /* caption for the log file */
							 range_ptr r) /* range list to print */
{
	int	i,nc = 0 /* track line length */;

	nc = fprintf (log,"%s ",caption);
	for (i = 0; i < r->n; i++){
		if(i)nc += fprintf(log,", "); /* add a comma if more */
		if (nc > 50){ /* time for a new line */
			nc = 0;
			fprintf (log,"\n");
		}
		if (r->r[i].from == r->r[i].to)
			/* range beginning and ending points are the same */
			nc += fprintf (log,"%ld",r->r[i].from);
		else nc += fprintf (log,"%ld-%ld",r->r[i].from,r->r[i].to);
	}
	if (r->is_more) fprintf (log,". . . + %ld more\n",r->is_more);
	else fprintf (log,"\n");
}


/*****************************************************************
Start a logfile. If the logfile can't be opened use stdout
	name -- name for the logfile
	access -- "w" create new file, "a" append to old file
	comment -- Comment entry for log file
	stats[0] -- SCCS entry for program
	stats[1] -- date program compiled
	stats[2] -- time program compiled
	np -- number of command line parameters
	p[0] -- program name
	p[1] -- test case name
	p[2] -- computer host hame
*****************************************************************/
FILE	*log_open(char *name, char *access, char *comment, char **stats,
	int np, char **p)
{
	FILE	*log;
	int	  i;

	/* If no comment string then prompt to insert a log floppy and type
	a comment for the log file. If there is a comment already (i.e., from
	the command line, then use the floppy disk in the drive for the log file */
	if (strlen(comment) == 0){
		printf ("Insert a log disk, type a descriptive comment and press ENTER\n");
		gets(comment);
	}
	log = fopen (name,access);
	if (log == NULL) log = stdout; /* use stdout if no floppy */
	fprintf (log,"%s %s\ncompiled on %s at %s\n",p[0],stats[0],stats[1],stats[2]);
	fprintf (log,"%s\n",SCCS_Z); /* log support lib and header file version */
	fprintf (log,"cmd:"); /* copy command line to log file */
	for (i = 0; i < np; i++) fprintf (log," %s",p[i]);
	fprintf (log,"\n");
	fprintf (log,"TEST %s HOST %s\nComment: %s\n",p[1],p[2],comment);
	return log;
}


/*****************************************************************
Compute elapsed time and close a log file
*****************************************************************/
void log_close (FILE *log, /* log file */
					 time_t from /* time program started running */)
{
	time_t	till; /* time program finished (actually time log_close called) */
	unsigned long et, /* elapsed time (program run time: seconds of wall clock) */
					  tmin, /* total minutes running */
					  min, /* minutes for elapsed time after whole hours deducted */
					  sec, /* seconds of elapsed run time after hours & minutes */
					  hours; /* hours of elapsed time */

	time(&till); /* get current time */
	et = till - from; /* elapsed time in seconds */
	tmin = et/60; /* elapsed time in minutes */
	sec = et%60; /* fraction of last minute in seconds */
	hours = tmin/60; /* hours elapsed time */
	min = tmin%60; /* fraction of last hour in whole minutes */
   
	fprintf (log,"run start %s",ctime(&from));
	fprintf (log,"run finish %s",ctime(&till));

	fprintf (log,"elapsed time %ld:%ld:%ld\n",hours,min,sec);
	fprintf (log,"Normal exit\n"); 

	printf ("elapsed time %ld:%ld:%ld\n",hours,min,sec);
	printf ("Normal exit\n");
}


/*****************************************************************
print disk geometry for a disk described on disk_control_ptr d
*****************************************************************/
void print_dcb(disk_control_ptr d)
{
	printf ("Drive 0x%X %s \n",d->drive,d->use_bios_x?"Use extensions":
		"No extensions");
		printf ("Interrupt 13  bios  %04ld/%03ld/%02ld (max cyl/hd values)\n",
			d->logical_disk.cylinder,
			d->logical_disk.head,
			d->logical_disk.sector);
		printf ("Interrupt 13  ext  %05ld/%03ld/%02ld (number of cyl/hd)\n",
			d->disk_max.cylinder,
			d->disk_max.head,
			d->disk_max.sector);
		printf ("%ld total sectors accessible via BIOS\n",d->n_sectors);

}
/*****************************************************************
Record information about a disk to a log file
	Type of BIOS: Legacy or extended
	Disk geometry reported by Legacy BIOS: int 13/cmd 8 (max values)
	Disk geometry reported by XBIOS: int 13/cmd 48 (number of values)
	For IDE disks connected to a standard IDE controler:
		Disk Model number
		Disk Serial number
		for disks that support LBA: max number of user accessable sectors
		reported by the controler.
*****************************************************************/
void log_disk(FILE *log, /* log file */
				 char *caption, /* caption for log file */
				 disk_control_ptr d) /* disk to log information about */
{
	printf ("%s Drive 0x%X, BIOS: %s \n",caption,
		d->drive,d->use_bios_x?"Extensions Present":"Legacy");
	fprintf (log,"%s Drive 0x%x, BIOS: %s\n",caption,d->drive,
		(d->use_bios_x)?"Extensions Present":"Legacy");
		fprintf (log,"Interrupt 13  bios  %04ld/%03ld/%02ld (max cyl/hd values)\n",
			d->logical_disk.cylinder,
			d->logical_disk.head,
			d->logical_disk.sector);
		fprintf (log,"Interrupt 13  ext  %05ld/%03ld/%02ld (number of cyl/hd)\n",
			d->disk_max.cylinder,
			d->disk_max.head,
			d->disk_max.sector);
		fprintf (log,"%ld total number of sectors reported via interrupt 13 from the BIOS\n",
			d->n_sectors);
		if (d->logical_disk.sector != 63) fprintf (log,
			"WARNING: disk is not supported, it has other than 63 sectors per track.\n");
		if (d->ide_info){
			fprintf (log,"IDE disk: Model (%s) serial # (%s)\n",
				d->ide_info->model_no,d->ide_info->serial_no);
			fprintf (log,"Max number of user addressable sectors reported by ATA identify device command %lu\n",
				d->ide_info->max_user_sectors);
		}
		else fprintf (log,"Non-IDE disk\n");

}

/*****************************************************************
Dump the command block for interrupt 13 command 48
*****************************************************************/
void dap_print (unsigned char dap[16])
{
	printf ("count %2X %2X\n",dap[0],dap[1]);
	printf ("B O:S %2X %2X %2X %2X (%02X%02X:%02X%02X)\n",
		dap[4],dap[5],dap[6],dap[7],dap[5],dap[4],dap[7],dap[6])/
	printf ("LBA   %2X %2X %2X %2X\n",dap[8],dap[9],dap[10],dap[11]);
}

/*****************************************************************
Give progress and completion time feedback to a user
	Intent is to call this on each pass of a loop that goes
	from "from" upto "to".
	start -- time the loop entered
	from  -- starting value of loop index
	at    -- current value of loop index
	to    -- terminal value of loop
*****************************************************************/
void feedback (time_t start, unsigned long from, unsigned long at,
					unsigned long to)
{
	time_t	now;
	unsigned long	every_pc = 5, n_feed = 100/every_pc; /* give feedback every n_feed% */
	unsigned long feed = (to - from)/n_feed;
	unsigned long et,hr,min,tmin,sec; /* estimated elapsed time */
	float pc, /* percent completed */
			fat,ns; /* number completed */
	static int first = 1; /* first time called switch */

	at++;
   if (feed == 0) feed = 1;
	if (first){/* first time through tell the user how often to
						expect feed back */
		first = 0;
		printf ("Feedback every %ld sectors (%ld%%) of %ld\n",feed,every_pc,
			to-from);
	}
	if (((at - from)%feed) && (at != to)) return;
	if (from)
		printf ("at %lu (%lu) of %lu (%lu) from %lu",
					at,at-from,to,to-from,from);
	else printf ("at %lu of %lu",at,to);

	fat = at - from;
	ns = to - from;
	pc = (100.00*fat)/ns; /* pc = percent completed so far */
	time(&now);
	et = (now - start)*(ns - fat)/fat; /* et estimates time remaining */
	tmin = et/60;
	hr = tmin/60;
	min = tmin%60;
	sec = et%60;

	printf (" %5.1f%%    %lu:%02lu:%02lu remains on %s",pc,hr,min,
		sec,ctime(&now));
	return;
}

/*****************************************************************
Write a track (63 sectors) to disk d at address a
See the ATA BIOS extensions documents for details
*****************************************************************/

int disk_write (disk_control_ptr d, chs_addr *a)
{
	static int	error_code = 0,drive,bad_write;
	static unsigned long lba;
	static unsigned char disk_addr_packet [16];
	static unsigned int *dap_count = (unsigned int *)disk_addr_packet,
		*dap_n = (unsigned int *)&disk_addr_packet[2],
		*dap_buffer_offset = (unsigned int *)&disk_addr_packet[4],
		*dap_buffer_segment = (unsigned int *)&disk_addr_packet[6];
	static unsigned long *dap_lba = (unsigned long *)&disk_addr_packet[8],
		*dap_fill = (unsigned long *)&disk_addr_packet[12];
	static unsigned int dap_segment = FP_SEG(disk_addr_packet),
				dap_offset = FP_OFF(disk_addr_packet);


	if (d->drive == 0) return 1;
	error_code = 0;/*
	printf ("Write %d/%d/%d on drive %x\n",
		(int)a->cylinder,(int)a->head,(int)a->sector,d->drive);   */
	if (d->use_bios_x){
/*****************************************************************
Write a track (63 sectors) to disk d at address a
Using the ATA BIOS extensions
*****************************************************************/
			/* convert a to LBA */
			lba = (a->cylinder*d->disk_max.head + a->head)*63 + a->sector - 1;
			drive = d->drive;
			*dap_count = (int) 16;
			/* don't try to write past the last disk sector!!! */
			if ((lba < n_sectors(d)) && ((lba + 63) > n_sectors(d)))
					*dap_n = (int)(n_sectors(d) - lba);
			else 	*dap_n = (int) d->valid_sectors;
			*dap_buffer_offset = FP_OFF(d->buffer); /* data buffer address */
			*dap_buffer_segment = FP_SEG(d->buffer);
			*dap_lba = lba;  /* disk address */
			*dap_fill = (long) 0;/*
			printf ("Write to drive %x (xbios)\n",d->drive);
			dap_print(disk_addr_packet);   */
			bad_write = 0;
			error_code = 0;
			asm {
				mov ah,0x43         /* set write command */
				mov al,0x0          /* MBZ */
				mov dl,BYTE PTR drive  /* set BIOS drive number */
				mov ds,dap_segment     /* set address of command block */
				mov si,dap_offset
				int 0x13               /* write to disk */
				mov BYTE PTR error_code,ah
				jnc write_ok          /* carry flag indicates error */
			}
			bad_write = 1;
			write_ok:
	}
	else { /*
	printf ("Write %d/%d/%d on drive %x\n",
		(int)a->cylinder,(int)a->head,(int)a->sector,d->drive); */
		error_code = 0;
/*****************************************************************
Write a track (63 sectors) to disk d at address a
using the ATA Legacy BIOS
*****************************************************************/
		error_code = biosdisk (_DISK_WRITE,d->drive,(int)a->head,(int)a->cylinder,
			(int)a->sector,d->valid_sectors,d->buffer);
	}
	if (error_code && d->use_bios_x) printf ("bad write %d error code %d\n",
		bad_write,error_code);
	return error_code;
}

/*****************************************************************
Read a track from disk d at address a
*****************************************************************/
int disk_read (disk_control_ptr d, chs_addr *a)
{
	static int	error_code,drive;
	static unsigned long lba;
	static unsigned char disk_addr_packet [16];
	static unsigned int *dap_count = (unsigned int *)disk_addr_packet,
		*dap_n = (unsigned int *)&disk_addr_packet[2],
		*dap_buffer_offset = (unsigned int *)&disk_addr_packet[4],
		*dap_buffer_segment = (unsigned int *)&disk_addr_packet[6];
	static unsigned long *dap_lba = (unsigned long *)&disk_addr_packet[8],
		*dap_fill = (unsigned long *)&disk_addr_packet[12];
	static unsigned int dap_segment = FP_SEG(disk_addr_packet),
				dap_offset = FP_OFF(disk_addr_packet);


	if (d->drive == 0) return 1;
	if (d->use_bios_x){
/*****************************************************************
Read a track (63 sectors) from disk d at address a
Using the ATA BIOS extensions
*****************************************************************/
			/* convert a to LBA */
		lba = (a->cylinder*d->disk_max.head + a->head)*63 + a->sector - 1;
		drive = d->drive;
		*dap_count = (int) 16;
		/* don't try to read past the end of the disk */
		if ((lba < n_sectors(d)) && ((lba + 63) > n_sectors(d)))
			  *dap_n = (int)(n_sectors(d) - lba);
		else *dap_n = (int) d->valid_sectors;
		*dap_buffer_offset = FP_OFF(d->buffer);
		*dap_buffer_segment = FP_SEG(d->buffer);
		*dap_lba = lba;
		*dap_fill = (long) 0;
		error_code = 0;
		asm {
			mov ah,0x42          /* set read command */
			mov dl,BYTE PTR drive  /* set BIOS drive number */
			mov ds,dap_segment    /* set address of command block */
			mov si,dap_offset
			int 0x13             /* read */
			mov BYTE PTR error_code,ah
		}
	}
	else {
/*****************************************************************
Read a track (63 sectors) from disk d at address a
using the ATA Legacy BIOS
*****************************************************************/
			error_code = biosdisk (_DISK_READ,d->drive,(int)a->head,(int)a->cylinder,
				(int)a->sector,d->valid_sectors,d->buffer);
	}
	d->this_track.cylinder = a->cylinder;
	d->this_track.head = a->head;
   d->this_track.sector = a->sector;

	return error_code;
}

/*****************************************************************
Test for ATA BIOS extensions on drive
See BIOS Enhanced Disk Drive Services (EDD) for details
*****************************************************************/
int c13(int drive)
{
  int carry;
  int drive_number=drive;
  int   x13 = 0;

  unsigned int ah_register = 0;
  unsigned int bx_register = 0;
  unsigned int cx_register = 0;
	 carry=0;

	 asm{
		mov ah,0x41               /* test for extensions command */
		mov bx,0x55aa
		mov dl,BYTE PTR drive_number
		INT 0x13

		mov BYTE PTR ah_register,ah
		mov WORD PTR bx_register,bx
		mov WORD PTR cx_register,cx

		jnc carry_flag_not_set   /* Jump if the carry flag is clear  */
	  }                         /* If the carry flag is clear, then */
										/* the extensions exist.            */
	 carry=1;


	 carry_flag_not_set:
	 if( (carry==0) && (bx_register==0xaa55))
		  x13 = 1;

		 return x13;
}

/*****************************************************************
Assemble a number in pieces
*****************************************************************/
unsigned long Decimal_Number(unsigned long hex1, unsigned long hex2,
			 unsigned long hex3, unsigned long hex4)
{
  return((hex1) + (hex2*256ul) +
	(hex3*65536ul) + (hex4*16777216ul));
}
/*****************************************************************
Get geometry for a disk using ATA BIOS extensions
*****************************************************************/
int Get_Hard_Drive_Parameters(int physical_drive,
	unsigned long *cylinders, /* number of .. */
	unsigned long *heads,     /* number of .. */
	unsigned long *sectors,   /* number of .. */
	unsigned long *tsl,       /* number of .. */
	unsigned long *tsh)
{
  static unsigned char result_buffer[26];
  static unsigned int result_buffer_segment=FP_SEG(result_buffer);
  static unsigned int result_buffer_offset=FP_OFF(result_buffer);
  int error_code=0;
  int i;

  for (i = 1; i < 26; i++) result_buffer[i] = 0;
  result_buffer[0]=26;
  result_buffer_segment=FP_SEG(result_buffer);
  result_buffer_offset=FP_OFF(result_buffer);


	 asm {
		mov ah,0x48
		mov dl,BYTE PTR physical_drive
		mov ds,result_buffer_segment
		mov si,result_buffer_offset
		int 0x13

		mov BYTE PTR error_code,ah
		}

	 *cylinders=Decimal_Number(result_buffer[4],result_buffer[5],
		result_buffer[6],result_buffer[7]);
    *heads=Decimal_Number(result_buffer[8],result_buffer[9],
		result_buffer[10],result_buffer[11]);
	 *sectors=Decimal_Number(result_buffer[12],result_buffer[13],
		result_buffer[14],result_buffer[15]);
		*tsl =   Decimal_Number(result_buffer[16],result_buffer[17],
		result_buffer[18],result_buffer[19]);
		*tsh =   Decimal_Number(result_buffer[20],result_buffer[21],
		result_buffer[22],result_buffer[23]); /* should be zero */

  return(error_code);
}
 
/*****************************************************************
Open a disk, return a pointer to a disk_control_rec
The disk_control_rec contains a description of the disk ...
	legacy BIOS geometry
	extended BIOS geometry
	IDE or not
	number of sectors addressable
	number of sectors reported by BIOS
	IDE disk model and serial numbers
*****************************************************************/
disk_control_ptr open_disk (int drive, int *err)
{
	static unsigned char b[600];
	unsigned long cylinders = 0;
	unsigned long heads = 0;
	unsigned long sectors = 0;
	unsigned long tsl,tsh;
	disk_control_ptr d;
	int	ec;

	/* Get Legacy BIOS disk geometry */
	ec = biosdisk (GET_DISK_PARMS,drive,0,0,0,0,b);
	printf ("BIOS get_disk_parms %d\n",ec);
   *err = 0;
	if (ec){*err = ec; return NULL;}
	d = (disk_control_ptr) malloc (sizeof(disk_control_block));
	if (d == NULL){*err = 3000; return NULL;} /* out of memory */
	d->logical_disk.head = b[3];
	d->logical_disk.sector = b[0]&0x3f;
	d->logical_disk.cylinder = ((b[0]&0xC0)<<2)|b[1];

	d->valid_sectors = 63;
	d->use_bios_x = c13(drive); /* test for XBIOS */
	printf ("Open disk 0x%x, of %d disks, BIOS: %s\n",drive,b[2],
		(d->use_bios_x)?"Extensions Present":"Legacy");
	d->drive = drive;
	d->ide_info = probe_ide(drive); /* get model and serial numbers */
	if (d->ide_info) printf ("Open %s %s %lu on drive 0x%x\n",
		d->ide_info->model_no,d->ide_info->serial_no,
		d->ide_info->max_user_sectors,drive);
	else printf ("Open non-IDE (SCSI?) disk on drive 0x%x\n",drive);
	if (d->use_bios_x){
		ec = Get_Hard_Drive_Parameters(drive, /* get XBIOS disk geometry */
		&cylinders,&heads,&sectors,&tsl,&tsh);
		if (ec){*err = 1000 + ec;
			printf ("Disk open failed to get extended drive parameters");
			 return NULL;}
		d->disk_max.cylinder = cylinders;
		d->disk_max.head = heads;
		d->disk_max.sector = sectors;
		d->n_sectors = tsl;
		if (tsh){*err = 2000; return NULL;} /* really big disk */
		if ((cylinders == 0) || (heads == 0) || (sectors == 0)){
			/* some SCSI cards return zeros in the geometry field so
				we have to fake it */
			d->disk_max = d->logical_disk;
			d->disk_max.cylinder++;
			d->disk_max.head++;
		}
	}
	else { /* Legacy BIOS only (no XBIOS) so, fake XBIOS */
		d->disk_max = d->logical_disk;
		d->disk_max.cylinder++;
		d->disk_max.head++;
		d->n_sectors = d->disk_max.cylinder*d->disk_max.head*63;
	}
	d->this_track.cylinder = 0;
	d->this_track.head = 0;
	d->this_track.sector = 1;
	ec = disk_read (d,&d->this_track);
	*err = ec;
	/* if disk does not support LBA, fake it */
	if  (d->ide_info) if(d->ide_info->max_user_sectors < d->n_sectors)
		d->ide_info->max_user_sectors = d->n_sectors;
	return d;
}

/*****************************************************************
Convert LBA value to C/H/S
*****************************************************************/
void lba_to_chs (disk_control_block *d,unsigned long lba, chs_addr *a)
{
	unsigned long track = lba/63;
	a->sector = lba%63 + 1;
	a->head = track%n_heads(d);
	a->cylinder = track/n_heads(d);
}

/*****************************************************************
Read sector "lba" of disk d set b to point to start of sector
(An easier interface than C/H/S)
*****************************************************************/
int read_lba (disk_control_ptr d,unsigned long lba, unsigned char **b)
{
	chs_addr	a;

	*b = (unsigned char *)&(d->buffer[lba%63][0]);
	lba_to_chs (d,lba,&a);  /* convert lba to CHS */
	if ((a.cylinder == d->this_track.cylinder) && /* don't bother to do */
		(a.head == d->this_track.head)) return 0;  /* the read if already done */
	a.sector = 1;
	return disk_read(d,&a); /* need to do the read */

}

/*****************************************************************
Get a nested partition table entry
*****************************************************************/
pte_ptr get_sub_part (disk_control_block *d,unsigned long start,
	unsigned long base,int *status)
{
	mbr_sector *mbr;
	pte_ptr	p,q;
	int 		i;
	unsigned long at = base + start;
	chs_addr a;

	lba_to_chs(d,at,&a);
	*status = disk_read (d,&a);
	mbr = (mbr_sector *)(&(d->buffer));
	if (*status) return NULL;
	if (mbr->sig == 0xAA55) {
		p = q = (pte_ptr) malloc (sizeof(pte_rec));
		for (i = 0; i < 2; i++){
			p->is_boot = mbr->pe[i].bootid;
			p->type = mbr->pe[i].type_code;
			p->lba_start = mbr->pe[i].starting_lba_sector;
			p->lba_length = mbr->pe[i].n_sectors;
			p->start.cylinder = mbr->pe[i].start_cylinder |
				 ((mbr->pe[i].start_sector&0xC0)<<2);
			p->start.head =  mbr->pe[i].start_head;
			p->start.sector = mbr->pe[i].start_sector & 0x3F;
			p->end.cylinder = mbr->pe[i].end_cylinder |
				 ((mbr->pe[i].end_sector&0xC0)<<2);
			p->end.head =  mbr->pe[i].end_head;
			p->end.sector = mbr->pe[i].end_sector & 0x3F;
			if (i == 0){
				p->next = (pte_ptr) malloc (sizeof(pte_rec));
				p = p->next;
			}
			else {
				if ((mbr->pe[1].type_code == 0x05) ||
					 (mbr->pe[1].type_code == 0x0F) ||
					 (mbr->pe[1].type_code == 0x0C)){
					p->next = get_sub_part(d,p->lba_start,base,status);
				}
				else p->next = NULL;
			}
		}
	}
	else return NULL;
	return q;
}

/*****************************************************************
Get the partition table for disk d and save in pt
*****************************************************************/
int get_partition_table(disk_control_block *d,pte_ptr pt)
{
	mbr_sector *mbr;
	chs_addr  boot = {0ul,0ul,1ul};
	int	status,i;

	status = disk_read (d,&boot);
	mbr = (mbr_sector *) &(d->buffer);
	if (status) return status;
	if (mbr->sig != 0xAA55) return -1;
	else {
		for (i = 0; i < 4; i++){
			status = disk_read (d,&boot); /* really needed */
			pt[i].is_boot = mbr->pe[i].bootid;
			pt[i].type = mbr->pe[i].type_code;
			pt[i].lba_start = mbr->pe[i].starting_lba_sector;
			pt[i].lba_length = mbr->pe[i].n_sectors;
			pt[i].start.cylinder = mbr->pe[i].start_cylinder |
				 ((mbr->pe[i].start_sector&0xC0)<<2);
			pt[i].start.head =  mbr->pe[i].start_head;
			pt[i].start.sector = mbr->pe[i].start_sector & 0x3F;
			pt[i].end.cylinder = mbr->pe[i].end_cylinder |
				 ((mbr->pe[i].end_sector&0xC0)<<2);
			pt[i].end.head =  mbr->pe[i].end_head;
			pt[i].end.sector = mbr->pe[i].end_sector & 0x3F;
			if ((mbr->pe[i].type_code == 0x05) ||
				 (mbr->pe[i].type_code == 0x0F) ||
				 (mbr->pe[i].type_code == 0x0C)){
					pt[i].next = get_sub_part (d,0ul,pt[i].lba_start,&status);
					if (status) return status;
			}
			else {
				pt[i].next = NULL;
			}
		 }

	}
	return status;

}

/*****************************************************************
Map common partition type codes to an ASCII string
*****************************************************************/
char *partition_type(unsigned char code)
{
	char	*pt;

	if (code == 0) pt = "empty entry";
	else if ((code == 0x05) ||
				(code == 0x0F) ||
				(code == 0x0C)) pt = "extended";
	else if ((code == 0x04) ||
				(code == 0x06) ||
				(code == 0x0E)) pt = "Fat16";
	else if ((code == 0x0B)) pt = "Fat32"; 
	else if ((code == 0x01)) pt = "Fat12";
	else if ((code == 0x07)) pt = "NTFS";
	else if ((code == 0x82)) pt = "Linux swap";
	else if ((code == 0x81) ||
				(code == 0x83)) pt = "Linux";
	else pt = "other";
	return pt;
}
 
/*****************************************************************
Print a partition table
	full=1 -- print extended table entries too
*****************************************************************/
void print_partition_table(FILE *log,/* output file: either log file or stdout */
						pte_rec *p, /* partition table */
						int index, /* print the index numbers */
						int full) /* print all table entries, else omit empty entries
										 and extended partition entries */
{
	int	i,j = 0;
	char	type_code;
	pte_ptr	sub;

	if (index) fprintf (log," N ");
	fprintf (log,"  %-9s %-9s %-11s %-11s boot Partition type\n",
   	"Start LBA","Length","Start C/H/S","End C/H/S");
	for (i = 0; i < 4; i++){
		if ((p[i].type == 0x05) ||
			 (p[i].type == 0x0F) ||
			 (p[i].type == 0x0C)) type_code = 'X';
		else type_code = 'P';
		j++;
		if (full || ((type_code != 'X') && (p[i].type))){
			if (index) fprintf (log,"%2d ",j);
			fprintf (log,"%c %09ld %09ld %04ld/%03ld/%02ld %04ld/%03ld/%02ld %4s %02X",
				type_code,
				p[i].lba_start,p[i].lba_length,
				p[i].start.cylinder,p[i].start.head,p[i].start.sector,
				p[i].end.cylinder,p[i].end.head,p[i].end.sector,
				p[i].is_boot?"Boot":"",p[i].type);
			fprintf (log," %s",partition_type(p[i].type));
			fprintf(log,"\n");
		}
		sub = p[i].next;
		while(sub){
			if ((sub->type == 0x05) ||
				 (sub->type == 0x0F) ||
				 (sub->type == 0x0C)) type_code = 'x';
			else type_code = 'S';
			j++;
			if (full || ((type_code != 'x') && (sub->type))){
				if (index) fprintf (log,"%2d ",j);
					fprintf (log,"%c %09ld %09ld %04ld/%03ld/%02ld %04ld/%03ld/%02ld %4s %02X",
						type_code,
						sub->lba_start,sub->lba_length,
						sub->start.cylinder,sub->start.head,sub->start.sector,
						sub->end.cylinder,sub->end.head,sub->end.sector,
						sub->is_boot?"Boot":"",sub->type);
					fprintf (log," %s",partition_type(sub->type));
					fprintf(log,"\n");
			}
			sub = sub->next;
		}
	}
	if (full) {
		fprintf (log,"P primary partition (1-4)\n"); 
		fprintf (log,"S secondary (sub) partition\n");
		fprintf (log,"X primary extended partition (1-4)\n");
		fprintf (log,"x secondary extended partition\n");
	}
}

