/*
** bt_support.ex.c :
**
** Bluetooth model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      Frederic Mouveaux
** Module description:  Bluetooth Common MAC Function Support
** Last Modification:   January, 16th, 2003 (Olivier Rebala)
*/

/* Include header */
#include "bt_support.h"


/*
 * Function:	bt_init_bt_support
 *
 * Description:	Initialization routine to call at the Initialization
 *				of a process in order to build all the structures and
 * 				variables associated with this module.
 *
 * No Parameter
 */

void	bt_init_bt_support ()
{
	int i,j; // loop variable
	
	FIN (bt_init_bt_support ());
	
	/*
	 * Setup the priorities for the self event of this process
	 * The higher will be processed first 
	 */
	op_intrpt_priority_set (OPC_INTRPT_STRM, LOWER_LAYER_INPUT_STREAM, 40);
	op_intrpt_priority_set (OPC_INTRPT_STRM, HIGHER_LAYER_VC_INPUT_STREAM, 35);
	op_intrpt_priority_set (OPC_INTRPT_STRM, HIGHER_LAYER_INPUT_STREAM, 30);
	op_intrpt_priority_set (OPC_INTRPT_SELF, PKT_DELIVERY, 20);
	op_intrpt_priority_set (OPC_INTRPT_SELF, PKT_WAITING, 15);

	op_intrpt_priority_set (OPC_INTRPT_SELF, MASTER_CLK_TIMER, 10);
	op_intrpt_priority_set (OPC_INTRPT_SELF, SLAVE_CLK_TIMER, 5);

	/* 
	 * Set the base hop number, and slot number
	 * Note that one hop frequency is a complex scheme base on the clock 
	 * and addresses (UAP/LAP) as describe in the Bluetooth specification.
	 * This is calculated in the bt_hop_selection () function in the
	 * "bt_freq_hopping" files
	 */
	if (number_of_piconet)
		{
		bluetooth_packet_delivery = (Boolean *) op_prg_mem_alloc (number_of_piconet * sizeof(Boolean));
		txrx_slot_number = (int *) op_prg_mem_alloc (number_of_piconet * sizeof(int));
		txrx_hop_freq_offset = (int *) op_prg_mem_alloc (number_of_piconet * sizeof(int));
		packet_delivery_clk = (int *) op_prg_mem_alloc (number_of_piconet * sizeof(int));
	
		for (i=0; i<number_of_piconet; i++)
			{
			bluetooth_packet_delivery[i] = OPC_FALSE;
			txrx_slot_number[i] = -1;
			txrx_hop_freq_offset[i] = 0;
			packet_delivery_clk[i] = 0;
			}

		/* Initialize the packet_counter */
		for (i=0; i<STATION_NUMBER; i++)
			{
			packet_counter[i] = (int *) op_prg_mem_alloc (number_of_piconet * sizeof (int));
		
			for (j=0; j<number_of_piconet; j++)
				packet_counter[i][j] = 0;
			}
		}
	else
		bt_mac_error ("No Bluetooth piconet in the topology", OPC_NIL, OPC_NIL);
	
	if (number_of_wlan)
		{
		wlan_packet_delivery = (Boolean *) op_prg_mem_alloc (number_of_wlan * sizeof(Boolean));
		
		for (i=0; i<number_of_wlan; i++)
			wlan_packet_delivery[i] = OPC_FALSE;
		}
	
	FOUT;
}

/* 
 * Function:	bt_mac_error
 *
 * Description:	Error handling procedure
 *
 * ParamIn:		char * fcnt
 *				The function in the process where
 *				the error occured
 *
 *				char * msg
 *				The description of the error
 *
 *		  		char * opt
 *				An optional message
 */

void	bt_mac_error (char * fcnt, char * msg, char * opt)
{
	/* Terminates simulation and print an error message. */
	FIN (bt_mac_error (fcnt, msg, opt));

	op_sim_end ("Error in BlueTooth MAC process: ", fcnt, msg, opt);

	FOUT;
}

/*
 * Function:	bt_generate_next_self_event
 *
 * Description:	Schedule a self-interrupt event
 *
 * ParamIn:		double next_evt_time
 *				The next creation time for the event
 *
 *				int code
 *				The code descriptor of the event
 */

void	bt_generate_next_self_event(double next_evt_time, int code)
{
	FIN(bt_generate_next_self_event(next_evt_time, code));


	if ( op_ev_valid (op_intrpt_schedule_self (next_evt_time, code)) == OPC_FALSE)
	  	bt_mac_error ("bt_generate_next_self_event", "Unable to schedule self interrupt", "");


	FOUT;
}

/*
 * Function:	bt_get_number_of_ts_from_type
 *
 * Description:	Determine from the type number of the packet
 *				the # of TS that it will take. There is 16 types
 *				of packets. the number of TS is determined
 *				reguarding to the specs BBand p54
 *
 * ParamIn:		int packet_type
 *				The type of Bluetooth packet (DM1,DM3,DM5,...)
 *
 * ParamOut:	int number_of_ts
 *				The Number of TS that the packet takes.
 */

int	bt_get_number_of_ts_from_type(int packet_type)
{
  int	number_of_ts;			/* The number of TS for this packet type */

  FIN(bt_get_number_of_ts_from_type(packet_type));


  switch (packet_type)
  {
	case 10:
	case 11:
	case 12:
	case 13:
	  number_of_ts = 3;
	break;
	case 14:
	case 15:
	  number_of_ts = 5;
	break;
	default:
	  number_of_ts = 1;
	break;
  }


  FRET(number_of_ts);
}

/*
 * Function:	bt_get_overhead_from_type
 *
 * Description:	Determine from the type number of the packet
 *				the overhead
 *
 * ParamIn:		int packet_type
 *				The type of Bluetooth packet (DM1,DM3,DM5,...)
 *
 * ParamOut:	int overhead
 *				The overhead introduced
 */

int     bt_get_overhead_from_type(int packet_type)
{
  int   overhead;

  FIN(bt_get_overhead_from_type(packet_type));


  switch (packet_type)
  {
        case 10:
        case 11:
        case 12:
        case 13:
        case 14:
        case 15:
          overhead = 158;
	break;
        default:
          overhead = 150;
        break;
  }


  FRET(overhead);
}

/*
 * Function:	bt_next_slot
 *
 * Description:	Calculate the next slot number based
 *				on the current slot information
 *
 * ParamIn:		int * txrx_s_num
 *				Pointer to the current slot number
 */

void	bt_next_slot (int * txrx_s_num)
{
	FIN (bt_next_slot (txrx_s_num));


	(*txrx_s_num)++;
	(*txrx_s_num) %= (NUMBER_OF_SLOTS);


	FOUT;
}


/*
 * Function:	bt_clock
 *
 * Description:	This state include a simple clock based on the TS time.
 *				Each time that a clock tick occurs the pair TX/RX must
 *				keep track of the hop to the next frequency. If a packet
 *				transmission or reception is in progress, the current
 *				frequency is kept constant until the end of the transmis-
 *				sion or reception.
 *
 * ParamIn:		int ulap
 *				in general, the mac address of the master to initialize
 *				the random generator for the frequency hopping.
 *
 *				int piconet_number
 *				number of the current piconet using the function
 */

void    bt_clock (int ulap, int piconet_number)
{
	int clock;			// master clock 
	Boolean clock_debug = op_prg_odb_ltrace_active ("hopping");
	char	msg1[128], msg2[128];
		
	
	FIN (bt_clock (slot_number, ulap));
  
	/* Also update the slot number          */
	bt_next_slot (&txrx_slot_number[piconet_number]);
	
	/* Compute the master clock */
	clock = txrx_slot_number[piconet_number] << 1;
	
	/* compute the next frequency */
	txrx_hop_freq_offset[piconet_number] = (int) bt_hop_selection ((Bt_Hop_Long) clock, (Bt_Hop_Long) ulap);
  
	if (clock_debug)
		{
		sprintf (msg1, "Current time: %f s; Frequency: %d MHz", op_sim_time (), 2400 + txrx_hop_freq_offset[piconet_number]);
		sprintf (msg2, "Slot number: %d; master clock: %d", txrx_slot_number[piconet_number], clock);
		op_prg_odb_print_major ("Bt_clock ():", msg1, msg2, OPC_NIL);
		}
	
	/* And the packet delivery clk if sat to something else than Zero */
	if (packet_delivery_clk[piconet_number] > 0)
		packet_delivery_clk[piconet_number]--;
  
	/* Generate a Clock self interrupt at the next time slot */
	bt_generate_next_self_event (op_sim_time ()+TS, MASTER_CLK_TIMER);
  
	FOUT;
}


/*
 * Function:	bt_set_master_transmission
 *
 * Description:	Set the master transmission by make its TX hopping
 *				to the current frequency and make the RX of all the
 *				slaves hopping to the same frequency
 *
 * ParamIn:		Packet * pkptr
 *				Pointer to the packet to set up
 *
 *				int piconet_number
 *				number of the piconet using the function
 */

void	bt_set_master_transmission (Packet *pkptr, int piconet_number)
{
	FIN (bt_set_master_transmission (pkptr, freq_offset));

	op_pk_nfd_set (pkptr, "frequency", 2400.0 + txrx_hop_freq_offset[piconet_number]);

	FOUT;
}

/*
 * Function:	bt_set_slave_transmission
 *
 * Description:	Set the slave transmission by make its TX hopping
 *				to the current frequency and make the RX of the
 *				master hopping to the same frequency
 *
 * ParamIn:		Packet * pkptr
 *				Pointer to the packet to set up
 *
 *				int piconet_number
 *				number of the piconet using the function
 */

void	bt_set_slave_transmission (Packet *pkptr, int piconet_number)
{
	FIN (bt_set_slave_transmission (pkptr, freq_offset));

	op_pk_nfd_set (pkptr, "frequency", 2400.0 + txrx_hop_freq_offset[piconet_number]);

	FOUT;
}

/*
 * Function:	bt_set_tx_packet_delivery_clk
 *
 * Description:	set the packet delivery clk variable to the number of
 *				clock cycles that the packet takes in Transmission
 *
 * ParamIn:		Packet * pkptr
 *				pointer to the current packet
 *
 *				int piconet_number
 *				number of the piconet using the function
 */

void	bt_set_tx_packet_delivery_clk (Packet * pkptr, int piconet_number)
{
	int			packet_type; /* Packet type */


	FIN (bt_set_tx_packet_delivery_clk (pkptr, piconet_number));

	/* Get the packet information */
	op_pk_nfd_get(pkptr, "TYPE", &packet_type);
	
	/* set the packet delivery clk variable */
	packet_delivery_clk[piconet_number] = bt_get_number_of_ts_from_type (packet_type);

	FOUT;
}

/*
 * Function:	bt_set_rx_packet_delivery_clk
 *
 * Description:	set the packet delivery clk variable to the number of
 *				clock cycles that the packet takes in Reception
 *
 * ParamIn:		Packet * pkptr
 *				pointer to the current packet
 */

void	bt_set_rx_packet_delivery_clk (Packet * pkptr)
{
	int			packet_type; /* Packet type */

	FIN (bt_set_rx_packet_delivery_clk (pkptr));

	/* Get the packet information */
	op_pk_nfd_get (pkptr, "TYPE", &packet_type);

	FOUT;
}


/*
 * Function:	bt_destroy_packet
 *
 * Description:	destroy a packet with its content to free
 *				the memory
 *
 * ParamIn:		Packet * pkptr
 *				pointer to the packet to destroy
 */

void	bt_destroy_packet (Packet * pkptr)
{
        int		packet_type;
	Packet *	frame_body;
	Packet *	fec;
	
	FIN (bt_destroy_packet (pkptr));
	
	/* Get the packet type information */
	op_pk_nfd_get(pkptr, "TYPE", &packet_type);
	
	switch (packet_type)
	  {
	  case	NULL_TYPE:
	  case	POLL_TYPE:
	    op_pk_destroy(pkptr);
	    break;
	    
	  case	HV1_TYPE:
	  case	HV2_TYPE:
	  case	HV3_TYPE:
	    op_pk_nfd_get(pkptr, "Frame Body", &frame_body);
	    op_pk_destroy(pkptr);
	    break;
	    
	  case    DM1_TYPE:
	  case    DM3_TYPE:
	  case    DM5_TYPE:
	    op_pk_nfd_get(pkptr, "Frame Body", &frame_body);
	    op_pk_nfd_get(pkptr, "FEC", &fec);
	    op_pk_destroy(frame_body);
	    op_pk_destroy(fec);
	    op_pk_destroy(pkptr);
	    break;

	  default:
	    op_pk_nfd_get(pkptr, "Frame Body", &frame_body);
	    op_pk_destroy(frame_body);
	    op_pk_destroy(pkptr);
	    break;
	  }
	
	FOUT;
}


/*
 * Function:	bt_print_debug (const char* message)
 *
 * Description: a little function to display some messages when
 *				program's working on to debug a function
 *
 * ParamIn:		const char* message
 *				message to display
 */

void bt_print_debug (const char* message)
{
  FIN(bt_print_debug (message));

  printf ("\n\n\n*------------------------------------------*\n");
  printf ("%s \n",message);
  printf ("*------------------------------------------*\n\n\n");

  FOUT;
}


/*
 * Function:	bt_print_count_packet(int address)
 *
 * Description: count the number of packet received by the device
 *				with the mac address 'address' and print the value
 *				on the screen
 *
 * ParamIn:		int address
 *				mac address of the devices 
 */

void bt_print_count_packet(int address, int piconet_number)
{
  int i,j;

  FIN ( bt_print_count_packet(address, piconet_number) );

  packet_counter[address][piconet_number]++; /* array which contain the number of packet */

  if (bt_message_print)
	  {
	  for (i=0; i<STATION_NUMBER; i++)
		  for (j=0; j<number_of_piconet; j++)
			  {
			  if (packet_counter[i][j]) /* if the array is not empty */
				  printf ("| number of packet sent to the address %d of the piconet %d : %d\n",i, j, packet_counter[i][j]);
			  }
	  }
  
  
  FOUT;
}


/*
 * Function:	bt_print_type_packet
 *
 * Description: convert the integer code type of a packet to
 *				the text type
 *
 * ParamIn:		int type
 *				integer code type of the packet
 */

char *bt_print_type_packet (int type)
{
  char *type_char;

  FIN(bt_print_type_packet (type));

  type_char = (char *) op_prg_mem_alloc ( 5 * sizeof(char));

  switch ( type )
    {
    case	NULL_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_NULL);
      break;
    case	POLL_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_POLL);
      break;
    case	HV1_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_HV1);
      break;
    case	HV2_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_HV2);
      break;
    case	HV3_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_HV3);
      break;
      
    case DM1_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_DM1);
      break;
      
    case DM3_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_DM3);
      break;
      
    case DM5_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_DM5);
      break;
      
    case DH1_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_DH1);
      break;

    case DH3_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_DH3);
      break;
      
    case DH5_TYPE:
      strcpy (type_char,BLUETOOTH_PKT_DH5);
      break;
      
    default:
      strcpy (type_char,"???");
      break;
      
    }
  
  FRET(type_char);
}


/*
 * Function:	bt_size_of_pkt_type
 *
 * Description: get the size in slot time of a packet
 *				according to its packet type
 *
 * ParamIn:		const char *packet_type
 *				Type of the packet : "DM1", "DM3" or "DM5"
 *
 * ParamOut:	int size
 *				size of the packet in slot time
 */

int bt_size_of_pkt_type (const char *packet_type)
{
  int size;

  FIN(bt_size_of_pkt_type (packet_type));

  if (packet_type != NULL)
    {
      if ((strcmp(packet_type,"DM1") == 0) || (strcmp(packet_type,"DH1") == 0) || 
	  (strcmp(packet_type, "POLL") == 0) || (strcmp(packet_type, "NULL") == 0) )
	size = 1;
      else if ((strcmp(packet_type,"DM3") == 0) || (strcmp(packet_type,"DH3") == 0))
	size = 3;
      else if ((strcmp(packet_type,"DM5") == 0) || (strcmp(packet_type,"DH5") == 0))
	size = 5;
      else
	size = 0;
    }
  else
    bt_mac_error ("bt_size_of_pkt_type() :", "pointer to packet_type is NULL", "");
  
  if (size == 0)
    bt_mac_error ("bt_size_of_pkt_type() :", "Packet type unknown", "");

  FRET(size);
}

/*
 * Function : 	bt_get_report_name
 *
 * Description:	get the current path set in the channel node
 * 				and compute the entire path of the file to generate
 *
 * ParamIn : 	const char * name_of_file
 *				name of the file to generate
 *
 * ParamOut:	char * name
 *				name of the entire path generated
 */

char *	bt_get_report_name (const char * name_of_file, Objid obid)
{
	char *	name;
	char	buffer[1024];
	Objid	report_objid;
	int		name_size;
	
	
	FIN (bt_get_report_name (name_of_file, obid));
	
	/* get the path set in the entity with the objid in parameter */
	op_ima_obj_attr_get (obid, "Report", &report_objid);
	op_ima_obj_attr_get (op_topo_child (report_objid, OPC_OBJTYPE_GENERIC,0),
		"path", buffer);
	
	/* get the size of the name */
	name_size = strlen (buffer) + strlen (name_of_file) + 1;
	
	/* allocate some memory */
	name = (char *) op_prg_mem_alloc (name_size * sizeof (char));
	
	/* get the name */
	strcpy (name, buffer);
	strcat (name, name_of_file);
	
	FRET (name);
}


/*
 * Function:	bt_end_bt_support
 *
 * Description:	free the memory at the end of the simulation
 *
 * No parameter
 */

void	bt_end_bt_support ()
{
	int i; // loop variable
	
	FIN (bt_end_bt_support ());

	/* Delete the memory */
	op_prg_mem_free (bluetooth_packet_delivery);
	op_prg_mem_free (txrx_slot_number);
	op_prg_mem_free (txrx_hop_freq_offset);
	op_prg_mem_free (packet_delivery_clk);
	
	/* Delete the packet_counter */
	for (i=0; i<STATION_NUMBER; i++)
		op_prg_mem_free (packet_counter[i]);
			
	FOUT;
}
