/*
** bt_stats_write.ex.c :
**
** Bluetooth model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      Olivier Rebala
** Module description:  Bluetooth Common MAC Function Support
** Last Modification:   July, 24th 2002
*/

/* include header */
#include	"bt_stats_write.h"


/*
 * $Function:    bt_stats_error
 *
 * $Description: stop the simulation and print an error
 *
 * $ParamIn:     char * fcnt, char * msg, char * opt
 *               Error messages to display
 */

void bt_stats_error (char * fcnt, char * msg, char * opt)
{
  FIN (bt_stats_error (fcnt, msg, opt));

  op_sim_end ("Error in statistic writing:", fcnt, msg, opt);

  FOUT;
}


/*
 * Function:	bt_stats_read_path
 *
 * Description:	
 */

char *	bt_stats_read_path (Objid process_id)
{
	Objid	file_objid;
	char	buffer[256];
	char *	path;
	
	
	FIN (bt_stats_read_path (process_id));
	
	/* get the objid of the Report compound */
	op_ima_obj_attr_get (process_id, "Report", &file_objid);
	
	
	/* Get the path of the output file */
	op_ima_obj_attr_get (op_topo_child (file_objid, OPC_OBJTYPE_GENERIC,0), "path", buffer);
	
	if (strcmp(buffer,"Default") == 0)
		{
		op_ima_obj_attr_get (topo_get_channel(), "Report", &file_objid);
		op_ima_obj_attr_get (op_topo_child (file_objid, OPC_OBJTYPE_GENERIC,0), "path", buffer);
		}
	
	path = (char *) op_prg_mem_alloc ( (strlen(buffer) + 1) * sizeof(char) );
	strcpy (path,buffer);
	
	FRET (path);
}


/*
 * Function:	
 */

bt_TCP_stats *	bt_stats_init_tcp_structure ()
{
	bt_TCP_stats *	tcp_stats;
	
	
	FIN (bt_stats_init_structure ());
	
	/* allocate some memory */
	tcp_stats = (bt_TCP_stats *) op_prg_mem_alloc (sizeof(bt_TCP_stats));
	
	/* Init the values */
	tcp_stats->objid 		= op_topo_parent (op_id_self ());
	tcp_stats->delay 		= 0.0;
	tcp_stats->bytes 		= 0.0;
	tcp_stats->nb_packet 	= 0;
	tcp_stats->next 		= NULL;
	
	
	FRET (tcp_stats);
}


/*
 * Function:	
 */

bt_UDP_stats *	bt_stats_init_udp_structure ()
{
	bt_UDP_stats *	udp_stats;
	
	
	FIN (bt_stats_init_udp_structure ());
	
	/* allocate some memory */
	udp_stats = (bt_UDP_stats *) op_prg_mem_alloc (sizeof(bt_UDP_stats));
	
	/* Init the values */
	udp_stats->objid 		= op_topo_parent (op_id_self ());
	udp_stats->bytes 		= 0.0;
	udp_stats->nb_packet 	= 0;
	udp_stats->next 		= NULL;
	
	
	FRET (udp_stats);
}


/*
 * Function:	
 */

void	bt_stats_update_tcp_stats (double delay, double bytes, int nb_packet)
{
	bt_TCP_stats *	current_ptr;
	Objid			node_id;
	
	
	FIN (bt_stats_update_tcp_stats (delay, bytes, nb_packet));
	
	/* Get the node ID*/
	node_id = op_topo_parent (op_id_self ());
	
	if (bt_tcp_stats_write == NULL)
		{
		/* if the structure is not initialized */
		bt_tcp_stats_write = bt_stats_init_tcp_structure ();
		bt_tcp_stats_write->delay += delay;
		bt_tcp_stats_write->bytes += bytes;
		bt_tcp_stats_write->nb_packet += nb_packet;
		}
	else
		{
		/* we're looking for the good cell */
		current_ptr = bt_tcp_stats_write;
		
		while ( (current_ptr->next != NULL) && (current_ptr->objid != node_id) )
			current_ptr = current_ptr->next;
		
		if (current_ptr->objid != node_id)
			{
			/* if the cell doesn't exist yet, we create it */
			current_ptr->next = bt_stats_init_tcp_structure ();
			current_ptr = current_ptr->next;
			}
		
		/* update the value */
		current_ptr->delay += delay;
		current_ptr->bytes += bytes;
		current_ptr->nb_packet += nb_packet;
		}
	
		
	FOUT;
}


/*
 * Function:	
 */

void	bt_stats_update_udp_stats (double bytes, int nb_packet)
{
	bt_UDP_stats *	current_ptr;
	Objid			node_id;
	
	
	FIN (bt_stats_update_udp_stats (bytes, nb_packet));
	
	/* Get the node ID*/
	node_id = op_topo_parent (op_id_self ());
	
	if (bt_udp_stats_write == NULL)
		{
		/* if the structure is not initialized */
		bt_udp_stats_write = bt_stats_init_udp_structure ();
		bt_udp_stats_write->bytes += bytes;
		bt_udp_stats_write->nb_packet += nb_packet;
		}
	else
		{
		/* we're looking for the good cell */
		current_ptr = bt_udp_stats_write;
		
		while ( (current_ptr->next != NULL) && (current_ptr->objid != node_id) )
			current_ptr = current_ptr->next;
		
		if (current_ptr->objid != node_id)
			{
			/* if the cell doesn't exist yet, we create it */
			current_ptr->next = bt_stats_init_udp_structure ();
			current_ptr = current_ptr->next;
			}
		
		/* update the value */
		current_ptr->bytes += bytes;
		current_ptr->nb_packet += nb_packet;
		}
	
		
	FOUT;
}


/*
 * Function:	
 */

bt_TCP_stats *	bt_stats_get_tcp_stats ()
{
	bt_TCP_stats *	current_ptr;
	Objid			node_id;
	
	
	FIN (bt_stats_get_tcp_stats ());
	
	/* Get the node ID */
	node_id = op_topo_parent (op_id_self ());
	
	/* we're looking for the good cell */
	current_ptr = bt_tcp_stats_write;
		
	while ( (current_ptr != NULL) && (current_ptr->objid != node_id))
		current_ptr = current_ptr->next;
	
		
	FRET (current_ptr);
}


/*
 * Function:	
 */

bt_UDP_stats *	bt_stats_get_udp_stats ()
{
	bt_UDP_stats *	current_ptr;
	Objid			node_id;
	
	
	FIN (bt_stats_get_udp_stats ());
	
	/* Get the node ID */
	node_id = op_topo_parent (op_id_self ());
	
	/* we're looking for the good cell */
	current_ptr = bt_udp_stats_write;
		
	while ( (current_ptr != NULL) && (current_ptr->objid != node_id))
		current_ptr = current_ptr->next;
	
		
	FRET (current_ptr);
}


/*
 * Function:	
 */

void	bt_stats_tcp_end_of_simulation ()
{
	
	double	time_of_simulation;
	double	delay, bytes;
	
	bt_TCP_stats *	tcp_stats_ptr;
	const char *	comment = "#Delay\t\tbytes/s";
	char			results[128]; 
	
	
	FIN (bt_stats_tcp_end_of_simulation ());
	
	/* init the variable */
	time_of_simulation = op_sim_time ();
	
	/* compute the stats */
	if ((tcp_stats_ptr = bt_stats_get_tcp_stats ()) != NULL)
		{
		/* print info about the statistics writing */
		printf ("\n|-------------------------------------------------------------\n");
		printf ("| TCP statistics writing:\n");
		
		if (tcp_stats_ptr->nb_packet)
			delay = tcp_stats_ptr->delay / (double) tcp_stats_ptr->nb_packet;
		else
			delay = 0.0;
		
		if (time_of_simulation)
			bytes = tcp_stats_ptr->bytes / time_of_simulation;
		else
			bytes = 0.0;
		
		/* write the stats */
		sprintf (results, "%f\t\t%f\n", delay, bytes);

		bt_stats_get_file_name ("_tcp_stats.txt", comment, results);
		}
	
	FOUT;
}

/*
 * Function:	
 */

void	bt_stats_udp_end_of_simulation ()
{
	double	bytes;
	double  time_of_simulation;
	
	bt_UDP_stats *	udp_stats_ptr;
	const char *	comment = "#bytes/s";
	char			results[128]; 
	
	
	FIN (bt_stats_tcp_end_of_simulation ());
	
	/* init the variable */
	time_of_simulation = op_sim_time ();

	/* compute the stats */
	if ((udp_stats_ptr = bt_stats_get_udp_stats ()) != NULL)
		{
		/* print info about the statistics writing */
		printf ("\n|-------------------------------------------------------------\n");
		printf ("| UDP statistics writing:\n");
				
		if (time_of_simulation)
			bytes = udp_stats_ptr->bytes / time_of_simulation;
		else
			bytes = 0.0;
		
		/* write the stats */
		sprintf (results, "%f", bytes);
		
		bt_stats_get_file_name ("_udp_stats.txt", comment, results);
		}
	
	FOUT;
}


/*
 * Function:	
 */

void	bt_stats_get_file_name (const char * file_name, const char * line1, const char * line2)
{
	FILE *	fp;
	char *	path;
	char *	file_total_name;
	char	node_name[64];
	Objid	process_id;
	Objid	node_id;
	int		name_size;
	
	
	FIN (bt_stats_tcp_end_of_simulation ());
	
	/* init the variable */
	process_id = op_id_self ();
	node_id = op_topo_parent (process_id);
	
	/* Set the path */
	path = bt_stats_read_path (op_id_self ());
	
	/* get the name of the node */
	op_ima_obj_attr_get (node_id, "name", node_name);
	
	/* get the name of the file */
	name_size = strlen (path) + strlen (node_name) + strlen (file_name) + 1;
	file_total_name = (char *) op_prg_mem_alloc ( name_size * sizeof (char) );
	strcpy (file_total_name, path);
	strcat (file_total_name, node_name);
	strcat (file_total_name, file_name);
	
	/* open the file */
	if ((fp = fopen (file_total_name, "a")) == NULL)
		bt_stats_error ("Unable to open the file:", file_total_name, OPC_NIL);
	
	/* print info about the statistics writing */
	printf ("|\tPath where the file is write : %s\n", path);
	printf ("|\tName of file : %s%s\n", node_name, file_name);
	printf ("|-------------------------------------------------------------\n");
	
	/* write the stats */
	fprintf (fp, "%s\n", line1);
	fprintf (fp, "%s\n", line2);
	
	/* close the file */
	fclose (fp);
	
	op_prg_mem_free (file_total_name);
	op_prg_mem_free (path);
	
	FOUT;
}

/*
 * Function : 	bt_stats_get_report_name
 *
 * Description:	get the current path set in the channel node
 * 				and compute the entire path of the file to generate
 *
 * ParamIn : 	const char * name_of_file
 *				name of the file to generate
 *
 * ParamOut:	char * name
 *				name of the entire path generated
 */

char *	bt_stats_get_report_name (const char * name_of_file)
{
	char *	name;
	char	buffer[1024];
	Objid	report_objid;
	int		name_size;
	
	
	FIN (bt_stats_get_report_name (name_of_file));
	
	/* get the path according to the id of the node */
	if (op_ima_obj_attr_get (op_id_self (), "Report", &report_objid) == OPC_COMPCODE_FAILURE)
		op_ima_obj_attr_get (topo_get_channel (), "Report", &report_objid);
	
	op_ima_obj_attr_get (op_topo_child (report_objid, OPC_OBJTYPE_GENERIC, 0),
		"path", buffer);
	
	if (strcmp (buffer, "Default") == 0)
		{
		op_ima_obj_attr_get (topo_get_channel (), "Report", &report_objid);
		op_ima_obj_attr_get (op_topo_child (report_objid, OPC_OBJTYPE_GENERIC, 0),
			"path", buffer);
		}
	
	/* get the size of the name */
	name_size = strlen (buffer) + strlen (name_of_file) + 1;
	
	/* allocate some memory */
	name = (char *) op_prg_mem_alloc (name_size * sizeof (char));
	
	/* get the name */
	strcpy (name, buffer);
	strcat (name, name_of_file);
	
	FRET (name);
}

/*
 *
 */

void	bt_stats_print_report (const char * file_name, const char * header, const char * data, Boolean only_one_header)
{
	FILE *	fp;
	char *	absolute_file_name;

	
	FIN (bt_stats_print_report (file_name, header, data, only_one_header));
	
	/* Get the absolute name of the file */
	absolute_file_name = bt_stats_get_report_name (file_name);
	
	/* check if the file is already there in case we don't want print the header all the time */
	if ((fp = fopen (absolute_file_name, "r")) == NULL)
		{
		if ((fp = fopen (absolute_file_name, "w")) == NULL)
			bt_mac_error ("bt_print_csv_report ()", "Cannot open the file", OPC_NIL);
		
		if (header != OPC_NIL)
			fprintf (fp, "%s\n", header);
		
		fprintf (fp, "%s\n", data);
		}
	else
		{
		fclose (fp);
		
		if ((fp = fopen (absolute_file_name, "a")) == NULL)
			bt_mac_error ("bt_print_csv_report ()", "Cannot open the file", OPC_NIL);
		
		if ((!only_one_header) && (header != OPC_NIL)) fprintf (fp, "%s\n", header);
		fprintf (fp, "%s\n", data);
		}
	
	/* we close the file */
	fclose (fp);
	
	/* free the memory */
	op_prg_mem_free (absolute_file_name);
	
	FOUT;
}
