/*
** $RCSfile:
**
** Bluetooth model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      Frederic Mouveaux
** Module description:  Manage a list of device present in the topology
** Last Modification:   December, 11, 2002 (Olivier Rebala)
*/

/* Standard Includes */
#include "topo_discovery.h"


/* Internal Variables */

int	mac_address = 0;	/* The MAC address start at 0 */
                                /* 0 is reserved for broadcast */
/*
 * $Function:		init_topo_discovery
 * #Description:	Initialization routine to call at the Initialization
 *			of a process in order to build all the structures and
 *			variables associated with this module.
 */

void	init_topo_discovery ()
{
  FIN (init_topo_discovery ());
  
  if (master_device_list_ptr == NULL)
    master_device_list_ptr = op_prg_list_create();
  
  if (device_list_wlan_ptr == NULL)
    device_list_wlan_ptr = op_prg_list_create();	/* Start topology discovery */
  
  FOUT;
}


/*
 * Function :    topo_init_master
 *
 * Description : 
 */

void topo_init_master(char* name_of_slave, int piconet_number, Objid mac_objid, int piconet_read, int my_address)
{
  Master_Device *master_ptr;
  Slave_Device  *slave_ptr;

  
  FIN(topo_init_master(Objid mac_objid, piconet_number, my_address));

  master_ptr = topo_set_master_device (NULL, piconet_number, piconet_read, -2, mac_objid);
  op_prg_list_insert (master_device_list_ptr, master_ptr, OPC_LISTPOS_TAIL);
  slave_ptr = topo_set_slave_device (name_of_slave, piconet_number, piconet_read, 1, my_address, mac_objid);
  op_prg_list_insert (master_ptr->slave_device_list, slave_ptr, OPC_LISTPOS_TAIL);

  FOUT;
}

/*
 * Function: topo_set_master_device
 *
 * Description:
 */

Master_Device *topo_set_master_device(char* name_of_device, int piconet_number, int piconet_read, int my_address, Objid mac_objid)
{
  Master_Device *master;
  int address;
 

  FIN(topo_set_master_device(piconet_read, my_address, mac_objid));

  if ( (master = (Master_Device *) op_prg_mem_alloc (sizeof(Master_Device))) == OPC_NIL )
    topo_discovery_error ("topo_set_master_device :", "Can't register device", "Not enough memory");

  address = (my_address == -2)?mac_address++:my_address;
  
  if (name_of_device != NULL)
	  {
	  master->name_of_master = (char *) op_prg_mem_alloc ( (strlen(name_of_device) + 1 ) * sizeof(char) );
	  strcpy (master->name_of_master, name_of_device);
	  }
  else
	  master->name_of_master = NULL;
  
  master->piconet_number = piconet_number;
  master->piconet_real = piconet_read;
  master->mac_address = address;
  master->address_device = 0;
  master->mac_objid = mac_objid;
  master->slave_device_list = op_prg_list_create();

  FRET(master);
}


/*
 * Function: topo_set_slave_device
 *
 * Description: 
 */

Slave_Device *topo_set_slave_device(char* name_of_device, int piconet_number, int piconet_read, int address_device, int my_address, Objid mac_objid)
{
  Slave_Device *slave;

  FIN(topo_set_slave_device(piconet_number, piconet_read, address_device, mac_objid));

  if ((slave = (Slave_Device *) op_prg_mem_alloc (sizeof(Slave_Device))) == OPC_NIL )
      topo_discovery_error("topo_set_slave_device :", "Can't register device", "Not enough memory");

  if (name_of_device != NULL)
	  {
	  slave->name_of_slave = (char *) op_prg_mem_alloc ( (strlen(name_of_device) + 1 ) * sizeof(char));
	  strcpy (slave->name_of_slave, name_of_device);
	  }
  else
	  slave->name_of_slave = NULL;
  
  slave->piconet_number = piconet_number;
  slave->piconet_real = piconet_read;
  slave->mac_address = (my_address == -2)?mac_address++:my_address;
  slave->address_device = address_device;
  slave->mac_objid = mac_objid;

  FRET(slave);
}


/*
 *
 */

Wlan_Device *topo_set_wlan_device (Objid mac_objid, int address_device, int piconet_real)
{
  Wlan_Device *	wlan;
  int			piconet_number;
  

  FIN(topo_set_wlan_device (mac_objid, address_device, piconet_real));

  if ( (wlan = (Wlan_Device *) op_prg_mem_alloc (sizeof(Wlan_Device))) == OPC_NIL )
    topo_discovery_error ("topo_device_registration", "Can't register wlan device", "Not enough memory");
  
  if ((piconet_number = topo_get_piconet_wlan(piconet_real)) == -1)
	  piconet_number = number_of_wlan++;
  
  wlan->piconet_number = piconet_number;
  wlan->piconet_real = piconet_real;
  wlan->mac_objid = mac_objid;
  wlan->node_objid = op_topo_parent(mac_objid);
  wlan->mac_address = address_device;
  wlan->address_device = address_device;

  FRET(wlan);
}

/*
 * $Function:           topo_discovery_error
 * $Description:        Error handling procedure
 * $ParamIn:            char * : fcnt
 *                               The function in the process
 *                               where the error occured
 *                      char * : msg
 *                               The description of the error
 * $ParamIn:            char * : opt
 *                               An optional message
 */

void    topo_discovery_error (char * fcnt, char * msg, char * opt)
{
  /* Terminates simulation and print an error message. */
  FIN (topo_discovery_error (fcnt, msg, opt));
  
  op_sim_end ("Error in Topology Discovery: ", fcnt, msg, opt);
  
  FOUT;
}

/*
 * $Function:		topo_device_registration
 *
 * $Description:	register a new slave device in a list 'slave_device_list_ptr[]'
 *
 * $ParamIn:		Objid : mac
 *				The object id of the MAC process
 * 		        Objid : node
 *				The object id of the node
 *                      int : piconet_number
 *                            number of the current piconet
 *                      int * : address
 *				pointer to the MAC address for the device
 */

int	topo_slave_device_registration (char* name_of_device, Objid mac_objid, int piconet_read, int my_address)
{
  Slave_Device   *slave_ptr;
  Master_Device  *master_ptr;
  int address_device;
  int nb_of_piconet;
  int access;
  
  FIN (topo_slave_device_registration (mac_objid, piconet_read, my_address));
  
  if ( master_device_list_ptr == NULL )
    {
      init_topo_discovery ();
      topo_init_master(name_of_device, 0, mac_objid, piconet_read, my_address);
	  address_device = 1;
    }
  else
    {
      nb_of_piconet = op_prg_list_size (master_device_list_ptr);
	        
      for (access=0; access<nb_of_piconet; access++)
		  {
		  master_ptr = (Master_Device *) op_prg_list_access (master_device_list_ptr, access);
	  
		  if ( master_ptr->piconet_real == piconet_read )
			  {
			  address_device = op_prg_list_size(master_ptr->slave_device_list)+1;
		  
			  if ( address_device > NUMBER_OF_ACTIVE_STATIONS )
				  topo_discovery_error("Network Error",
					  "A bluetooth piconet can contain only 7 slaves",
					  "Check the number of slaves in each piconet");
			  slave_ptr = topo_set_slave_device (name_of_device, master_ptr->piconet_number, piconet_read, address_device, my_address, mac_objid);
			  op_prg_list_insert (master_ptr->slave_device_list, slave_ptr, OPC_LISTPOS_TAIL);
			  break;
			  }
		  }
      if ( access == nb_of_piconet )
		  {
		  address_device = 1;
		  topo_init_master(name_of_device, nb_of_piconet, mac_objid, piconet_read, my_address);
		  }
	  }
  
  FRET(address_device);
}

/*
 * $Function:		topo_device_wlan_registration
 *
 * $Description:	register a new WLAN device in a list 'device_list_wlan_ptr'
 *
 * $ParamIn:		Objid : mac
 *				The object id of the MAC process
 * 		        Objid : node
 *				The object id of the node
 *                      int : my_address
 *                            mac address of the WLAN device
 *                      int : address
 *			      The MAC address for the device
 */

int	topo_device_wlan_registration (Objid mac_objid, int my_address, int piconet_real)
{
  Wlan_Device	*	wlan_ptr;
  int				piconet_number;
  
  FIN (topo_device_wlan_registration (mac_objid, my_address, piconet_real));
  
  if (device_list_wlan_ptr == NULL)	/* If the discovery hasn't started yet */
    init_topo_discovery();	/* Start topology discovery */
  
  /* Generate a new entry in the list */
  wlan_ptr = topo_set_wlan_device (mac_objid, my_address, piconet_real);
  
  /* Add the new device entry to the list */
  op_prg_list_insert (device_list_wlan_ptr, wlan_ptr, OPC_LISTPOS_TAIL);
  
  FRET (wlan_ptr->piconet_number);
}

/*
 * $Function:           topo_master_device_registration
 * $Description:        register a new master device in the list 'master_device_list_ptr'
 * $ParamIn:            Objid : mac
 *                              The object id of the MAC process
 *                      Objid : node
 *                              The object id of the node
 *                      int : piconet_number
 *                            number of the current piconet
 *                      int * : address
 *                              pointer to the MAC address for the device
 */

void     topo_master_device_registration (char* name_of_device, Objid mac_objid, int piconet_read, int my_address)
{
  Master_Device* master_ptr;
  int access;
  int address_device;
  int nb_of_piconet;
  
  FIN (topo_master_device_registration (name_of_device, mac_objid, piconet_read, my_address));
  
  printf ("\n|---------------------------------------------------\n");
  printf ("|\tregistration of the master of the piconet %d\n", piconet_read);
  printf ("|---------------------------------------------------\n");


  address_device = 0;
  
  if ( master_device_list_ptr == NULL )
    {
      init_topo_discovery();
      master_ptr = topo_set_master_device (name_of_device, 0, piconet_read, my_address, mac_objid);
      op_prg_list_insert (master_device_list_ptr, master_ptr, OPC_LISTPOS_TAIL);
    }
  else
    {
      nb_of_piconet = op_prg_list_size (master_device_list_ptr);
      
      for ( access=0; access<nb_of_piconet; access++ )
	{
	  master_ptr = (Master_Device *) op_prg_list_access (master_device_list_ptr, access);
	  if ( master_ptr->piconet_real == piconet_read )
	    {
	      master_ptr->name_of_master = (char *) op_prg_mem_alloc ( (strlen(name_of_device) + 1) * sizeof(char));
		  strcpy (master_ptr->name_of_master, name_of_device);
	      master_ptr->mac_address = (my_address == -2)?mac_address++:my_address;
	      master_ptr->address_device = address_device;
	      master_ptr->mac_objid = mac_objid;
	      break;
	    }
	}
      if ( access == nb_of_piconet )
	{
	  master_ptr = topo_set_master_device (name_of_device, nb_of_piconet, piconet_read, my_address, mac_objid);
	  op_prg_list_insert (master_device_list_ptr, master_ptr, OPC_LISTPOS_TAIL);
	}
    }
  
  FOUT;
}

/*
 * $Function:           topo_channel_registration
 *
 * $Description:        register a new device in the list
 *
 * $ParamIn:            Objid : channel
 *                              The object id of the Channel process
 *                      int : piconet_number
 *                            number of the current piconet
 */

void	topo_channel_registration (Objid channel)
{
  FIN (topo_channel_registration (channel));
  
  channel_objid = channel;
  
  FOUT;
}

/*
 * $Function:    topo_get_master_device
 *
 * $Description: Get a pointer to the master device of 
 *               a piconet number
 *
 * $ParamIn :    int piconet_number
 *               number of the piconet
 *
 * $ParamOut:    Master_Device *master_ptr
 *               pointer to the master device
 */

Master_Device *topo_get_master_device (int piconet_number)
{
  Master_Device *master_ptr;
  int	         nb_of_master = op_prg_list_size (master_device_list_ptr); /* compute the number of master on the list */
  
  FIN (topo_get_master_device (piconet_number));
  
  if ( nb_of_master <= piconet_number ) /* if the piconet_number is unconsistent */
    topo_discovery_error ("topo_get_master_device:", "This number of piconet doesn't exist", "");
  
  /* get the pointer to the master device on the list */
  master_ptr = (Master_Device *) op_prg_list_access (master_device_list_ptr, piconet_number);
  
  FRET (master_ptr);
}

/*
 * $Function:    topo_get_slave_device_from_address
 *
 * $Description: Get a pointer to the slave device corresponding to
 *               a MAC address ans a piconet number
 *
 * $ParamIn:     int address
 *               The device address we're seeking
 *
 *               int piconet_number
 *               number of the current piconet
 *
 * $ParamOut:    Slave_Device *slave_ptr
 *               Pointer to the slave device
 */

Slave_Device *topo_get_slave_device_from_address (int address, int piconet_number)
{
  Master_Device *master_ptr;
  Slave_Device	*slave_ptr;
  int		 nb_of_master = op_prg_list_size (master_device_list_ptr);
  
  
  FIN (topo_get_slave_device_from_address (address, piconet_number));
  
  if ( nb_of_master <= piconet_number )
    topo_discovery_error ("topo_get_slave_device_from_address:", "This number of piconet doesn't exist", "");
  
  master_ptr = (Master_Device *) op_prg_list_access (master_device_list_ptr, piconet_number);
  slave_ptr = (Slave_Device *) op_prg_list_access (master_ptr->slave_device_list, address-1);
  
  FRET (slave_ptr);
}

/*
 * $Function:    topo_get_slave_device_from_address
 *
 * $Description: Get a pointer to the slave device corresponding to
 *               a MAC address ans a piconet number
 *
 * $ParamIn:     int address
 *               The device address we're seeking
 *
 *               int piconet_number
 *               number of the current piconet
 *
 * $ParamOut:    Slave_Device *slave_ptr
 *               Pointer to the slave device
 */

int	topo_index_slave_from_address (int mac_address, int piconet_number)
{
  Master_Device *	master_ptr;
  Slave_Device *	slave_ptr;
  int		 		nb_of_master = op_prg_list_size (master_device_list_ptr);
  int		 		nb_of_slave;
  int				index = 0;
  int				i;
  
  
  FIN (topo_index_slave_from_address (mac_address, piconet_number));
  
  if ( nb_of_master <= piconet_number )
    topo_discovery_error ("topo_get_slave_device_from_address:", "This number of piconet doesn't exist", "");
  
  master_ptr = (Master_Device *) op_prg_list_access (master_device_list_ptr, piconet_number);
  
  nb_of_slave = op_prg_list_size (master_ptr->slave_device_list);
  
  for (i=0; i<nb_of_slave; i++)
	  {
	  slave_ptr = (Slave_Device *) op_prg_list_access (master_ptr->slave_device_list, i);
	  
	  if (slave_ptr->mac_address == mac_address)
		  {
		  index = slave_ptr->address_device;
		  break;
		  }
	  }
  
  FRET (index);
}


/*
 * $Function:    topo_get_wlan_device_from_address
 *
 * $Description: Get the id of a device from its MAC address
 *
 * $ParamIn:     int address
 *               The device address we're seeking
 *
 * $ParamOut:    Wlan_Device *wlan_device_ptr
 *               Pointer to the WLAN device
 */

Wlan_Device *topo_get_wlan_device_from_address (int address)
{
  Wlan_Device *wlan_ptr;
  int	       access;
  int	       nb_elements = op_prg_list_size (device_list_wlan_ptr); /* compute the number of element */
                                                                      /* on the WLAN's list            */
  
  
  FIN (topo_get_wlan_device_from_address (address));
  
  /* we're looking for the WLAN device on the list */
  for ( access = 0; access < nb_elements; access++ )
    {
      wlan_ptr = (Wlan_Device *) op_prg_list_access (device_list_wlan_ptr, access);
      if ( wlan_ptr->address_device == address )
	break;
    }	
  
  /* if the WLAN device is not on the list */
  if ( access == nb_elements )
    topo_discovery_error ("Function topo_get_wlan_device_from_address",
			    "No device found for this address",
			    "Check the address you give in parameter");

  FRET (wlan_ptr);
}


/*
 * $Function:    topo_get_channel
 *
 * $Description: Get the object ID of the channel process
 *
 * $ParamOut:    Objid channel_objid
 *               The channel object ID
 */

Objid	topo_get_channel()
{
  FIN (topo_get_channel());
    
  FRET (channel_objid);
}


/*
 * $Function:    topo_get_number_of_slaves
 *
 * $Description: Get the number of slave devices present in the piconet
 *
 * $ParamIn :    int piconet_number
 *               number of the current piconet
 *
 * $ParamOut:    int nb_of_slaves
 *		 The number of slaves present in the piconet
 */

int	topo_get_number_of_slaves (int piconet_number)
{
  Master_Device *master;
  int nb_of_master;
  int nb_of_slaves;

  FIN (topo_get_number_of_slaves (piconet_number));
  
  nb_of_master = op_prg_list_size (master_device_list_ptr);

  if ( nb_of_master <= piconet_number)
    topo_discovery_error ("topo_get_number_of_slaves:", "This number of piconet doesn't exist", "");
  
  master = (Master_Device *) op_prg_list_access (master_device_list_ptr, piconet_number);

  nb_of_slaves = op_prg_list_size (master->slave_device_list);

  FRET (nb_of_slaves);
}


/*
 * $Function:    topo_get_number_of_piconet
 *
 * $Description: compute the number of piconet of the current topology
 *
 * $ParamOut:    number_of_piconet
 *               number of piconet of the topology
 */

int topo_get_number_of_piconet()
{
  int number_of_piconet;

  FIN(topo_get_number_of_piconet());

  number_of_piconet = op_prg_list_size (master_device_list_ptr);

  FRET(number_of_piconet);
}


/*
 * $Function:    topo_get_piconet_numbe
 *
 * $Description: give the number of piconet compute corresponding to
 *               the number of piconet read.
 *
 * $ParamIn:     int piconet_real
 *               Real number of piconet read in the topology
 *
 * $ParamOut:    int piconet_number
 *               piconet number registered for the piconet
 */

int topo_get_piconet_number (int piconet_real)
{
  Master_Device *master_ptr;
  int            piconet_number;
  int            access;
  int            nb_of_piconet = topo_get_number_of_piconet();

  FIN(topo_get_piconet_number (int piconet_real));

  for ( access = 0; access < nb_of_piconet; access++)
    {
      /* for each device on the list of master device */
      master_ptr = (Master_Device *) op_prg_list_access (master_device_list_ptr, access);

      if ( master_ptr->piconet_real == piconet_real )
	{
	  piconet_number = master_ptr->piconet_number;
	  break;
	}
    }

  FRET(piconet_number);
}

/*
 * $Function:    topo_get_piconet_wlan
 *
 * $Description: give the number of piconet compute corresponding to
 *               the number of piconet read.
 *
 * $ParamIn:     int piconet_real
 *               Real number of piconet read in the topology
 *
 * $ParamOut:    int piconet_number
 *               piconet number registered for the piconet
 */

int topo_get_piconet_wlan (int piconet_real)
{
  Wlan_Device *	wlan_ptr;
  int			piconet_number = -1;
  int			access;
  int			device_number = op_prg_list_size (device_list_wlan_ptr);

  FIN(topo_get_piconet_wlan (int piconet_real));

  for (access = 0; access < device_number; access++)
	  {
      /* for each device on the list of master device */
      wlan_ptr = (Wlan_Device *) op_prg_list_access (device_list_wlan_ptr, access);

      if ( wlan_ptr->piconet_real == piconet_real )
		  {
		  piconet_number = wlan_ptr->piconet_number;
		  break;
		  }
	  }

  FRET (piconet_number);
}

/*
 * $Function:    topo_choose_dest_address
 *
 * $Description: choose the destination address for the master
 *
 * $ParamIn:     int dest_address_read
 *               address read in the topology
 *
 *               int number_of_slave
 *               number of slave in the piconet
 *
 *               int old_dest_address
 *               old destination address (in case of round robin)
 *
 * $ParamOut:    int dest_address
 *               destination address for the master
 */

int topo_choose_dest_address (int dest_address_read, int number_of_slave, int old_dest_address)
{
  int dest_address;

  FIN(topo_choose_dest_address (destination_address, number_of_slave));

  /* Choose a correct destination if the Random generation is not set */
  
  if (dest_address_read == -2) /* in case of round robin */
    {
      dest_address = old_dest_address;
      dest_address++;
      if ( (dest_address %= number_of_slave + 1 ) == 0 )
	dest_address = 1;
    }
  
  if(dest_address_read >= 0 )
    dest_address = dest_address_read;  /* Fix address */
  
  if( dest_address_read <= -2 ) /* in case of broadcasting */
    topo_discovery_error ("topo_choose_destination:", "This model does not support broadcasting yet, sorry", "");
  
  FRET(dest_address);
}


/*
 *  $Function:    topo_print_device_list
 *
 *  $Description: display the content of the devices present
 *                in the current topology for a piconet
 *
 *  $ParamIn:     int piconet_number
 *                number of the current piconet
 */

void topo_print_device_list(int piconet_number)
{

  Slave_Device	*slave_ptr;
  Master_Device	*master_ptr;
  int            access;
  int            nb_of_slave;
  int		 nb_of_master = op_prg_list_size (master_device_list_ptr);

  FIN( topo_print_device_list(piconet_number) );
  
  if ( nb_of_master <= piconet_number)
    topo_discovery_error ("topo_print_device_list:", "This number of piconet doesn't exist", "");

  master_ptr = (Master_Device *) op_prg_list_access (master_device_list_ptr, piconet_number);

  printf ("\n|-----------------------------------------------------------\n");
  printf ("| %s :\n",master_ptr->name_of_master);
  printf ("| Master of the piconet number %d:\n", piconet_number);
  printf ("| The real number of this piconet is %d:\n", master_ptr->piconet_real);
  printf ("| mac_objid is %d \n",master_ptr->mac_objid);
  printf ("| mac_address is %d \n",master_ptr->mac_address);
  printf ("| address of the device : %d\n", master_ptr->address_device);

  nb_of_slave = op_prg_list_size (master_ptr->slave_device_list);

  for ( access = 0; access < nb_of_slave; access++ )
    {
      slave_ptr = (Slave_Device *) op_prg_list_access (master_ptr->slave_device_list, access);

      printf("|\n| %s :\n", slave_ptr->name_of_slave);
      printf("|\tSlave number %d (address of device)\n",slave_ptr->address_device);
      printf("|\tmac objid of this slave is %d \n",slave_ptr->mac_objid);
      printf("|\tmac_address of this slave is %d \n",slave_ptr->mac_address);
    }

  printf ("|-----------------------------------------------------------\n");

  FOUT;
}


/*
 */

void topo_print_wlan_list ()
{
  Wlan_Device*	wlan_device_ptr;
  int			access;
  char			name_of_node[64];
  int			nb_of_wlan;

  
  FIN( topo_print_device_list() );
  
  printf ("\n|-----------------------------------------------------------\n");
  printf ("| WLAN devices :\n");
  
  if ( (device_list_wlan_ptr != NULL) && ( (nb_of_wlan = op_prg_list_size (device_list_wlan_ptr)) != 0) )
	  {
	  for ( access = 0; access < nb_of_wlan; access++ )
		  {
		  wlan_device_ptr = (Wlan_Device *) op_prg_list_access (device_list_wlan_ptr, access);
		  
		  op_ima_obj_attr_get (wlan_device_ptr->node_objid, "name", name_of_node);
		  
		  printf ("|\n");
		  printf ("| ** The node %s **\n", name_of_node);
		  printf ("| \tPiconet in attribute: %d\n", wlan_device_ptr->piconet_real);
		  printf ("| \tPiconet number in the simulation: %d\n", wlan_device_ptr->piconet_number);
		  printf ("| \tmac_objid is %d \n",wlan_device_ptr->mac_objid);
		  printf ("| \tmac_address is %d \n",wlan_device_ptr->mac_address);
		  printf ("| \taddress of the device : %d\n", wlan_device_ptr->address_device);
		  }
	  }
  else
	  printf ("| \tNo WLAN device\n");
  
  printf ("|-----------------------------------------------------------\n");

  FOUT;
}

/*
 * $Function:    topo_get_address_of_slave
 *
 * $Description: Get the address of a slave name devices for a given piconet
 *
 * $ParamIn :    char* name_of_slave
 *               name of the slave device
 *
 *               int piconet_number
 *               number of the current piconet
 *
 * $ParamOut:    int slave_address
 *		 The slave address sought
 */

int	topo_get_address_of_slave (const char* name_of_slave, int piconet_number)
{
  Master_Device *master_ptr;
  Slave_Device  *slave_ptr;
  int nb_of_masters;
  int nb_of_slaves;
  int slave_address = -1;
  int access;

  FIN (topo_get_number_of_slaves (piconet_number));
  
  nb_of_masters = op_prg_list_size (master_device_list_ptr);

  if ( nb_of_masters <= piconet_number)
    topo_discovery_error ("topo_get_number_of_slaves:", "This number of piconet doesn't exist", "");
  
  master_ptr = (Master_Device *) op_prg_list_access (master_device_list_ptr, piconet_number);

  nb_of_slaves = op_prg_list_size (master_ptr->slave_device_list);

  for ( access = 0; access < nb_of_slaves; access++ )
    {
      slave_ptr = (Slave_Device *) op_prg_list_access (master_ptr->slave_device_list, access);

      if (strcmp(slave_ptr->name_of_slave, name_of_slave) == 0)
	{
	slave_address = slave_ptr->address_device;
	break;
	}
    }

  FRET (slave_address);
}

/*
 * Function : topo_discovery_free_memory ()
 *
 * Description : free memory
 */

void topo_discovery_free_memory ()
{
	Master_Device	*master_ptr;
	int				 access;
	int				 nb_of_master = op_prg_list_size (master_device_list_ptr);

	FIN(topo_free_memory ());

	for (access=0; access<nb_of_master; access++)
		{
		master_ptr = (Master_Device *) op_prg_list_access (master_device_list_ptr, access);

	    op_prg_list_free(master_ptr->slave_device_list);
		}

	op_prg_list_free (master_device_list_ptr);
	op_prg_list_free (device_list_wlan_ptr);

	FOUT;
}


/*
 * Function:	topo_register_slave_id ()
 *
 * Description:	record the mac objid of each slave in a global table
 */

void		topo_register_slave_id ()
{
	int i,j; // loop variable
	Master_Device *master_ptr;
	Slave_Device  *slave_ptr;
	int nb_of_slaves, nb_of_master = op_prg_list_size (master_device_list_ptr);
	
	
	FIN (topo_register_slave_id ());

	/* allocate some memory */
	slave_mac_objid = (Objid **) op_prg_mem_alloc (nb_of_master * sizeof(Objid *));
	
	for (i=0; i<nb_of_master; i++)
		{
		master_ptr = (Master_Device *) op_prg_list_access (master_device_list_ptr, i);

		nb_of_slaves = op_prg_list_size (master_ptr->slave_device_list);
		
		/* allocate some memory */
		slave_mac_objid[i] = (Objid *) op_prg_mem_alloc (nb_of_slaves * sizeof(Objid));
		
		for (j=0; j<nb_of_slaves; j++ )
			{
			slave_ptr = (Slave_Device *) op_prg_list_access (master_ptr->slave_device_list, j);

			slave_mac_objid[slave_ptr->piconet_number][slave_ptr->address_device] = slave_ptr->mac_objid;
			}
		}

	FOUT;
}
