/*
** rr_support.ex.c:
**
** Bluetooth model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      Frederic Mouveaux
** Module description:  RoundRobin scheduling algorithm for Bluetooth Device
** Last Modification:   January, 16th, 2003 (Olivier Rebala)
*/

#include "rr_support.h"

/*
 * Function:	rr_init_rr_support
 *
 * Description:	Initialization routine to call at the Initialization
 *				of a process in order to build all the structures and
 *				variables associated with this module.
 *
 * ParamIn:		int piconet_number
 *				number of the piconet using the function
 *
 * ParamOut:	List ** rr_bt_queue
 *				pointer to a list containing the packets
 */

List **	rr_init_support (int piconet_number)
{
  List **	rr_bt_queue;
  int 		number_of_piconet;
  int 		i;

  FIN (rr_init_support (piconet_number));
  
  /*
   * Initialize subqueues and scheduling algorithm
   * Check the first station first in the RoundRobin
   * scheduler
   */

  rr_bt_queue = (List **) op_prg_mem_alloc (RR_NB_MAX_QUEUE * sizeof(List *));
  
  for (i=0; i<RR_NB_MAX_QUEUE; i++)
	  {
	  rr_bt_queue[i] = op_prg_list_create ();
	  }
  
  number_of_piconet = topo_get_number_of_piconet();

  if (current_dest_station == NULL)
    current_dest_station = (int *) op_prg_mem_alloc (number_of_piconet * sizeof(int));

  current_dest_station[piconet_number] = SUBQ_ACTIVE_1;
  
  FRET (rr_bt_queue);
}


/*
 * $Function:		rr_pkt_enqueue
 *
 * $Description:	Enqueue a packet into the matching
 *               	subqueue reguarding to its destination
 *               	address
 *
 * $ParamIn:		List **rr_bt_queue
 *					queue in which enqueue the packet
 *
 *					Packet *pkptr
 * 					The packet to enqueue
 *
 * 					int queue
 *					The number of subqueue to modify
 */

void    rr_pkt_enqueue (List **rr_bt_queue, Packet * pkptr, int queue)
{
  FIN (rr_pkt_enqueue (rr_bt_queue, pkptr, queue));
  
  /* Check the consistence of the queue number */
  if (queue >= RR_NB_MAX_QUEUE)
	  bt_mac_error ("rr_pkt_enqueue", "Can not insert packet in subqueue", "queue selection error");
  
  /* Insert the packet in the queue */
  op_prg_list_insert (rr_bt_queue[queue], pkptr, OPC_LISTPOS_TAIL);
  
  FOUT;
}


/*
 * $Function:		rr_request_pkt_to_scheduler
 *
 * $Description: 	Use a scheduling algorithm in order
 *					to retreive the L2CAP packets put
 *					in the different subqueues.
 *					The scheduler skip the Broadcast
 *					queue as it has another priority
 *					scheme defined in the bluetooth spec
 *
 * $ParamOut:    	Packet *pkptr
 *					NULL if no packets are present
 *					in the suqueues, or the pointer
 *					to a packet to send.
 */

Packet * rr_request_pkt_to_scheduler (List **rr_bt_queue, int piconet_number)
{
  int	   old_dest_station;
  Packet * pkptr;
  
  
  FIN (rr_request_pkt_to_scheduler(rr_bt_queue, piconet_number));
  
  
  old_dest_station = current_dest_station[piconet_number]-1;
  while ( (rr_bt_queue_empty(rr_bt_queue, current_dest_station[piconet_number]) == OPC_TRUE) &&
	  (old_dest_station != current_dest_station[piconet_number]) )
    {
      /*
       * Update the current_destination_station variable
       * As a Round Robin scheduler algorithm, we
       * have to choose the next station available
       */
      current_dest_station[piconet_number]++;
      /*
       * Get the real value for the current destination
       * station, reguarding to the number of stations
       * present in the network, and take in account the
       * broadcast queue
       */
      current_dest_station[piconet_number] %= (STATION_NUMBER);
      if ((old_dest_station != SUBQ_BROADCAST) && (current_dest_station[piconet_number] == SUBQ_BROADCAST))
	current_dest_station[piconet_number]++;
    }
  
  if (current_dest_station[piconet_number] == SUBQ_BROADCAST)
    current_dest_station[piconet_number]++;
  
  if (rr_bt_queue_empty(rr_bt_queue, current_dest_station[piconet_number]) == OPC_FALSE)
    { /* Access the first packet in the subqueue */
      pkptr = rr_pk_queue_remove (rr_bt_queue, current_dest_station[piconet_number], OPC_QPOS_HEAD);
      
      if (bt_message_print) printf ("|\tRemove a Packet from the subqueue\n");
      
      /* Update the current_destination_station variable */
      current_dest_station[piconet_number]++;
      
      /* Get the real value for the current destination */
      current_dest_station[piconet_number] %= (NUMBER_OF_ACTIVE_STATIONS+1);
      if (current_dest_station[piconet_number] == SUBQ_BROADCAST)
	current_dest_station[piconet_number]++;
      
      /* START	DEBUG INFORMATION */
      if (op_sim_debug() == OPC_TRUE)
	{
	  double time_now;	        /* Keep Track of the simulation time */
	  int	 destination_station;	/* Destination station for this packet */
	  int	 packet_type;		/* The type of the packet */
	  
	  /* Get the simulation time */
	  time_now = op_sim_time();
	  /*
	   * Get the TYPE information on the packet in order
	   * to determine if it is a 1,3 or 5 TS packet
	   */
	  op_pk_nfd_get(pkptr, "TYPE", &packet_type);
	  op_pk_nfd_get(pkptr, "AM_ADDR", &destination_station);
	  if (bt_message_print) printf ("  BlueTooth MAC Scheduler Info: Packet delivery (type %d, dest %d, time %.6f).\n", packet_type, destination_station, time_now);
	}
      /* END OF	DEBUG INFORMATION */
      
      /* Now, just send the packet back to the caller */
      FRET(pkptr);
    }
  else
    {
      if (bt_message_print) printf ("|\tBlueTooth MAC Scheduler Info: No packet in subqueue.\n");
      FRET(NULL);		/* No packet to send at that time */
    }
}

/*
 * $Function:		rr_request_pkt_to_scheduler_with_poll
 *
 * $Description:	Use a scheduling algorithm in order
 *					to retreive the L2CAP packets put
 *					in the next subqueue.
 *					The scheduler skip the Broadcast
 *					queue as it has another priority
 *					scheme defined in the bluetooth spec
 *
 * $ParamIn:		List **rr_bt_queue
 *					Pointer to the list where are the packets
 *
 *					int piconet_number
 *					number of the piconet using the function
 *
 *					int number_of_slave
 *					total number of slaves in this piconet
 *
 * $ParamOut:		Packet *pkptr
 *					NULL if no packets are present
 *					in the suqueues, or the pointer
 *					to a packet to send.
 */

Packet * rr_request_pkt_to_scheduler_with_poll (List **rr_bt_queue, int piconet_number, int number_of_slave)
{
  Packet * pkptr = NULL;
  
  
  FIN (rr_request_pkt_to_scheduler_with_poll(rr_bt_queue, piconet_number, number_of_slave));

  /* increase the number of slave */
  number_of_slave++;

  /* Update the current_destination_station variable */
  current_dest_station[piconet_number]++;
  current_dest_station[piconet_number] %= (number_of_slave);
  if (current_dest_station[piconet_number] == SUBQ_BROADCAST)
    current_dest_station[piconet_number]++;
  
  if (rr_bt_queue_empty(rr_bt_queue, current_dest_station[piconet_number]) == OPC_FALSE)
    { /* Access the first packet in the subqueue */
      pkptr = rr_pk_queue_remove (rr_bt_queue, current_dest_station[piconet_number], OPC_QPOS_HEAD);
      
      if (bt_message_print) printf ("|\tRemove a Packet from the subqueue\n");
    }
  else if (bt_message_print) 
      printf ("|\tBlueTooth MAC Scheduler Info: No packet in subqueue.\n");
    

  FRET(pkptr);
}

/*
 * Function:	rr_bt_queue_empty
 *
 * Description:	Check if the selected queue is empty
 *
 * ParamIn:		List **rr_bt_queue
 *				pointer to the list where are the packets
 *
 *				int queue
 *				number of the subqueue to check
 *
 * ParamOut:	Boolean value
 *				True if the queue is empty, false otherwise
 */

Boolean rr_bt_queue_empty (List **rr_bt_queue, int queue)
{
	Boolean		value;
	int			list_size;
	
	FIN(rr_bt_queue_empty (rr_bt_queue, queue));
	
	/* Check the consistence of the queue number */
	if (queue >= RR_NB_MAX_QUEUE)
		bt_mac_error ("rr_bt_queue_empty", "Can not insert packet in subqueue", "queue selection error");
		
	/* compute the list size */
	list_size = op_prg_list_size(rr_bt_queue[queue]);
	
	/* Check if the list is empty */
	if (list_size)
		value = OPC_FALSE;
	else
		value = OPC_TRUE;
	
	FRET(value);
}


/*
 * Function:	rr_pk_queue_remove
 *
 * Description:	remove a packet from a queue
 *
 * ParamIn:		List **rr_bt_queue
 *				pointer to the list where are the packets
 *
 *				int queue
 *				number of the subqueue to access
 *
 *				int pos_index
 *				same argument as the op_prg_list_remove function
 *				give the position to access in the queue
 *
 * ParamOut:	Packet * pkptr
 *				pointer to the packet to remove of the queue
 */

Packet	*rr_pk_queue_remove (List **rr_bt_queue, int queue, int pos_index)
{
	Packet *pkptr = OPC_NIL;

	
	FIN(rr_pk_queue_remove (rr_bt_queue, queue, pos_index));
	
	/* remove the packet if the queue is not empty */
	if (rr_bt_queue_empty(rr_bt_queue, queue) == OPC_FALSE)
		pkptr = (Packet *) op_prg_list_remove (rr_bt_queue[queue], pos_index);
	
	FRET(pkptr);
}


/*
 * Function : rr_pk_queue_remove
 *
 * Description : access a packet from a queue without remove it
 *
 * ParamIn:		List **rr_bt_queue
 *				pointer to the list where are the packets
 *
 *				int queue
 *				number of the subqueue to access
 *
 *				int pos_index
 *				same argument as the op_prg_list_remove function
 *				give the position to access in the queue
 *
 * ParamOut:	Packet * pkptr
 *				pointer to the packet to access in the queue
 */

Packet	*rr_pk_queue_access (List **rr_bt_queue, int queue, int pos_index)
{
	Packet *pkptr = OPC_NIL;

	
	FIN(rr_pk_queue_access (rr_bt_queue, queue, pos_index));
	
	/* remove the packet if the queue is not empty */
	if (rr_bt_queue_empty (rr_bt_queue, queue) == OPC_FALSE)
		pkptr = (Packet *) op_prg_list_access (rr_bt_queue[queue], pos_index);
	
	FRET(pkptr);
}


/*
 * Function: 	rr_end_free
 *
 * Description:	desallocate the memory for this module
 *
 * ParamIn:		List **rr_bt_queue
 *				pointer to the list array to free
 */

void	rr_end_free (List **rr_bt_queue)
{
	int i; /* loop variable */
	
	
	FIN(rr_end_free (rr_bt_queue));
	
	/* for each list pointer in the array, we free it */
	for (i=0; i<RR_NB_MAX_QUEUE; i++)
	  op_prg_list_free (rr_bt_queue[i]);
	  
	/* free the list pointer array */
	op_prg_mem_free (rr_bt_queue);
	
	FOUT;
}
