/*
** $RCSfile:
**
** Bluetooth model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      Frederic Mouveaux
** Module description:  L2CAP Segmentation and Reassembly Support
** Last Modification:   May, 18, 2000
*/

/* Standard Includes */
#include <opnet.h>
#include "l2cap_support.h"
#include "bt_support.h"
#include "fifo_support.h"


/* Define some constants */
#define	DM1_LENGTH	17		/* Length of a DM1 packet in bytes (inc. l2cap header) */
#define	DM3_LENGTH	121		/* Length of a DM3 packet in bytes (inc. l2cap header) */
#define	DM5_LENGTH	224		/* Length of a DM5 packet in bytes (inc. l2cap header) */
#define	DM1_HEADER	1		/* Header size of a DM1 packet in bytes */
#define	DM3_HEADER	2		/* Header size of a DM3 packet in bytes */
#define	DM5_HEADER	2		/* Header size of a DM5 packet in bytes */
#define	CRC_SIZE	2		/* Size of the CRC in bytes */
#define	FIRST_PKT_CODE	10		/* L_CH code of the first packet in a sequence */
#define	NEXT_PKT_CODE	01		/* L_CH code of the next packets in a sequence */
#define	L2CAP_HEADER	32		/* L2CAP Header in bits */



/* $Function:		l2cap_segmentation				*/
/* $Description:	Process the segmentation of an L2CAP packet	*/
/*			into a DM sequence of packets.			*/
/* $ParamIn:		Packet * : pkptr				*/
/*				The packet from the higher layer	*/
/* $ParamIn:		int : src					*/
/*				The source (slave) or destination	*/
/*				(master) MAC address of the packet	*/
/*			int : subq					*/
/*				The output sub queue to update		*/
int l2cap_segmentation (Packet * pkptr, int src, int subq)
{
	int		packet_length = op_pk_total_size_get(pkptr)/8;
	int		number_of_packets = 0;
	int		last_packet_length = 0;
	int		last_pkt = 0;

	Packet *	the_packet;
	Packet *	encap_pk_ptr;
	Packet *	fec_pk_ptr;

	int 		code;					/* L2CAP code of the packet */
	int		type;					/* Type of the packet */
        int		i;					/* Loop variable */
	int		stuffing;


	FIN(l2cap_segmentation(pkptr,src,subq));


       	if ( packet_length <= DM1_LENGTH )
        {
                type = DM1_TYPE;                        /* Unique DM1 packet */
                last_packet_length = packet_length;
        }
	else
	{
		type = DM1_TYPE;
		number_of_packets = (packet_length / DM1_LENGTH);
		last_packet_length = packet_length - (number_of_packets * DM1_LENGTH);
	}

	/* Generate plain DM1 packets */
	for (i=0; i<number_of_packets; i++)
	{
        	code = (i == 0)?FIRST_PKT_CODE:NEXT_PKT_CODE;
		last_pkt = ( (last_packet_length==0) && ((i+1)==number_of_packets) )?1:0;

		encap_pk_ptr = op_pk_create(DM1_LENGTH * 8);
		fec_pk_ptr = op_pk_create (95);
		the_packet = op_pk_create_fmt ("DM1");

		/* Set packet fields */
		if ( (the_packet == OPC_NIL) ||
		(op_pk_nfd_set (the_packet, "AM_ADDR", src) == OPC_COMPCODE_FAILURE) ||
		(op_pk_nfd_set (the_packet, "Frame Body", encap_pk_ptr) == OPC_COMPCODE_FAILURE) ||
		(op_pk_nfd_set (the_packet, "FEC", fec_pk_ptr) == OPC_COMPCODE_FAILURE) ||
		(op_pk_nfd_set (the_packet, "TYPE",type) == OPC_COMPCODE_FAILURE) ||
        	(op_pk_nfd_set (the_packet, "L_CH",code ) == OPC_COMPCODE_FAILURE) ||
		(op_pk_nfd_set (the_packet, "Time Stamp", op_pk_creation_time_get(pkptr)) == OPC_COMPCODE_FAILURE) ||
		(op_pk_nfd_set (the_packet, "Last", last_pkt) == OPC_COMPCODE_FAILURE) )
			bt_mac_error ("L2CAP Segmentation", "Unable to segment packet", "");
	
		fifo_pkt_enqueue (the_packet, subq);
	}

	/* Generate single or last packets */
	if (last_packet_length > 0) 
	{
		int	info;				/* Used for stuffing */


		last_pkt = 1;				/* End of a packet sequence */
		code = (number_of_packets++)?NEXT_PKT_CODE:FIRST_PKT_CODE;

		encap_pk_ptr = op_pk_create(last_packet_length * 8);
		switch (type)
		{
			case DM1_TYPE:
				the_packet = op_pk_create_fmt (BLUETOOTH_PKT_DM1);
				info = (last_packet_length + DM1_HEADER + CRC_SIZE) * 8;
			break;
			case DM3_TYPE:
				the_packet = op_pk_create_fmt (BLUETOOTH_PKT_DM3);
				info = (last_packet_length + DM3_HEADER + CRC_SIZE) * 8;
			break;
			case DM5_TYPE:
				the_packet = op_pk_create_fmt (BLUETOOTH_PKT_DM5);
				info = (last_packet_length + DM5_HEADER + CRC_SIZE) * 8;
			break;
		}


		/* Introduce stuffing */
		stuffing = info % 10;
		if (stuffing == 0)
			fec_pk_ptr = op_pk_create( info/2 );
		else
			fec_pk_ptr = op_pk_create( (info + 10 - stuffing)/2 + (10 - stuffing) );

		/* Set packet fields */
		if ( (the_packet == OPC_NIL) ||
			(op_pk_nfd_set (the_packet, "AM_ADDR", src) == OPC_COMPCODE_FAILURE) ||
			(op_pk_nfd_set (the_packet, "Frame Body", encap_pk_ptr) == OPC_COMPCODE_FAILURE) ||
			(op_pk_nfd_set (the_packet, "TYPE",type) == OPC_COMPCODE_FAILURE) ||
			(op_pk_nfd_set (the_packet, "FEC", fec_pk_ptr) == OPC_COMPCODE_FAILURE) ||
			(op_pk_nfd_set (the_packet, "L_CH",code ) == OPC_COMPCODE_FAILURE) ||
			(op_pk_nfd_set (the_packet, "Time Stamp", op_pk_creation_time_get(pkptr)) == OPC_COMPCODE_FAILURE) ||
			(op_pk_nfd_set (the_packet, "Last", last_pkt) == OPC_COMPCODE_FAILURE) )
				bt_mac_error ("L2CAP Segmentation", "Unable to create last segment packet", "");

		fifo_pkt_enqueue (the_packet, subq);
	}

/*printf ("L2CAP Segmentation (%d, length %d, nbp %d)\n", type, packet_length, number_of_packets);*/
	FRET(number_of_packets);
}




/*	Note:	The "Last" Parameter of a packet simulate the mechanism */
/*		of packet length checking in the L2CAP.			*/
Packet * l2cap_reassembly (Packet * pkptr, int * number_of_bits, double * ctime)
{
	int	 code = -1;
	int	 last;
	Packet * new_packet = NULL;
	Packet * packet_body;
	Packet * frame_body;
	Packet * fec;
	

	FIN(l2cap_reassembly (pkptr, number_of_bits, ctime));
   

	/* Get information over the packet */
	op_pk_nfd_get(pkptr,"L_CH", &code);
	op_pk_nfd_get(pkptr,"Last", &last);

	/* If it is a packet start, then get the creation time and initialize the */
	/* number of bits received */
	if (code == 10)
	{
/*printf ("L2CAP start new packet reassembly\n");*/
		op_pk_nfd_get (pkptr, "Time Stamp", ctime);
		*number_of_bits = 0;
	}

	/* Add the number of bits received for this packet */
	*number_of_bits += op_pk_nfd_size(pkptr,"Frame Body");
/*printf ("Current pkt size: %d (code %d last %d)\n", *number_of_bits, code, last);*/

	/* Check if this packet is the last of the sequence */
	/* If it is the case, then perform the reassembly */
	if ( last )
	{
/*printf ("L2CAP Reassembly\n");*/
		new_packet = op_pk_create_fmt(BLUETOOTH_PKT_L2CAP);
		packet_body= op_pk_create(*number_of_bits-L2CAP_HEADER);
		op_pk_nfd_set(new_packet,"Frame Body", packet_body);
	}

	bt_destroy_packet(pkptr);


	FRET(new_packet);
}
