/*
** $RCSfile:
**
** Bluetooth model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      Frederic Mouveaux
** Module description:  L2CAP Segmentation and Reassembly Support
** Last Modification:   May, 18, 2000
*/

/* Standard Includes */
#include "l2cap_support.h"


/*
 * Function:	l2cap_init_support
 *
 * Description:	initialize the queue storing the L2CAP created packets
 *				This initialize takes place in the Channel module
 */

void l2cap_init_support ()
{
	FIN(l2cap_init_support ());
	
	/* Initialize the list to store the higher layer packets */
	l2cap_bt_queue = op_prg_list_create ();
	
	FOUT;
}


/*
 * Function: l2cap_init_parameters_master
 *
 * Description:	initialize the L2CAP parameters
 *				this initialize takes place in each masters
 *
 * ParamIn:		double 	*segmentation_size
 *				int 	*flag_pkt_type
 *
 *				address of the variables where the parameters
 *				will be stored				
 */

void l2cap_init_parameters (double *segmentation_size, int *flag_pkt_type)
{
	Objid	attr_compound;
	Objid	attr_l2cap_param;
	
	
	FIN(l2cap_init_parameters (segmentation_size, flag_pkt_type));
	
	/* Get the objid of the L2CAP parameter compound */
	op_ima_obj_attr_get (op_id_self(), "Bluetooth L2CAP Parameters", &attr_compound);
	attr_l2cap_param = op_topo_child(attr_compound, OPC_OBJTYPE_GENERIC, 0);
	
	/* Get the L2CAP parameters */
	op_ima_obj_attr_get (attr_l2cap_param, "L2CAP segmentation size", segmentation_size);
	op_ima_obj_attr_get (attr_l2cap_param, "L2CAP segmentation packet type", flag_pkt_type);
	
	FOUT;
}

/*
 * Function:	l2cap_segmentation
 *
 * Description:	Process the segmentation of an L2CAP packet
 *				into a DM sequence of packets.
 *
 * ParamIn:		List ** : bt_queue
 *				Pointer to the array of queue where
 *				the packets will be stored.
 *
 *				int : src
 *				The source (slave) or destination
 *				(master) MAC address of the packet
 *
 *				Device_type : device_type
 *				Type of device : master or slave
 *
 * ParamOut:	int : total_number_of_pkt
 *				number of created packets
 *
 * Modif.:		Olivier rebala
 */

int l2cap_segmentation (List **bt_queue, int src, Device_type device_type, int flag_packet_type)
{
	int		packet_length;
	int		list_size = op_prg_list_size (l2cap_bt_queue);
	int		number_of_packets;
	int		total_number_of_pkt = 0;
	int		last_packet_length = 0;
	int		last_pkt = 0;
	int		info;				/* Used for stuffing */
	int     l2cap_last;
			
	Packet *	higher_layer_pkt;
	Packet *	l2cap_packet;
	
	Packet *	the_packet;
	Packet *	encap_pk_ptr;
	Packet *	fec_pk_ptr;

	int		code;					/* L2CAP code of the packet */
	int		type;					/* Type of the packet */
    int		i;					/* Loop variable */
	int		stuffing;
	int     subq;                                   /* Subqueue number according to the src address */


	FIN(l2cap_segmentation(bt_queue, src, type));

	if (bt_message_print)
		{
		printf ("\n|---------------------------------------------------------------\n");
		printf ("| L2CAP Segmentation Infos :\n");
		printf ("| \t\tdevice type : %s\n", (device_type == MASTER_TYPE)?"Master":"Slave");
		}
	
	if (device_type == MASTER_TYPE)
	    subq = src;
	else
	    subq = SUBQ_INPUT;
	  	
	/* start the segmentation of the L2CAP packets of the list */
	while (list_size)
		{
		number_of_packets = 0;
		l2cap_packet = (Packet *) op_prg_list_remove (l2cap_bt_queue, OPC_LISTPOS_HEAD);
		
		list_size = op_prg_list_size (l2cap_bt_queue);
		packet_length = (int) op_pk_total_size_get (l2cap_packet)/8;
		
		/*printf ("| \t\tSize of the L2CAP packet: %d bytes\n", packet_length);*/
				
		/* look if this l2cap packet is the last of a data packet */
		op_pk_nfd_get (l2cap_packet, "Last", &l2cap_last);
		
		if (l2cap_last)
			op_pk_nfd_get (l2cap_packet, "HLPacket", &higher_layer_pkt);
				
		if (higher_layer_pkt == OPC_NIL)
			bt_mac_error ("L2CAP Segmentation", "Unable to get higher layer packets", "");
		
		/* Need to set the type of packet in terms of the packet size */
		if (flag_packet_type)
			{
			if ( packet_length <= DM1_LENGTH )
				{
				type = DM1_TYPE;			/* Unique DM1 packet */
				last_packet_length = packet_length;
				}
			else if ((packet_length > DM1_LENGTH) && (packet_length <= DM3_LENGTH))
				{
				type = DM3_TYPE;		/* Unique DM3 packet */
				last_packet_length = packet_length;
				}
			else if ((packet_length > DM3_LENGTH) && (packet_length <= DM5_LENGTH))      
				{
				type = DM5_TYPE;		/* Unique DM5 packet */
				last_packet_length = packet_length;
				}
			else if (packet_length > DM5_LENGTH)
				{	/* Need to generate a number of DM5 packets */
				type = DM5_TYPE;
				number_of_packets = (packet_length / DM5_LENGTH);
				last_packet_length = packet_length - (number_of_packets * DM5_LENGTH);
				}
			}
		else
			{
			if ( packet_length <= DH1_LENGTH )
				{
				type = DH1_TYPE;			/* Unique DM1 packet */
				last_packet_length = packet_length;
				}
			else if ((packet_length > DH1_LENGTH) && (packet_length <= DH3_LENGTH))
				{
				type = DH3_TYPE;		/* Unique DM3 packet */
				last_packet_length = packet_length;
				}
			else if ((packet_length > DH3_LENGTH) && (packet_length <= DH5_LENGTH))      
				{
				type = DH5_TYPE;		/* Unique DM5 packet */
				last_packet_length = packet_length;
				}
			else if (packet_length > DH5_LENGTH)
				{	/* Need to generate a number of DM5 packets */
				type = DH5_TYPE;
				number_of_packets = (packet_length / DH5_LENGTH);
				last_packet_length = packet_length - (number_of_packets * DH5_LENGTH);
				}
			}
		
		/*printf("| \t\tNumber of packets after l2cap segmentation : %d\n", number_of_packets + 1);*/
		
		
		/* Generate plain DM5 or DH5 packets */
		for (i=0; i<number_of_packets; i++)
			{
			if (flag_packet_type)
				{
				code = (i == 0)?FIRST_PKT_CODE:NEXT_PKT_CODE;
				last_pkt = ( (last_packet_length==0) && ((i+1)==number_of_packets) )?1:0;
				
				encap_pk_ptr = op_pk_create(DM5_LENGTH * 8);
				fec_pk_ptr = op_pk_create (6 + 915);
				the_packet = op_pk_create_fmt ("DM5");
				
				/* Set packet fields */
				if ( (the_packet == OPC_NIL) ||
					(op_pk_nfd_set (the_packet, "AM_ADDR", src) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "Frame Body", encap_pk_ptr) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "FEC", fec_pk_ptr) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "TYPE",type) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "L_CH",code ) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "Time Stamp", op_pk_creation_time_get(l2cap_packet)) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "Last", last_pkt) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "L2CAP Last", l2cap_last) == OPC_COMPCODE_FAILURE) )
					bt_mac_error ("L2CAP Segmentation", "Unable to segment packet", "");
				
				/* If the packet is the last of the last L2CAP packet : we put the higher layer packet in there */
				if ( (last_pkt) && (l2cap_last) )
					{
					if (op_pk_nfd_set (the_packet, "HLPacket", higher_layer_pkt) == OPC_COMPCODE_FAILURE)
						bt_mac_error ("L2CAP Segmentation", "Unable to segment packet", "");
					}
				else
					{
					if (op_pk_nfd_set (the_packet, "HLPacket", OPC_NIL) == OPC_COMPCODE_FAILURE)
						bt_mac_error ("L2CAP Segmentation", "Unable to segment packet", "");
					}
				
				/*printf("|\t the packet %d size is %f bits\n", i, op_pk_total_size_get(the_packet));*/
				
				l2cap_pkt_enqueue (bt_queue, the_packet, subq, device_type);
				}
			else
				{
				code = (i == 0)?FIRST_PKT_CODE:NEXT_PKT_CODE;
				last_pkt = ( (last_packet_length==0) && ((i+1)==number_of_packets) )?1:0;
				
				encap_pk_ptr = op_pk_create(DH5_LENGTH * 8);
				the_packet = op_pk_create_fmt ("DH5");
				
				/* Set packet fields */
				if ( (the_packet == OPC_NIL) ||
					(op_pk_nfd_set (the_packet, "AM_ADDR", src) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "Frame Body", encap_pk_ptr) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "TYPE",type) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "L_CH",code ) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "Time Stamp", op_pk_creation_time_get(l2cap_packet)) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "Last", last_pkt) == OPC_COMPCODE_FAILURE) ||
					(op_pk_nfd_set (the_packet, "L2CAP Last", l2cap_last) == OPC_COMPCODE_FAILURE) )
					bt_mac_error ("L2CAP Segmentation", "Unable to segment packet", "");
				
				/* If the packet is the last of the last L2CAP packet : we put the higher layer packet in there */
				if ( (last_pkt) && (l2cap_last) )
					{
					if (op_pk_nfd_set (the_packet, "HLPacket", higher_layer_pkt) == OPC_COMPCODE_FAILURE)
						bt_mac_error ("L2CAP Segmentation", "Unable to segment packet", "");
					}
				
				/*printf("| \t\tthe packet %d size is %f bits\n", i, op_pk_total_size_get(the_packet));*/
				
				l2cap_pkt_enqueue (bt_queue, the_packet, subq, device_type);
				}
			}
		
		/* Generate single or last packets */
		if (last_packet_length > 0) 
			{
			code = (number_of_packets++)?NEXT_PKT_CODE:FIRST_PKT_CODE;
			
			encap_pk_ptr = op_pk_create(last_packet_length * 8);
			
			switch (type)
				{
				case DM1_TYPE:
				the_packet = op_pk_create_fmt (BLUETOOTH_PKT_DM1);
				info = (last_packet_length + DM1_HEADER + CRC_SIZE) * 8;
				break;
				
				case DM3_TYPE:
				the_packet = op_pk_create_fmt (BLUETOOTH_PKT_DM3);
				info = (last_packet_length + DM3_HEADER + CRC_SIZE) * 8;
				break;
				
				case DM5_TYPE:
				the_packet = op_pk_create_fmt (BLUETOOTH_PKT_DM5);
				info = (last_packet_length + DM5_HEADER + CRC_SIZE) * 8;
				break;
				
				case DH1_TYPE:
				the_packet = op_pk_create_fmt (BLUETOOTH_PKT_DH1);
				info = (last_packet_length + DH1_HEADER + CRC_SIZE) * 8;
				break;
				
				case DH3_TYPE:
				the_packet = op_pk_create_fmt (BLUETOOTH_PKT_DH3);
				info = (last_packet_length + DH3_HEADER + CRC_SIZE) * 8;
				break;
				
				case DH5_TYPE:
				the_packet = op_pk_create_fmt (BLUETOOTH_PKT_DH5);
				info = (last_packet_length + DH5_HEADER + CRC_SIZE) * 8;
				break;
				}
			
			/* Introduce stuffing */
			stuffing = info % 10;
			if (stuffing == 0)
				fec_pk_ptr = op_pk_create( info/2 );
			else
				fec_pk_ptr = op_pk_create( (info + 10 - stuffing)/2 + (10 - stuffing) );
			
			/* Set packet fields */
			if ( (the_packet == OPC_NIL) ||
				(op_pk_nfd_set (the_packet, "AM_ADDR", src) == OPC_COMPCODE_FAILURE) ||
				(op_pk_nfd_set (the_packet, "Frame Body", encap_pk_ptr) == OPC_COMPCODE_FAILURE) ||
				(op_pk_nfd_set (the_packet, "TYPE",type) == OPC_COMPCODE_FAILURE) ||
				(op_pk_nfd_set (the_packet, "L_CH",code ) == OPC_COMPCODE_FAILURE) ||
				(op_pk_nfd_set (the_packet, "Time Stamp", op_pk_creation_time_get(l2cap_packet)) == OPC_COMPCODE_FAILURE) ||
				(op_pk_nfd_set (the_packet, "Last", 1) == OPC_COMPCODE_FAILURE) ||
				(op_pk_nfd_set (the_packet, "L2CAP Last", l2cap_last) == OPC_COMPCODE_FAILURE) )
				bt_mac_error ("L2CAP Segmentation", "Unable to create last segment packet", "");
			
			if ( (flag_packet_type) && (op_pk_nfd_set (the_packet, "FEC", fec_pk_ptr) == OPC_COMPCODE_FAILURE) )
				bt_mac_error ("L2CAP Segmentation", "Unable to create last segment packet", "");
			
			/* If the packet is the last of the last L2CAP packet : we put the higher layer packet in there */
			if (l2cap_last)
				{
				if (op_pk_nfd_set (the_packet, "HLPacket", higher_layer_pkt) == OPC_COMPCODE_FAILURE)
					bt_mac_error ("L2CAP Segmentation", "Unable to segment packet", "");
				}

			/*printf ("| \t\tthe last packet size is %f bits\n",op_pk_total_size_get(the_packet));*/
			
			l2cap_pkt_enqueue (bt_queue, the_packet, subq, device_type);
			}
		
		total_number_of_pkt += number_of_packets;
		}
	
	if (bt_message_print)
		{
		printf ("| \t\tTotal number of packets = %d\n", total_number_of_pkt);
		printf ("|---------------------------------------------------------------\n");
		}

	FRET(total_number_of_pkt);
}



/*
 * Function: l2cap_reassembly
 *
 * Description:	reassemble the data packets to construct the L2CAP_packets
 *
 * ParamIn:		Packet * pkptr
 *				Data packets to reassemble (DH and DM)
 *
 *				int * number_of_bits
 *				number of bits of the reassembled packets
 *
 *				double * ctime
 *				time stamp of the first data packet
 *
 * Note:		The "Last" Parameter of a packet simulate the mechanism
 *				of packet length checking in the L2CAP.
 *
 * Modif.:		Olivier Rebala
 */

Packet * l2cap_reassembly (Packet * pkptr, int * number_of_bits, double * ctime)
{
	int	 code = -1;
	int	 last;
	int      l2cap_last;
	Packet * new_packet = NULL;
	Packet * packet_body;
	Packet * higher_layer_pkt;
	/*Packet * frame_body;
	  Packet * fec;*/
	

	FIN(l2cap_reassembly (pkptr, number_of_bits, ctime));
   

	/* Get information over the packet */
	op_pk_nfd_get(pkptr,"L_CH", &code);
	op_pk_nfd_get(pkptr,"Last", &last);

	/* If it is a packet start, then get the creation time and initialize the */
	/* number of bits received */
	if (code == 10)
		{
		op_pk_nfd_get (pkptr, "Time Stamp", ctime);
		*number_of_bits = 0;
		}

	/* Add the number of bits received for this packet */
	*number_of_bits += op_pk_nfd_size(pkptr,"Frame Body");

	/* Check if this packet is the last of the sequence */
	/* If it is the case, then perform the reassembly */
	if ( last )
		{
		op_pk_nfd_get (pkptr, "L2CAP Last", &l2cap_last);
		new_packet = op_pk_create_fmt(BLUETOOTH_PKT_L2CAP);
		packet_body= op_pk_create(*number_of_bits-L2CAP_HEADER);
		
		if ( (new_packet == OPC_NIL) ||
			(op_pk_nfd_set (new_packet, "Last", l2cap_last) == OPC_COMPCODE_FAILURE) ||
			(op_pk_nfd_set (new_packet,"Frame Body", packet_body) == OPC_COMPCODE_FAILURE) ||
			(op_pk_nfd_set (new_packet,"Time Stamp", *ctime) == OPC_COMPCODE_FAILURE) )
			bt_mac_error ("L2CAP Segmentation", "Unable to reassemble the L2CAP packet", "");
		
		if (l2cap_last)
			{
			op_pk_nfd_get (pkptr, "HLPacket", &higher_layer_pkt);
			
			if (higher_layer_pkt == OPC_NIL)
				bt_mac_error ("L2CAP Segmentation", "Unable to get the higher layer packet", "");
			
			if ( (op_pk_nfd_set (new_packet, "HLPacket", higher_layer_pkt)) == OPC_COMPCODE_FAILURE)
				bt_mac_error ("L2CAP Segmentation", "Unable to reassemble the L2CAP packet", "");
			}	  
		}
	
	bt_destroy_packet(pkptr);


	FRET(new_packet);
}


/*
 * Function:	l2cap_destroy_packet
 *
 * Description:	destroy a L2CAP packet
 *
 * ParamIn:		Packet * pkptr
 *				pointer to the packet we want to destroy
 *
 * Author:		Olivier rebala
 */

void  l2cap_destroy_packet (Packet * pkptr)
{
  Packet * frame_body;


  FIN(l2cap_destroy_packet (pkptr));

  op_pk_nfd_get (pkptr,"Frame Body", &frame_body);
  op_pk_destroy (frame_body);
  op_pk_destroy (pkptr);

  FOUT;
}


/*
 * Function:	l2cap_pkt_enqueue
 *
 * Description:	enqueue the packet using the good function
 *				according to the device type (master or slave)
 *
 * ParamIn:		List **bt_queue
 *				pointer to the list array where the packets
 *				will be stored
 *
 *				Packet *pkptr
 *				pointer to the packet to be stored
 *
 *				int queue
 *				number of the queue (list)
 *
 *				Device_type device_type
 *				slave or master
 *				if the device is a master, we use rr_support
 *				otherwise we use fifo_support
 *
 * Author:		Olivier Rebala
 */

void l2cap_pkt_enqueue (List **bt_queue, Packet *pkptr, int queue, Device_type device_type)
{
	FIN(l2cap_pkt_enqueue (bt_queue, pkptr, queue, device_type));
	
	/* Choice the good function to enqueue the packets */
	if (device_type == MASTER_TYPE)
		rr_pkt_enqueue (bt_queue, pkptr, queue);
	else
		fifo_pkt_enqueue (bt_queue, pkptr, queue);
	
	FOUT;
}


/*
 * Function: l2cap_higher_layer_arrival
 *
 * Description:	segmentation of the higher layer into
 *				several L2CAP packets
 *				The created packets are stored in the
 *				l2cap_bt_queue list
 *
 * ParamIn:		Packet *pkptr
 *				pointer to the higher layer packet
 *
 *				double l2cap_seg_size
 *				size (in bits) of the L2CAP packets
 *				to create
 *
 * Author:		Olivier Rebala
 */

void l2cap_higher_layer_arrival (Packet *pkptr, double l2cap_seg_size)
{
	double	pkt_size, last_pkt_size;	/* Packet sizes in bits */
	int		nb_of_pkt;					/* Number of L2CAP packets to create */
	int		last;						/* flag to distinguish the last packet */
	int		i;							/* loop variable */
	Packet *the_packet, *encap_pk_ptr;	/* Packet to be generated */
	
	
	FIN(l2cap_higher_layer_arrival (pkptr, l2cap_seg_size));
		
	/* Compute the current packet size */
	pkt_size = op_pk_total_size_get (pkptr);
	
	
	/* Compute the number of packet to create */
	if (pkt_size > l2cap_seg_size)
		{
		nb_of_pkt = (int) (pkt_size/l2cap_seg_size);
		last_pkt_size = pkt_size - ((double) nb_of_pkt * l2cap_seg_size);
		}
	else
		{
		nb_of_pkt = 0;
		last_pkt_size = pkt_size;
		}
	
	
	/* Create L2CAP packets */
	for (i=0; i<nb_of_pkt; i++)
		{
		last = ( (last_pkt_size==0) && ((i+1)==nb_of_pkt) )?1:0;

		if ( ((the_packet = op_pk_create_fmt ("L2CAP")) == OPC_NIL) ||
			 ((encap_pk_ptr = op_pk_create(l2cap_seg_size)) == OPC_NIL) ||
			 (op_pk_nfd_set (the_packet, "Frame Body", encap_pk_ptr) == OPC_COMPCODE_FAILURE) ||
			 (op_pk_nfd_set (the_packet, "Last", last) == OPC_COMPCODE_FAILURE))
			bt_mac_error ("l2cap_higher_layer_arrival () :", "Unable to create L2CAP packet.", "");
		
		if (last)
			{
			if ( (op_pk_nfd_set (the_packet, "HLPacket", pkptr)) == OPC_COMPCODE_FAILURE)
				bt_mac_error ("l2cap_higher_layer_arrival () :", "Unable to create L2CAP packet.", "");
			}
		
		op_prg_list_insert (l2cap_bt_queue, the_packet, OPC_LISTPOS_TAIL);
		}
	
	if (last_pkt_size > 0)
		{
		if ( ((the_packet = op_pk_create_fmt ("L2CAP")) == OPC_NIL) ||
			 ((encap_pk_ptr = op_pk_create(last_pkt_size)) == OPC_NIL) ||
			 (op_pk_nfd_set (the_packet, "Frame Body", encap_pk_ptr) == OPC_COMPCODE_FAILURE) ||
			 (op_pk_nfd_set (the_packet, "Last", 1) == OPC_COMPCODE_FAILURE) ||
			 (op_pk_nfd_set (the_packet, "HLPacket", pkptr) == OPC_COMPCODE_FAILURE))
			bt_mac_error ("l2cap_higher_layer_arrival () :", "Unable to create L2CAP packet.", "");
				
		op_prg_list_insert (l2cap_bt_queue, the_packet, OPC_LISTPOS_TAIL);
   		}
		
	if (bt_message_print)
		{
		printf ("\n|--------------------------------------------------------\n");
		printf ("| L2CAP higher layer arrival info :\n");
		printf ("| \t\tPacket size : %f bits\n", pkt_size);
		printf ("| \t\tNumber of packet to create: %d\n", nb_of_pkt + 1 );
		printf ("|--------------------------------------------------------\n");
		}

	FOUT;
}
