/*
** bt_freq_hopping.ex.c :
**
** Bluetooth model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      TI
** Module description:  Bluetooth Common MAC Function Support
** Last Modification:   December, 3rd, 2002 (Olivier Rebala)
*/

/* Include header file */
#include "bt_freq_hopping.h"


/*
 * Function:	bt_hop_extract_short
 *
 * Description: get the bit from a 32 bits word between the ith 
 *		and the jth position in a 8 bits integer
 *
 * ParamIn:	Bt_Hop_Long word:
 *		32 bits word.
 *
 *		Bt_Hop_Short i, j:
 *		the both positions.
 *
 * ParamOut:	Bt_Hop_Short
 *		8 bits integer
 */

Bt_Hop_Short bt_hop_extract_short (Bt_Hop_Long word, Bt_Hop_Short i, Bt_Hop_Short j)
{
	Bt_Hop_Short	result;
	
	FIN (bt_hop_extract_short (word, i, j));
	
	/* shift the word twice to extract to good bit sequence */
	result = (Bt_Hop_Short)((word << (i = WORD_LENGTH_MINUS_1 - i)) >> (i+j));
	
	FRET (result)
}


/*
 * Function:	bt_hop_extract_long
 *
 * Description: get the bits from a 32 bits word between the ith
 *		and the jth position in another 32 bits word
 *
 * ParamIn:	Bt_Hop_Long word:
 *		32 bits word.
 *
 *		Bt_Hop_Short i, j:
 *		the both positions.
 *
 * ParamOut:	Bt_Hop_Long
 *		32 bits word
 */

Bt_Hop_Long bt_hop_extract_long (Bt_Hop_Long word, Bt_Hop_Short i, Bt_Hop_Short j)
{
	Bt_Hop_Long	result;
	
	FIN (bt_hop_extract_long (word, i, j));
	
	/* Shift the word twice to extract the bit sequence */
	result = (word << (i=WORD_LENGTH_MINUS_1-i)) >> (i+j);
	
	FRET (result);
}


/*
 * Function: bt_hop_extract_even
 *
 * Description: extract the first bits in a word
 *		in an even position
 *
 * ParamIn:	Bt_Hop_Long word:
 *		32 bits word
 *
 *		Bt_Hop_Short how_many_bit
 *		how many bits we want to extract
 *
 * ParamOut:	Bt_Hop_Short
 *		8 bits integer
 */

Bt_Hop_Short bt_hop_extract_even (Bt_Hop_Long word, Bt_Hop_Short how_many_bits)
{
  Bt_Hop_Short i; // loop variable
  Bt_Hop_Short	result = 0; // return value
  
  FIN (bt_hop_extract_even (word, how_many_bits));
  
  for (i=0; i < how_many_bits; i++)
    {
      result |= (word & 1) << i;
      word >>= 2;
    }
  
  FRET (result);
}

/*
 * Function: bt_hop_extract_odd
 *
 * Description: extract the first bits in a word
 *		in an odd position
 *
 * ParamIn:	Bt_Hop_Long word:
 *		32 bits word
 *
 *		Bt_Hop_Short how_many_bit
 *		how many bits we want to extract
 *
 * ParamOut:	Bt_Hop_Short
 *		8 bits integer
 */

Bt_Hop_Short	bt_hop_extract_odd	(Bt_Hop_Long word, Bt_Hop_Short how_many_bits)
{
  Bt_Hop_Short result = 0;

  FIN (bt_hop_extract_odd (word, how_many_bits));
  
  result = bt_hop_extract_even (word >> 1, how_many_bits);

  FRET (result);
}

/*
 * Function: bt_hop_connection_permute
 *
 * Description: permute the 5 first bits of an integer
 *		according to the control word as describe
 *		in the Bluetooth specification for a 
 */

Bt_Hop_Short bt_hop_connection_permute (Bt_Hop_Short z, Bt_Hop_Long control_word)
{
	FIN (permute (z, control_word));
	
	/* 
	 * permutation describe in the Bluetooth specification
	 * in CONNECTION state
	 */
	
	z = bt_hop_permute (z, 0, 3, 1, 2, (Bt_Hop_Short)((control_word >> 12) & 3));
	z = bt_hop_permute (z, 2, 4, 1, 3, (Bt_Hop_Short)((control_word >> 10) & 3));
	z = bt_hop_permute (z, 1, 4, 0, 3, (Bt_Hop_Short)((control_word >> 8) & 3));
	z = bt_hop_permute (z, 0, 2, 3, 4, (Bt_Hop_Short)((control_word >> 6) & 3));
	z = bt_hop_permute (z, 0, 4, 1, 3, (Bt_Hop_Short)((control_word >> 4) & 3));
	z = bt_hop_permute (z, 1, 2, 3, 4, (Bt_Hop_Short)((control_word >> 2) & 3));
	z = bt_hop_permute (z, 0, 1, 2, 3, (Bt_Hop_Short)((control_word >> 0) & 3));
  
	FRET (z);
}


/*
 * Function: bt_hop_permute
 *
 * Description: permute two bits of a binary numbers
 *		according to the control number
 *
 * ParamIn:	Bt_Hop_Short z:
 *		binary number for the permutation
 *
 *		Bt_Hop_Short i1, j1:
 *		first positions for the first permutation
 *
 *		Bt_Hop_Short i2, j2:
 *		second positions for the second permutation
 *
 *		Bt_Hop_Short control
 *		control number: if control % 4 equal:
 *		0: no permutation
 *		1: first permutation
 *		2: second permutation
 *		3: both permutations
 */

Bt_Hop_Short bt_hop_permute (Bt_Hop_Short z, Bt_Hop_Short i1, Bt_Hop_Short j1, Bt_Hop_Short i2, Bt_Hop_Short j2, Bt_Hop_Short control)
{
	FIN (bt_hop_permute (z, i1, j1, i2, j2, control));
	
	/* if the last bit of control is 1 */
	if ((control & 1) == 1)
		z = bt_hop_flip (z, i1, j1);

	/* if the second bit of control is 1 */
	if ((control & 2) == 2)
		z = bt_hop_flip (z, i2, j2);

	FRET (z);
}


/*
 * Function: bt_hop_flip
 *
 * Description: switch the two bits at the ith and jth position
 *		in a binary number
 *
 * ParamIn:	Bt_Hop_Short z:
 *		binary number (8 bits)
 *
 *		Bt_Hop_Short i:
 *		first position to switch
 *
 *		Bt_Hop_Short j:
 *		second position to switch
 */

Bt_Hop_Short bt_hop_flip (Bt_Hop_Short z, Bt_Hop_Short i, Bt_Hop_Short j)
{
  Bt_Hop_Short templete;
 
  FIN (bt_hop_flip (z, i, j));
  
  /* construct a binary number with a 1 at the jth and ith position, 0 elsewhere */
  templete = (1 << i) | (1 << j);
  
  /* 
   * if the both bit in the ith and jth position are not the same,
   * we switch them thanks to the number above
   */
  if ( ((z & templete) != 0) && ((z & templete) != templete) )
    z ^= templete;
  
  FRET (z);
}


/*
 * Function: bt_hop_selection
 */

Bt_Hop_Short bt_hop_selection (Bt_Hop_Long Clock, Bt_Hop_Long ULAP)
{
  Bt_Hop_Short	A,B,E,F,Y1,Y2,X,z,frequency;
  Bt_Hop_Long	C,D; 
  
  FIN (bt_hop_selection (Clock, ULAP));
  
  /*
   * Select a hop frequency according to the Bluetooth specification
   * for a 79 hop system in a connection state
   */

  X		= bt_hop_extract_short (Clock, 6, 2);
  Y1	= bt_hop_extract_short (Clock, 1, 1);
  Y2	= Y1 << 5;
  A		= bt_hop_extract_short (ULAP, 27, 23) ^ bt_hop_extract_short (Clock, 25, 21);
  B		= bt_hop_extract_short (ULAP, 22, 19);
  C		= bt_hop_extract_even (ULAP, 5) ^ bt_hop_extract_short (Clock, 20, 16);

  if (Y1 == 1)
	  C = ~C;

  D	= bt_hop_extract_long (ULAP, 18, 10) ^ bt_hop_extract_long (Clock, 15, 7);
  E	= bt_hop_extract_odd (ULAP, 7);
  F	= (bt_hop_extract_long (Clock, 27, 7) << 4) % NUMBER_OF_HOPS;
  
  /* Compute the position z in the hop table */
  z = (X+A) & BT_HOP_MODULO;
  z = bt_hop_connection_permute (z ^ B, (C << 9) | D);
  z = (z + Y2 + E + F) % NUMBER_OF_HOPS;

  /* Compute the frequency according to the position z */
  frequency = 2*z - NUMBER_OF_HOPS*(z/40);
  
  FRET (frequency);
}
