/*
** fifo_support.ex.c:
**
** Bluetooth model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      Frederic Mouveaux
** Module description:  FIFO scheduling algorithm for Bluetooth Device
** Last Modification:   January, 16th, 2003 (Olivier Rebala)
*/

/* Standard Includes */
#include "fifo_support.h"


/*
 * Function:	fifo_init_fifo_support
 *
 * Description:	Initialization routine to call at the Initialization
 *				of a process in order to build all the structures and
 *				variables associated with this module.
 *
 * ParamOut:	List ** fifo_bt_queue
 *				pointer to the new list array to enqueue the packets
 */

List **	fifo_init_support ()
{
	List **	fifo_bt_queue;
	int 	i;
	
	
	FIN (fifo_init_support (fifo_bt_queue));
	
	/* Allocation  memory */
	fifo_bt_queue = (List **) op_prg_mem_alloc (FIFO_NB_MAX_QUEUE * sizeof(List *));
	
	/* for each list pointer in the array, we initialize it */
	for (i=0; i<FIFO_NB_MAX_QUEUE; i++)
		{
			fifo_bt_queue[i] = op_prg_list_create ();
		}
			

	FRET(fifo_bt_queue);
}


/*
 * Function:	fifo_pkt_enqueue
 *
 * Description:	Enqueue a packet into the matching
 *				subqueue reguarding to its dest address
 *
 * ParamIn:		List ** fifo_bt_queue
 *				pointer to the list array where are the packets
 *
 *				Packet * pkptr
 *				The packet to enqueue
 *
 *				int subq
 *				The subqueue to modify
 */

void    fifo_pkt_enqueue (List **fifo_bt_queue, Packet * pkptr, int queue)
{
        FIN (fifo_pkt_enqueue (fifo_bt_queue, pkptr, queue));

		/* Check the consistence of the queue number */
		if (queue >= FIFO_NB_MAX_QUEUE)
			bt_mac_error ("fifo_pkt_enqueue", "Can not insert packet in subqueue", "subqueue selection error");
		
        /* Insert the packet in the subqueue */
        op_prg_list_insert (fifo_bt_queue[queue], pkptr, OPC_LISTPOS_TAIL);
	
        FOUT;
}

/*
 * Function:	fifo_pkt_enqueue_in_head
 *
 * Description:	Enqueue a packet into the matching
 *				subqueue reguarding to its dest address
 *
 * ParamIn:		List ** fifo_bt_queue
 *				pointer to the list where are the packets
 *
 *				Packet * pkptr
 *				The packet to enqueue
 *
 *				int subq
 *				The subqueue to modify
 */

void    fifo_pkt_enqueue_in_head (List **fifo_bt_queue, Packet * pkptr, int queue)
{        
	FIN (fifo_pkt_enqueue_in_head (fifo_bt_queue, pkptr, queue));
	
	/*Check the consistence of the queue number */
	if (queue > FIFO_NB_MAX_QUEUE)
			bt_mac_error ("fifo_pkt_enqueue_in_head", "Can not insert packet in queue", "queue selection error");
	
	/* Insert the packet in the queue */
	op_prg_list_insert (fifo_bt_queue[queue], pkptr, OPC_LISTPOS_HEAD);

	FOUT;
}

/*
 * Function:	fifo_request_pkt_to_scheduler
 *
 * Description:	Use a scheduling algorithm in order to retreive
 *				the L2CAP packets put in the different subqueues.
 *
 * ParamIn:		List **fifo_bt_queue
 *				pointer to the list array where are the packets
 *
 * ParamOut:	Packet * pkptr
 *				NULL if no packets are present in the suqueues,
 *				or the pointer to a packet to send.
 */

Packet *	fifo_request_pkt_to_scheduler (List **fifo_bt_queue)
{
	Packet * pkptr;


	FIN (fifo_request_pkt_to_scheduler(fifo_bt_queue));


	if (fifo_bt_queue_empty(fifo_bt_queue, SUBQ_INPUT) == OPC_FALSE)
	{ /* access the first packet in the subqueue */
	  pkptr = fifo_pk_queue_remove (fifo_bt_queue, SUBQ_INPUT, OPC_QPOS_HEAD);
	  if (bt_message_print) printf("packet remove from the subqueue\n");
	  //sleep(1);

	  /* START	DEBUG INFORMATION */
	  if (op_sim_debug() == OPC_TRUE)
	  {
		double		time_now;		/* Keep Track of the simulation time */
		int		destination_station;	/* The destination station for this packet */
		int		packet_type;		/* The type of the packet */
	
		/* Get the simulation time */
		time_now = op_sim_time();
		/* Get the TYPE information on the packet in order */
		/* to determine if it is a 1,3 or 5 TS packet      */
		op_pk_nfd_get(pkptr, "TYPE", &packet_type);
		op_pk_nfd_get(pkptr, "AM_ADDR", &destination_station);
		if (bt_message_print) printf ("  BlueTooth MAC Scheduler Info: Packet delivery (type %d, dest %d, time %.6f).\n", packet_type, destination_station, time_now);
	  }
	  /* END OF	DEBUG INFORMATION */

	  {
		double		time_now;		/* Keep Track of the simulation time */
		int		destination_station;	/* The destination station for this packet */
		int		packet_type;		/* The type of the packet */

		/* Get the simulation time */
		time_now = op_sim_time();
		/* Get the TYPE information on the packet in order */
		/* to determine if it is a 1,3 or 5 TS packet      */
		op_pk_nfd_get(pkptr, "TYPE", &packet_type);
		op_pk_nfd_get(pkptr, "AM_ADDR", &destination_station);
/*		printf ("  BlueTooth MAC Scheduler Info: Packet delivery (type %d, dest %d, time %.6f).\n", packet_type, destination_station, time_now);*/
	  }

	  /* Now, just send the packet back to the caller */
	  FRET(pkptr);
	}
	else
	{
/*	  printf ("  BlueTooth MAC Scheduler Info: No packet in subqueues.\n");*/
	  FRET(NULL);		/* No packet to send at that time */
	}
}


/*
 * Function:	fifo_bt_queue_empty
 *
 * Description:	Check if the selected queue is empty
 *
 * ParamIn:		List ** fifo_bt_queue
 *				pointer to the list where are the packets
 *
 *				int queue
 *				number of the subqueue to check
 *
 * ParamOut:	Boolean value
 *				true if the queue is empty, false otherwise
 */

Boolean fifo_bt_queue_empty (List **fifo_bt_queue, int queue)
{
	Boolean		value;
	int			list_size;
	
	FIN(fifo_bt_queue_empty (fifo_bt_queue, queue));
	
	/* Check the consistence of the queue number */
	if (queue >= FIFO_NB_MAX_QUEUE)
		bt_mac_error ("fifo_bt_queue_empty", "Can not insert packet in subqueue", "subqueue selection error");
		
	/* compute the list size */
	list_size = op_prg_list_size(fifo_bt_queue[queue]);
	
	/* Check if the list is empty */
	if (list_size)
		value = OPC_FALSE;
	else
		value = OPC_TRUE;
	
	FRET(value);
}


/*
 * Function : fifo_pk_queue_remove
 *
 * Description : remove a packet from a queue
 *
 * ParamIn:		List ** fifo_bt_queue
 *				pointer to the list where are the packets
 *
 *				int queue
 *				number of the subqueue to check
 *
 *				int pos_index
 *				same argument as the op_prg_list_remove function
 *				position in the subqueue
 *
 * ParamOut:	Packet * pkptr
 *				pointer to the packet to remove of the queue
 */

Packet	*fifo_pk_queue_remove (List **fifo_bt_queue, int queue, int pos_index)
{
	Packet *pkptr = OPC_NIL;

	
	FIN(fifo_pk_queue_remove (fifo_bt_queue, queue, pos_index));
	
	/* remove the packet if the queue is not empty */
	if (fifo_bt_queue_empty(fifo_bt_queue, queue) == OPC_FALSE)
		pkptr = (Packet *) op_prg_list_remove (fifo_bt_queue[queue], pos_index);
	
	FRET(pkptr);
}


/*
 * Function : fifo_pk_queue_access
 *
 * Description : access a packet from a queue
 *
 * ParamIn:		List ** fifo_bt_queue
 *				pointer to the list where are the packets
 *
 *				int queue
 *				number of the subqueue to check
 *
 *				int pos_index
 *				same argument as the op_prg_list_remove function
 *				position in the subqueue
 *
 * ParamOut:	Packet * pkptr
 *				pointer to the packet to access of the queue
 */

Packet	*fifo_pk_queue_access (List **fifo_bt_queue, int queue, int pos_index)
{
	Packet *pkptr = OPC_NIL;

	
	FIN(fifo_pk_queue_access (fifo_bt_queue, queue, pos_index));
	
	/* remove the packet if the queue is not empty */
	if (fifo_bt_queue_empty (fifo_bt_queue, queue) == OPC_FALSE)
		pkptr = (Packet *) op_prg_list_access (fifo_bt_queue[queue], pos_index);
	
	FRET(pkptr);
}


/*
 * Function: 	fifo_end_free
 *
 * Description:	desallocate the memory for this module
 *
 * ParamIn:		List **fifo_bt_queue
 *				pointer to the list array to free
 */

void	fifo_end_free (List **fifo_bt_queue)
{
	int i; /* loop variable */
	
	FIN(fifo_end_free (fifo_bt_queue));
	
	/* for each list pointer in the array, we free it */
	for (i=0; i<FIFO_NB_MAX_QUEUE; i++)
		op_prg_list_free (fifo_bt_queue[i]);
	
	/* free memory */
	op_prg_mem_free (fifo_bt_queue);
	
	
	FOUT;
}
