/*
** bt_support.h :
**
** Bluetooth model in Opnet
** National Institute of Standards and Technology
**
** This model was developed at the National Institute of Standards
** and Technology by employees of the Federal Government in the course
** of their official duties. Pursuant to title 17 Section 105 of the
** United States Code this software is not subject to copyright
** protection and is in the public domain. This is an experimental
** system.  NIST assumes no responsibility whatsoever for its use by
** other parties, and makes no guarantees, expressed or implied,
** about its quality, reliability, or any other characteristic.
**
** We would appreciate acknowledgement if the model is used.
**
** NIST ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION
** AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
** RESULTING FROM THE USE OF THIS SOFTWARE.
**
** Primary Author:      Frederic Mouveaux
** Module description:  Bluetooth Common MAC Function Support
** Last Modification:   January, 16th 2003 (Olivier Rebala)
*/

#ifndef	__BT_SUPPORT_H__
#define	__BT_SUPPORT_H__

/* Standard Includes */
#include <opnet.h>
#include <stdlib.h>
#include <math.h>
#include "bt_freq_hopping.h"

/* define some macros */
#ifndef         max
#define	        max(a,b)                ((a)>(b)?(a):(b))
#endif

#ifndef         min
#define	        min(a,b)                ((a)>(b)?(b):(a))
#endif

/* Define packets format */
#define		BLUETOOTH_PKT_POLL		"POLL"
#define		BLUETOOTH_PKT_NULL		"NULL"
#define		BLUETOOTH_PKT_DM1		"DM1"
#define		BLUETOOTH_PKT_DM3		"DM3"
#define		BLUETOOTH_PKT_DM5		"DM5"
#define		BLUETOOTH_PKT_DH1		"DH1"
#define		BLUETOOTH_PKT_DH3		"DH3"
#define		BLUETOOTH_PKT_DH5		"DH5"
#define		BLUETOOTH_PKT_HV1		"HV1"
#define		BLUETOOTH_PKT_HV2		"HV2"
#define		BLUETOOTH_PKT_HV3		"HV3"
#define         BLUETOOTH_WLAN_PKT_MAC          "wlan_mac"
#define         BLUETOOTH_WLAN_PKT_CONTROL      "wlan_control"


/* Define some packet types */
#define NULL_TYPE       0               /* Type of the bluetooth NULL packet */
#define POLL_TYPE       1               /* Type og the bluetooth POLL packet */
#define DM1_TYPE        3               /* Bluetooth spec type of a DM1 packet */
#define DM3_TYPE        10              /* Bluetooth spec type of a DM3 packet */
#define DM5_TYPE        14              /* Bluetooth spec type of a DM5 packet */
#define DH1_TYPE        4               /* Bluetooth spec type of a DH1 packet */
#define DH3_TYPE        11              /* Bluetooth spec type of a DH3 packet */
#define DH5_TYPE        15              /* Bluetooth spec type of a DH5 packet */
#define	HV1_TYPE	5		/* Bluetooth spec type of a HV1 packet */
#define	HV2_TYPE	6		/* Bluetooth spec type of a HV2 packet */
#define	HV3_TYPE	7		/* Bluetooth spec type of a HV3 packet */

/* Incoming Statistics and Stream Wires */
#define 	TRANSMITTER_BUSY_INSTAT		0
#define		RECEIVER_BUSY_INSTAT	    	1
#define		RECEIVER_BER_INSTAT		2
#define 	LOWER_LAYER_OUTPUT_STREAM	0
#define		LOWER_LAYER_INPUT_STREAM	0
#define 	HIGHER_LAYER_OUTPUT_STREAM	1
#define		HIGHER_LAYER_INPUT_STREAM	1
#define		HIGHER_LAYER_VC_OUTPUT_STREAM	2
#define		HIGHER_LAYER_VC_INPUT_STREAM	2
#define		BLUETOOTH_INPUT_STREAM		0
#define		WLAN_INPUT_STREAM		1


/* Self-Interrupt modes */
#define		MASTER_CLK_TIMER		5
#define		SLAVE_CLK_TIMER			4
#define		PKT_WAITING			1
#define		PKT_DELIVERY			2


/* Define some constant */
#define	NUMBER_OF_ACTIVE_STATIONS	7	/* Maximum number of stations in a Piconet */
#define BLUETOOTH_STA				0

#define NUMBER_OF_SLOTS		 134217728	/* 2^27 modulo for slots counter */
#define	TS			   .000625	/* The TS unit is 625uS */
#define	AM_ADDR_BROADCAST		 0	/* The MAC address for Broadcast Packets */
#define	TSCO_2SLOTS		2*TS		/* The interarrival time of packets, every 2 slots */
#define	TSCO_4SLOTS		4*TS		/* A time slot is 625uS */
#define	TSCO_6SLOTS		6*TS

#define	TSCO_RAW_HV1	80			/* The number of data bits for an HV1 packet */
#define	TSCO_RAW_HV2	2*TSCO_RAW_HV1		/* Same for HV2 */
#define	TSCO_RAW_HV3	3*TSCO_RAW_HV1		/* and HV3 */

#define         MAX_MASTER_NUMBER       10      /* define the maximum number of master */

#define   bt_bit_rate 1000000.0    /* Bluetooth channel bit rate */

//#define NUMBER_OF_PICONET  15 /* define the maximum number of piconets */
//#define NUMBER_OF_WLAN 15 /* define the maximum number of wlan */

#define TIME_SHIFT  0.0000025 /* shift the time for each piconet */

#define	STATION_NUMBER   (NUMBER_OF_ACTIVE_STATIONS+1)	/* 7 actives stations max + 1 Master */

#define  DS  1          /* Direct Sequence  */
#define  FH  2          /*Frequency Hopping */
#define  IR  3          /*Infra Red         */

/* Boolean to know when the first packet is created */
Boolean bt_txrx_first_packet;

/* type for distinguishing the different device types */
typedef enum
{
  MASTER_TYPE,
  SLAVE_TYPE,
  WLAN_TYPE
} Device_type;

typedef enum
{
  TYPE_MP3 = 1,
  TYPE_FTP = 2,
  TYPE_HTTP = 3,
  TYPE_VOICE = 4
} BT_profile_type;

/* structure for the channel buffer */
typedef struct
{
	double delivery_time;
	double pkt_size;
	double type;
	double power;
	double freq_transmission;
	double tx_x;
	double tx_y;
	double rx_x;
	double rx_y;
	int    piconet;
	double pkt_id;
} Channel_buffer;


double    wlan_bit_rate ;          /* Wlan channel bit rate */
double    WLAN;
int Phy_Char;       /*wlan mode of transmission*/

/* Get the number of Bluetooth piconet */
int	number_of_piconet;
/* Get the number of WLAN */
int	number_of_wlan;

/* Enable or disable the printf message */
int	bt_message_print;

int *packet_counter	[STATION_NUMBER];

Boolean	*	bluetooth_packet_delivery;	/* Packet delivery flag */
Boolean	*	wlan_packet_delivery;		/* Packet delivery flag */
int *		txrx_slot_number;		/* Slot counter */
int *		packet_delivery_clk;		/* Clock cycles remaining in delivery */
int *		txrx_hop_freq_offset;		/* The current from base offset hop frequency */


/* is a pointer where all the information */
/* about interfernece will be stocked     */
double  **interference_ptr;

/* Define the macros for the events */
/* Clock Event Generated each TS */
#define	MASTER_CLK_TIMER_EXPIRED	((op_intrpt_type () == OPC_INTRPT_SELF ||\
					  op_intrpt_type () == OPC_INTRPT_REMOTE) &&\
					 op_intrpt_code() == MASTER_CLK_TIMER)

#define	SLAVE_CLK_TIMER_EXPIRED		(op_intrpt_type () == OPC_INTRPT_REMOTE && \
					op_intrpt_code() == SLAVE_CLK_TIMER)


#define PACKET_DELIVERY                 (bluetooth_packet_delivery[piconet_number] || packet_delivery_clk[piconet_number] > 0)

/* Test the parity of a TS following its number */
/* OPC_FALSE = Even, OPC_TRUE = Odd */
#define	TIME_SLOT_PARITY_ODD		(txrx_slot_number[piconet_number]%2)

/* Test is a packet is waiting to be transmitted */
#define	PACKET_WAITING			(packets_in_subqueues != 0)

/* End a packet delivery */
#define	DELIVERY_ENDED			(op_intrpt_type () == OPC_INTRPT_SELF && \
					op_intrpt_code() == PKT_DELIVERY)

/* A packet has arrived from the higher layer */
#define	PACKET_FROM_HIGHER_LAYER	(op_intrpt_type () == OPC_INTRPT_STRM && \
					(op_intrpt_strm() == HIGHER_LAYER_INPUT_STREAM || \
					op_intrpt_strm() == HIGHER_LAYER_VC_INPUT_STREAM))

/* A packet has arrived from the lower layer */
#define	PACKET_FROM_LOWER_LAYER		(op_intrpt_type () == OPC_INTRPT_STRM && \
					op_intrpt_strm() == LOWER_LAYER_INPUT_STREAM)

/* We are ready to receive a packet */
#define READY_TO_RECEIVE		(PACKET_FROM_LOWER_LAYER)


/* Define the way the packets are going thru the PHY Channel */
enum { MASTER_TO_SLAVE, SLAVE_TO_MASTER };

/* Define function macros */
#define	convert_time_to_ts(t)	(floor((t+0.000001)/TS)*TS)


/* Function prototypes declaration */
void	bt_init_bt_support ();
void	bt_mac_error (char * fcnt, char * msg, char * opt);
void	bt_generate_next_self_event (double next_evt_time, int evt_code);
int	bt_get_number_of_ts_from_type (int packet_type);
int	bt_get_overhead_from_type (int packet_type);
void	bt_next_slot (int * txrx_s_num);
void	bt_clock (int ulap, int piconet_number);
void	bt_set_tx_objid (int index, Objid tx_id, int piconet_num);
void	bt_set_rx_objid (int index, Objid rx_id, int piconet_num);
void	bt_set_master_transmission (Packet *pkptr, int piconet_number);
void	bt_set_slave_transmission (Packet *pkptr, int piconet_number);
void	bt_set_tx_packet_delivery_clk (Packet * pkptr, int piconet_number);
void	bt_set_rx_packet_delivery_clk (Packet * pkptr);
void	bt_destroy_packet (Packet * pkptr);
void	bt_print_debug(const char* message);
void	bt_print_count_packet(int address, int piconet_number);
char*	bt_print_type_packet (int type);
int	bt_size_of_pkt_type (const char *packet_type);
char *	bt_get_report_name (const char * name_of_file, Objid obid);
void	bt_end_bt_support ();


#endif	/* __BT_SUPPORT_H__ */
