// This software was developed at the National Institute of Standards 
// and Technology by employees of the Federal Government in the course 
// of their official duties. Pursuant to title 17 Section 105 of the 
// United States Code this software is not subject to copyright 
// protection and is in the public domain. 
// 
// We would appreciate acknowledgement if the software is used.



// File: ieee802.11b.h
// Last modified: 11 July 2001
// Author(s):
//   Amir Soltanian
//   T. A. Hall
//   Wireless Communications Technologies Group
//   National Institute of Standards and Technology
//   100 Bureau Drive, STOP 8920
//   Gaithersburg, MD 20899-8920
//   {amirs,timhall}@nist.gov
//
#ifndef  __ieee802_11b_h__
#define  __ieee802_11b_h__

#include "basetype.h"

// Number of samples per CCK bit (NsCCK)
// Number of bits per CCK code (codeLength)
enum { NsCCK=4, codeLength=8 };
//** End 11 Mb/s (CCK) only **//

// IEEE 802.11b receiver and transmitter classes:

class IEEE802_11b_Receiver : public Receiver {
public:
    IEEE802_11b_Receiver(int bitrate);
    virtual ~IEEE802_11b_Receiver();

    virtual Bits receive(const Signal& input);
    virtual void reset();
    virtual int delay() const;

    void setRollOff(double rollOff);
    double getRollOff() const;

private:
    //** All bitrates **//
	FIRFilter m_H;                      // RSRC pulse shaping filter
    double m_rollOff;                   // roll-off factor for RSRC pulse-shaping filter
	int m_bitDelay;                     // delay in terms of bit    
	const int m_delay;                  // total system delay in terms of number of samples
    //** End all bitrates **//

    //** 1 Mb/s only **//
    Bits receive_1Mb_s(const Signal& input);
	bool receiveBit(const Signal& inputSlice);
	Sample despreader(const Signal& X); //despreading filter
	bool diffDecoder(const Sample& X);  // differential decoder

	Sample m_diffDecMem;                // differential decoder memory
	static const double m_nchip[11];    // PN Barker code at the receiver
	bool m_receiveFlag;					// indicates the recieption of the first bit
	Signal m_receiverBuffer;			// reciver input buffer
    //** End 1 Mb/s only **//

    //** 11 Mb/s (CCK) only **//
    Bits receive_11Mb_s(const Signal& input);
	Bits receiveCode(const Signal& inputSlice);
};

class IEEE802_11b_Transmitter : public Transmitter {
public:
    IEEE802_11b_Transmitter(int bitrate, double phase_shift=0.0);
    virtual ~IEEE802_11b_Transmitter();

    virtual Signal transmit(const Bits& input, double df);
    virtual int minInputLength() const;
    void reset();

    void setRollOff(double rollOff);
    double getRollOff() const;

private:
    //** All bitrates **//
    FIRFilter m_H;      // RSRC pulse shaping filter  
    double m_rollOff;   //raised cosine filter roll-off factor
    //** End all bitrates **//

    //** 1 Mb/s only **//
    Signal modulate_1Mb_s(const Bits& input, double df);
    Bits diffEncode(const Bits& input);     // differential encoder
    Bits addChips(const Bits& input);       // spread filter
    bool m_diffEncMem;          // differential encoder memory
    static const Bits m_chip;   // Barker code
    //** End 1 Mb/s only **//

    //** 11 Mb/s (CCK) only **//
    Signal modulate_11Mb_s(const Bits& input);
    void initCodeTable();

    Signal m_codeTable;     // 8*256 code table for all combinations of CCK code
    //** End 11 Mb/s (CCK) only **//

    int m_freqTrack; // memory for the instantaneous phase generated by the frequency offset
};

#endif  //__ieee802_11b_h__

