"""
Bipartite Graph Generation
"""
__author__ = 'Brian Cloteaux (brian.cloteaux@nist.gov)'

# Mathematical and Computational Sciences Division
# National Institute of Standards and Technology,
# Gaithersburg, MD USA
#
# This software was developed at the National Institute of Standards and
# Technology (NIST) by employees of the Federal Government in the course
# of their official duties. Pursuant to title 17 Section 105 of the
# United States Code, this software is not subject to copyright protection
# and is in the public domain. NIST assumes no responsibility whatsoever for
# its use by other parties, and makes no guarantees, expressed or implied,
# about its quality, reliability, or any other characteristic.

import networkx as nx

def make_bipartite_guillaume_latapy(G, create_using=None, name=None):
    """ 
    Returns a two-mode graph A whose projection is a given graph G. 

    Notes
    -----
    The nodes of the graph G are retained as the "Bottom" nodes of 
    returned bipartite graph A, and the nodes of the affiliations are
    return as the "Top" nodes.
    The algorithm is based on Guillaume and Latapy [1]_.

    References
    ----------
    ..[1] J.L. Guillaume & M. Latapy, "Bipartite structure of
        all complex networks",  Information Processing Letters, 90,
        pg. 215-221, 2004.
        http://dx.doi.org/10.1016/j.ipl.2004.03.007
    """ 
    if create_using:
        A = create_using
        A.clear()
    else:
        A = nx.Graph()
    if name is not None:
        A.name = name

    A.add_nodes_from(G)
    A.node_type = {}
    for n in A.nodes():
        A.node_type[n] = "Bottom" 

    cliq = list(nx.find_cliques(G))
    top = set([]) 
    for e in G.edges():
        max = []
        n1, n2 = e
        for c in cliq:
            if n1 in c and n2 in c:
                if len(c) > len(max):
                    max = c
        top.add(frozenset(max))

    a = -1
    for t in top:
        A.add_node(a)
        A.node_type[a] = "Top" 
        for n in t:
            A.add_edge(n, a)
        a -= 1
    return A
