--=============================================================================
--
-- Name:		CRC32
--
-- Purpose:	This module adds the crc32 at the end of each packet passing through
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. TX_FRAME
-- 2. MUX4_1
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================

LIBRARY ieee;
LIBRARY UNISIM;
USE ieee.std_logic_1164.ALL;
USE ieee.numeric_std.ALL;
USE UNISIM.Vcomponents.ALL;

entity CRC32 is
    Port (
					CLK_CRC32						: in std_logic;
					RESET								: in std_logic;

					DATA_CRC32_IN 			: in std_logic_vector(7 downto 0);
					EN_DATA_CRC32_IN		: in std_logic;
					DATA_CRC32_OUT 			: out std_logic_vector(7 downto 0);
					EN_DATA_CRC32_OUT		: out std_logic
					);
end CRC32;

architecture RTL of CRC32 is

type stateType is	(idle, send_data, send_crc1,send_crc2,send_crc3,send_crc4);
signal next_state : stateType;

SIGNAL P						:  std_logic_vector(7 downto 0);
SIGNAL next_crc 				:  std_logic_vector(31 downto 0);

begin

--================================================================================================================================
-- These flip-flops are to delay of one clock edge the data in order to start computation of the CRC32 right on the first data.
--================================================================================================================================
I_Q10 : FD
   PORT MAP (C=> CLK_CRC32, D=>DATA_CRC32_IN(0), Q=>P(0));
I_Q11 : FD
   PORT MAP (C=> CLK_CRC32, D=>DATA_CRC32_IN(1), Q=>P(1));
I_Q12 : FD
   PORT MAP (C=> CLK_CRC32, D=>DATA_CRC32_IN(2), Q=>P(2));
I_Q13 : FD
   PORT MAP (C=> CLK_CRC32, D=>DATA_CRC32_IN(3), Q=>P(3));
I_Q14 : FD
   PORT MAP (C=> CLK_CRC32, D=>DATA_CRC32_IN(4), Q=>P(4));
I_Q15 : FD
   PORT MAP (C=> CLK_CRC32, D=>DATA_CRC32_IN(5), Q=>P(5));
I_Q16 : FD
   PORT MAP (C=> CLK_CRC32, D=>DATA_CRC32_IN(6), Q=>P(6));
I_Q17 : FD
   PORT MAP (C=> CLK_CRC32, D=>DATA_CRC32_IN(7), Q=>P(7));



--================================================================================================================================
-- In this process, I am computing the CRC32 at each new clock edge and new set of 8 bits data.
-- The state machine starts when the is data on then bus: when EN_DATA_CRC32_IN goes up.
-- The computation is based on the application note of xilinx :http://www.xilinx.com/xapp/xapp209.pdf
-- When all the data passed through, this module adds the computed CRC32 at the end.
--================================================================================================================================
unpack : process(CLK_CRC32,RESET)
begin
if RESET ='0' then
	next_state <= idle;
elsif rising_edge(CLK_CRC32) then
		
		case next_state is
			when idle =>
				next_crc<="11111111111111111111111111111111";
				if EN_DATA_CRC32_IN = '1' then
					next_state <= send_data;
				else
					next_state <= idle;
				end if;
			when send_data =>
				next_crc(0) <= next_crc(30) XOR P(7) XOR next_crc(24) XOR P(1);
				next_crc(1) <= next_crc(30) XOR next_crc(31) XOR P(6) XOR P(7) XOR next_crc(24) XOR P(0) XOR next_crc(25) XOR P(1);
				next_crc(2) <= next_crc(30) XOR next_crc(31) XOR P(5) XOR P(6) XOR P(7) XOR next_crc(24) XOR next_crc(25) XOR next_crc(26) XOR P(0) XOR P(1);
				next_crc(3) <= P(4) XOR next_crc(31) XOR P(5) XOR P(6) XOR next_crc(25) XOR next_crc(26) XOR next_crc(27) XOR P(0);
				next_crc(4) <= next_crc(30) XOR P(4) XOR P(5) XOR P(7) XOR next_crc(24) XOR next_crc(26) XOR next_crc(27) XOR next_crc(28) XOR P(1) XOR P(3);
				next_crc(5) <= next_crc(30) XOR P(4) XOR next_crc(31) XOR P(6) XOR P(7) XOR next_crc(24) XOR next_crc(25) XOR next_crc(27) XOR next_crc(28) XOR next_crc(29) XOR P(0) XOR P(1) XOR P(2) XOR P(3);
				next_crc(6) <= next_crc(30) XOR next_crc(31) XOR P(5) XOR P(6) XOR next_crc(25) XOR next_crc(26) XOR next_crc(28) XOR next_crc(29) XOR P(0) XOR P(1) XOR P(2) XOR P(3);
				next_crc(7) <= P(4) XOR P(5) XOR next_crc(31) XOR P(7) XOR next_crc(24) XOR next_crc(26) XOR next_crc(27) XOR next_crc(29) XOR P(0) XOR P(2);
				next_crc(8) <= P(4) XOR P(6) XOR P(7) XOR next_crc(24) XOR next_crc(25) XOR next_crc(27) XOR next_crc(28) XOR next_crc(0) XOR P(3);
				next_crc(9) <= P(5) XOR P(6) XOR next_crc(25) XOR next_crc(26) XOR next_crc(28) XOR next_crc(29) XOR P(2) XOR next_crc(1) XOR P(3);
				next_crc(10) <= P(4) XOR P(5) XOR P(7) XOR next_crc(24) XOR next_crc(26) XOR next_crc(27) XOR next_crc(29) XOR P(2) XOR next_crc(2);
				next_crc(11) <= P(4) XOR next_crc(3) XOR P(6) XOR P(7) XOR next_crc(24) XOR next_crc(25) XOR next_crc(27) XOR next_crc(28) XOR P(3);
				next_crc(12) <= next_crc(30) XOR P(5) XOR next_crc(4) XOR P(6) XOR P(7) XOR next_crc(24) XOR next_crc(25) XOR next_crc(26) XOR next_crc(28) XOR next_crc(29) XOR P(1) XOR P(2) XOR P(3);
				next_crc(13) <= P(4) XOR next_crc(30) XOR next_crc(31) XOR P(5) XOR P(6) XOR next_crc(5) XOR next_crc(25) XOR next_crc(26) XOR next_crc(27) XOR next_crc(29) XOR P(0) XOR P(1) XOR P(2);
				next_crc(14) <= P(4) XOR next_crc(30) XOR P(5) XOR next_crc(31) XOR next_crc(6) XOR next_crc(26) XOR next_crc(27) XOR next_crc(28) XOR P(0) XOR P(1) XOR P(3);
				next_crc(15) <= P(4) XOR next_crc(31) XOR next_crc(7) XOR next_crc(27) XOR next_crc(28) XOR next_crc(29) XOR P(0) XOR P(2) XOR P(3);
				next_crc(16) <= P(7) XOR next_crc(24) XOR next_crc(8) XOR next_crc(28) XOR next_crc(29) XOR P(2) XOR P(3);
				next_crc(17) <= next_crc(30) XOR P(6) XOR next_crc(25) XOR next_crc(9) XOR next_crc(29) XOR P(1) XOR P(2);
				next_crc(18) <= next_crc(30) XOR P(5) XOR next_crc(31) XOR next_crc(26) XOR P(0) XOR P(1) XOR next_crc(10);
				next_crc(19) <= P(4) XOR next_crc(31) XOR next_crc(27) XOR P(0) XOR next_crc(11);
				next_crc(20) <= next_crc(12) XOR next_crc(28) XOR P(3);
				next_crc(21) <= next_crc(13) XOR next_crc(29) XOR P(2);
				next_crc(22) <= next_crc(14) XOR P(7) XOR next_crc(24);
				next_crc(23) <= next_crc(30) XOR P(6) XOR P(7) XOR next_crc(24) XOR next_crc(15) XOR next_crc(25) XOR P(1);
				next_crc(24) <= next_crc(31) XOR P(5) XOR P(6) XOR next_crc(25) XOR next_crc(16) XOR next_crc(26) XOR P(0);
				next_crc(25) <= P(4) XOR P(5) XOR next_crc(26) XOR next_crc(17) XOR next_crc(27);
				next_crc(26) <= next_crc(30) XOR P(4) XOR P(7) XOR next_crc(24) XOR next_crc(27) XOR next_crc(18) XOR next_crc(28) XOR P(1) XOR P(3);
				next_crc(27) <= next_crc(31) XOR P(6) XOR next_crc(25) XOR next_crc(28) XOR next_crc(19) XOR next_crc(29) XOR P(0) XOR P(2) XOR P(3);
				next_crc(28) <= next_crc(30) XOR P(5) XOR next_crc(26) XOR next_crc(29) XOR P(1) XOR P(2) XOR next_crc(20);
				next_crc(29) <= P(4) XOR next_crc(30) XOR next_crc(21) XOR next_crc(31) XOR next_crc(27) XOR P(0) XOR P(1);
				next_crc(30) <= next_crc(31) XOR next_crc(22) XOR next_crc(28) XOR P(0) XOR P(3);
				next_crc(31) <= next_crc(23) XOR next_crc(29) XOR P(2);
				if EN_DATA_CRC32_IN = '1' then
					next_state <= send_data;
				else
					next_state <= send_crc1;
				end if;
			when send_crc1 =>
				next_state <= send_crc2;
			when send_crc2 =>
				next_state <= send_crc3;
			when send_crc3 =>
				next_state <= send_crc4;
			when send_crc4 =>
				next_state <= idle;
		end case;
end if;
end process unpack;

--================================================================================================================================
-- Creation of the new DATA ENABLE for the next module TX_FRAME.
--================================================================================================================================
with next_state select
EN_DATA_CRC32_OUT <=	'0'	 when idle,
											'1'	 when others;

--================================================================================================================================
-- Creation of DATA OUT from the DATA IN but with the computed CRC32 at the end.
--================================================================================================================================
with next_state select
DATA_CRC32_OUT <=	P when send_data,
									not (next_crc(24) & next_crc(25) & next_crc(26) & next_crc(27) & next_crc(28) & next_crc(29) & next_crc(30) & next_crc(31)) when send_crc1,
									not (next_crc(16) & next_crc(17) & next_crc(18) & next_crc(19) & next_crc(20) & next_crc(21) & next_crc(22) & next_crc(23)) when send_crc2,
									not (next_crc(8) & next_crc(9) & next_crc(10) & next_crc(11) & next_crc(12) & next_crc(13) & next_crc(14) & next_crc(15)) when send_crc3,
									not (next_crc(0) & next_crc(1) & next_crc(2) & next_crc(3) & next_crc(4) & next_crc(5) & next_crc(6) & next_crc(7)) when send_crc4,
									"00000000" when others;

end RTL;
