--=============================================================================
--
-- Name:		SRAM_INTERFACE
--
-- Purpose:	This module takes the data in the input memory 
--					and place it in the external SRAM memory 
--					and retrive it when asked.
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. CAPTURE
-- 2. MEM_READ_INCOMING_MSG
-- 3. CAPTURE_UDP_FRAME
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

library UNISIM;
use UNISIM.vcomponents.ALL;

entity SRAM_INTERFACE is
    Port (
					CLK_SRAM_INTERFACE				: in std_logic;
					CLK_SRAM_INTERFACE90			: in std_logic;
					CLKD2_SRAM_INTERFACE			: in std_logic;

					RESET_SRAM_INTERFACE			: in std_logic;
					START_CAPTURE							: in std_logic;

					SRAM_ADDR									: out std_logic_vector(18 downto 0);
					SRAM_r_w									: out std_logic;
					SRAM_OE										: out std_logic;
					SRAM_CE										: out std_logic;
					SRAM_IO01									: inout std_logic_vector(7 downto 0);
					SRAM_IO02									: inout std_logic_vector(7 downto 0);
					SRAM_IO03									: inout std_logic_vector(7 downto 0);
					SRAM_IO04									: inout std_logic_vector(7 downto 0);

					REQUEST_OLD_PACKET				: in std_logic;
					SELECT_REQUEST_OLD_PACKET	: out std_logic;
					OLD_NUMERO_PACKET					: in std_logic_vector(10 downto 0);

					PACKET_READY_CAPTURE_UDP	: out std_logic;
					CAPTURE_UDP_BUZZY					: in std_logic;
					NUMERO_PACKET_CAPTURE_UDP	: out std_logic_vector(15 downto 0);
					ADDR_CAPTURE_UDP					: in std_logic_vector(9 downto 0);
					EN_DATA_CAPTURE_UDP				: in std_logic;
					DATA_CAPTURE_UDP					: out std_logic_vector(7 downto 0);

					PACKET_READY_CAPTURE			: in std_logic;
					ADDR_CAPTURE							: out std_logic_vector(8 downto 0);
					EN_DATA_CAPTURE						: out std_logic;
					DATA_CAPTURE							: in std_logic_vector(15 downto 0)
					);
end SRAM_INTERFACE;

architecture RTL of SRAM_INTERFACE is

type stateType is	(STATE0, STATE1, STATE2, STATE3, STATE4, STATE5, STATE6, STATE7, STATE8);
signal next_state : stateType;

signal reset												:std_logic;

signal s_PACKET_READY_CAPTURE				: std_logic;
signal s_EN_DATA_CAPTURE						: std_logic;
signal s_write											: std_logic;
signal s_OLD_NUMERO_PACKET					: std_logic_vector(10 downto 0);
signal EXT_MEMORY_IN_ENABLE					: std_logic;

signal addr_current									: std_logic_vector(15 downto 0);
signal addr_write										: std_logic_vector(7 downto 0);
signal addr_write_tmp								: std_logic_vector(7 downto 0);
signal addr_read_old_packet					: std_logic_vector(7 downto 0);
signal addr_read_current_packet			: std_logic_vector(7 downto 0);

signal packet_current_ready					: std_logic;

signal MEM_ADDR 										: std_logic_vector (8 downto 0);
signal MEM 													: std_logic_vector (15 downto 0);
signal ENABLE_GAUCHE 								: std_logic;

signal DATA_CAPTURE0								: std_logic_vector (15 downto 0);
signal DATA_CAPTURE1								: std_logic_vector (15 downto 0);

--signal DATA_CAPTURE_UDP_tmp0				: std_logic_vector(7 downto 0);
--signal DATA_CAPTURE_UDP_tmp1				: std_logic_vector(7 downto 0);

signal MEM_DATA1										: std_logic_vector(15 downto 0);
signal MEM_DATA0										: std_logic_vector(15 downto 0);
signal MEM_DATA1_tmp								: std_logic_vector(15 downto 0);
signal MEM_DATA0_tmp								: std_logic_vector(15 downto 0);

signal A														: std_logic_vector (15 downto 0);
signal B														: std_logic_vector (15 downto 0);

--signal clkd2												: std_logic;
signal ENA1 												: std_logic;
signal ENA2													: std_logic;
signal ENA3 												: std_logic;
signal ENA4 												: std_logic;

signal ENB1 												: std_logic;
signal ENB2 												: std_logic;
signal ENB3 												: std_logic;
signal ENB4 												: std_logic;

signal ENB1_i												: std_logic;
signal ENB2_i												: std_logic;
signal ENB3_i												: std_logic;

signal DOB01 												: std_logic_vector (7 downto 0);
signal DOB02 												: std_logic_vector (7 downto 0);
signal DOB03												: std_logic_vector (7 downto 0);
signal DOB04 												: std_logic_vector (7 downto 0);

begin

reset <= RESET_SRAM_INTERFACE AND START_CAPTURE;
s_write <= CLKD2_SRAM_INTERFACE and not CLK_SRAM_INTERFACE;
EN_DATA_CAPTURE<=s_EN_DATA_CAPTURE;

--================================================================================================================================
-- process : this process is a state machine which :	- write current packet in SRAM memory 
--																										- read current packet in SRAM memory
--																										- read old packet in SRAM memory
--						The different indermediate state are setup times. 
--================================================================================================================================
unpack : process (CLK_SRAM_INTERFACE,reset)
begin
if rising_edge(CLK_SRAM_INTERFACE) then
	if CLKD2_SRAM_INTERFACE = '1' then

		if EN_DATA_CAPTURE_UDP='1' then
			PACKET_READY_CAPTURE_UDP<='0';
		end if;

		if PACKET_READY_CAPTURE = '1' then
			s_PACKET_READY_CAPTURE <='1';
		end if;

		if reset='0' then
			next_state <= STATE0;
		else
			case next_state is
				WHEN STATE0 =>
					addr_write <= (others => '0');
					addr_read_old_packet <= (others => '0');
					addr_read_current_packet <= (others => '0');
					addr_current <= (others => '0');
					next_state <= STATE1;

				WHEN STATE1 =>													-- write current packet + 1 empty state
					if s_PACKET_READY_CAPTURE = '1' then
						next_state <= STATE2;  
					else
						next_state <= STATE3;
					end if;

				WHEN STATE2 =>													 -- write current packet in SRAM memory
					packet_current_ready <= '1';
					if addr_write = "11110100" then
						next_state <= STATE3;
						addr_write <= (others => '0');
						s_PACKET_READY_CAPTURE <= '0';
					else			
						addr_write <= addr_write + '1';
						next_state <= STATE2;
					end if;
			   WHEN STATE3 =>												-- read current packet in SRAM memory
					if packet_current_ready = '1' then
						next_state <= STATE4;					
					else
						next_state <= STATE6;
					end if;

				WHEN STATE4 =>
					if addr_read_current_packet = "11110100" then
						next_state <= STATE5;
						addr_current <= addr_current + '1';
						addr_read_current_packet <= (others => '0');
						packet_current_ready<= '0';
					else
						next_state <= STATE4;
						addr_read_current_packet <= addr_read_current_packet +'1';
					end if;

				WHEN STATE5 =>
					NUMERO_PACKET_CAPTURE_UDP <= addr_current;
					if CAPTURE_UDP_BUZZY ='1' then
						next_state <= STATE5;
					else
						ENABLE_GAUCHE<= not ENABLE_GAUCHE;
						PACKET_READY_CAPTURE_UDP<='1';
						next_state <= STATE6;
					end if;

	 			WHEN STATE6 =>													-- read old packet SRAM in memory
					if REQUEST_OLD_PACKET = '1' then
						s_OLD_NUMERO_PACKET <= OLD_NUMERO_PACKET;
						next_state <= STATE7;
					else
						next_state <= STATE1;
					end if;

				WHEN STATE7 =>
					if addr_read_old_packet = "11110100" then
						next_state <= STATE8;
						addr_read_old_packet <= (others => '0');
					else
						addr_read_old_packet <= addr_read_old_packet + '1';
						next_state <= STATE7;
					end if;
	
				WHEN STATE8 =>
					NUMERO_PACKET_CAPTURE_UDP <= addr_current(15 downto 11) &s_OLD_NUMERO_PACKET;
					if CAPTURE_UDP_BUZZY ='1' then
						next_state <= STATE8;
					else
						ENABLE_GAUCHE<= not ENABLE_GAUCHE;
						PACKET_READY_CAPTURE_UDP<='1';
						next_state <= STATE1;
					end if;

			END CASE;
		end if;
	end if;
end if;
END PROCESS unpack;

--================================================================================================================================
-- signal : SELECT_REQUEST_OLD_PACKET goes up when the request of an old packet is considered.
--================================================================================================================================
with next_state select
SELECT_REQUEST_OLD_PACKET <=	'1' when STATE7,
															'0' when others;

--================================================================================================================================
-- signal : EN_DATA_CAPTURE goes up to get data form the memory of the CAPTURE module
--================================================================================================================================
with next_state select
s_EN_DATA_CAPTURE <=	'1' when STATE2,
											'0' when others;

--================================================================================================================================
-- signal : ADDR_CAPTURE values to get data form the memory of the CAPTURE module
--================================================================================================================================
with next_state select
ADDR_CAPTURE <=	addr_write & not CLKD2_SRAM_INTERFACE WHEN STATE2,
								(others => '1') when others;

--================================================================================================================================
-- signal : MEM_ADDR values of the SRAM memory
--					In state7 address of the reasked packet
--					In state4 address of the current packet
--================================================================================================================================
with next_state select 
MEM_ADDR<=  addr_read_old_packet & CLKD2_SRAM_INTERFACE when STATE7,
				addr_read_current_packet & CLKD2_SRAM_INTERFACE when STATE4,
				(others => '1') when others;

--================================================================================================================================
-- signal : addr_write delayed of one CLK_SRAM_INTERFACE edge
--================================================================================================================================
I_Q0 : FD PORT MAP (C=>CLK_SRAM_INTERFACE, D=>addr_write(0), Q=>addr_write_tmp(0));
I_Q1 : FD PORT MAP (C=>CLK_SRAM_INTERFACE, D=>addr_write(1), Q=>addr_write_tmp(1));
I_Q2 : FD PORT MAP (C=>CLK_SRAM_INTERFACE, D=>addr_write(2), Q=>addr_write_tmp(2));
I_Q3 : FD PORT MAP (C=>CLK_SRAM_INTERFACE, D=>addr_write(3), Q=>addr_write_tmp(3));
I_Q4 : FD PORT MAP (C=>CLK_SRAM_INTERFACE, D=>addr_write(4), Q=>addr_write_tmp(4));
I_Q5 : FD PORT MAP (C=>CLK_SRAM_INTERFACE, D=>addr_write(5), Q=>addr_write_tmp(5));
I_Q6 : FD PORT MAP (C=>CLK_SRAM_INTERFACE, D=>addr_write(6), Q=>addr_write_tmp(6));
I_Q7 : FD PORT MAP (C=>CLK_SRAM_INTERFACE, D=>addr_write(7), Q=>addr_write_tmp(7));

--================================================================================================================================
-- signal :	SRAM_ADDR: values of the address of the SRAM memory for the different states.
--					SRAM_ADDR is divided in 2 parts:	- 11 bits different between 2 packets
--																						- 8 bits for the different values in one packet
--					(48 is the size of a packet of 1* 64 channels in 24 bits)
--================================================================================================================================
with next_state select
SRAM_ADDR <=	addr_current(10 downto 0) & addr_write_tmp when STATE2,
							s_OLD_NUMERO_PACKET & addr_read_old_packet when STATE7,		
							addr_current(10 downto 0) & addr_read_current_packet when STATE4,
							(others => '0') when others;

--================================================================================================================================
-- signal : SRAM_CE is SRAM chip enable, goes up when the memory is used. High = standby
--================================================================================================================================
SRAM_CE <= 	'0' when ((next_state=STATE2) or (next_state=STATE4) or (next_state=STATE7)) else
						'1';

--================================================================================================================================
-- signal : EXT_MEMORY_IN_ENABLE is the signal that transforms the IO into inputs or ouputs
--					inputs when high and ouputs when down.
--================================================================================================================================
with next_state select
EXT_MEMORY_IN_ENABLE <=	'0' when STATE2,
												'1' when others;

--================================================================================================================================
-- signal : SRAM_r_w is read/write control
--					The SRAM memory is written on rising edge of that signal.
--================================================================================================================================
with next_state select
SRAM_r_w <=	s_write when STATE2,
						'1' when others;

--================================================================================================================================
-- signal : SRAM_OE is SRAM ouputs enable
--					It has to be low to be able to read the memory.
--================================================================================================================================
with next_state select
SRAM_OE <=	'0' when STATE7,
						not packet_current_ready when STATE4,
						'1' when others;

--================================================================================================================================
-- signal : MEM_DATA is buffered
--					MEM_DATA0 is the first two memories
--					MEM_DATA0 is the last two memories
--================================================================================================================================
I_A0 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(0), Q=>MEM_DATA0_tmp(0));
I_A1 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(1), Q=>MEM_DATA0_tmp(1));
I_A2 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(2), Q=>MEM_DATA0_tmp(2));
I_A3 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(3), Q=>MEM_DATA0_tmp(3));
I_A4 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(4), Q=>MEM_DATA0_tmp(4));
I_A5 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(5), Q=>MEM_DATA0_tmp(5));
I_A6 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(6), Q=>MEM_DATA0_tmp(6));
I_A7 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(7), Q=>MEM_DATA0_tmp(7));
I_A8 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(8), Q=>MEM_DATA0_tmp(8));
I_A9 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(9), Q=>MEM_DATA0_tmp(9));
I_A10 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(10), Q=>MEM_DATA0_tmp(10));
I_A11 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(11), Q=>MEM_DATA0_tmp(11));
I_A12 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(12), Q=>MEM_DATA0_tmp(12));
I_A13 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(13), Q=>MEM_DATA0_tmp(13));
I_A14 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(14), Q=>MEM_DATA0_tmp(14));
I_A15 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA0(15), Q=>MEM_DATA0_tmp(15));

I_B0 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(0), Q=>MEM_DATA1_tmp(0));
I_B1 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(1), Q=>MEM_DATA1_tmp(1));
I_B2 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(2), Q=>MEM_DATA1_tmp(2));
I_B3 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(3), Q=>MEM_DATA1_tmp(3));
I_B4 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(4), Q=>MEM_DATA1_tmp(4));
I_B5 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(5), Q=>MEM_DATA1_tmp(5));
I_B6 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(6), Q=>MEM_DATA1_tmp(6));
I_B7 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(7), Q=>MEM_DATA1_tmp(7));
I_B8 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(8), Q=>MEM_DATA1_tmp(8));
I_B9 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(9), Q=>MEM_DATA1_tmp(9));
I_B10 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(10), Q=>MEM_DATA1_tmp(10));
I_B11 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(11), Q=>MEM_DATA1_tmp(11));
I_B12 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(12), Q=>MEM_DATA1_tmp(12));
I_B13 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(13), Q=>MEM_DATA1_tmp(13));
I_B14 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(14), Q=>MEM_DATA1_tmp(14));
I_B15 : FD_1 PORT MAP (C=>CLKD2_SRAM_INTERFACE, D=>MEM_DATA1(15), Q=>MEM_DATA1_tmp(15));

--================================================================================================================================
-- process :	this process associated with the previous bufferization transform 
--						a 32 bits input into 16 bits on a bus two times faster MEM
--================================================================================================================================
process(CLK_SRAM_INTERFACE)
begin
if falling_edge(CLK_SRAM_INTERFACE) then
	if CLKD2_SRAM_INTERFACE = '0' then
		MEM <= MEM_DATA1_tmp;
	else
		MEM <= MEM_DATA0_tmp;
	end if;
end if;
end process;

--================================================================================================================================
-- test vectors
--================================================================================================================================
--MEM_DATA0 <= 	SRAM_IO02_in & SRAM_IO01_in;
--MEM_DATA1 <= 	SRAM_IO04_in & SRAM_IO03_in;

--SRAM_IO01 <= DATA_CAPTURE0(7 downto 0);
--SRAM_IO02 <= DATA_CAPTURE0(15 downto 8);
--SRAM_IO03 <= DATA_CAPTURE1(7 downto 0);
--SRAM_IO04 <= DATA_CAPTURE1(15 downto 8);

--================================================================================================================================
-- signal : IOBUF interfacing with data ports of the SRAM memory
--					when EXT_MEMORY_IN_ENABLE is high the SRAM_IO0x is in input mode
--================================================================================================================================
SRAM_IO010: IOBUF port map (I=>DATA_CAPTURE0(0) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(0) ,IO=>SRAM_IO01(0));
SRAM_IO011: IOBUF port map (I=>DATA_CAPTURE0(1) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(1) ,IO=>SRAM_IO01(1));
SRAM_IO012: IOBUF port map (I=>DATA_CAPTURE0(2) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(2) ,IO=>SRAM_IO01(2));
SRAM_IO013: IOBUF port map (I=>DATA_CAPTURE0(3) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(3) ,IO=>SRAM_IO01(3));
SRAM_IO014: IOBUF port map (I=>DATA_CAPTURE0(4) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(4) ,IO=>SRAM_IO01(4));
SRAM_IO015: IOBUF port map (I=>DATA_CAPTURE0(5) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(5) ,IO=>SRAM_IO01(5));
SRAM_IO016: IOBUF port map (I=>DATA_CAPTURE0(6) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(6) ,IO=>SRAM_IO01(6));
SRAM_IO017: IOBUF port map (I=>DATA_CAPTURE0(7) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(7) ,IO=>SRAM_IO01(7));

SRAM_IO020: IOBUF port map (I=>DATA_CAPTURE0(8) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(8) ,IO=>SRAM_IO02(0));
SRAM_IO021: IOBUF port map (I=>DATA_CAPTURE0(9) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(9) ,IO=>SRAM_IO02(1));
SRAM_IO022: IOBUF port map (I=>DATA_CAPTURE0(10) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(10) ,IO=>SRAM_IO02(2));
SRAM_IO023: IOBUF port map (I=>DATA_CAPTURE0(11) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(11) ,IO=>SRAM_IO02(3));
SRAM_IO024: IOBUF port map (I=>DATA_CAPTURE0(12) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(12) ,IO=>SRAM_IO02(4));
SRAM_IO025: IOBUF port map (I=>DATA_CAPTURE0(13) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(13) ,IO=>SRAM_IO02(5));
SRAM_IO026: IOBUF port map (I=>DATA_CAPTURE0(14) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(14) ,IO=>SRAM_IO02(6));
SRAM_IO027: IOBUF port map (I=>DATA_CAPTURE0(15) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA0(15) ,IO=>SRAM_IO02(7));

SRAM_IO030: IOBUF port map (I=>DATA_CAPTURE1(0) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(0) ,IO=>SRAM_IO03(0));
SRAM_IO031: IOBUF port map (I=>DATA_CAPTURE1(1) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(1) ,IO=>SRAM_IO03(1));
SRAM_IO032: IOBUF port map (I=>DATA_CAPTURE1(2) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(2) ,IO=>SRAM_IO03(2));
SRAM_IO033: IOBUF port map (I=>DATA_CAPTURE1(3) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(3) ,IO=>SRAM_IO03(3));
SRAM_IO034: IOBUF port map (I=>DATA_CAPTURE1(4) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(4) ,IO=>SRAM_IO03(4));
SRAM_IO035: IOBUF port map (I=>DATA_CAPTURE1(5) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(5) ,IO=>SRAM_IO03(5));
SRAM_IO036: IOBUF port map (I=>DATA_CAPTURE1(6) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(6) ,IO=>SRAM_IO03(6));
SRAM_IO037: IOBUF port map (I=>DATA_CAPTURE1(7) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(7) ,IO=>SRAM_IO03(7));

SRAM_IO040: IOBUF port map (I=>DATA_CAPTURE1(8) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(8) ,IO=>SRAM_IO04(0));
SRAM_IO041: IOBUF port map (I=>DATA_CAPTURE1(9) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(9) ,IO=>SRAM_IO04(1));
SRAM_IO042: IOBUF port map (I=>DATA_CAPTURE1(10) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(10) ,IO=>SRAM_IO04(2));
SRAM_IO043: IOBUF port map (I=>DATA_CAPTURE1(11) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(11) ,IO=>SRAM_IO04(3));
SRAM_IO044: IOBUF port map (I=>DATA_CAPTURE1(12) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(12) ,IO=>SRAM_IO04(4));
SRAM_IO045: IOBUF port map (I=>DATA_CAPTURE1(13) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(13) ,IO=>SRAM_IO04(5));
SRAM_IO046: IOBUF port map (I=>DATA_CAPTURE1(14) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(14) ,IO=>SRAM_IO04(6));
SRAM_IO047: IOBUF port map (I=>DATA_CAPTURE1(15) ,T=>EXT_MEMORY_IN_ENABLE ,O=>MEM_DATA1(15) ,IO=>SRAM_IO04(7));


--================================================================================================================================
-- signal : transform the 16 bits from the memory of the module CAPTURE into 2 buffers of 16 bits
--================================================================================================================================
deserial01 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(0), Q=>A(0));
deserial02 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(0), Q=>B(0));
deserial11 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(1), Q=>A(1));
deserial12 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(1), Q=>B(1));
deserial21 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(2), Q=>A(2));
deserial22 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(2), Q=>B(2));
deserial31 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(3), Q=>A(3));
deserial32 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(3), Q=>B(3));
deserial41 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(4), Q=>A(4));
deserial42 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(4), Q=>B(4));
deserial51 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(5), Q=>A(5));
deserial52 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(5), Q=>B(5));
deserial61 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(6), Q=>A(6));
deserial62 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(6), Q=>B(6));
deserial71 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(7), Q=>A(7));
deserial72 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(7), Q=>B(7));
deserial81 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(8), Q=>A(8));
deserial82 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(8), Q=>B(8));
deserial91 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(9), Q=>A(9));
deserial92 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(9), Q=>B(9));
deserial101 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(10), Q=>A(10));
deserial102 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(10), Q=>B(10));
deserial111 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(11), Q=>A(11));
deserial112 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(11), Q=>B(11));
deserial121 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(12), Q=>A(12));
deserial122 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(12), Q=>B(12));
deserial131 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(13), Q=>A(13));
deserial132 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(13), Q=>B(13));
deserial141 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(14), Q=>A(14));
deserial142 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(14), Q=>B(14));
deserial151 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>DATA_CAPTURE(15), Q=>A(15));
deserial152 : FDE PORT MAP (C=> CLK_SRAM_INTERFACE,CE=> s_EN_DATA_CAPTURE, D=>A(15), Q=>B(15));

--================================================================================================================================
-- process:	this process use the previous transformation to make the 2 16-bits registers in 1 32-bits register for the SRAM memory
--================================================================================================================================
deserialmem: process(CLK_SRAM_INTERFACE)
begin
	if rising_edge(CLK_SRAM_INTERFACE) then
		if CLKD2_SRAM_INTERFACE = '1' then
			DATA_CAPTURE0<=B;
			DATA_CAPTURE1<=A;
		end if;
	end if;
end process;

--================================================================================================================================
-- signal :	this enable the double buffering of the memory interfacing with the module CAPTURE_UDP_FRAME.
--					each of the 4 memories is activatied at the right time
--================================================================================================================================
ENA1<=ENABLE_GAUCHE AND (not MEM_ADDR(8));
ENA2<=ENABLE_GAUCHE AND MEM_ADDR(8);
ENA3<=(not ENABLE_GAUCHE) AND (not MEM_ADDR(8));
ENA4<=(not ENABLE_GAUCHE) AND MEM_ADDR(8);

--================================================================================================================================
-- memory:	the 4 memories of the double buffering of the data packets of 960 bytes (2 memories per packet)
--================================================================================================================================
----------------------------------------------------------------------left memory
XLXI_1 : RAMB4_S8_S16 PORT MAP (WEA=>'0', ENA=>ENB1, RSTA=>'0', CLKA=>CLKD2_SRAM_INTERFACE, ADDRA=>ADDR_CAPTURE_UDP(8 downto 0), DIA=>"00000000", DOA=>DOB01,
										  WEB=>'1', ENB=>ENA1, RSTB=>'0', CLKB=>CLK_SRAM_INTERFACE90, ADDRB=>MEM_ADDR(7 downto 0), DIB=>MEM, DOB=>open
										  );

XLXI_2 : RAMB4_S8_S16 PORT MAP (WEA=>'0', ENA=>ENB2, RSTA=>'0', CLKA=>CLKD2_SRAM_INTERFACE, ADDRA=>ADDR_CAPTURE_UDP(8 downto 0), DIA=>"00000000", DOA=>DOB02,
										  WEB=>'1', ENB=>ENA2, RSTB=>'0', CLKB=>CLK_SRAM_INTERFACE90, ADDRB=>MEM_ADDR(7 downto 0), DIB=>MEM, DOB=>open
										  );
----------------------------------------------------------------------left memory
----------------------------------------------------------------------right memory
XLXI_3 : RAMB4_S8_S16 PORT MAP (WEA=>'0', ENA=>ENB3, RSTA=>'0', CLKA=>CLKD2_SRAM_INTERFACE, ADDRA=>ADDR_CAPTURE_UDP(8 downto 0), DIA=>"00000000", DOA=>DOB03,
										  WEB=>'1', ENB=>ENA3, RSTB=>'0', CLKB=>CLK_SRAM_INTERFACE90, ADDRB=>MEM_ADDR(7 downto 0), DIB=>MEM, DOB=>open
										  );

XLXI_4 : RAMB4_S8_S16 PORT MAP (WEA=>'0', ENA=>ENB4, RSTA=>'0', CLKA=>CLKD2_SRAM_INTERFACE, ADDRA=>ADDR_CAPTURE_UDP(8 downto 0), DIA=>"00000000", DOA=>DOB04,
										  WEB=>'1', ENB=>ENA4, RSTB=>'0', CLKB=>CLK_SRAM_INTERFACE90, ADDRB=>MEM_ADDR(7 downto 0), DIB=>MEM, DOB=>open
										  );
----------------------------------------------------------------------right memory

--================================================================================================================================
-- signal :	this enable the double buffering of the memory interfacing with the module CAPTURE_UDP_FRAME.
--					each of the 4 memories is activatied at the right time and in opposition to ENAx
--================================================================================================================================
ENB1<=(not ENABLE_GAUCHE) AND (not ADDR_CAPTURE_UDP(9)) and EN_DATA_CAPTURE_UDP;
ENB2<=(not ENABLE_GAUCHE) AND (ADDR_CAPTURE_UDP(9) or (ADDR_CAPTURE_UDP(8) and ADDR_CAPTURE_UDP(7) and ADDR_CAPTURE_UDP(6) and ADDR_CAPTURE_UDP(5) and ADDR_CAPTURE_UDP(4) and ADDR_CAPTURE_UDP(3) and ADDR_CAPTURE_UDP(2) and ADDR_CAPTURE_UDP(1) and ADDR_CAPTURE_UDP(0))) and EN_DATA_CAPTURE_UDP;
ENB3<=ENABLE_GAUCHE AND (not ADDR_CAPTURE_UDP(9)) and EN_DATA_CAPTURE_UDP;
ENB4<=ENABLE_GAUCHE AND (ADDR_CAPTURE_UDP(9) or (ADDR_CAPTURE_UDP(8) and ADDR_CAPTURE_UDP(7) and ADDR_CAPTURE_UDP(6) and ADDR_CAPTURE_UDP(5) and ADDR_CAPTURE_UDP(4) and ADDR_CAPTURE_UDP(3) and ADDR_CAPTURE_UDP(2) and ADDR_CAPTURE_UDP(1) and ADDR_CAPTURE_UDP(0))) and EN_DATA_CAPTURE_UDP;
----------------------------------------------------------------------recuperation donnees

--================================================================================================================================
-- signal : delay of ENBx of one clock edge otherwise "0000" appears on DATA_CAPTURE_MEM
--================================================================================================================================
process(CLKD2_SRAM_INTERFACE)
begin
	if rising_edge(CLKD2_SRAM_INTERFACE) then
		ENB1_i <= ENB1;
		ENB2_i <= ENB2;
		ENB3_i <= ENB3;
	end if;	
end process;

--================================================================================================================================
-- signal : muliplexing of the 4 memories to 1 data bus
--================================================================================================================================
DATA_CAPTURE_UDP <=	DOB01 when ENB1_i='1' else
										DOB02 when ENB2_i='1' else
										DOB03 when ENB3_i='1' else
										DOB04;
end RTL;