--=============================================================================
--
-- Name:		MUX4_1
--
-- Purpose:	This module is multiplexing different packets.
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. BOOTP_FRAME
-- 2. ARP_FRAME
-- 3. CAPTURE_UDP_FRAME
-- 4. RESPONSE_STATUS_UDP_FRAME
-- 5. CRC32
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

library UNISIM;
use UNISIM.vcomponents.ALL;

entity MUX4_1 is
    Port (
					CLK_MUX						: in std_logic;
					RESET							: in std_logic;
					TX_FRAME_READY		: in std_logic;

					REQ_IN_0					: in std_logic;
					SELECT_0					: out std_logic;
					EN_D_IN_0					: in std_logic;
					D_IN_0						: in std_logic_vector(7 downto 0);

					REQ_IN_1					: in std_logic;
					SELECT_1					: out std_logic;
					EN_D_IN_1					: in std_logic;
					D_IN_1						: in std_logic_vector(7 downto 0);

					REQ_IN_2					: in std_logic;
					SELECT_2					: out std_logic;
					EN_D_IN_2					: in std_logic;
					D_IN_2						: in std_logic_vector(7 downto 0);

					REQ_IN_3					: in std_logic;
					SELECT_3					: out std_logic;
					EN_D_IN_3					: in std_logic;
					D_IN_3						: in std_logic_vector(7 downto 0);

					EN_DATA_MUX_OUT		: out std_logic;
					DATA_MUX_OUT			: out std_logic_vector(7 downto 0)
					);
end MUX4_1;

architecture RTL of MUX4_1 is

type stateType is	(STATE0, STATE1, STATE2, STATE3);
signal next_state : stateType;

signal s_SELECT_0					:std_logic;
signal s_SELECT_1					:std_logic;
signal s_SELECT_2					:std_logic;
signal s_SELECT_3					:std_logic;

begin

SELECT_0	<= s_SELECT_0;
SELECT_1	<= s_SELECT_1;
SELECT_2	<= s_SELECT_2;
SELECT_3	<= s_SELECT_3;

--================================================================================================================================
-- process :	this is a simple rotating state machine.
--						each state corresponds to one entry . So each entry is tested circularly.
--						At each state , there is a test to know if the previous packet has been sent.
--						If there is no request from the entry then the state machine goes to the next entry.
--================================================================================================================================
comb_logic: process(CLK_MUX,RESET)
begin
if RESET='0' then
	next_state <= STATE0;
elsif rising_edge(CLK_MUX) then

		case next_state is

			WHEN STATE0 =>
				if REQ_IN_0 = '1' then
					if TX_FRAME_READY='0' then
						s_SELECT_0 <= '1';
					else
						s_SELECT_0 <= s_SELECT_0;
					end if;
					next_state <= STATE0;
				else
					s_SELECT_0 <= '0';
					next_state <= STATE1;
				end if;

			WHEN STATE1 =>
				if REQ_IN_1 = '1' then
					if TX_FRAME_READY='0' then
						s_SELECT_1 <= '1';
					else
						s_SELECT_1 <= s_SELECT_1;
					end if;
					next_state <= STATE1;
				else
					s_SELECT_1 <= '0';
					next_state <= STATE2;
				end if;

			WHEN STATE2 =>
				if REQ_IN_2 = '1' then
					if TX_FRAME_READY='0' then
						s_SELECT_2 <= '1';
					else
						s_SELECT_2 <= s_SELECT_2;
					end if;
					next_state <= STATE2;
				else
					s_SELECT_2 <= '0';
					next_state <= STATE3;
				end if;

			WHEN STATE3 =>
				if REQ_IN_3 = '1' then
					if TX_FRAME_READY='0' then
						s_SELECT_3 <= '1';
					else
						s_SELECT_3 <= s_SELECT_3;
					end if;
					next_state <= STATE3;
				else
					s_SELECT_3 <= '0';
					next_state <= STATE0;
				end if;

			WHEN OTHERS =>
		END CASE;
	end if;
END PROCESS comb_logic;

--================================================================================================================================
-- signal : multiplexing of the data bus depending of the entry
--================================================================================================================================
with next_state select
DATA_MUX_OUT <=	D_IN_0 when STATE0,
								D_IN_1 when STATE1,
								D_IN_2 when STATE2,
								D_IN_3 when STATE3,
								"00000000" when others;

--================================================================================================================================
-- signal : multiplexing of the enable-data bus depending of the entry
--================================================================================================================================
with next_state select
EN_DATA_MUX_OUT <=  		EN_D_IN_0 when STATE0,
								EN_D_IN_1 when STATE1,
			 					EN_D_IN_2 when STATE2,
								EN_D_IN_3 when STATE3,
								'0' when others;

end RTL;
