--=============================================================================
--
-- Name:		MEM_READ_INCOMING_MSG
--
-- Purpose:	This module reads in the memory provided and determines the
--					action to take corresponding to the message.
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. INCOMING_MESSAGE_MEM
-- 2. ARP_FRAME
-- 3. CAPTURE
-- 4. RESPONSE_STATUS_UDP_FRAME
-- 5. SRAM_INTERFACE
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity MEM_READ_INCOMING_MSG is
    Port (
					CLK_MEM_READ							: in std_logic;
					RESET											: in std_logic;

					ADDR_INCOMING_MSG_MEM			: out std_logic_vector(7 downto 0);
					ENABLE_INCOMING_MSG_MEM		: out std_logic;
					DATA_INCOMING_MSG_MEM			: in std_logic_vector(15 downto 0);

					SRC_IP_ARP_REQ						: out  std_logic_vector(31 downto 0);
					MY_IP											: in std_logic_vector(31 downto 0);
					ARP_FRAME_BUZZY						: in std_logic;
					REQ_ARP										: out std_logic;

					ID												: in std_logic_vector(31 downto 0);
					BOOTP_IP									: out std_logic_vector(31 downto 0);
					REQ_BOOTP									: out std_logic;

					SYNC_SLAVE_ON							: out std_logic;
					DOUBLE_FRQ_AD							: out std_logic; 
					START_CAPTURE							: out std_logic;

					REQUEST_OLD_PACKET				: out std_logic;
					SELECT_REQUEST_OLD_PACKET	: in std_logic;
					OLD_NUMERO_PACKET					: out std_logic_vector(10 downto 0);

					RESPONSE_REQUEST_BUZZY		: in std_logic;
					REQ_RESPONSE_REQUEST			: out std_logic;
					TYPE_RESPONSE_REQUEST			: out std_logic_vector(2 downto 0);
					DATA_REQUEST							: out std_logic_vector(9 downto 0);
					IP_SENDER									: out std_logic_vector(31 downto 0);
					MAC_SENDER								: out std_logic_vector(47 downto 0);
					PRIORITY_SENDER						: out std_logic;

--visu_Frame_UDP									: out std_logic;
--visu_Frame_RESPONSE							: out std_logic;
--visu_Frame_request06 						: out std_logic;
--visu_Frame_request09 						: out std_logic;
--visu_start_identification_msg				: out std_logic;

					RECV_PACKET_MSG						: in std_logic;
					END_ADDR_MSG							: in std_logic_vector(7 downto 0));
end Mem_Read_incoming_msg;

architecture RTL of MEM_READ_INCOMING_MSG is

signal s_SYNC_SLAVE_ON						: std_logic;
signal s_DOUBLE_FRQ_AD						: std_logic;
signal s_START_CAPTURE						: std_logic;
signal s_REQUEST_OLD_PACKET				: std_logic;

signal start_identification_msg		: std_logic;	
signal addr 											: std_logic_vector (7 downto 0);

signal Frame_ARP									: std_logic;
signal Frame_BOOTP								: std_logic;
signal Frame_UDP									: std_logic;
signal Frame_RESPONSE							: std_logic;
signal Frame_request01						: std_logic;
signal Frame_request02						: std_logic;
signal Frame_request03						: std_logic;
signal Frame_request04						: std_logic;
signal Frame_request05						: std_logic;
signal Frame_request06						: std_logic;
signal Frame_request07						: std_logic;
signal Frame_request08						: std_logic;
signal Frame_request09						: std_logic;

signal tmp_mac_sender							: std_logic_vector(47 downto 0);
signal tmp_ip_sender							: std_logic_vector(31 downto 0); 

signal old_numero_packet_begin		: std_logic_vector(10 downto 0);
signal old_numero_packet_end			: std_logic_vector(10 downto 0);
signal count_packet								: std_logic_vector(10 downto 0);
signal send_old_packet_actif			: std_logic;
type stateType is	(idle, wait_selection, send_packet_number);
signal next_state : stateType;

signal clkd2											: std_logic;

begin

--================================================================================================================================
-- test vectors
--================================================================================================================================
--visu_Frame_UDP				<= Frame_UDP;
--visu_Frame_RESPONSE		<= Frame_RESPONSE;			
--visu_Frame_request06 	<= Frame_request06; 						
--visu_Frame_request09 	<= Frame_request09; 					
--visu_start_identification_msg <= start_identification_msg;

ADDR_INCOMING_MSG_MEM<=addr;
SYNC_SLAVE_ON<= s_SYNC_SLAVE_ON;
DOUBLE_FRQ_AD <= s_DOUBLE_FRQ_AD;
START_CAPTURE <= s_START_CAPTURE;

--================================================================================================================================
-- process :	divide CLK_MEM_READ by 2
--================================================================================================================================
process(CLK_MEM_READ)
begin
if rising_edge(CLK_MEM_READ) then
	clkd2 <= not clkd2;
end if;
end process;

--================================================================================================================================
-- process : 	this process read the message filled in the memory of INCOMING_MESSAGE_MEM.
--						The memory is read in 16 bits large.
--						At each address corresponds 1 or 2 fields that are tested and understood.
--						At the end everything is initialized
--================================================================================================================================
identification_msg : process(CLK_MEM_READ)
begin
if rising_edge(CLK_MEM_READ) then
		if RECV_PACKET_MSG='1' then
			start_identification_msg<='1';
		end if;

		if start_identification_msg='1' then
			ENABLE_INCOMING_MSG_MEM<='1';
			addr <= addr + '1';
			case addr is
				when "00000011" =>--3
					tmp_mac_sender(47 downto 32) <= DATA_INCOMING_MSG_MEM(15 downto 0);--memory mac sender		
				when "00000100" =>--4
					tmp_mac_sender(31 downto 16) <= DATA_INCOMING_MSG_MEM(15 downto 0);--memory mac sender		
				when "00000101" =>--5
					tmp_mac_sender(15 downto 0) <= DATA_INCOMING_MSG_MEM(15 downto 0);--memory mac sender		
				when "00000110" =>--6
					if DATA_INCOMING_MSG_MEM /= "0000100000000110" then --- arp test ethernet type x0806
						Frame_ARP <= '1';
					elsif DATA_INCOMING_MSG_MEM /= "0000100000000000" then --- udp test ethernet type x0800
						Frame_UDP <= '1';
					end if;
				when "00001010" =>--10
					if DATA_INCOMING_MSG_MEM /= "00000000000000001" then --- arp test opcode request 0x0001
						Frame_ARP <= '1';
					end if;
				when "00001011" =>--11
					if DATA_INCOMING_MSG_MEM(7 downto 0) /= "00010001" then -- udp test protocol UDP 11
						Frame_UDP <= '1';
					end if;
				when "00001101" =>--13 
					if Frame_UDP = '0' then
						tmp_ip_sender(31 downto 16) <= DATA_INCOMING_MSG_MEM(15 downto 0);  -- put in mem udp IP sender
					end if;
				when "00001110" =>--14
					if Frame_ARP = '0' then
						SRC_IP_ARP_REQ(31 downto 16) <= DATA_INCOMING_MSG_MEM(15 downto 0); -- put in mem arp IP sender
					end if;
					if Frame_UDP = '0' then
						tmp_ip_sender(15 downto 0) <= DATA_INCOMING_MSG_MEM(15 downto 0);	-- put in mem udp IP sender
					end if;
				when "00001111" =>--15
					if Frame_ARP = '0' then
						SRC_IP_ARP_REQ(15 downto 0) <= DATA_INCOMING_MSG_MEM(15 downto 0);-- put in mem arp IP sender
					end if;
					if Frame_UDP = '1' then
						Frame_BOOTP <= '1';
						Frame_RESPONSE <= '1';
					else
						if DATA_INCOMING_MSG_MEM /= MY_IP(31 downto 16) then ---- udp test if good ip target
							Frame_RESPONSE <= '1';
						end if;
					end if;
				when "00010000" =>--16
					if DATA_INCOMING_MSG_MEM /= MY_IP(15 downto 0) then ----udp test if good ip target
						Frame_RESPONSE <= '1';
					end if;
				when "00010010" =>--18
					if DATA_INCOMING_MSG_MEM /= "0111111111111111" then -- test destination port = 32767 = x7fff
						Frame_RESPONSE <= '1';
					elsif DATA_INCOMING_MSG_MEM /= "0000000001000100" then -- bootp test destination port = 68 = x0044 
						Frame_BOOTP <= '1';
					end if;
				when "00010011" =>--19
					if DATA_INCOMING_MSG_MEM /= MY_IP(31 downto 16) then --arp test if good ip target
						Frame_ARP <= '1';
					end if;
				when "00010100" =>--20
					if DATA_INCOMING_MSG_MEM /= MY_IP(15 downto 0) then ---- arptest if good ip target 
						Frame_ARP <= '1';
					end if;
				when "00010101" =>--21
					if DATA_INCOMING_MSG_MEM(15 downto 8) /= "00000010" then   ---bootp test message type: boot reply 02
						Frame_BOOTP <= '1';
					end if;
					if DATA_INCOMING_MSG_MEM /= "0000000000000001" or Frame_RESPONSE = '1' then -- test message type led request = 0001
						Frame_request01 <= '1';
					end if;
					if DATA_INCOMING_MSG_MEM /= "0000000000000010" or Frame_RESPONSE = '1' then -- test message type led request = 0002 
						Frame_request02 <= '1';
					end if;
					if DATA_INCOMING_MSG_MEM /= "0000000000000011" or Frame_RESPONSE = '1' then -- test message type led request = 0003 
						Frame_request03 <= '1';
					end if;
					if DATA_INCOMING_MSG_MEM /= "0000000000000100" or Frame_RESPONSE = '1' then -- test message type led request = 0004
						Frame_request04 <= '1';
					end if;
					if DATA_INCOMING_MSG_MEM /= "0000000000000101" or Frame_RESPONSE = '1' then -- test message type led request = 0005
						Frame_request05 <= '1';
					end if;
					if DATA_INCOMING_MSG_MEM /= "0000000000000110" or Frame_RESPONSE = '1' then -- test message type led request = 0006
						Frame_request06 <= '1';
					end if;
					if DATA_INCOMING_MSG_MEM /= "0000000000000111" or Frame_RESPONSE = '1' then -- test message type led request = 0007
						Frame_request07 <= '1';
					end if;
					if DATA_INCOMING_MSG_MEM /= "0000000000001000" or Frame_RESPONSE = '1' then -- test message type led request = 0008
						Frame_request08 <= '1';
					end if;
					if DATA_INCOMING_MSG_MEM /= "0000000000001001" or Frame_RESPONSE = '1' then -- test message type led request = 0009
						Frame_request09 <= '1';
					end if;
				when "00010110" =>--22
					IP_SENDER 	<= tmp_ip_sender;
					MAC_SENDER 	<=	tmp_mac_sender;
					if Frame_ARP = '0' and arp_frame_buzzy = '0' then
						REQ_ARP <= '1';
						PRIORITY_SENDER <= '0';	
					end if;
					if Frame_request01 = '0' then																-- sync on = x0001 ; off = x 0000
						s_SYNC_SLAVE_ON <= DATA_INCOMING_MSG_MEM(8);
						PRIORITY_SENDER <= '1';		 
					elsif Frame_request02 = '0' and RESPONSE_REQUEST_BUZZY = '0' then			-- ask status sync : slave or master
						TYPE_RESPONSE_REQUEST <= "010";
						DATA_REQUEST <= "000000000" & s_SYNC_SLAVE_ON;
						REQ_RESPONSE_REQUEST <= '1';
						PRIORITY_SENDER <= '0';		 
					elsif Frame_request03 = '0' and RESPONSE_REQUEST_BUZZY = '0' then			-- ask ID
						TYPE_RESPONSE_REQUEST <= "011";
						DATA_REQUEST <= ID(9 downto 0);
						REQ_RESPONSE_REQUEST <= '1';
						PRIORITY_SENDER <= '0';
					elsif Frame_request04 = '0' then														-- capture on = x0001 ; off = x 0000
				      	s_START_CAPTURE <= DATA_INCOMING_MSG_MEM(0);
						PRIORITY_SENDER <= '1';		
	 				elsif Frame_request05 = '0' and RESPONSE_REQUEST_BUZZY = '0' then			-- ask status capture
						TYPE_RESPONSE_REQUEST <= "101";
						DATA_REQUEST <= "000000000" & s_START_CAPTURE;
						REQ_RESPONSE_REQUEST <= '1';
						PRIORITY_SENDER <= '0';
					elsif Frame_request06 = '0' then														-- ask packets number
						if send_old_packet_actif='0' then
							s_REQUEST_OLD_PACKET <= '1';
							PRIORITY_SENDER <= '1';
							old_numero_packet_begin <= DATA_INCOMING_MSG_MEM(10 downto 0);
							old_numero_packet_end <= DATA_INCOMING_MSG_MEM(10 downto 0);
						else
							TYPE_RESPONSE_REQUEST <= "110";
							DATA_REQUEST <= "0001101110";
							REQ_RESPONSE_REQUEST <= '1';
							PRIORITY_SENDER <= '0';
						end if;
					elsif Frame_request07 = '0' then														-- frequency doubled on = x0001 ; off = x 0000
				     	if s_START_CAPTURE='0' then
							s_DOUBLE_FRQ_AD <= DATA_INCOMING_MSG_MEM(0);
						end if;
						PRIORITY_SENDER <= '1';		
	 				elsif Frame_request08 = '0' and RESPONSE_REQUEST_BUZZY = '0' then			-- ask if frequency double activated
						TYPE_RESPONSE_REQUEST <= "111";
						DATA_REQUEST <= "000000000" & s_DOUBLE_FRQ_AD;
						REQ_RESPONSE_REQUEST <= '1';
						PRIORITY_SENDER <= '0';
					elsif Frame_request09 = '0' then														-- ask packets number
						if send_old_packet_actif='0' then
							s_REQUEST_OLD_PACKET <= '1';
							PRIORITY_SENDER <= '1';
							old_numero_packet_begin <= DATA_INCOMING_MSG_MEM(10 downto 0);
						else
							TYPE_RESPONSE_REQUEST <= "110";
							DATA_REQUEST <= "0001101110";
							REQ_RESPONSE_REQUEST <= '1';
							PRIORITY_SENDER <= '0';
						end if;
					end if;
				when "00010111" =>--23
					if DATA_INCOMING_MSG_MEM /= ID(31 downto 16) then
						Frame_BOOTP <= '1';
					end if;
					if Frame_request09 = '0' then														-- ask packets number
						old_numero_packet_end <= DATA_INCOMING_MSG_MEM(10 downto 0);
					end if; 
				when "00011000" =>--24
					if DATA_INCOMING_MSG_MEM /= ID(15 downto 0) then
						Frame_BOOTP <= '1';
					end if;
					s_REQUEST_OLD_PACKET <= '0';
				when "00011101" =>--29
					if Frame_BOOTP = '0' then
						BOOTP_IP(31 downto 16) <= DATA_INCOMING_MSG_MEM(15 downto 0);-- put in mem bootp IP
	 			   end if;
					when "00011110" =>--30
					if Frame_BOOTP = '0' then
						BOOTP_IP(15 downto 0) <= DATA_INCOMING_MSG_MEM(15 downto 0);-- put in mem bootp IP
 				   end if;
	 			when "00011111" =>--31
					if Frame_BOOTP = '0' then
						REQ_BOOTP <= '1';
					end if;
				when others => 
			end case;
			if addr = END_ADDR_MSG then
				start_identification_msg<='0';
			end if;
		else
			ENABLE_INCOMING_MSG_MEM<='0';
			addr <= "00000000";
			REQ_ARP <= '0';
			REQ_BOOTP <= '0';
			REQ_RESPONSE_REQUEST <= '0';
			PRIORITY_SENDER <= '0';
			Frame_ARP <= '0';
			Frame_UDP <= '0';
			Frame_BOOTP <= '0';
			Frame_RESPONSE <= '0';
			Frame_request01 <= '0';
			Frame_request02 <= '0';
			Frame_request03 <= '0';
			Frame_request04 <= '0';
			Frame_request05 <= '0';
			Frame_request06 <= '0';
			Frame_request07 <= '0';
			Frame_request08 <= '0';
			Frame_request09 <= '0';
		end if;
end if;
end process identification_msg;


--================================================================================================================================
-- process:	This process is used when one or a range of packets are lost and the computer ask them back
--================================================================================================================================
unpack : process(CLK_MEM_READ,RESET)
begin
if RESET='0' then
	next_state <= idle;
elsif rising_edge(CLK_MEM_READ) then
	if clkd2='1' then
		case next_state is
			when idle =>
				if s_REQUEST_OLD_PACKET = '1' then
					next_state <= wait_selection;
					count_packet <= old_numero_packet_begin;
				else
					next_state <= idle;
				end if;
			when wait_selection =>
				if SELECT_REQUEST_OLD_PACKET = '1' then
					next_state <= send_packet_number;
				else
					next_state <= wait_selection;
				end if;
			when send_packet_number =>
				if count_packet = old_numero_packet_end then 
					next_state <= idle;
				elsif SELECT_REQUEST_OLD_PACKET = '1' then
				  next_state <= send_packet_number;
				else
					count_packet <= count_packet+1;
					next_state <= wait_selection;
				end if;
		end case;
	end if;
end if;
end process unpack;

OLD_NUMERO_PACKET<=count_packet;

--================================================================================================================================
-- signal :	this signal goes up when a packet is reasked.
--================================================================================================================================
with next_state select
send_old_packet_actif <=	'0' when idle,
													'1' when others;

--================================================================================================================================
-- signal :	this signal goes up for a request of an old packet.
--================================================================================================================================
with next_state select
REQUEST_OLD_PACKET <=	'1' when wait_selection,
											'0' when others;

end RTL;
