--=============================================================================
--
-- Name:		INCOMING_MESSAGE_MEM
--
-- Purpose:	This module checks if the packet arriving live is not too big,
--					have the right destination mac address and the right crc32.
--					After that it makes it available to another module to read it.
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. MEM_READ_INCOMING_MSG
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================

LIBRARY ieee;
LIBRARY UNISIM;
USE ieee.std_logic_1164.ALL;
USE ieee.numeric_std.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
USE UNISIM.Vcomponents.ALL;

entity INCOMING_MESSAGE_MEM is
    Port (
					CLK_INCOMING_MESSAGE_MEM			: IN STD_LOGIC;
					ADDR_INCOMING_MESSAGE_MEM			: IN std_logic_vector (7 downto 0);
					ENABLE_INCOMING_MESSAGE_MEM		: IN STD_LOGIC;
					DATA_INCOMING_MESSAGE_MEM			: OUT std_logic_vector (15 downto 0);

					RECV_PACKET										: OUT STD_LOGIC;
					END_ADDR											: OUT std_logic_vector (7 downto 0);

					SA														: in std_logic_vector(47 downto 0);

					RXD														: IN STD_LOGIC_VECTOR (3 DOWNTO 0); 
					RX_CLK												: IN STD_LOGIC; 
					RX_DV													: IN STD_LOGIC
					);
end INCOMING_MESSAGE_MEM;

architecture RTL of INCOMING_MESSAGE_MEM is

signal prev_RX_DV			:std_logic;
signal debut_preamb		:std_logic;
signal crc_store			:std_logic_vector(31 downto 0);
signal next_crc				:std_logic_vector(31 downto 0);
signal addr1					:std_logic_vector (9 downto 0);
signal addr2					:std_logic_vector (9 downto 0);
signal DA_good				:std_logic;

signal Too_big_msg : std_logic;

SIGNAL dec01	:	STD_LOGIC_VECTOR (31 DOWNTO 0);
SIGNAL dec02	:	STD_LOGIC_VECTOR (31 DOWNTO 0);
SIGNAL dec03	:	STD_LOGIC_VECTOR (31 DOWNTO 0);
SIGNAL dec04	:	STD_LOGIC_VECTOR (31 DOWNTO 0);
SIGNAL dec05	:	STD_LOGIC_VECTOR (31 DOWNTO 0);
SIGNAL dec06	:	STD_LOGIC_VECTOR (31 DOWNTO 0);
SIGNAL dec07	:	STD_LOGIC_VECTOR (31 DOWNTO 0);
SIGNAL dec08	:	STD_LOGIC_VECTOR (31 DOWNTO 0);


COMPONENT FD32CE is
	 Port ( C : in std_logic;
           D : in std_logic_vector(31 downto 0);
           Q : out std_logic_vector(31 downto 0));
end COMPONENT;

begin

--================================================================================================================================
-- RXD is the data bus recieved from the ethernet physical layer device.
-- In order to be able to compare the CRC32 recieved with the CRC32 computed from the data recieved,
-- I am transforming the 4 bits data bus into 32 bits data bus called crc_store which is the CRC32 recieved at the end of the message.
--================================================================================================================================
ST01_P0 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>not RXD(3), Q=>crc_store(0));
ST01_P1 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(0), Q=>crc_store(4));
ST01_P2 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(4), Q=>crc_store(8));
ST01_P3 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(8), Q=>crc_store(12));
ST01_P4 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(12), Q=>crc_store(16));
ST01_P5 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(16), Q=>crc_store(20));
ST01_P6 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(20), Q=>crc_store(24));
ST01_P7 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(24), Q=>crc_store(28));

ST02_P0 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>not RXD(2), Q=>crc_store(1));
ST02_P1 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(1), Q=>crc_store(5));
ST02_P2 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(5), Q=>crc_store(9));
ST02_P3 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(9), Q=>crc_store(13));
ST02_P4 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(13), Q=>crc_store(17));
ST02_P5 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(17), Q=>crc_store(21));
ST02_P6 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(21), Q=>crc_store(25));
ST02_P7 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(25), Q=>crc_store(29));

ST03_P0 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>not RXD(1), Q=>crc_store(2));
ST03_P1 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(2), Q=>crc_store(6));
ST03_P2 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(6), Q=>crc_store(10));
ST03_P3 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(10), Q=>crc_store(14));
ST03_P4 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(14), Q=>crc_store(18));
ST03_P5 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(18), Q=>crc_store(22));
ST03_P6 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(22), Q=>crc_store(26));
ST03_P7 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(26), Q=>crc_store(30));

ST04_P0 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>not RXD(0), Q=>crc_store(3));
ST04_P1 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(3), Q=>crc_store(7));
ST04_P2 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(7), Q=>crc_store(11));
ST04_P3 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(11), Q=>crc_store(15));
ST04_P4 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(15), Q=>crc_store(19));
ST04_P5 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(19), Q=>crc_store(23));
ST04_P6 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(23), Q=>crc_store(27));
ST04_P7 : FDE PORT MAP (C=>RX_CLK, CE=>RX_DV, D=>crc_store(27), Q=>crc_store(31));

--================================================================================================================================
--Delaying of 8 clock edge ( 8 clock edges to fill the crc_store signal) the signal next_crc on 32 bits
--================================================================================================================================
Basc01 : FD32CE
	PORT MAP(C=>RX_CLK, D=>next_crc, Q=>dec01);
Basc02 : FD32CE
	PORT MAP(C=>RX_CLK, D=>dec01, Q=>dec02);
Basc03 : FD32CE
	PORT MAP(C=>RX_CLK, D=>dec02, Q=>dec03);
Basc04 : FD32CE
	PORT MAP(C=>RX_CLK, D=>dec03, Q=>dec04);
Basc05 : FD32CE
	PORT MAP(C=>RX_CLK, D=>dec04, Q=>dec05);
Basc06 : FD32CE
	PORT MAP(C=>RX_CLK, D=>dec05, Q=>dec06);
Basc07 : FD32CE
	PORT MAP(C=>RX_CLK, D=>dec06, Q=>dec07);
Basc08 : FD32CE
	PORT MAP(C=>RX_CLK, D=>dec07, Q=>dec08);

--================================================================================================================================
-- process :	detect if the CRC32 is good
--											the destination MAC address is good
--											the message not too big
--================================================================================================================================
increm_addr : process(RX_CLK)
begin
if rising_edge(RX_CLK) then
	prev_RX_DV <= RX_DV;
-- detection of the rising edge of RX_DV ( beginning of the packet) to start the understanding of the received packet.
	if RX_DV /= prev_RX_DV  and RX_DV = '1' then
		debut_preamb<='1';
-- detection of the falling edge of RX_DV ( end of the packet) and filter the one that are too big, the one with a bad CRC32 and the ones with a bad destination adress.
-- Gives to the next module MEM_READ_INCOMING_MSG the adress of the end of the packet and the signal RECV_PACKET when something that passed the filter arrived.
	elsif RX_DV /= prev_RX_DV  and RX_DV = '0' and Too_big_msg='0' and crc_store= dec08 and DA_good='1' then
		END_ADDR <= addr1(9 downto 2);
		RECV_PACKET <= '1';
		DA_good <='0';
	else
-- Each packet comes with a preambule "0101" * 15 and "1101" and the end of the preambule
		if debut_preamb ='1' and RXD = "1101" then
			 addr1 <="0000000000";
			 Too_big_msg<='0';
			 debut_preamb<='0';
			 next_crc<= "11111111111111111111111111111111";
		else
-- increase the addr in the memory
			addr1 <= addr1 + '1';
-- computation of the CRC32 like in the module CRC32
			next_crc(0) <= next_crc(28) XOR RXD(3);
			next_crc(1) <= RXD(2) XOR next_crc(28) XOR RXD(3) XOR next_crc(29);
			next_crc(2) <= next_crc(30) XOR RXD(1) XOR RXD(2) XOR next_crc(28) XOR RXD(3) XOR next_crc(29);
			next_crc(3) <= next_crc(30) XOR RXD(0) XOR RXD(1) XOR next_crc(31) XOR RXD(2) XOR next_crc(29);
			next_crc(4) <= next_crc(30) XOR RXD(0) XOR RXD(1) XOR next_crc(31) XOR next_crc(28) XOR next_crc(0) XOR RXD(3);
			next_crc(5) <= RXD(0) XOR next_crc(31) XOR RXD(2) XOR next_crc(28) XOR RXD(3) XOR next_crc(29) XOR next_crc(1);
			next_crc(6) <= next_crc(30) XOR next_crc(2) XOR RXD(1) XOR RXD(2) XOR next_crc(29);
			next_crc(7) <= next_crc(30) XOR RXD(0) XOR RXD(1) XOR next_crc(31) XOR next_crc(3) XOR next_crc(28) XOR RXD(3);
			next_crc(8) <= RXD(0) XOR next_crc(31) XOR RXD(2) XOR next_crc(28) XOR RXD(3) XOR next_crc(4) XOR next_crc(29);
			next_crc(9) <= next_crc(30) XOR next_crc(5) XOR RXD(1) XOR RXD(2) XOR next_crc(29);
			next_crc(10) <= next_crc(30) XOR RXD(0) XOR RXD(1) XOR next_crc(31) XOR next_crc(6) XOR next_crc(28) XOR RXD(3);
			next_crc(11) <= RXD(0) XOR next_crc(31) XOR RXD(2) XOR next_crc(7) XOR next_crc(28) XOR RXD(3) XOR next_crc(29);
			next_crc(12) <= next_crc(30) XOR RXD(1) XOR RXD(2) XOR next_crc(28) XOR RXD(3) XOR next_crc(8) XOR next_crc(29);
			next_crc(13) <= next_crc(30) XOR RXD(0) XOR next_crc(9) XOR RXD(1) XOR next_crc(31) XOR RXD(2) XOR next_crc(29);
			next_crc(14) <= next_crc(30) XOR RXD(0) XOR RXD(1) XOR next_crc(31) XOR next_crc(10);
			next_crc(15) <= RXD(0) XOR next_crc(31) XOR next_crc(11);
			next_crc(16) <= next_crc(28) XOR RXD(3) XOR next_crc(12);
			next_crc(17) <= next_crc(13) XOR RXD(2) XOR next_crc(29);
			next_crc(18) <= next_crc(30) XOR RXD(1) XOR next_crc(14);
			next_crc(19) <= RXD(0) XOR next_crc(31) XOR next_crc(15);
			next_crc(20) <= next_crc(16);
			next_crc(21) <= next_crc(17);
			next_crc(22) <= next_crc(18) XOR next_crc(28) XOR RXD(3);
			next_crc(23) <= RXD(2) XOR next_crc(19) XOR next_crc(28) XOR RXD(3) XOR next_crc(29);
			next_crc(24) <= next_crc(30) XOR RXD(1) XOR RXD(2) XOR next_crc(20) XOR next_crc(29);
			next_crc(25) <= next_crc(30) XOR RXD(0) XOR RXD(1) XOR next_crc(31) XOR next_crc(21);
			next_crc(26) <= RXD(0) XOR next_crc(22) XOR next_crc(31) XOR next_crc(28) XOR RXD(3);
			next_crc(27) <= next_crc(23) XOR RXD(2) XOR next_crc(29);
			next_crc(28) <= next_crc(30) XOR RXD(1) XOR next_crc(24);
			next_crc(29) <= RXD(0) XOR next_crc(31) XOR next_crc(25);
			next_crc(30) <= next_crc(26);
			next_crc(31) <= next_crc(27);
-- if the packet is too big, the signal Too_big_msg goes up.
			if addr1 = "1111111100" then
			 Too_big_msg<='1';
			end if;
-- test if the received packet is broadcast or have the right destination IP.
			if addr1 = "0000000000" and (RXD="1111" or RXD = SA (43 downto 40)) then
					DA_good<='1';
			elsif addr1 = "0000000001" and not (RXD ="1111" or RXD = SA(47 downto 44)) then
					DA_good<='0';
			elsif addr1 = "0000000010" and not (RXD ="1111" or RXD = SA (35 downto 32)) then
					DA_good<='0';
			elsif addr1 = "0000000011" and not (RXD ="1111" or RXD = SA (39 downto 36)) then
					DA_good<='0';
			elsif addr1 = "0000000100" and not (RXD ="1111" or RXD = SA (27 downto 24)) then
					DA_good<='0';
			elsif addr1 = "0000000101" and not (RXD ="1111" or RXD = SA (31 downto 28)) then
					DA_good<='0';
			elsif addr1 = "0000000110" and not (RXD ="1111" or RXD = SA (19 downto 16)) then
					DA_good<='0';
			elsif addr1 = "0000000111" and not (RXD ="1111" or RXD = SA (23 downto 20)) then
					DA_good<='0';
			elsif addr1 = "0000001000" and not (RXD ="1111" or RXD = SA (11 downto 8)) then
					DA_good<='0';
			elsif addr1 = "0000001001" and not (RXD ="1111" or RXD = SA (15 downto 12)) then
					DA_good<='0';
			elsif addr1 = "0000001010" and not (RXD ="1111" or RXD = SA (3 downto 0)) then
					DA_good<='0';
			elsif addr1 = "0000001011" and not (RXD ="1111" or RXD = SA (7 downto 4)) then
				DA_good<='0';
			end if;
		end if;
	end if;
-- if the memory is read by the module MEM_READ_INCOMING_MSG 
	if ENABLE_INCOMING_MESSAGE_MEM = '1' then
		RECV_PACKET <= '0';
	end if;
end if; 
end process;

--================================================================================================================================
-- adapation of addr2 to match timing problems
--================================================================================================================================
addr2<=addr1(9 downto 2) & not addr1(1) &addr1(0);

--================================================================================================================================
-- Memory where is recorded the mesage
--================================================================================================================================
mem_packet_in : RAMB4_S4_S16
      PORT MAP (
		ADDRA=>addr2, CLKA=>RX_CLK, DIA(3)=>RXD(3), DIA(2)=>RXD(2), DIA(1)=>RXD(1), DIA(0)=>RXD(0), ENA=>RX_DV, RSTA=>'0', WEA=>'1', DOA=>open,
	   ADDRB=>ADDR_INCOMING_MESSAGE_MEM, CLKB=>CLK_INCOMING_MESSAGE_MEM, DIB=>"0000000000000000", ENB=>ENABLE_INCOMING_MESSAGE_MEM, RSTB=>'0', WEB=>'0', DOB=>DATA_INCOMING_MESSAGE_MEM);

end RTL;
