--=============================================================================
--
-- Name:		Microphone_Array
--
-- Purpose:	Main module that calls all the other ones
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. CAPTURE
-- 2. SRAM_INTERFACE
-- 3. CAPTURE_UDP_FRAME
-- 4. INCOMING_MESSAGE_MEM
-- 5. MEM_READ_INCOMING_MSG
-- 6. RESPONSE_STATUS_UDP_FRAME
-- 7. ARP_FRAME
-- 8. BOOTP_FRAME
-- 9. MUX4_1
-- 10. CRC32
-- 11. TX_FRAME
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

library UNISIM;
use UNISIM.vcomponents.ALL;

entity Microphone_Array is
    Port (
					CLK										: in  std_logic;	-- tx clk	  

					CLOCKS_ENABLE					: out std_logic;
					CAP_CLK								: in	std_logic;	-- 33.8696MHz
					CAP_CLK_SLAVE					: in	std_logic;	-- 33.8696MHz
					SYNC_CAP_CLK_SLAVE		: in std_logic;
					SYNC_CAP_CLK_MASTER		: out std_logic;
					BCK										: out std_logic;	-- 0.705616	=33.8696/3 /16	= 48 fs
					SCKI									: out std_logic;	-- 11.2896	=33.8696/3		= 512 fs
					LRCK									: out std_logic;	-- 0.022050	=33.8696/3 /512	= fs

					STDIN01								: in std_logic;
					STDIN02								: in std_logic; 
					STDIN03								: in std_logic;
					STDIN04								: in std_logic;
					STDIN05								: in std_logic;
					STDIN06								: in std_logic;
					STDIN07								: in std_logic;
					STDIN08								: in std_logic;
					STDIN09								: in std_logic;
					STDIN10								: in std_logic; 
					STDIN11								: in std_logic;
					STDIN12								: in std_logic;
					STDIN13								: in std_logic;
					STDIN14								: in std_logic;
					STDIN15								: in std_logic;
					STDIN16								: in std_logic;
					STDIN17								: in std_logic;
					STDIN18								: in std_logic; 
					STDIN19								: in std_logic;
					STDIN20								: in std_logic;
					STDIN21								: in std_logic;
					STDIN22								: in std_logic;
					STDIN23								: in std_logic;
					STDIN24								: in std_logic;
					STDIN25								: in std_logic;
					STDIN26								: in std_logic;
					STDIN27								: in std_logic;
					STDIN28								: in std_logic;
					STDIN29								: in std_logic;
					STDIN30								: in std_logic;
					STDIN31								: in std_logic;
					STDIN32								: in std_logic;

					SRAM_ADDR							: out std_logic_vector(18 downto 0);
					SRAM_r_w							: out std_logic;
					SRAM_OE								: out std_logic;
					SRAM_CE								: out std_logic;
					SRAM_IO01							: inout std_logic_vector(7 downto 0);
					SRAM_IO02							: inout std_logic_vector(7 downto 0);
					SRAM_IO03							: inout std_logic_vector(7 downto 0);
					SRAM_IO04							: inout std_logic_vector(7 downto 0);

					RXD										: in std_logic_vector(3 downto 0); 
					RX_CLK								: in std_logic; 
					RX_DV									: in std_logic;

					LED_ACTIVE						: out std_logic_vector(4 downto 0);
					MAC_ADDR							: in  std_logic_vector(7 downto 0);
					DATA_OUT							: out std_logic_vector(3 downto 0);
					EN_DATA_OUT						: out std_logic
					);

attribute LOC   : string;
attribute LOC of CLOCKS_ENABLE				: signal is "P205";
attribute LOC of CAP_CLK							: signal is "P185";	-- GCK3
attribute LOC of CAP_CLK_SLAVE				: signal is "P182";	-- GCK2
attribute LOC of SYNC_CAP_CLK_SLAVE		: signal is "P194";
attribute LOC of SYNC_CAP_CLK_MASTER	: signal is "P193";
attribute LOC of LRCK									: signal is "P199";
attribute LOC of BCK									: signal is "P201";
attribute LOC of SCKI									: signal is "P203";

attribute LOC of STDIN01			: signal is "P4";
attribute LOC of STDIN02			: signal is "P5";
attribute LOC of STDIN03			: signal is "P6";
attribute LOC of STDIN04			: signal is "P7";
attribute LOC of STDIN05			: signal is "P8";
attribute LOC of STDIN06			: signal is "P9";
attribute LOC of STDIN07			: signal is "P10";
attribute LOC of STDIN08			: signal is "P14";
attribute LOC of STDIN09			: signal is "P15";
attribute LOC of STDIN10			: signal is "P16";
attribute LOC of STDIN11			: signal is "P17";
attribute LOC of STDIN12			: signal is "P18";
attribute LOC of STDIN13			: signal is "P20";
attribute LOC of STDIN14			: signal is "P21";
attribute LOC of STDIN15			: signal is "P22";
attribute LOC of STDIN16			: signal is "P23";
attribute LOC of STDIN17			: signal is "P27";
attribute LOC of STDIN18			: signal is "P29";
attribute LOC of STDIN19			: signal is "P30";
attribute LOC of STDIN20			: signal is "P31";
attribute LOC of STDIN21			: signal is "P33";
attribute LOC of STDIN22			: signal is "P34";
attribute LOC of STDIN23			: signal is "P35";
attribute LOC of STDIN24			: signal is "P36";
attribute LOC of STDIN25			: signal is "P37";
attribute LOC of STDIN26			: signal is "P41";
attribute LOC of STDIN27			: signal is "P42";
attribute LOC of STDIN28			: signal is "P43";
attribute LOC of STDIN29			: signal is "P44";
attribute LOC of STDIN30			: signal is "P45";
attribute LOC of STDIN31			: signal is "P46";
attribute LOC of STDIN32			: signal is "P47";

attribute LOC of CLK					: signal is "P77";	-- GCK1
attribute LOC of DATA_OUT			: signal is "P68,P69,P70,P71";
attribute LOC of EN_DATA_OUT	: signal is "P63";
attribute LOC of RX_CLK				: signal is "P80";	-- GCK0
attribute LOC of RXD					: signal is "P87,P86,P84,P83";
attribute LOC of RX_DV				: signal is "P89";
attribute LOC of LED_ACTIVE		: signal is "P102,P101,P100,P99,P98";
attribute LOC of MAC_ADDR			: signal is "P48,P49,P57,P58,P59,P60,P61,P62";

attribute LOC of SRAM_ADDR		: signal is "P162,P161,P160,P152,P151,P150,P149,P148,P147,P146,P141,P140,P139,P138,P136,P135,P134,P133,P132";
attribute LOC of SRAM_r_w			: signal is "P163";
attribute LOC of SRAM_OE			: signal is "P164";
attribute LOC of SRAM_CE			: signal is "P165";
attribute LOC of SRAM_IO01		: signal is "P178,P179,P180,P181,P187,P188,P189,P191";
attribute LOC of SRAM_IO02		: signal is "P166,P167,P168,P172,P173,P174,P175,P176";
attribute LOC of SRAM_IO03		: signal is "P119,P120,P121,P122,P123,P125,P126,P127";
attribute LOC of SRAM_IO04		: signal is "P108,P109,P110,P111,P112,P113,P114,P115";

attribute ALLCLOCKNETS : string; 
attribute ALLCLOCKNETS of Microphone_Array: entity is "25ns";

end Microphone_Array;

-- Behavioral is used for simulation and testbench. No timing concern.
-- RTL (Register Transfer Level) is for real design.
architecture RTL of Microphone_Array is

component CLKDLL
	generic( CLKDV_DIVIDE : real :=  3.000000);
  port ( CLKFB  : in    std_logic; 
         CLKIN  : in    std_logic; 
         RST    : in    std_logic; 
         CLK0   : out   std_logic; 
         CLK180 : out   std_logic; 
         CLK270 : out   std_logic; 
         CLK2X  : out   std_logic; 
         CLK90  : out   std_logic; 
         CLKDV  : out   std_logic; 
         LOCKED : out   std_logic);
end component;

COMPONENT CAPTURE is				
    Port (
					CAP_CLK								: in std_logic;		--33.8696MHz
					CAP_CLK_SLAVE					: in	std_logic;	-- 33.8696MHz
					SYNC_CAP_CLK_SLAVE		: in std_logic;
					SYNC_CAP_CLK_MASTER		: out std_logic;
					SYNC_SLAVE						: in std_logic;

					LOW_RESET							: in std_logic;

					BCK										: out std_logic;		--1.058425	=33.8696/32       = 48 fs
					SCKI									: out std_logic;		--11.2896	=33.8696/3       	= 512 fs
					LRCK									: out std_logic;		--0.022050	=33.8696/3 /512 	= fs
					START_CAPTURE					: in std_logic;
					START_CAPTURE_SLV			: out std_logic;
					DOUBLE_FRQ_AD					: in std_logic; 

					STD01									: in std_logic;
					STD02									: in std_logic; 
					STD03									: in std_logic;
					STD04									: in std_logic;
					STD05									: in std_logic;
					STD06									: in std_logic;
					STD07									: in std_logic;
					STD08									: in std_logic;
					STD09									: in std_logic;
					STD10									: in std_logic; 
					STD11									: in std_logic;
					STD12									: in std_logic;
					STD13									: in std_logic;
					STD14									: in std_logic;
					STD15									: in std_logic;
					STD16									: in std_logic;
					STD17									: in std_logic;
					STD18									: in std_logic; 
					STD19									: in std_logic;
					STD20									: in std_logic;
					STD21									: in std_logic;
					STD22									: in std_logic;
					STD23									: in std_logic;
					STD24									: in std_logic;
					STD25									: in std_logic;
					STD26									: in std_logic;
					STD27									: in std_logic;
					STD28									: in std_logic;
					STD29									: in std_logic;
					STD30									: in std_logic;
					STD31									: in std_logic;
					STD32									: in std_logic;

					ADDR_CAPTURE_MEM			: in std_logic_vector (8 downto 0);
					CLK_CAPTURE_MEM				: in std_logic;
					ENABLE_CAPTURE_MEM		: in std_logic;
					DATA_CAPTURE_MEM			: out std_logic_vector (15 downto 0);
					PACKET_READY					: out std_logic
					);
end COMPONENT;

COMPONENT SRAM_INTERFACE is
    Port (
					CLK_SRAM_INTERFACE				: in std_logic;
					CLK_SRAM_INTERFACE90			: in std_logic;
					CLKD2_SRAM_INTERFACE			: in std_logic;

					RESET_SRAM_INTERFACE			: in std_logic;
					START_CAPTURE							: in std_logic;

					SRAM_ADDR									: out std_logic_vector(18 downto 0);
					SRAM_r_w									: out std_logic;
					SRAM_OE										: out std_logic;
					SRAM_CE										: out std_logic;
					SRAM_IO01									: inout std_logic_vector(7 downto 0);
					SRAM_IO02									: inout std_logic_vector(7 downto 0);
					SRAM_IO03									: inout std_logic_vector(7 downto 0);
					SRAM_IO04									: inout std_logic_vector(7 downto 0);

					REQUEST_OLD_PACKET				: in std_logic;
					SELECT_REQUEST_OLD_PACKET	: out std_logic;
					OLD_NUMERO_PACKET					: in std_logic_vector(10 downto 0);

					PACKET_READY_CAPTURE_UDP	: out std_logic;
					CAPTURE_UDP_BUZZY					: in std_logic;
					NUMERO_PACKET_CAPTURE_UDP	: out std_logic_vector(15 downto 0);							
					ADDR_CAPTURE_UDP					: in std_logic_vector(9 downto 0);
					EN_DATA_CAPTURE_UDP				: in std_logic;
					DATA_CAPTURE_UDP					: out std_logic_vector(7 downto 0);

					PACKET_READY_CAPTURE			: in std_logic;
					ADDR_CAPTURE							: out std_logic_vector(8 downto 0);
					EN_DATA_CAPTURE						: out std_logic;
					DATA_CAPTURE							: in std_logic_vector(15 downto 0)
					);
end COMPONENT;

COMPONENT CAPTURE_UDP_FRAME is
    Port (
					CLK_CAPTURE_UDP						: in std_logic;
					CLKD2_CAPTURE_UDP					: in std_logic;	
					RESET											: in std_logic;

					START_CAPTURE_UDP					: in std_logic;
					BUZZY											: out std_logic;

					ADDR_CAPTURE_UDP					: OUT std_logic_vector (9 downto 0);
					EN_DATA_CAPTURE_UDP_IN		: OUT STD_LOGIC;
					DATA_CAPTURE_UDP_IN				: IN std_logic_vector (7 downto 0);

					SOURCE_MAC								: in std_logic_vector(47 downto 0);
					DESTINATION_MAC						: in std_logic_vector(47 downto 0);
					SOURCE_IP									: in std_logic_vector(31 downto 0);
					DESTINATION_IP						: in std_logic_vector(31 downto 0);
					NUMERO_PACKETS						: in std_logic_vector(15 downto 0);

					SELECT_CAPTURE_UDP_FRAME	: in std_logic;
					REQ_CAPTURE_UDP_FRAME			: out std_logic;		------ UP WHEN PACKET READY TO SEND AND GO DOWN AT END OF PACKET
					EN_DATA_CAPTURE_UDP_OUT		: out std_logic;
					DATA_CAPTURE_UDP_OUT			: out std_logic_vector(7 downto 0)
					);
end COMPONENT;

COMPONENT INCOMING_MESSAGE_MEM is
    Port (
					CLK_INCOMING_MESSAGE_MEM			: IN STD_LOGIC;
					ADDR_INCOMING_MESSAGE_MEM			: IN std_logic_vector (7 downto 0);
					ENABLE_INCOMING_MESSAGE_MEM		: IN STD_LOGIC;
					DATA_INCOMING_MESSAGE_MEM			: OUT std_logic_vector (15 downto 0);

					RECV_PACKET										: OUT STD_LOGIC;
					END_ADDR											: OUT std_logic_vector (7 downto 0);

					SA														: in std_logic_vector(47 downto 0);

					RXD														: IN STD_LOGIC_VECTOR (3 DOWNTO 0); 
					RX_CLK												: IN STD_LOGIC; 
					RX_DV													: IN STD_LOGIC
					);
end COMPONENT;


COMPONENT MEM_READ_INCOMING_MSG is
    Port (
					CLK_MEM_READ							: in std_logic;
					RESET											: in std_logic;

					ADDR_INCOMING_MSG_MEM			: out std_logic_vector(7 downto 0);
					ENABLE_INCOMING_MSG_MEM		: out std_logic;
					DATA_INCOMING_MSG_MEM			: in std_logic_vector(15 downto 0);

					SRC_IP_ARP_REQ						: out  std_logic_vector(31 downto 0);
					MY_IP											: in std_logic_vector(31 downto 0);
					ARP_FRAME_BUZZY						: in std_logic;
					REQ_ARP										: out std_logic;

					ID												: in std_logic_vector(31 downto 0);
					BOOTP_IP									: out std_logic_vector(31 downto 0);
					REQ_BOOTP									: out std_logic;

					SYNC_SLAVE_ON							: out std_logic;
					DOUBLE_FRQ_AD							: out std_logic; 
					START_CAPTURE							: out std_logic;

					REQUEST_OLD_PACKET				: out std_logic;
					SELECT_REQUEST_OLD_PACKET	: in std_logic;
					OLD_NUMERO_PACKET					: out std_logic_vector(10 downto 0);

					RESPONSE_REQUEST_BUZZY		: in std_logic;
					REQ_RESPONSE_REQUEST			: out std_logic;
					TYPE_RESPONSE_REQUEST			: out std_logic_vector(2 downto 0);
					DATA_REQUEST							: out std_logic_vector(9 downto 0);
					IP_SENDER									: out std_logic_vector(31 downto 0);
					MAC_SENDER								: out std_logic_vector(47 downto 0);
					PRIORITY_SENDER						: out std_logic;

					RECV_PACKET_MSG						: in std_logic;
					END_ADDR_MSG 							: in std_logic_vector(7 downto 0)
					);
end COMPONENT;

COMPONENT RESPONSE_STATUS_UDP_FRAME is
    Port (
					CLK_RESPONSE_STATUS						: in std_logic;
					RESET													: in std_logic;

					START_RESPONSE_STATUS					: in std_logic;
					BUZZY													: out std_logic;

					TYPE_REQUEST									: in std_logic_vector(2 downto 0);
					DATA_REQUEST									: in std_logic_vector(9 downto 0);

					SOURCE_MAC										: in std_logic_vector(47 downto 0);
					DESTINATION_MAC								: in std_logic_vector(47 downto 0);
					SOURCE_IP 										: in std_logic_vector(31 downto 0);
					DESTINATION_IP 								: in std_logic_vector(31 downto 0);

					SELECT_RESPONSE_STATUS_FRAME	: in std_logic;
					REQ_RESPONSE_STATUS_FRAME			: out std_logic;	------ UP WHEN PACKET READY TO SEND AND GO DOWN AT END OF PACKET
					EN_DATA_RESPONSE_STATUS_OUT		: out std_logic;
					DATA_RESPONSE_STATUS_OUT			: out std_logic_vector(7 downto 0)
					);
end COMPONENT;

COMPONENT ARP_FRAME is
    Port (
					CLK_ARP						: in std_logic;
					RESET							: in std_logic;

					START_ARP					: in std_logic;
					BUZZY							: out std_logic;

					SOURCE_MAC				: in std_logic_vector(47 downto 0);
					DESTINATION_MAC		: in std_logic_vector(47 downto 0);
					SOURCE_IP					: in std_logic_vector(31 downto 0);
					DESTINATION_IP		: in std_logic_vector(31 downto 0);

					SELECT_ARP_FRAME	: in std_logic;
					REQ_ARP_FRAME			: out std_logic;	------ UP WHEN PACKET READY TO SEND AND GO DOWN AT END OF PACKET
					EN_DATA_ARP_OUT		: out std_logic;
					DATA_ARP_OUT 			: out std_logic_vector(7 downto 0)
					);
end COMPONENT;

COMPONENT BOOTP_FRAME is
    Port (
					CLK_BOOTP						: in std_logic;
					RESET								: in std_logic;

					START_BOOTP					: in std_logic;
					BUZZY								: out std_logic;

					SOURCE_MAC					: in std_logic_vector(47 downto 0);
					SECONDS							: in std_logic_vector(7 downto 0);

					SELECT_BOOTP_FRAME	: in std_logic;
					REQ_BOOTP_FRAME			: out std_logic;	------ UP WHEN PACKET READY TO SEND AND GO DOWN AT END OF PACKET
					EN_DATA_BOOTP_OUT		: out std_logic;
					DATA_BOOTP_OUT			: out std_logic_vector(7 downto 0)
					);
end COMPONENT;

COMPONENT MUX4_1 is
    Port (
					CLK_MUX						: in std_logic;
					RESET							: in std_logic;
					TX_FRAME_READY		: in std_logic;

					REQ_IN_0					: in std_logic;
					SELECT_0					: out std_logic;
					EN_D_IN_0					: in std_logic;
					D_IN_0						: in std_logic_vector(7 downto 0);

					REQ_IN_1					: in std_logic;
					SELECT_1					: out std_logic;
					EN_D_IN_1					: in std_logic;
					D_IN_1						: in std_logic_vector(7 downto 0);

					REQ_IN_2					: in std_logic;
					SELECT_2					: out std_logic;
					EN_D_IN_2					: in std_logic;
					D_IN_2						: in std_logic_vector(7 downto 0);

					REQ_IN_3					: in std_logic;
					SELECT_3					: out std_logic;
					EN_D_IN_3					: in std_logic;
					D_IN_3						: in std_logic_vector(7 downto 0);

					EN_DATA_MUX_OUT		: out std_logic;
					DATA_MUX_OUT			: out std_logic_vector(7 downto 0)
					);
end COMPONENT;

COMPONENT CRC32 is
    Port (
					CLK_CRC32						: in std_logic;
					RESET								: in std_logic;

					DATA_CRC32_IN 			: in std_logic_vector(7 downto 0);
					EN_DATA_CRC32_IN		: in std_logic;
					DATA_CRC32_OUT 			: out std_logic_vector(7 downto 0);
					EN_DATA_CRC32_OUT		: out std_logic
					);
end COMPONENT;

COMPONENT TX_FRAME is
    Port (
					CLK_TX								: in std_logic;

					DATA_TX_IN						: in std_logic_vector(7 downto 0);
					EN_DATA_TX_IN					: in std_logic;

					DATA_TX_OUT						: out std_logic_vector(3 downto 0);
					EN_DATA_TX_OUT				: out std_logic
					);
end COMPONENT;

signal addr_capture_udp								: std_logic_vector(9 downto 0);
signal en_data_capture_udp						: std_logic;
signal data_capture_udp								: std_logic_vector(7 downto 0);

signal packet_present1								: std_logic;
signal packet_number									: std_logic_vector(15 downto 0);
signal en_data_capture0								: std_logic;
signal data_capture0									: std_logic_vector(15 downto 0);

signal addr0													: std_logic_vector(7 downto 0);
signal pres_data											: std_logic;
signal addr_fin												: std_logic_vector(7 downto 0);
signal endatainterm0									: std_logic;
signal datainterm0										: std_logic_vector(15 downto 0);
signal s_START_RESPONSE_STATUS				: std_logic;
signal s_START_ARP										: std_logic;
signal req_bootp_present							: std_logic;

signal buzzy_response_status					: std_logic;
signal buzzy_capture_udp							: std_logic;
signal buzzy_arp_frame								: std_logic;

signal addr_capture0									: std_logic_vector(8 downto 0);
signal packet_present0								: std_logic;

signal type_response_request0					: std_logic_vector(2 downto 0);
signal data_response_status0					: std_logic_vector(9 downto 0);

signal mac_addr_array									: std_logic_vector(47 downto 0);
signal ip_addr_array									: std_logic_vector(31 downto 0);

signal mac_addr_destination_response	: std_logic_vector(47 downto 0);
signal ip_addr_destination_response		: std_logic_vector(31 downto 0);
signal mac_addr_destination_response1	: std_logic_vector(47 downto 0);
signal ip_addr_destination_response1	: std_logic_vector(31 downto 0);
signal ip_addr_destination_arp				: std_logic_vector(31 downto 0);
signal priority												: std_logic;

signal s_start_bootp									: std_logic;
signal en_dout00											: std_logic;
signal dout00													: std_logic_vector(7 downto 0);
signal sel00													: std_logic;	
signal req00													: std_logic;

signal sel01													: std_logic;
signal req01													: std_logic;
signal en_dout01											: std_logic;
signal dout01													: std_logic_vector(7 downto 0);

signal mem_seconds										: std_logic_vector(7 downto 0);
signal seconds												: std_logic_vector(7 downto 0);

signal sel02													: std_logic;
signal req02													: std_logic;
signal en_dout02											: std_logic;
signal dout02													: std_logic_vector(7 downto 0);
signal clkd2													: std_logic;
signal buzzy_bootp_frame							: std_logic;

signal en_data_in_crc32								: std_logic;
signal data_in_crc32									: std_logic_vector(7 downto 0);

signal en_data_in_tx_frame						: std_logic;
signal data_in_tx_frame								: std_logic_vector(7 downto 0);

signal sel03													: std_logic;
signal req03													: std_logic;
signal en_dout03											: std_logic;
signal dout03													: std_logic_vector(7 downto 0);

signal s_EN_DATA_OUT									: std_logic;
signal s_clocks_enable								: std_logic;
signal int_START_CAPTURE_SLV					: std_logic;
signal s_START_CAPTURE_SLV						: std_logic;
signal s_double_frequency							: std_logic;
signal s_sync_slave_on								: std_logic;
signal req_old_packets								: std_logic;
signal sel_old_packets								: std_logic;
signal num_old_packets								: std_logic_vector(10 downto 0);

signal counttps												: std_logic_vector(23 downto 0);
signal srstn													: std_logic;

signal clkin2													: std_logic;
signal clkfb_2												: std_logic;
signal clk_2													: std_logic;
signal clkd90													: std_logic;
signal clkdll_ready										: std_logic;
signal clkdll_ready0									: std_logic;
signal clkdiv2												: std_logic;

signal clkin4													: std_logic;
signal clkfb_4												: std_logic;
signal clk_4													: std_logic;
signal clkd4													: std_logic;

attribute CLKDV_DIVIDE: string;
attribute CLKDV_DIVIDE of div_2_clkdll : label is "2";
attribute CLKDV_DIVIDE of div_4_clkdll : label is "4";

type stateType is	(
				START_ENGINE,
				REQUEST_BOOTP,
				WAIT_RESPONSE_BOOTP,
				SEND_DATA
			);
signal current_state : stateType;

begin

--================================================================================================================================
-- Outputs or Test point outputs
--================================================================================================================================
EN_DATA_OUT		<= s_EN_DATA_OUT;														-- for the verification of no output data
mac_addr_array	<= x"1000000003" & MAC_ADDR(7 downto 0);	-- Adress MAC plaque 10:00:00:00:03:XX
CLOCKS_ENABLE	<= int_START_CAPTURE_SLV;										-- Capture is on

--================================================================================================================================
-- process :	This process count time in seconds in order to give setup time to the 80225
--================================================================================================================================
Comptetemps: process (clkdll_ready, clkfb_2)
begin
	if clkdll_ready = '0' then
		srstn		<= '0';
		seconds	<= ( others => '0');
		counttps	<= ( others => '0');
	elsif rising_edge(clkfb_2) then
		if clkdiv2 = '1' then 
			if counttps = "101111101011110000100000" then
				seconds	<= seconds + 1;
				counttps	<= (others => '0');
			else
				counttps	<= counttps + 1;
			end if;
			if seconds = "00000010" then
				srstn	<= '1';
			end if;
		end if;
	end if;
end process;

--================================================================================================================================
-- process :	Divide the main clock by 2
--================================================================================================================================
process(clkfb_2)
begin
if rising_edge(clkfb_2) then
	if clkdll_ready = '0' then
		clkdiv2 <= '0';
	else
		clkdiv2 <= not clkdiv2;
	end if;
end if;
end process;

--================================================================================================================================
-- process :	Starting state machine
--						Do bootp request until an understood IP address has arrived
--						After that it's in working status and LED_ACTIVE(4) lights
--================================================================================================================================
state_machine: process(clkfb_2, srstn)
begin
	if (srstn='0') then
		current_state <= START_ENGINE;
	elsif rising_edge(clkfb_2) then
		if clkdiv2 = '1' then 
		case current_state is

			when START_ENGINE =>
					s_START_BOOTP<='0';
					current_state <= REQUEST_BOOTP;
					LED_ACTIVE(4)<='0';
			
			when REQUEST_BOOTP =>
					s_START_BOOTP<='1';
					mem_seconds<= seconds + "00000010";
					if buzzy_bootp_frame = '1' then
						current_state <= WAIT_RESPONSE_BOOTP;
						s_START_BOOTP<='0'; 
					end if;

			when WAIT_RESPONSE_BOOTP =>
					if seconds = mem_seconds then
						current_state <= REQUEST_BOOTP;
					end if;
					if req_bootp_present ='1' then
						current_state <= SEND_DATA;
					end if;

			when SEND_DATA =>
					current_state <= SEND_DATA;
					LED_ACTIVE(4)<='1';
			when others =>	
					current_state <= START_ENGINE;

		end case;
		end if;
	end if;
end process;

--================================================================================================================================
-- signals :	control signals that light a LED
--						int_START_CAPTURE_SLV when there is a capture active
--						req_old_packets when old packets are asked
--						s_double_frequency when the acquisition frequency is doubled
--						s_sync_slave_on when the slave mode is active
--================================================================================================================================
LED_ACTIVE(3) <= int_START_CAPTURE_SLV;
LED_ACTIVE(2) <= s_double_frequency;
LED_ACTIVE(1) <= s_sync_slave_on;
LED_ACTIVE(0) <= int_START_CAPTURE_SLV;

incomming_msg_mem: INCOMING_MESSAGE_MEM
    PORT MAP(
						CLK_INCOMING_MESSAGE_MEM		=> clkd90,
						ADDR_INCOMING_MESSAGE_MEM		=> addr0,
						ENABLE_INCOMING_MESSAGE_MEM	=> endatainterm0,
						DATA_INCOMING_MESSAGE_MEM		=> datainterm0,

						SA													=> mac_addr_array,
						RECV_PACKET									=> pres_data,
						END_ADDR										=> addr_fin,

						RXD													=> RXD,
						RX_CLK											=> clkfb_4,
						RX_DV												=> RX_DV
						);

mem_read_incom_msg: MEM_READ_INCOMING_MSG
    PORT MAP(
						CLK_MEM_READ							=> clkfb_2,
						RESET											=> clkdll_ready,
						ADDR_INCOMING_MSG_MEM			=> addr0, 
						ENABLE_INCOMING_MSG_MEM		=> endatainterm0,
						DATA_INCOMING_MSG_MEM			=> datainterm0,

						RECV_PACKET_MSG						=> pres_data,
						END_ADDR_MSG							=> addr_fin,

						SRC_IP_ARP_REQ						=> ip_addr_destination_arp,
						MY_IP											=> ip_addr_array,
						ARP_FRAME_BUZZY						=> buzzy_arp_frame,
						REQ_ARP										=> s_START_ARP,

						ID												=> mac_addr_array(31 downto 0),
						BOOTP_IP									=> ip_addr_array,
						REQ_BOOTP									=> req_bootp_present,

						SYNC_SLAVE_ON							=> s_sync_slave_on,
						START_CAPTURE							=> s_CLOCKS_ENABLE,
						DOUBLE_FRQ_AD							=> s_double_frequency,

						REQUEST_OLD_PACKET				=> req_old_packets, 
						SELECT_REQUEST_OLD_PACKET	=> sel_old_packets,
						OLD_NUMERO_PACKET					=> num_old_packets,

						RESPONSE_REQUEST_BUZZY		=> buzzy_response_status,
						REQ_RESPONSE_REQUEST			=> s_START_RESPONSE_STATUS,
						TYPE_RESPONSE_REQUEST			=> type_response_request0,
						DATA_REQUEST							=> data_response_status0,

						IP_SENDER									=> ip_addr_destination_response,
						MAC_SENDER								=> mac_addr_destination_response,
						PRIORITY_SENDER						=> priority
						);

--================================================================================================================================
-- CLKDLL :	This CLKDLL is the main clock of all the modules except capture
--					It's based on the TX_CLOCK of the 80225
--					It's dividing the main clock by 2
--================================================================================================================================
div_2_ibufg: IBUFG
	port map (I => CLK, O => clkin2);
div_2_bufg: BUFG
	port map (I => clk_2, O => clkfb_2);
div_2_clkdll: CLKDLL
    PORT MAP(CLKIN => clkin2, CLKFB => clkfb_2, RST => '0',CLK2X=>open, CLK180=>open, CLK270=>open,
						CLK0 => clk_2, CLK90 => clkd90, LOCKED => clkdll_ready0, CLKDV => clkd2);

--================================================================================================================================
-- FLIP_FLOP : resynchronized clkdll_ready
--================================================================================================================================
sync_clkdll_ready : FD	PORT MAP (C=>clkfb_2, D=>clkdll_ready0, Q=>clkdll_ready);

--================================================================================================================================
-- CLKDLL :	This CLKDLL is the clock used only in the module INCOMING_MESSAGE_MEM
--					It's based on the RX_CLOCK of the 80225
--					It's dividing the main clock by 2
--================================================================================================================================
div_4_ibufg: IBUFG
	port map (I => RX_CLK, O => clkin4); 
div_4_bufg: BUFG
	port map (I => clk_4, O => clkfb_4);
div_4_clkdll: CLKDLL 
    PORT MAP(CLKIN => clkin4, CLKFB => clkfb_4, RST => '0',CLK2X=>open, CLK90=>open, CLK180=>open, CLK270=>open,
						CLK0 => clk_4, LOCKED	=> open, CLKDV => clkd4);

-------------------------------------------------------------------------------------------emission packets
-------------------------------------------------------------------------------------------capture
the_capture: capture 
    PORT MAP(
							CAP_CLK								=> CAP_CLK,
							CAP_CLK_SLAVE					=> CAP_CLK_SLAVE,
							SYNC_CAP_CLK_SLAVE		=> SYNC_CAP_CLK_SLAVE,
							SYNC_CAP_CLK_MASTER		=> SYNC_CAP_CLK_MASTER,
							SYNC_SLAVE						=> s_sync_slave_on,
							LOW_RESET							=> clkdll_ready,

							BCK										=> BCK,
							SCKI									=> SCKI,
							LRCK									=> LRCK,

							START_CAPTURE					=> s_CLOCKS_ENABLE,
							START_CAPTURE_SLV			=> s_START_CAPTURE_SLV,
							DOUBLE_FRQ_AD					=> s_double_frequency,

							STD01									=> STDIN01,
							STD02									=> STDIN02,
							STD03									=> STDIN03,
							STD04									=> STDIN04,
							STD05									=> STDIN05,
							STD06									=> STDIN06,
							STD07									=> STDIN07,
							STD08									=> STDIN08,
							STD09									=> STDIN09,
							STD10									=> STDIN10,
							STD11									=> STDIN11,
							STD12									=> STDIN12,
							STD13									=> STDIN13,
							STD14									=> STDIN14,
							STD15									=> STDIN15,
							STD16									=> STDIN16,
							STD17									=> STDIN17,
							STD18									=> STDIN18,
							STD19									=> STDIN19,
							STD20									=> STDIN20,
							STD21									=> STDIN21,
							STD22									=> STDIN22,
							STD23									=> STDIN23,
							STD24									=> STDIN24,
							STD25									=> STDIN25,
							STD26									=> STDIN26,
							STD27									=> STDIN27,
							STD28									=> STDIN28,
							STD29									=> STDIN29,
							STD30									=> STDIN30,
							STD31									=> STDIN31,
							STD32									=> STDIN32,

							ADDR_CAPTURE_MEM			=> addr_capture0,
							CLK_CAPTURE_MEM				=> clkfb_2,
							ENABLE_CAPTURE_MEM		=> en_data_capture0,
							DATA_CAPTURE_MEM			=> data_capture0,
							PACKET_READY					=> packet_present0
							);

-------------------------------------------------------------------------------------------capture
-------------------------------------------------------------------------------------------mem buffer
the_sram_interface: SRAM_INTERFACE 
    PORT MAP(
						CLK_SRAM_INTERFACE				=> clkfb_2,
						CLK_SRAM_INTERFACE90			=> clkd90,
						CLKD2_SRAM_INTERFACE			=> clkd2,
						RESET_SRAM_INTERFACE			=> clkdll_ready,

						START_CAPTURE							=> int_START_CAPTURE_SLV,

						SRAM_ADDR									=> SRAM_ADDR,
						SRAM_r_w									=> SRAM_r_w,
						SRAM_OE										=> SRAM_OE,
						SRAM_CE										=> SRAM_CE,
						SRAM_IO01									=> SRAM_IO01,
						SRAM_IO02									=> SRAM_IO02,
						SRAM_IO03									=> SRAM_IO03,
						SRAM_IO04									=> SRAM_IO04,

						OLD_NUMERO_PACKET					=> num_old_packets,
						REQUEST_OLD_PACKET				=> req_old_packets,
						SELECT_REQUEST_OLD_PACKET	=> sel_old_packets,

						PACKET_READY_CAPTURE_UDP	=> packet_present1,
						CAPTURE_UDP_BUZZY					=> buzzy_capture_udp,
						NUMERO_PACKET_CAPTURE_UDP	=> packet_number,				
						ADDR_CAPTURE_UDP					=> addr_capture_udp,
						EN_DATA_CAPTURE_UDP				=> en_data_capture_udp,
						DATA_CAPTURE_UDP					=> data_capture_udp,

						PACKET_READY_CAPTURE			=> packet_present0,
						ADDR_CAPTURE							=> addr_capture0,
						EN_DATA_CAPTURE						=> en_data_capture0,
						DATA_CAPTURE							=> data_capture0
						);
-------------------------------------------------------------------------------------------mem buffer

the_bootp_frame: BOOTP_FRAME
    PORT MAP(
						CLK_BOOTP						=> clkfb_2,
						RESET								=> clkdll_ready,

						START_BOOTP					=> s_START_BOOTP,
						BUZZY								=> buzzy_bootp_frame,
						SOURCE_MAC					=> mac_addr_array,
						SECONDS							=> seconds,

						SELECT_BOOTP_FRAME	=> sel00,
						REQ_BOOTP_FRAME			=> req00,
						EN_DATA_BOOTP_OUT		=> en_dout00,
						DATA_BOOTP_OUT			=> dout00
						);

the_arp_frame: ARP_FRAME
    PORT MAP(
						CLK_ARP							=> clkfb_2,
						RESET								=> clkdll_ready,

						START_ARP						=> s_START_ARP,
						BUZZY								=> buzzy_arp_frame,
						SOURCE_MAC					=> mac_addr_array,
						DESTINATION_MAC			=> mac_addr_destination_response,
						SOURCE_IP						=> ip_addr_array,
						DESTINATION_IP			=> ip_addr_destination_arp,

						SELECT_ARP_FRAME		=> sel01,
						REQ_ARP_FRAME				=> req01,
						EN_DATA_ARP_OUT			=> en_dout01,
						DATA_ARP_OUT				=> dout01
						);

the_response_frame: RESPONSE_STATUS_UDP_FRAME
    PORT MAP(
						CLK_RESPONSE_STATUS						=> clkfb_2,
						RESET													=> clkdll_ready,

						START_RESPONSE_STATUS					=> s_START_RESPONSE_STATUS,
						BUZZY													=> buzzy_response_status,
						TYPE_REQUEST									=> type_response_request0,
						DATA_REQUEST									=> data_response_status0,
						SOURCE_MAC										=> mac_addr_array,
						DESTINATION_MAC								=> mac_addr_destination_response,
						SOURCE_IP											=> ip_addr_array,
						DESTINATION_IP								=> ip_addr_destination_response,

						SELECT_RESPONSE_STATUS_FRAME	=> sel02,
						REQ_RESPONSE_STATUS_FRAME			=> req02,
						EN_DATA_RESPONSE_STATUS_OUT		=> en_dout02,
						DATA_RESPONSE_STATUS_OUT			=> dout02
		);

--================================================================================================================================
-- BUFFER :	Buffer of the IP and MAC address of the computer which is commanding the Microphone array
--					Permit ar computer to ask status of the microphone array 
--					without recieving all the capture data asked by another computer
--					priority goes up when it has the IP and MAC address of the main ( giving commands) computer
--================================================================================================================================
mac_addr_destination_response1 <= mac_addr_destination_response when priority = '1';
ip_addr_destination_response1 <= ip_addr_destination_response when priority = '1';

--================================================================================================================================
-- SIGNAL :	goes up when a capture is started
--================================================================================================================================
int_START_CAPTURE_SLV <= ((s_CLOCKS_ENABLE and (not s_sync_slave_on)) or s_START_CAPTURE_SLV);

the_capture_frame: CAPTURE_UDP_FRAME
    PORT MAP(
						CLK_CAPTURE_UDP						=> clkfb_2,
						CLKD2_CAPTURE_UDP					=> clkd2,
						RESET											=> int_START_CAPTURE_SLV,

						START_CAPTURE_UDP					=> packet_present1,
						BUZZY											=> buzzy_capture_udp,
						ADDR_CAPTURE_UDP					=> addr_capture_udp,
						EN_DATA_CAPTURE_UDP_IN		=> en_data_capture_udp,
						DATA_CAPTURE_UDP_IN				=> data_capture_udp,
						SOURCE_MAC								=> mac_addr_array,
						DESTINATION_MAC						=> mac_addr_destination_response1,
						SOURCE_IP									=> ip_addr_array,
						DESTINATION_IP						=> ip_addr_destination_response1,
						NUMERO_PACKETS						=> packet_number,

						SELECT_CAPTURE_UDP_FRAME	=> sel03,
						REQ_CAPTURE_UDP_FRAME			=> req03,
						EN_DATA_CAPTURE_UDP_OUT		=> en_dout03,
						DATA_CAPTURE_UDP_OUT			=> dout03
						);

the_multiplex : MUX4_1
    PORT MAP(
						CLK_MUX					=> clkfb_2,
						RESET						=> clkdll_ready,
						TX_FRAME_READY	=> s_EN_DATA_OUT,

						REQ_IN_0				=> req00,
						SELECT_0				=> sel00,
						EN_D_IN_0				=> en_dout00,
						D_IN_0					=> dout00,

						REQ_IN_1				=> req01,
						SELECT_1				=> sel01,
						EN_D_IN_1				=> en_dout01,
						D_IN_1					=> dout01,

						REQ_IN_2				=> req02,
						SELECT_2				=> sel02,
						EN_D_IN_2				=> en_dout02,
						D_IN_2					=> dout02,

						REQ_IN_3				=> req03,
						SELECT_3				=> sel03,
						EN_D_IN_3				=> en_dout03,
						D_IN_3					=> dout03,

						EN_DATA_MUX_OUT	=> en_data_in_crc32,
						DATA_MUX_OUT		=> data_in_crc32
						);

the_crc_fct: crc32
    PORT MAP(
						CLK_CRC32					=> clkd2,
						RESET							=> clkdll_ready,

						DATA_CRC32_IN			=> data_in_crc32, 
						EN_DATA_CRC32_IN	=> en_data_in_crc32,

						DATA_CRC32_OUT		=> data_in_tx_frame, 
						EN_DATA_CRC32_OUT	=> en_data_in_tx_frame
						);

the_last_frame: TX_FRAME
    PORT MAP(
						CLK_TX					=> clkfb_2,

						DATA_TX_IN			=> data_in_tx_frame,
						EN_DATA_TX_IN		=> en_data_in_tx_frame,

						DATA_TX_OUT			=> DATA_OUT,
						EN_DATA_TX_OUT	=> s_EN_DATA_OUT
						);
end RTL;
