--=============================================================================
--
-- Name:		ARP_FRAME
--
-- Purpose:	This module makes an ARP reply packet.
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. MEM_READ_INCOMING_MSG
-- 2. MUX4_1
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity ARP_FRAME is
    Port (
					CLK_ARP						: in std_logic;
					RESET							: in std_logic;

					START_ARP					: in std_logic;
					BUZZY							: out std_logic;

					SOURCE_MAC				: in std_logic_vector(47 downto 0);
					DESTINATION_MAC		: in std_logic_vector(47 downto 0);
					SOURCE_IP					: in std_logic_vector(31 downto 0);
					DESTINATION_IP		: in std_logic_vector(31 downto 0);

					SELECT_ARP_FRAME	: in std_logic;
					REQ_ARP_FRAME			: out std_logic;
					EN_DATA_ARP_OUT		: out std_logic;
					DATA_ARP_OUT 			: out std_logic_vector(7 downto 0)
					);
end ARP_FRAME;

architecture RTL of ARP_FRAME is

type stateType is	(	idle,wait_selection,
										send_dest_addr1,send_dest_addr2, send_dest_addr3, send_dest_addr4, send_dest_addr5, send_dest_addr6,
										send_src_addr1, send_src_addr2, send_src_addr3, send_src_addr4, send_src_addr5, send_src_addr6,
										send_lenght_mac1,send_lenght_mac2,
										send_Source_IP_address1, send_Source_IP_address2, send_Source_IP_address3, send_Source_IP_address4,
										send_Destination_IP_address1, send_Destination_IP_address2, send_Destination_IP_address3, send_Destination_IP_address4,
										send_hardware_type1,send_hardware_type2,send_protocol_type1,send_protocol_type2,
										send_hardware_size,send_protocol_size,send_opcode1,send_opcode2,
										send_src_addr01, send_src_addr02, send_src_addr03, send_src_addr04, send_src_addr05, send_src_addr06,
										send_dest_addr01,send_dest_addr02, send_dest_addr03, send_dest_addr04, send_dest_addr05, send_dest_addr06,
										send_00);

signal next_state : stateType;

signal cpt 										: std_logic_vector(4 downto 0);
signal clkd2									: std_logic;
begin

--================================================================================================================================
-- process :	Divide the main clock by 2
--================================================================================================================================
process(CLK_ARP)
begin
if rising_edge(CLK_ARP) then
	clkd2 <= not clkd2;
end if;
end process;

--================================================================================================================================
-- In this process, I am using a linear state machine in order to create the different fields in an arp packet.
-- RESET is provided by the main clockdll with LOCKED which is a low reset and to give a starting point to the state machine.
-- START_ARP is the signal starting the state machine
-- The next module connected to this one is the mux4_1 which gives us a feedback in order to know when it's ready.
-- So the signal SELECT_ARP_FRAME is the one waiting the green light to start.
-- After that it's the different fields of an arp.
-- the counter at the end is here to match the minimum size packet
--================================================================================================================================
unpack : process(CLK_ARP,RESET)
begin
if RESET='0' then
	next_state <= idle;
elsif rising_edge(CLK_ARP) then
	if clkd2='1' then		
		case next_state is
			when idle =>
				if (START_ARP) = '1' then
					next_state <= wait_selection;
				else
					next_state <= idle;
				end if;
			when wait_selection =>
				if (SELECT_ARP_FRAME) = '1' then
					next_state <= send_dest_addr1;
				else
					next_state <= wait_selection;
				end if;
			when send_dest_addr1 =>
				next_state <= send_dest_addr2;
			when send_dest_addr2 =>
				next_state <= send_dest_addr3;
			when send_dest_addr3 =>
				next_state <= send_dest_addr4;
			when send_dest_addr4 =>
				next_state <= send_dest_addr5;
			when send_dest_addr5 =>
				next_state <= send_dest_addr6;
			when send_dest_addr6 =>
				next_state <= send_src_addr1;
			when send_src_addr1 =>
				next_state <= send_src_addr2;
			when send_src_addr2 =>
				next_state <= send_src_addr3;
			when send_src_addr3 =>
				next_state <= send_src_addr4;
			when send_src_addr4 =>
				next_state <= send_src_addr5;
			when send_src_addr5 =>
				next_state <= send_src_addr6;
			when send_src_addr6 =>
				next_state <= send_lenght_mac1;
			when send_lenght_mac1 =>
				next_state <= send_lenght_mac2;
			when send_lenght_mac2 =>
				next_state <= send_hardware_type1;
			when send_hardware_type1 =>
				next_state <= send_hardware_type2;
			when send_hardware_type2 =>
				next_state <= send_protocol_type1;
			when send_protocol_type1 =>
				next_state <= send_protocol_type2;
			when send_protocol_type2 =>
				next_state <= send_hardware_size;
			when send_hardware_size =>
				next_state <= send_protocol_size;
			when send_protocol_size =>
				next_state <= send_opcode1;
			when send_opcode1 =>
				next_state <= send_opcode2;
			when send_opcode2 =>
				next_state <= send_src_addr01;
			when send_src_addr01 =>
				next_state <= send_src_addr02;
			when send_src_addr02 =>
				next_state <= send_src_addr03;
			when send_src_addr03 =>
				next_state <= send_src_addr04;
			when send_src_addr04 =>
				next_state <= send_src_addr05;
			when send_src_addr05 =>
				next_state <= send_src_addr06;
			when send_src_addr06 =>
				next_state <= send_Source_IP_address1;
			when send_Source_IP_address1 =>
				next_state <= send_Source_IP_address2;
			when send_Source_IP_address2 =>
				next_state <= send_Source_IP_address3;
			when send_Source_IP_address3 =>
				next_state <= send_Source_IP_address4;
			when send_Source_IP_address4 =>
				next_state <= send_dest_addr01;
			when send_dest_addr01 =>
				next_state <= send_dest_addr02;
			when send_dest_addr02 =>
				next_state <= send_dest_addr03;
			when send_dest_addr03 =>
				next_state <= send_dest_addr04;
			when send_dest_addr04 =>
				next_state <= send_dest_addr05;
			when send_dest_addr05 =>
				next_state <= send_dest_addr06;
			when send_dest_addr06 =>
				next_state <= send_Destination_IP_address1;
			when send_Destination_IP_address1 =>
				next_state <= send_Destination_IP_address2;
			when send_Destination_IP_address2 =>
				next_state <= send_Destination_IP_address3;
			when send_Destination_IP_address3 =>
				next_state <= send_Destination_IP_address4;
			when send_Destination_IP_address4 =>
				next_state <= send_00;
			when send_00 =>
				if cpt = "10001" then
					next_state <= idle;
					cpt <= "00000";
				else
					next_state <= send_00;
					cpt<= cpt + '1';
				end if;	
		end case;
	end if;
end if;
end process unpack;

--================================================================================================================================
-- feeds DATA_ARP_OUT depending on the state of the state machine: value of the different fields of the ARP packets.
--================================================================================================================================
DATA_ARP_OUT <=		DESTINATION_MAC(7 downto 0) when ((next_state=send_dest_addr6) or (next_state=send_dest_addr06)) else
									DESTINATION_MAC(15 downto 8) when ((next_state=send_dest_addr5) or (next_state=send_dest_addr05)) else
									DESTINATION_MAC(23 downto 16) when ((next_state=send_dest_addr4) or (next_state=send_dest_addr04)) else
									DESTINATION_MAC(31 downto 24) when ((next_state=send_dest_addr3) or (next_state=send_dest_addr03)) else
									DESTINATION_MAC(39 downto 32) when ((next_state=send_dest_addr2) or (next_state=send_dest_addr02)) else
									DESTINATION_MAC(47 downto 40) when ((next_state=send_dest_addr1) or (next_state=send_dest_addr01)) else
									SOURCE_MAC(7 downto 0) when ((next_state=send_src_addr6) or (next_state=send_src_addr06)) else
									SOURCE_MAC(15 downto 8) when ((next_state=send_src_addr5) or (next_state=send_src_addr05)) else
									SOURCE_MAC(23 downto 16) when ((next_state=send_src_addr4) or (next_state=send_src_addr04)) else
									SOURCE_MAC(31 downto 24) when ((next_state=send_src_addr3) or (next_state=send_src_addr03)) else
									SOURCE_MAC(39 downto 32) when ((next_state=send_src_addr2) or (next_state=send_src_addr02)) else
									SOURCE_MAC(47 downto 40) when ((next_state=send_src_addr1) or (next_state=send_src_addr01)) else
									"00001000" when ((next_state=send_lenght_mac1) or (next_state=send_protocol_type1)) else
									"00000110" when ((next_state=send_lenght_mac2) or (next_state=send_hardware_size)) else
									"00000001" when (next_state=send_hardware_type2) else
									"00000100" when (next_state=send_protocol_size) else
									"00000010" when (next_state=send_opcode2) else
									SOURCE_IP(31 downto 24)  when (next_state=send_Source_IP_address1) else
									SOURCE_IP(23 downto 16) when (next_state=send_Source_IP_address2) else
									SOURCE_IP(15 downto 8) when (next_state=send_Source_IP_address3) else
									SOURCE_IP(7 downto 0) when (next_state=send_Source_IP_address4) else
									DESTINATION_IP(31 downto 24) when (next_state=send_Destination_IP_address1) else
									DESTINATION_IP(23 downto 16) when (next_state=send_Destination_IP_address2) else
									DESTINATION_IP(15 downto 8) when (next_state=send_Destination_IP_address3) else
									DESTINATION_IP(7 downto 0) when (next_state=send_Destination_IP_address4) else
									"00000000" ;

--================================================================================================================================
-- REQ_ARP_FRAME is high when there is something to feed the mux4_1
------ UP WHEN PACKET READY TO SEND AND GO DOWN AT END OF PACKET
--================================================================================================================================
with next_state select
REQ_ARP_FRAME <= 	'0' when idle,
									'1' when others;

--================================================================================================================================
-- BUZZY when this module is buzzy and cannot respond to an other request.
--================================================================================================================================
with next_state select
BUZZY <=	'0' when idle,
					'1' when others;

-- feeds EN_DATA_ARP_OUT depending on the state of the state machine, it tells when there is data on DATA_ARP_OUT
with next_state select
EN_DATA_ARP_OUT <=	'0' when idle,
										'0' when wait_selection,
										'1' when others;
end RTL;
