--=============================================================================
--
-- Name:		TX_FRAME
--
-- Purpose:	This module is transforming the packet to match 
--					the ethernet physical layer requirement.
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. CRC32
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity TX_FRAME is
    Port (
					CLK_TX								: in std_logic;

					DATA_TX_IN						: in std_logic_vector(7 downto 0);
					EN_DATA_TX_IN					: in std_logic;

					DATA_TX_OUT						: out std_logic_vector(3 downto 0);
					EN_DATA_TX_OUT				: out std_logic
					);
end TX_FRAME;

architecture RTL of TX_FRAME is

type stateType is	(idle, send_preamble, send_sfd, send_low, send_high);
signal state : stateType;
signal count : std_logic_vector(3 downto 0);

signal data: std_logic_vector(7 downto 0);
signal data_en: std_logic; 
 
SIGNAL dec00	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec01	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec02	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec03	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec04	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec05	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec06	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec07	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec08	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec09	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec10	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec11	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec12	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec13	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec14	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec15	:	STD_LOGIC_VECTOR (8 DOWNTO 0);
SIGNAL dec16	:	STD_LOGIC_VECTOR (8 DOWNTO 0);

COMPONENT fd9ce is
    Port (
					C : in std_logic;
					D : in std_logic_vector(8 downto 0);
					Q : out std_logic_vector(8 downto 0)
					);
end COMPONENT;

begin
--================================================================================================================================
-- Here I am gathering DATA_TX_IN and EN_DATA_TX_IN in order to delay them of 17 clock edge
--================================================================================================================================
dec00<=DATA_TX_IN & EN_DATA_TX_IN;
Basc01 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec00, Q=>dec01);
Basc02 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec01, Q=>dec02);
Basc03 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec02, Q=>dec03);
Basc04 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec03, Q=>dec04);
Basc05 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec04, Q=>dec05);
Basc06 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec05, Q=>dec06);
Basc07 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec06, Q=>dec07);
Basc08 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec07, Q=>dec08);
Basc09 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec08, Q=>dec09);
Basc10 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec09, Q=>dec10);
Basc11 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec10, Q=>dec11);
Basc12 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec11, Q=>dec12);
Basc13 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec12, Q=>dec13);
Basc14 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec13, Q=>dec14);
Basc15 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec14, Q=>dec15);
Basc16 : FD9CE
	PORT MAP(C=>CLK_TX, D=>dec15, Q=>dec16);

--================================================================================================================================
-- recontruction of the signal data_en and the bus data.
--================================================================================================================================
retard :process(CLK_TX)
begin
	if rising_edge(CLK_TX) then
		EN_DATA_TX_OUT<=EN_DATA_TX_IN or dec16(0);
		data<=dec16(8 downto 1);
		data_en<=dec16(0);
	end if;
end process retard;

--================================================================================================================================
-- process:	in this process, I am adding at the beginning of the data a preambule
--					That is asked by the 80225
--					After it's transforming a 8 bits data bus into 4 bits but at a double rate (25 Mhz).
--================================================================================================================================
unpack : process(CLK_TX)
begin
	if rising_edge(CLK_TX) then

		if (data_en OR EN_DATA_TX_IN) = '0' then
			state <= idle;
		else
			case state is
				when idle =>
					count <= "0000";
					if (data_en OR EN_DATA_TX_IN) = '1' then
						state <= send_preamble;
					end if;

				when send_preamble =>
					count <= count + 1;
					if count = "1110" then
						state <= send_sfd;
					end if;

				when send_sfd =>
					state <= send_high;
						
				when send_high =>
					if data_en = '1' then
						state <= send_low;
					else 
						state <= idle;
					end if;

				when send_low =>
					state <= send_high;

			end case;

		end if;
	end if;
end process unpack;

--================================================================================================================================
-- Creation of DATA_TX_OUT from the data but with the preambule at the beginning.
--================================================================================================================================
with state select
DATA_TX_OUT <=	"0000" when idle,
								"0101" when send_preamble,
								"1101" when send_sfd,
								data(3 downto 0) when send_high,
								data(7 downto 4) when send_low;

end RTL;
