--=============================================================================
--
-- Name:		CAPTURE_UDP_FRAME
--
-- Purpose:	This module makes a UDP packet from the data of the microphone
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. SRAM_INTERFACE
-- 2. MUX4_1
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================

library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
USE ieee.numeric_std.ALL;
USE UNISIM.Vcomponents.ALL;

entity CAPTURE_UDP_FRAME is
    Port (	CLK_CAPTURE_UDP						: in std_logic;
						CLKD2_CAPTURE_UDP					: in std_logic;	
						RESET											: in std_logic;

						START_CAPTURE_UDP					: in std_logic;
						BUZZY											: out std_logic;

						ADDR_CAPTURE_UDP					: OUT std_logic_vector (9 downto 0);
						EN_DATA_CAPTURE_UDP_IN		: OUT STD_LOGIC;
						DATA_CAPTURE_UDP_IN				: IN std_logic_vector (7 downto 0);

						SOURCE_MAC								: in std_logic_vector(47 downto 0);
						DESTINATION_MAC						: in std_logic_vector(47 downto 0);
						SOURCE_IP									: in std_logic_vector(31 downto 0);
						DESTINATION_IP						: in std_logic_vector(31 downto 0);

						NUMERO_PACKETS						: in std_logic_vector(15 downto 0);

						SELECT_CAPTURE_UDP_FRAME	: in std_logic;
						REQ_CAPTURE_UDP_FRAME			: out std_logic;
						EN_DATA_CAPTURE_UDP_OUT		: out std_logic;
						DATA_CAPTURE_UDP_OUT			: out std_logic_vector(7 downto 0));

end CAPTURE_UDP_FRAME;

architecture RTL of CAPTURE_UDP_FRAME is

type stateType is	(	idle0, wait_selection,
										send_dest_addr6,send_dest_addr5,send_dest_addr4,send_dest_addr3,send_dest_addr2,send_dest_addr1,
										send_src_addr6,send_src_addr5,send_src_addr4,send_src_addr3,send_src_addr2,send_src_addr1,send_lenght_mac1,send_lenght_mac2,
										send_Version_IHL,send_TOS,
										send_Total_length1,send_Total_length2,
										send_Identification1,send_Identification2,
										send_Flags_Fragment_offset1,send_Fragment_offset2,
										send_TTL,send_Protocol,
										send_Header_checksum1,send_Header_checksum2,
										send_Source_IP_address1,send_Source_IP_address2,send_Source_IP_address3,send_Source_IP_address4,
										send_Destination_IP_address1,send_Destination_IP_address2,send_Destination_IP_address3,send_Destination_IP_address4,
										send_Source_Port1,send_Source_Port2,
										send_Destination_Port1,send_Destination_Port2,
										send_Length1,send_Length2,
										send_Checksum1,send_Checksum2,
										send_Type_Capture,
										send_packetsNb1,send_packetsNb2,
										send_data);
signal next_state : stateType;

signal Hchecksum								: std_logic_vector(15 downto 0);
signal Hchecksum00							: std_logic_vector(31 downto 0);

signal addr											: std_logic_vector(9 downto 0);

begin

ADDR_CAPTURE_UDP <= addr;

--================================================================================================================================
-- This is computed as the 16-bit one's complement of the one's complement sum of a pseudo header of information from the IP header,
-- the UDP header, and the data, padded as needed with zero bytes at the end to make a multiple of two bytes.
-- If the checksum is set to zero, then checksuming is disabled. If the computed checksum is zero, then this field must be set to 0xFFFF.
--================================================================================================================================
Hchecksum00 <= (("0000000000000000"&SOURCE_IP(15 downto 0)) + ("0000000000000000"&SOURCE_IP(31 downto 16))) + (("0000000000000000"&DESTINATION_IP(15 downto 0)) + ("0000000000000000"&DESTINATION_IP(31 downto 16)));
Hchecksum <= not ( "0111000011110001" + ((Hchecksum00(31 downto 16)) + Hchecksum00(15 downto 0)));


--================================================================================================================================
-- In this process, I am using a linear state machine in order to create the different fields in an UDP packet.
-- RESET is provided by the main clockdll with LOCKED which is a low reset and to give a starting point to the state machine.
-- START_CAPTURE_UDP is the signal starting the state machine

-- The next module connected to this one is the mux4_1 which gives us a feedback in order to know when it's ready.
-- So the signal SELECT_CAPTURE_UDP_FRAME is the one waiting the green light to start.
-- After that it's the different fields of an UDP packet.
-- NUMERO_PACKETS is given by the SRAM_INTERFACE module and is the packet number in order to know if there is loss of packet or not.
-- At the end there is an address counter to pick up the data from the double buffering made in the SRAM_INTERFACE module. 
-- The address counter doesn't start at 0 because of the setup time produced in the SRAM_INTERFACE module.
-- for more information : http://www.networksorcery.com/enp/protocol/udp.htm
--================================================================================================================================
unpack : process(CLK_CAPTURE_UDP,RESET)
begin
if RESET='0' then
	next_state <= idle0;
elsif rising_edge(CLK_CAPTURE_UDP) then
	if CLKD2_CAPTURE_UDP='1' then		
		case next_state is
			when idle0 =>
				if (START_CAPTURE_UDP) = '1' then
					next_state <= wait_selection;
				else
					next_state <= idle0;
				end if;
			when wait_selection =>
				if (SELECT_CAPTURE_UDP_FRAME) = '1' then
					next_state <= send_dest_addr1;
				else
					next_state <= wait_selection;
				end if;
			when send_dest_addr1 =>
				next_state <= send_dest_addr2;
			when send_dest_addr2 =>
				next_state <= send_dest_addr3;
			when send_dest_addr3 =>
				next_state <= send_dest_addr4;
			when send_dest_addr4 =>
				next_state <= send_dest_addr5;
			when send_dest_addr5 =>
				next_state <= send_dest_addr6;
			when send_dest_addr6 =>
				next_state <= send_src_addr1;
			when send_src_addr1 =>
				next_state <= send_src_addr2;
			when send_src_addr2 =>
				next_state <= send_src_addr3;
			when send_src_addr3 =>
				next_state <= send_src_addr4;
			when send_src_addr4 =>
				next_state <= send_src_addr5;
			when send_src_addr5 =>
				next_state <= send_src_addr6;
			when send_src_addr6 =>
				next_state <= send_lenght_mac1;
			when send_lenght_mac1 =>
				next_state <= send_lenght_mac2;
			when send_lenght_mac2 =>								---------fin mac debut ip
				next_state <= send_Version_IHL;
			when send_Version_IHL =>
				next_state <= send_TOS;
			when send_TOS =>
				next_state <= send_Total_length1;
			when send_Total_length1 =>
				next_state <= send_Total_length2;
			when send_Total_length2 =>
				next_state <= send_Identification1;
			when send_Identification1 =>
				next_state <= send_Identification2;
			when send_Identification2 =>
				next_state <= send_Flags_Fragment_offset1;
			when send_Flags_Fragment_offset1 =>
				next_state <= send_Fragment_offset2;
			when send_Fragment_offset2 =>
				next_state <= send_TTL;
			when send_TTL =>
				next_state <= send_Protocol;
			when send_Protocol =>
				next_state <= send_Header_checksum1;
			when send_Header_checksum1 =>
				next_state <= send_Header_checksum2;
			when send_Header_checksum2 =>
				next_state <= send_Source_IP_address1;
			when send_Source_IP_address1 =>
				next_state <= send_Source_IP_address2;
			when send_Source_IP_address2 =>
				next_state <= send_Source_IP_address3;
			when send_Source_IP_address3 =>
				next_state <= send_Source_IP_address4;
			when send_Source_IP_address4 =>
				next_state <= send_Destination_IP_address1;
			when send_Destination_IP_address1 =>
				next_state <= send_Destination_IP_address2;
			when send_Destination_IP_address2 =>
				next_state <= send_Destination_IP_address3;
			when send_Destination_IP_address3 =>
				next_state <= send_Destination_IP_address4;
			when send_Destination_IP_address4 =>					----fin ip debut udp
				next_state <= send_Source_Port1;
			when send_Source_Port1 =>
				next_state <= send_Source_Port2;
			when send_Source_Port2 =>
				next_state <= send_Destination_Port1;
			when send_Destination_Port1 =>
				next_state <= send_Destination_Port2;
			when send_Destination_Port2 =>
				next_state <= send_Length1;
			when send_Length1 =>
				next_state <= send_Length2;
			when send_Length2 =>
				next_state <= send_Checksum1;
			when send_Checksum1 =>
				next_state <= send_Checksum2;
			when send_Checksum2 =>	
				next_state <= send_Type_Capture;
			when send_Type_Capture =>
				next_state <= send_packetsNb1;
			when send_packetsNb1 =>
				next_state <= send_packetsNb2;
			when send_packetsNb2 =>
				next_state <= send_data;
			when send_data =>
				if addr = "1111001101" then
					next_state <= idle0;
					addr<="0000001010";
				else
					next_state <= send_data;
					addr<= addr + '1';
				end if;
 		end case;
	end if;
end if;
end process unpack;

--================================================================================================================================
-- feeds DATA_CAPTURE_UDP_OUT depending on the state of the state machine: value of the different fields of the UDP packets.
--================================================================================================================================
DATA_CAPTURE_UDP_OUT <=	DESTINATION_MAC(7 downto 0) when (next_state=send_dest_addr6) else
												DESTINATION_MAC(15 downto 8) when (next_state=send_dest_addr5) else
												DESTINATION_MAC(23 downto 16) when (next_state=send_dest_addr4) else
												DESTINATION_MAC(31 downto 24) when (next_state=send_dest_addr3) else
												DESTINATION_MAC(39 downto 32) when (next_state=send_dest_addr2) else
												DESTINATION_MAC(47 downto 40) when (next_state=send_dest_addr1) else
												SOURCE_MAC(7 downto 0) when (next_state=send_src_addr6) else
												SOURCE_MAC(15 downto 8) when (next_state=send_src_addr5) else
												SOURCE_MAC(23 downto 16) when (next_state=send_src_addr4) else
												SOURCE_MAC(31 downto 24) when (next_state=send_src_addr3) else
												SOURCE_MAC(39 downto 32) when (next_state=send_src_addr2) else
												SOURCE_MAC(47 downto 40) when (next_state=send_src_addr1) else
												"00001000" when (next_state=send_lenght_mac1) else
												"01000101" when (next_state=send_Version_IHL) else
												"00000011" when ((next_state=send_Total_length1) or (next_state=send_Length1)) else
												"11100000" when (next_state=send_Total_length2) else
												"00101000" when (next_state=send_TTL) else
												"00010001" when (next_state=send_Protocol) else
												Hchecksum(15 downto 8) when (next_state=send_Header_checksum1) else
												Hchecksum(7 downto 0) when (next_state=send_Header_checksum2) else
												SOURCE_IP(31 downto 24)  when (next_state=send_Source_IP_address1) else
												SOURCE_IP(23 downto 16) when (next_state=send_Source_IP_address2) else
												SOURCE_IP(15 downto 8) when (next_state=send_Source_IP_address3) else
												SOURCE_IP(7 downto 0) when (next_state=send_Source_IP_address4) else
												DESTINATION_IP(31 downto 24) when (next_state=send_Destination_IP_address1) else
												DESTINATION_IP(23 downto 16) when (next_state=send_Destination_IP_address2) else
												DESTINATION_IP(15 downto 8) when (next_state=send_Destination_IP_address3) else
												DESTINATION_IP(7 downto 0) when (next_state=send_Destination_IP_address4) else
												"01111111" when ((next_state=send_Source_Port1) or (next_state=send_Destination_Port1)) else
												"11111111" when ((next_state=send_Source_Port2) or (next_state=send_Destination_Port2)) else
												"11001100" when (next_state=send_Length2) else			-- 960 (data) + 2 (nb packet) + 1 (type packet) + 8 (udp)
												"10000110" when (next_state=send_Type_Capture) else
												NUMERO_PACKETS(15 downto 8) when (next_state=send_packetsNb1) else
												NUMERO_PACKETS(7 downto 0)  when (next_state=send_packetsNb2) else			
												DATA_CAPTURE_UDP_IN(7 downto 0) when (next_state=send_data) else
												"00000000";

--================================================================================================================================
-- REQ_CAPTURE_UDP_FRAME is high when there is something to feed the mux4_1
-- UP WHEN PACKET READY TO SEND AND GO DOWN AT END OF PACKET
--================================================================================================================================
with next_state select
REQ_CAPTURE_UDP_FRAME <=	'0' when idle0,
													'1' when others;

--================================================================================================================================
-- BUZZY when this module is buzzy and cannot respond to an other request.
--================================================================================================================================
with next_state select
BUZZY <=	'0' when idle0,
					'1' when others;

--================================================================================================================================
-- feeds EN_DATA_CAPTURE_UDP_OUT depending on the state of the state machine, it tells when there is data on DATA_CAPTURE_UDP_OUT
--================================================================================================================================
with next_state select
EN_DATA_CAPTURE_UDP_OUT <=	'0' when idle0,
														'0' when wait_selection,
														'1' when others;

--================================================================================================================================
-- feeds EN_DATA_CAPTURE_UDP_IN depending on the state of the state machine,
-- This is connected to the memory of the SRAM_INTERFACE module and goes up when it's time to activate the memory: so one state before the time we need the data.
--================================================================================================================================
with next_state select
EN_DATA_CAPTURE_UDP_IN <=	'1' when send_data,
													'1' when send_packetsNb2,
													'0' when others;

end RTL;

