--=============================================================================
--
-- Name:		BOOTP_FRAME
--
-- Purpose:	This module makes a BOOTP request packet.
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. MEM_READ_INCOMING_MSG
-- 2. MUX4_1
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================

library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
USE ieee.numeric_std.ALL;
USE UNISIM.Vcomponents.ALL;

entity BOOTP_FRAME is
    Port (
					CLK_BOOTP						: in std_logic;
					RESET								: in std_logic;

					START_BOOTP					: in std_logic;
					BUZZY								: out std_logic;

					SOURCE_MAC					: in std_logic_vector(47 downto 0);
					SECONDS							: in std_logic_vector(7 downto 0);

					SELECT_BOOTP_FRAME	: in std_logic;
					REQ_BOOTP_FRAME			: out std_logic;	------ UP WHEN PACKET READY TO SEND AND GO DOWN AT END OF PACKET
					EN_DATA_BOOTP_OUT		: out std_logic;
					DATA_BOOTP_OUT			: out std_logic_vector(7 downto 0)
					);
end BOOTP_FRAME;

architecture RTL of BOOTP_FRAME is


type stateType is (	idle, wait_selection,
										send_dest_addr6,send_dest_addr5,send_dest_addr4,send_dest_addr3,send_dest_addr2,send_dest_addr1,
										send_src_addr6,send_src_addr5,send_src_addr4,send_src_addr3,send_src_addr2,send_src_addr1,send_lenght_mac1,send_lenght_mac2,
										send_Version_IHL,send_TOS,
										send_Total_length1,send_Total_length2,
										send_Identification1,send_Identification2,
										send_Flags_Fragment_offset1,send_Fragment_offset2,
										send_TTL,send_Protocol,
										send_Header_checksum1,send_Header_checksum2,
										send_Source_IP_address1,send_Source_IP_address2,send_Source_IP_address3,send_Source_IP_address4,
										send_Destination_IP_address1,send_Destination_IP_address2,send_Destination_IP_address3,send_Destination_IP_address4,
										send_Source_Port1,send_Source_Port2,
										send_Destination_Port1,send_Destination_Port2,
										send_Length1,send_Length2,
										send_Checksum1,send_Checksum2,
										send_opcode,send_Hardware_type,
										send_Hardware_address_length,send_Hop_count,
										send_Transaction_ID1,send_Transaction_ID2,send_Transaction_ID3,send_Transaction_ID4,
										send_Number_seconds1,send_Number_seconds2,
										send_Flags1,send_Flags2,
										send_client_ip_address1,send_client_ip_address2,send_client_ip_address3,send_client_ip_address4,
										send_your_client_ip_address1,send_your_client_ip_address2,send_your_client_ip_address3,send_your_client_ip_address4,
										send_server_ip_address1,send_server_ip_address2,send_server_ip_address3,send_server_ip_address4,
										send_relay_ip_address1,send_relay_ip_address2,send_relay_ip_address3,send_relay_ip_address4,
										send_client_hardware_address1,send_client_hardware_address2,send_client_hardware_address3,send_client_hardware_address4,send_client_hardware_address5,send_client_hardware_address6,
										send_end_request);

signal next_state : stateType;

signal count									: std_logic_vector(8 downto 0);
signal clkd2									: std_logic;
begin

--================================================================================================================================
-- process to divide the main clock by 2
--================================================================================================================================
process(CLK_BOOTP)
begin
if rising_edge(CLK_BOOTP) then
	clkd2 <= not clkd2;
end if;
end process;

--================================================================================================================================
-- In this process, I am using a linear state machine in order to create the different fields in an BOOTP packet.
-- RESET is provided by the main clockdll with LOCKED which is a low reset and to give a starting point to the state machine.
-- START_BOOTP is the signal starting the state machine
-- The next module connected to this one is the mux4_1 which gives us a feedback in order to know when it's ready.
-- So the signal SELECT_BOOTP_FRAME is the one waiting the green light to start.
-- After that it's the different fields of an BOOTP packet.
-- the counter at the end is here to match the minimum size packet

-- for more information : http://www.networksorcery.com/enp/protocol/bootp.htm
--================================================================================================================================
unpack : process(CLK_BOOTP,RESET)
begin
if RESET='0' then
	next_state <= idle;
elsif rising_edge(CLK_BOOTP) then
	if clkd2='1' then		
		
		case next_state is
			when idle =>
				if (START_BOOTP) = '1' then
					next_state <= wait_selection;
				else
					next_state <= idle;
				end if;
			when wait_selection =>
				if (SELECT_BOOTP_FRAME) = '1' then
					next_state <= send_dest_addr1;
				else
					next_state <= wait_selection;
				end if;
			when send_dest_addr1 =>
				next_state <= send_dest_addr2;
			when send_dest_addr2 =>
				next_state <= send_dest_addr3;
			when send_dest_addr3 =>
				next_state <= send_dest_addr4;
			when send_dest_addr4 =>
				next_state <= send_dest_addr5;
			when send_dest_addr5 =>
				next_state <= send_dest_addr6;
			when send_dest_addr6 =>
				next_state <= send_src_addr1;
			when send_src_addr1 =>
				next_state <= send_src_addr2;
			when send_src_addr2 =>
				next_state <= send_src_addr3;
			when send_src_addr3 =>
				next_state <= send_src_addr4;
			when send_src_addr4 =>
				next_state <= send_src_addr5;
			when send_src_addr5 =>
				next_state <= send_src_addr6;
			when send_src_addr6 =>
				next_state <= send_lenght_mac1;
			when send_lenght_mac1 =>
				next_state <= send_lenght_mac2;
			when send_lenght_mac2 =>								---------fin mac debut ip
				next_state <= send_Version_IHL;
			when send_Version_IHL =>
				next_state <= send_TOS;
			when send_TOS =>
				next_state <= send_Total_length1;
			when send_Total_length1 =>
				next_state <= send_Total_length2;
			when send_Total_length2 =>
				next_state <= send_Identification1;
			when send_Identification1 =>
				next_state <= send_Identification2;
			when send_Identification2 =>
				next_state <= send_Flags_Fragment_offset1;
			when send_Flags_Fragment_offset1 =>
				next_state <= send_Fragment_offset2;
			when send_Fragment_offset2 =>
				next_state <= send_TTL;
			when send_TTL =>
				next_state <= send_Protocol;
			when send_Protocol =>
				next_state <= send_Header_checksum1;
			when send_Header_checksum1 =>
				next_state <= send_Header_checksum2;
			when send_Header_checksum2 =>
				next_state <= send_Source_IP_address1;
			when send_Source_IP_address1 =>
				next_state <= send_Source_IP_address2;
			when send_Source_IP_address2 =>
				next_state <= send_Source_IP_address3;
			when send_Source_IP_address3 =>
				next_state <= send_Source_IP_address4;
			when send_Source_IP_address4 =>
				next_state <= send_Destination_IP_address1;
			when send_Destination_IP_address1 =>
				next_state <= send_Destination_IP_address2;
			when send_Destination_IP_address2 =>
				next_state <= send_Destination_IP_address3;
			when send_Destination_IP_address3 =>
				next_state <= send_Destination_IP_address4;
			when send_Destination_IP_address4 =>					----fin ip debut udp
				next_state <= send_Source_Port1;
			when send_Source_Port1 =>
				next_state <= send_Source_Port2;
			when send_Source_Port2 =>
				next_state <= send_Destination_Port1;
			when send_Destination_Port1 =>
				next_state <= send_Destination_Port2;
			when send_Destination_Port2 =>
				next_state <= send_Length1;
			when send_Length1 =>
				next_state <= send_Length2;
			when send_Length2 =>
				next_state <= send_Checksum1;
			when send_Checksum1 =>
				next_state <= send_Checksum2;
			when send_Checksum2 =>
				next_state <= send_opcode;
			when send_opcode =>
				next_state <= send_Hardware_type;
			when send_Hardware_type =>
				next_state <= send_Hardware_address_length;
			when send_Hardware_address_length =>
				next_state <= send_Hop_count;
			when send_Hop_count =>
				next_state <= send_Transaction_ID1;
			when send_Transaction_ID1 =>
				next_state <= send_Transaction_ID2;
			when send_Transaction_ID2 =>
				next_state <= send_Transaction_ID3;
			when send_Transaction_ID3 =>
				next_state <= send_Transaction_ID4;
			when send_Transaction_ID4 =>
				next_state <= send_Number_seconds1;
			when send_Number_seconds1 =>
				next_state <= send_Number_seconds2;
			when send_Number_seconds2 =>
				next_state <= send_Flags1;
			when send_Flags1 =>
				next_state <= send_Flags2;
			when send_Flags2 =>
				next_state <= send_client_ip_address1;
			when send_client_ip_address1 =>
				next_state <= send_client_ip_address2;
			when send_client_ip_address2 =>
				next_state <= send_client_ip_address3;
			when send_client_ip_address3 =>
				next_state <= send_client_ip_address4;
			when send_client_ip_address4=>
				next_state <= send_your_client_ip_address1;
			when send_your_client_ip_address1 =>
				next_state <= send_your_client_ip_address2;
			when send_your_client_ip_address2 =>
				next_state <= send_your_client_ip_address3;
			when send_your_client_ip_address3 =>
				next_state <= send_your_client_ip_address4;
			when send_your_client_ip_address4=>
				next_state <= send_server_ip_address1;
			when send_server_ip_address1 =>
				next_state <= send_server_ip_address2;
			when send_server_ip_address2 =>
				next_state <= send_server_ip_address3;
			when send_server_ip_address3 =>
				next_state <= send_server_ip_address4;
			when send_server_ip_address4=>
				next_state <= send_relay_ip_address1;
			when send_relay_ip_address1 =>
				next_state <= send_relay_ip_address2;
			when send_relay_ip_address2 =>
				next_state <= send_relay_ip_address3;
			when send_relay_ip_address3 =>
				next_state <= send_relay_ip_address4;
			when send_relay_ip_address4=>
				next_state <= send_client_hardware_address1;
			when send_client_hardware_address1 =>
				next_state <= send_client_hardware_address2;
			when send_client_hardware_address2 =>
				next_state <= send_client_hardware_address3;
			when send_client_hardware_address3 =>
				next_state <= send_client_hardware_address4;
			when send_client_hardware_address4=>
				next_state <= send_client_hardware_address5;
			when send_client_hardware_address5 =>
				next_state <= send_client_hardware_address6;
			when send_client_hardware_address6 =>
				next_state <= send_end_request;
			when send_end_request =>
				if count = "11111101" then 
					next_state <= idle;
					count <= "000000000";
				else
					count <= count+1;
					next_state <= send_end_request;
				end if;
		end case;
	end if;
end if;
end process unpack;

--================================================================================================================================
-- signal :feeds DATA_BOOTP_OUT depending on the state of the state machine: value of the different fields of the BOOTP packets.
--================================================================================================================================
with next_state select
DATA_BOOTP_OUT <=	"11111111" when send_dest_addr1,
									"11111111" when send_dest_addr2,
									"11111111" when send_dest_addr3,
									"11111111" when send_dest_addr4,
									"11111111" when send_dest_addr5,
									"11111111" when send_dest_addr6,
									SOURCE_MAC(47 downto 40) when send_src_addr1,
									SOURCE_MAC(39 downto 32) when send_src_addr2,
									SOURCE_MAC(31 downto 24) when send_src_addr3,
									SOURCE_MAC(23 downto 16) when send_src_addr4,
									SOURCE_MAC(15 downto 8) when send_src_addr5,
									SOURCE_MAC(7 downto 0) when send_src_addr6,				
									"00001000" when send_lenght_mac1,
									"01000101" when send_Version_IHL,
									"00000001" when send_Total_length1,
									"00111100" when send_Total_length2,
									"00101000" when send_TTL,
									"00010001" when send_Protocol,
									"10010001" when send_Header_checksum1,
									"10110010" when send_Header_checksum2,
									"11111111" when send_Destination_IP_address1,
									"11111111" when send_Destination_IP_address2,
									"11111111" when send_Destination_IP_address3,
									"11111111" when send_Destination_IP_address4,
									"01000100" when send_Source_Port2,
									"01000011" when send_Destination_Port2,
									"00000001" when send_Length1,
									"00101000" when send_Length2,
									"00000001" when send_opcode,
									"00000001" when send_Hardware_type,
									"00000110" when send_Hardware_address_length,
									SOURCE_MAC(31 downto 24) when send_Transaction_ID1,
									SOURCE_MAC(23 downto 16) when send_Transaction_ID2, 
									SOURCE_MAC(15 downto 8) when send_Transaction_ID3,
									SOURCE_MAC(7 downto 0) when send_Transaction_ID4,
									SECONDS(7 downto 0) when send_Number_seconds2,
									"10000000" when send_Flags1,
									SOURCE_MAC(47 downto 40) when send_client_hardware_address1,
									SOURCE_MAC(39 downto 32) when send_client_hardware_address2,
									SOURCE_MAC(31 downto 24) when send_client_hardware_address3,
									SOURCE_MAC(23 downto 16) when send_client_hardware_address4,
									SOURCE_MAC(15 downto 8) when send_client_hardware_address5,
									SOURCE_MAC(7 downto 0) when send_client_hardware_address6,
									"00000000" when others;

--================================================================================================================================
-- REQ_BOOTP_FRAME is high when there is something to feed the mux4_1
-- UP WHEN PACKET READY TO SEND AND GO DOWN AT END OF PACKET
--================================================================================================================================
with next_state select
REQ_BOOTP_FRAME <= 	'0' when idle,
										'1' when others;

--================================================================================================================================
-- BUZZY when this module is buzzy and cannot respond to an other request.
--================================================================================================================================
with next_state select
BUZZY <= 	'0' when idle,
					'1' when others;

--================================================================================================================================
-- feeds EN_DATA_BOOTP_OUT depending on the state of the state machine, it tells when there is data on DATA_BOOTP_OUT
--================================================================================================================================
with next_state select
EN_DATA_BOOTP_OUT <=	'0' when idle,
											'0' when wait_selection,
											'1' when others;
end RTL;


