--=============================================================================
--
-- Name:		CAPTURE
--
-- Purpose:	This module gives the clocks for the microphone boards and
--					captures the datas from the ADCs
--
-- Note:		This code is in the public domain, subject to the restriction
--					given in the official notice below
--
-- Author:	Cedrick Rochet, NIST Smart Space Project
--
-- Contact:	crochet@nist.gov or cedrick.rochet@yahoo.fr
--
-- Dependencies:
--
-- 1. 
-- 2. 
--
--=============================================================================
-- Revision History
-------------------------------------------------------------------------------
-- August 5, 2003 
-- 
-- - initial version
--
--
--========================= Official Notice ===================================
--
-- "This software was developed at the National Institute of Standards and 
-- Technology by employees of the Federal Government in the course of their 
-- official duties. Pursuant to Title 17 Section 105 of the United States Code 
-- this software is not subject to copyright protection and is in the public 
-- domain.
-- 
-- Capture is an experimental system and is offered AS IS. NIST assumes no 
-- responsibility whatsoever for its use by other parties, and makes no 
-- guarantees and NO WARRANTIES, EXPRESS OR IMPLIED, about its quality, 
-- reliability, fitness for any purpose, or any other characteristic. We would 
-- appreciate acknowledgement if the software is used.
-- 
-- This software can be redistributed and/or modified freely provided that any 
-- derivative works bear some notice that they are derived from it, and any 
-- modified versions bear some notice that they have been modified from the 
-- original."
-- 
--=============================================================================

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use ieee.numeric_std.ALL;

library UNISIM;
use UNISIM.vcomponents.ALL;

entity CAPTURE is
    Port (
					CAP_CLK								: in std_logic;		--33.8696MHz
					CAP_CLK_SLAVE					: in	std_logic;	-- 33.8696MHz
					SYNC_CAP_CLK_SLAVE		: in std_logic;
					SYNC_CAP_CLK_MASTER		: out std_logic;
					SYNC_SLAVE						: in std_logic;

					BCK										: out std_logic;		--1.058425	=33.8696/32       = 48 fs
					SCKI									: out std_logic;		--11.2896	=33.8696/3       	= 512 fs
					LRCK									: out std_logic;		--0.022050	=33.8696/3 /512 	= fs

					LOW_RESET							: in std_logic;
					START_CAPTURE					: in std_logic;
					START_CAPTURE_SLV			: out std_logic;
					DOUBLE_FRQ_AD					: in std_logic;

					STD01									: in std_logic;
					STD02									: in std_logic; 
					STD03									: in std_logic;
					STD04									: in std_logic;
					STD05									: in std_logic;
					STD06									: in std_logic;
					STD07									: in std_logic;
					STD08									: in std_logic;
					STD09									: in std_logic;
					STD10									: in std_logic; 
					STD11									: in std_logic;
					STD12									: in std_logic;
					STD13									: in std_logic;
					STD14									: in std_logic;
					STD15									: in std_logic;
					STD16									: in std_logic;
					STD17									: in std_logic;
					STD18									: in std_logic; 
					STD19									: in std_logic;
					STD20									: in std_logic;
					STD21									: in std_logic;
					STD22									: in std_logic;
					STD23									: in std_logic;
					STD24									: in std_logic;
					STD25									: in std_logic;
					STD26									: in std_logic;
					STD27									: in std_logic;
					STD28									: in std_logic;
					STD29									: in std_logic;
					STD30									: in std_logic;
					STD31									: in std_logic;
					STD32									: in std_logic;

--MEM_visu							: out std_logic_vector (7 downto 0);
--MEM_ADDR_visu	 				: out std_logic_vector (9 downto 0);
--incrmt_addr_visu			: out std_logic;
--incrmt_shift_visu			: out std_logic;
--incrmt_par_visu				: out std_logic;
--count32_visu 					: out std_logic_vector (8 downto 0);
--count512_visu 				: out std_logic_vector(8 downto 0);
--startdata_visu				: out std_logic;
--clkfb_3_visu	 				: out std_logic;
--count1_visu						: out std_logic_vector (4 downto 0);

					ADDR_CAPTURE_MEM			: in std_logic_vector (8 downto 0);
					CLK_CAPTURE_MEM				: in std_logic;
					ENABLE_CAPTURE_MEM		: in std_logic;
					DATA_CAPTURE_MEM			: out std_logic_vector (15 downto 0);
					PACKET_READY					: out std_logic
					);
end CAPTURE;

architecture RTL of CAPTURE is

signal MEM 								: std_logic_vector (7 downto 0);
signal MEM_ADDR 					: std_logic_vector (9 downto 0);

signal start_capture_0		: std_logic;

signal startdata					: std_logic;
signal incrmt_addr				: std_logic;
signal incrmt_shift				: std_logic;
signal incrmt_par					: std_logic;
signal incrmt_addr_mem		: std_logic;

signal count32	 					: std_logic_vector (8 downto 0);
signal count512		 				: std_logic_vector(8 downto 0);
signal sPDN32							: std_logic;
signal sPDN512						: std_logic;
signal sPDN512_0					: std_logic;
signal sPDN512_2					: std_logic;
signal sPDN512_3					: std_logic;
signal sPDN512_5					: std_logic;

signal DOB01	 					: std_logic_vector (15 downto 0);
signal DOB02 						: std_logic_vector (15 downto 0);
signal DOB03						: std_logic_vector (15 downto 0);
signal DOB04 						: std_logic_vector (15 downto 0);
							 
signal par01 						: std_logic_vector (7 downto 0);
signal par02 						: std_logic_vector (7 downto 0);
signal par03	 					: std_logic_vector (7 downto 0);
signal par04 						: std_logic_vector (7 downto 0);
signal par05 						: std_logic_vector (7 downto 0);
signal par06 						: std_logic_vector (7 downto 0);
signal par07 						: std_logic_vector (7 downto 0);
signal par08	 					: std_logic_vector (7 downto 0);
signal par09 						: std_logic_vector (7 downto 0);
signal par10 						: std_logic_vector (7 downto 0);
signal par11 						: std_logic_vector (7 downto 0);
signal par12 						: std_logic_vector (7 downto 0);
signal par13	 					: std_logic_vector (7 downto 0);
signal par14 						: std_logic_vector (7 downto 0);
signal par15 						: std_logic_vector (7 downto 0);
signal par16 						: std_logic_vector (7 downto 0);
signal par17 						: std_logic_vector (7 downto 0);
signal par18 						: std_logic_vector (7 downto 0);
signal par19 						: std_logic_vector (7 downto 0);
signal par20 						: std_logic_vector (7 downto 0);
signal par21 						: std_logic_vector (7 downto 0);
signal par22 						: std_logic_vector (7 downto 0);
signal par23 						: std_logic_vector (7 downto 0);
signal par24 						: std_logic_vector (7 downto 0);
signal par25 						: std_logic_vector (7 downto 0);
signal par26 						: std_logic_vector (7 downto 0);
signal par27 						: std_logic_vector (7 downto 0);
signal par28 						: std_logic_vector (7 downto 0);
signal par29 						: std_logic_vector (7 downto 0);
signal par30 						: std_logic_vector (7 downto 0);
signal par31 						: std_logic_vector (7 downto 0);
signal par32 						: std_logic_vector (7 downto 0);
			
signal ENA1 						: std_logic;
signal ENA2							: std_logic;
signal ENA3 						: std_logic;
signal ENA4 						: std_logic;

signal ENB1 						: std_logic;
signal ENB2 						: std_logic;
signal ENB3 						: std_logic;
signal ENB4 						: std_logic;

signal ENB1_i						: std_logic;
signal ENB2_i						: std_logic;
signal ENB3_i						: std_logic;

signal count1 					: std_logic_vector (4 downto 0);
signal count2 					: std_logic_vector (2 downto 0);

signal counter_wait			: std_logic_vector (4 downto 0);
signal nbpaquet 				: std_logic_vector (2 downto 0);

signal ENABLE_GAUCHE 		: std_logic;

signal clkin3 					: std_logic;
signal clkd3 						: std_logic;
signal clkfb_3 					: std_logic;
signal clk_3 						: std_logic;

signal clkin5 					: std_logic;
signal clkd5 						: std_logic;
signal clkfb_5 					: std_logic;
signal clk_5 						: std_logic;

signal clkdiv						: std_logic;
signal clkfb						: std_logic;

signal clkdll_ready3		: std_logic;
signal clkdll_ready3_0	: std_logic;

type stateTypeMaster is	(wait_start_capture, top_clkdll_reset, wait_start_count, top_start_count, wait_stop_capture, top_stop_capture);
signal next_state_master : stateTypeMaster;

type stateTypeSlave is	(wait_start_slave_on, wait_top_clkdll_reset_slave, wait_top_start_count_slave, wait_top_stop_capture_slave);
signal next_state_slave : stateTypeSlave;

signal clkdll_reset3		: std_logic;
signal clkdll_reset5		: std_logic;

signal int_START_CAPTURE		: std_logic;
signal int1_START_CAPTURE		: std_logic;
signal s_START_CAPTURE_SLV	: std_logic;

signal out_divA	: std_logic;
signal out_divB	: std_logic;
signal in_divA	: std_logic;
signal in_divB	: std_logic;

signal out_fbA	: std_logic;
signal out_fbB	: std_logic;
signal in_fbA	: std_logic;
signal in_fbB	: std_logic;

signal out_LRCKA	: std_logic;
signal out_LRCKB	: std_logic;
signal in_LRCKA	: std_logic;
signal in_LRCKB	: std_logic;

signal out_BCKA	: std_logic;
signal out_BCKB	: std_logic;
signal in_BCKA	: std_logic;
signal in_BCKB	: std_logic;
			
component CLKDLL
  port ( CLKFB  : in    std_logic; 
         CLKIN  : in    std_logic; 
         RST    : in    std_logic; 
         CLK0   : out   std_logic; 
         CLK180 : out   std_logic; 
         CLK270 : out   std_logic; 
         CLK2X  : out   std_logic; 
         CLK90  : out   std_logic; 
         CLKDV  : out   std_logic; 
         LOCKED : out   std_logic);
end component;

attribute CLKDV_DIVIDE: string;
attribute CLKDV_DIVIDE of div_3_clkdll : label is "3";
attribute CLKDV_DIVIDE of div_5_clkdll : label is "3";

--attribute clock_signal : string;
--attribute clock_signal of clkdiv : signal is yes; 
--attribute clock_signal of clkfb : signal is yes; 

attribute LOC: string;
attribute LOC of XLXI_1 : label is "RAMB4_R1C0";
attribute LOC of XLXI_2 : label is "RAMB4_R0C0";
attribute LOC of XLXI_3 : label is "RAMB4_R2C0";
attribute LOC of XLXI_4 : label is "RAMB4_R3C0";

begin
--================================================================================================================================
-- test vectors
--================================================================================================================================
--count1_visu	<= count1;
--MEM_ADDR_visu <= MEM_ADDR;
--MEM_ADDR_visu(0) <= sPDN512_2;
--MEM_ADDR_visu(1) <= clkfb_3;
--MEM_ADDR_visu(2) <= clk_3;
--MEM_ADDR_visu(3) <= clkdll_reset3;
--MEM_ADDR_visu(4) <= clkdll_ready3;
--MEM_ADDR_visu(5) <= clkd3;
--MEM_ADDR_visu(6) <= clkdiv;
--MEM_ADDR_visu(7) <= clkfb;
--MEM_ADDR_visu(8) <= sPDN512;
--MEM_ADDR_visu(9) <= sPDN32;
--MEM_visu <= MEM;
--incrmt_addr_visu <= incrmt_addr;
--incrmt_shift_visu <= incrmt_shift;
--incrmt_par_visu	<= incrmt_par;
--count32_visu <= count32;
--count512_visu <= count512;
--clkfb_3_visu <= clkfb_3;
--startdata_visu <= startdata;

--================================================================================================================================

START_CAPTURE_SLV <= s_START_CAPTURE_SLV;

--================================================================================================================================
-- CLKDLL :	This CLKDLL is the one using the oscillator on the board.
--					So it's the one used in master mode
--					It's dividing the main clock by 3
--================================================================================================================================
div_3_ibufg: IBUFG port map (I=>CAP_CLK, O=>clkin3); 
div_3_bufg: BUFG port map (I=>clk_3, O=>clkfb_3);
div_3_clkdll: CLKDLL 
port map (CLKIN=>clkin3, CLKFB=>clkfb_3, RST=>clkdll_reset3,CLK2X=>open, CLK90=>open, CLK180=>open, CLK270=>open, 
					CLK0=>clk_3, LOCKED=>clkdll_ready3, CLKDV=>clkd3);

--================================================================================================================================
-- CLKDLL :	This CLKDLL is the one using the clock signal from another board and distributed.
--					So it's the one used in slave mode
--					It's dividing the main clock by 3
--================================================================================================================================
div_5_ibufg: IBUFG port map (I=>CAP_CLK_SLAVE, O=>clkin5); 
div_5_bufg: BUFG port map (I=>clk_5, O=>clkfb_5);
div_5_clkdll: CLKDLL 
port map (CLKIN=>clkin5, CLKFB=>clkfb_5, RST=>clkdll_reset5,CLK2X=>open, CLK90=>open, CLK180=>open, CLK270=>open, 
					CLK0=>clk_5, LOCKED=>open, CLKDV=>clkd5);

--================================================================================================================================
-- signals :	2 main clocks of this module depending if we are in slave mode or not.
--						clkfb is 3 times faster than clkdiv.
--						Trouble-Free Switching Between Clocks
--						cf http://www.xilinx.com/xcell/xl24/xl24_20.pdf
--================================================================================================================================
in_divA <=	not SYNC_SLAVE and not out_divB;
in_divB <=	SYNC_SLAVE and not out_divA;

Adiv : FD_1
	port map (C=>clkd3, D=>in_divA, Q=>out_divA);
Bdiv : FD_1
	port map (C=>clkd5, D=>in_divB, Q=>out_divB);
clkdiv <= (out_divA and clkd3 ) or (out_divB and clkd5);  

in_fbA <=	not SYNC_SLAVE and not out_fbB;
in_fbB <=	SYNC_SLAVE and not out_fbA;

Afb : FD_1
	port map (C=>clkfb_3, D=>in_fbA, Q=>out_fbA);
Bfb : FD_1
	port map (C=>clkfb_5, D=>in_fbB, Q=>out_fbB);
clkfb <= (out_fbA and clkfb_3 ) or (out_fbB and clkfb_5);  

--================================================================================================================================
-- signals :	3 control clocks of the ADCs on the Microboards
--						Trouble-Free Switching Between Clocks
--						cf http://www.xilinx.com/xcell/xl24/xl24_20.pdf
--================================================================================================================================
in_BCKA <=	not DOUBLE_FRQ_AD and not out_BCKB;
in_BCKB <=	DOUBLE_FRQ_AD and not out_BCKA;

ABCK : FD
	port map (C=>count32(4), D=>in_BCKA, Q=>out_BCKA);
BBCK : FD_1
	port map (C=>count32(3), D=>in_BCKB, Q=>out_BCKB);
BCK <= (out_BCKA and count32(4) ) or (out_BCKB and not count32(3));  

in_LRCKA <=	not DOUBLE_FRQ_AD and not out_LRCKB;
in_LRCKB <=	DOUBLE_FRQ_AD and not out_LRCKA;

ALRCK : FD
	port map (C=>count512(8), D=>in_LRCKA, Q=>out_LRCKA);
BLRCK : FD
	port map (C=>count512(7), D=>in_LRCKB, Q=>out_LRCKB);
LRCK <= not((out_LRCKA and count512(8) ) or (out_LRCKB and count512(7)));  

SCKI <= clkdiv;

--================================================================================================================================
-- process :	Starts the 9 bits counter count512 with the signal sPDN512
--						sPDN512 is generating depending if we are in slave mode or not.
--================================================================================================================================

sPDN512_0 <= (sPDN512_3 and not SYNC_SLAVE) or (sPDN512_5 and SYNC_SLAVE);--48fs

makesPDN512_2 : FD
	port map (C=>clkdiv, D=>sPDN512_0, Q=>sPDN512_2);

makesPDN512_3 : FD
	port map (C=>clkdiv, D=>sPDN512_2, Q=>sPDN512);
 
divide_512 : process(clkdiv,sPDN512)
begin
if rising_edge(clkdiv) then
	if sPDN512 ='0' then
		count512 <= "000000000";
	else
		count512 <= count512 + '1';
	end if;
end if;
end process divide_512;

--================================================================================================================================
-- process :	Make a synchronized signal sPDN32.
--================================================================================================================================
makespdn32_2 : FDE
	port map (C=>clkfb, CE=>not clkdiv, D=>sPDN512, Q=>sPDN32);

--================================================================================================================================
-- process :	Starts the 9 bits counter count512 with the signal sPDN32.
--================================================================================================================================
divide2 : process(clkfb,sPDN32)
begin
if rising_edge(clkfb) then
	if sPDN32 ='0' then
		count32 <= "000000000";
	else
		count32 <= count32 + '1';
	end if;
end if;
end process divide2;


--================================================================================================================================
-- process :	Make a synchronized signal int_START_CAPTURE which is used to start the capture.
--================================================================================================================================
process(clkfb)
begin
if rising_edge(clkfb) then
	int1_START_CAPTURE <= ( s_START_CAPTURE_SLV and SYNC_SLAVE) or START_CAPTURE;
end if;
end process;

makeint_START_CAPTURE : FD
	port map (C=>clkfb,  D=>int1_START_CAPTURE, Q=>int_START_CAPTURE);

--================================================================================================================================
-- process :	Make a synchronized signal startdata wich is used to start the address process 
--						depending on the frequency asked.
--================================================================================================================================
debut : process(clkfb)
begin
	if rising_edge(clkfb) then
		if int_START_CAPTURE = '0' then
				startdata<='0';
		else
			if incrmt_shift ='1' then
				if int_START_CAPTURE = '0' then
					startdata<='0';
				else
					if DOUBLE_FRQ_AD ='0' then
						if count512="001010000" then
							startdata<='1';
						end if;
					else
						if count512="000101011" then
							startdata<='1';
						end if;
					end if;
				end if;
			end if;
		end if;
	end if;
end process;

--================================================================================================================================
-- process :	This counter starting when the CLKDLL is ready, is just here to delay of some clock edge 
--						the signal to start the capture in order to be sure that every slave board CLKDLL are ready.
--================================================================================================================================
makeclkdll_ready3 : FD
	port map (C=>clkin3, D=>clkdll_ready3, Q=>clkdll_ready3_0);

--makesPDN512_2 : FD
--	port map (C=>clkdiv, D=>clkdll_ready3_0, Q=>clkdll_ready3_1);

--makespdn512 : process(clkdiv,clkdll_ready3_1)
--begin
--if rising_edge(clkdiv) then
--		if clkdll_ready3 = '1' then
--			counter_wait <= counter_wait + '1';
--		else
--			counter_wait <= "00000";
--		end if;
--end if;
--end process makespdn512;

--================================================================================================================================
-- process :	This state machine is here to start the different phases of a capture
--						in order to synchronized the slave and master boards.
--						This process is only used in master mode.
--================================================================================================================================
makeSTART_CAPTURE_0 : FD
	port map (C=>clkin3, D=>START_CAPTURE, Q=>start_capture_0);

state_machine_master : process(clkin3,LOW_RESET)
begin
if LOW_RESET = '0' then
	next_state_master <= wait_start_capture;
elsif rising_edge(clkin3) then
	case next_state_master is

		when wait_start_capture =>
			if start_capture_0 = '1' then
				next_state_master <= top_clkdll_reset;
			else
				next_state_master <= wait_start_capture;
			end if;

		when top_clkdll_reset =>
			next_state_master <= wait_start_count;

		when wait_start_count =>
			if clkdll_ready3_0 = '1' then
				counter_wait <= counter_wait + '1';
				if counter_wait = "11111" then
					next_state_master <= top_start_count;
				else
					next_state_master <= wait_start_count;
				end if;
			else
				counter_wait <= "00000";
				next_state_master <= wait_start_count;
			end if;

		when top_start_count =>
			next_state_master <= wait_stop_capture;

		when wait_stop_capture =>
			if start_capture_0 = '1' then
				next_state_master <= wait_stop_capture;
			else
				next_state_master <= top_stop_capture;
			end if;

		when top_stop_capture =>
			next_state_master <= wait_start_capture;

	end case;
end if;
end process  state_machine_master;

--================================================================================================================================
-- signal :	sPDN512_3 goes up when the capture is active in master mode
--================================================================================================================================
with next_state_master select
sPDN512_3		 <= 	'1' when top_start_count,
									'1' when wait_stop_capture,
									'0' when others;

--================================================================================================================================
-- signal :	clkdll_reset3 resets the CLKDLL3 
--================================================================================================================================
with next_state_master select
clkdll_reset3 <= 	'1' when top_clkdll_reset,
									'0' when others;
--clkdll_reset3 <= '0';
--================================================================================================================================
-- signal :	SYNC_CAP_CLK_MASTER is the 3 edges signal distributed to the slave boards
--================================================================================================================================
with next_state_master select
SYNC_CAP_CLK_MASTER  <= 	'1' when top_clkdll_reset,
													'1' when top_start_count,
													'1' when top_stop_capture,
													'0' when others;


--================================================================================================================================
-- process :	This state machine is here to start the different phases of a capture
--						in order to synchronized the slave board after the master board.
--						This process is only used in slave mode.
--						It's using the input made as previously but from antoher board.
--================================================================================================================================
state_machine_slave : process(clkin5,LOW_RESET)
begin
if LOW_RESET = '0' then
	next_state_slave <= wait_start_slave_on;
elsif rising_edge(clkin5) then
	case next_state_slave is

		when wait_start_slave_on  =>
			if SYNC_SLAVE = '1' then
				next_state_slave <= wait_top_clkdll_reset_slave;
			else
				next_state_slave <= wait_start_slave_on;
			end if;

		when wait_top_clkdll_reset_slave =>
			if SYNC_CAP_CLK_SLAVE = '1' then
				next_state_slave <= wait_top_start_count_slave;
			else
				next_state_slave <= wait_top_clkdll_reset_slave;
			end if;	

		when wait_top_start_count_slave =>
			if SYNC_CAP_CLK_SLAVE = '1' then
				next_state_slave <= wait_top_stop_capture_slave;
			else
				next_state_slave <= wait_top_start_count_slave;
			end if;	
		
		when wait_top_stop_capture_slave =>
			if SYNC_CAP_CLK_SLAVE = '1' then
				next_state_slave <= wait_start_slave_on;
			else
				next_state_slave <= wait_top_stop_capture_slave;
			end if;

	end case;
end if;
end process  state_machine_slave;

--================================================================================================================================
-- signal :	s_START_CAPTURE_SLV goes up when the capture is active in slave mode.
--================================================================================================================================
with next_state_slave select
s_START_CAPTURE_SLV <=	'1' when wait_top_stop_capture_slave,
												'0' when others;

--================================================================================================================================
-- signal :	clkdll_reset5 resets the CLKDLL5 
--================================================================================================================================
with next_state_slave select
clkdll_reset5 <=	SYNC_CAP_CLK_SLAVE when wait_top_clkdll_reset_slave,
									'0' when others;

--================================================================================================================================
-- signal :	sPDN512_5 goes up when the capture is active in slave mode 
--					but one clock in advance on s_START_CAPTURE_SLV.
--================================================================================================================================
with next_state_slave select
sPDN512_5 <=	SYNC_CAP_CLK_SLAVE when wait_top_start_count_slave,
							'1' when wait_top_stop_capture_slave,
							'0' when others;

--================================================================================================================================
-- signal :	incrmt_addr goes up when the address process should advance of one
--================================================================================================================================
incrmt_addr <= count32(0) and count32(1) and (count32(2) or DOUBLE_FRQ_AD);
--================================================================================================================================
-- signal :	incrmt_shift goes up when the a shift in the inputs should be done
--================================================================================================================================
incrmt_shift<= count32(0) and count32(1) and count32(2) and count32(3) and (not count32(4) or DOUBLE_FRQ_AD);
--================================================================================================================================
-- signal :	incrmt_par goes up when the 8 bits shift is done and the buffer is ready to be put in the memory
--================================================================================================================================
incrmt_par  <= (count32(4) xnor DOUBLE_FRQ_AD) and count32(5) and count32(6) and (count32(7) or DOUBLE_FRQ_AD);

--================================================================================================================================
-- process :	This state machine is here to order each 8 bits 
--						buffer created form the serial input in the right place in the memory
--						ENABLE_GAUCHE is here to do the double buffering between the 4 memories ( 2 bloc of 2)
--						PACKET_READY goes up when a packet is ready in memory for the SRAM module
--================================================================================================================================
engine : process(clkfb)
begin
	if rising_edge(clkfb) then
		if startdata='0' then
			MEM_ADDR<="0000000110";
			count1<="00000";
			count2<="000";
			nbpaquet<="000";
		else
			if incrmt_addr ='1' then
				count1<=count1 + '1';			    				-- line counter
				if count1 = "11111" then							-- 32-1 lines
					case count2 is
						when "101" =>											-- 6-1 space between 2 colonnes
							count2<="000";
							if nbpaquet="100" then						-- 5-1 packets per buffer
								nbpaquet<="000";
								MEM_ADDR<="0000000011";
								ENABLE_GAUCHE<= not ENABLE_GAUCHE;
								PACKET_READY<='1';
							else
								MEM_ADDR<=MEM_ADDR+"111";
						   	nbpaquet<=nbpaquet+'1';
							end if;
						when "010" =>
							MEM_ADDR<=MEM_ADDR-"10111111";		--	MEM_ADDR- 191
							count2 <= count2 + '1';	
					  when others  => 
							MEM_ADDR<=MEM_ADDR-"10111001";		--	MEM_ADDR- 185
							count2 <= count2 + '1';						--	column counter
					end case;
				else
					MEM_ADDR<=MEM_ADDR + "110";					-- 110=6 between 2 lines
				end if;
			end if;
		end if;
		if ENABLE_CAPTURE_MEM='1' then
			PACKET_READY<='0';
		end if;
	end if;
end process engine;

--================================================================================================================================
-- process :	this process is just creating a 8 bits large buffer from a serial input data
--						This on the 32 input signals from the 32 ADC
--================================================================================================================================
deserial: process (clkfb)
variable STD01_tmp : std_logic_vector(7 downto 0);
variable STD02_tmp : std_logic_vector(7 downto 0);
variable STD03_tmp : std_logic_vector(7 downto 0);
variable STD04_tmp : std_logic_vector(7 downto 0);
variable STD05_tmp : std_logic_vector(7 downto 0);
variable STD06_tmp : std_logic_vector(7 downto 0);
variable STD07_tmp : std_logic_vector(7 downto 0);
variable STD08_tmp : std_logic_vector(7 downto 0);
variable STD09_tmp : std_logic_vector(7 downto 0);
variable STD10_tmp : std_logic_vector(7 downto 0);
variable STD11_tmp : std_logic_vector(7 downto 0);
variable STD12_tmp : std_logic_vector(7 downto 0);
variable STD13_tmp : std_logic_vector(7 downto 0);
variable STD14_tmp : std_logic_vector(7 downto 0);
variable STD15_tmp : std_logic_vector(7 downto 0);
variable STD16_tmp : std_logic_vector(7 downto 0);
variable STD17_tmp : std_logic_vector(7 downto 0);
variable STD18_tmp : std_logic_vector(7 downto 0);
variable STD19_tmp : std_logic_vector(7 downto 0);
variable STD20_tmp : std_logic_vector(7 downto 0);
variable STD21_tmp : std_logic_vector(7 downto 0);
variable STD22_tmp : std_logic_vector(7 downto 0);
variable STD23_tmp : std_logic_vector(7 downto 0);
variable STD24_tmp : std_logic_vector(7 downto 0);
variable STD25_tmp : std_logic_vector(7 downto 0);
variable STD26_tmp : std_logic_vector(7 downto 0);
variable STD27_tmp : std_logic_vector(7 downto 0);
variable STD28_tmp : std_logic_vector(7 downto 0);
variable STD29_tmp : std_logic_vector(7 downto 0);
variable STD30_tmp : std_logic_vector(7 downto 0);
variable STD31_tmp : std_logic_vector(7 downto 0);
variable STD32_tmp : std_logic_vector(7 downto 0);
begin
if rising_edge(clkfb) then
	if incrmt_shift ='1' then
		STD01_tmp := STD01_tmp(6 downto 0) & STD01;
		STD02_tmp := STD02_tmp(6 downto 0) & STD02;
		STD03_tmp := STD03_tmp(6 downto 0) & STD03;
		STD04_tmp := STD04_tmp(6 downto 0) & STD04;
		STD05_tmp := STD05_tmp(6 downto 0) & STD05;
		STD06_tmp := STD06_tmp(6 downto 0) & STD06;
		STD07_tmp := STD07_tmp(6 downto 0) & STD07;
		STD08_tmp := STD08_tmp(6 downto 0) & STD08;
		STD09_tmp := STD09_tmp(6 downto 0) & STD09;
		STD10_tmp := STD10_tmp(6 downto 0) & STD10;
		STD11_tmp := STD11_tmp(6 downto 0) & STD11;
		STD12_tmp := STD12_tmp(6 downto 0) & STD12;
		STD13_tmp := STD13_tmp(6 downto 0) & STD13;
		STD14_tmp := STD14_tmp(6 downto 0) & STD14;
		STD15_tmp := STD15_tmp(6 downto 0) & STD15;
		STD16_tmp := STD16_tmp(6 downto 0) & STD16;
		STD17_tmp := STD17_tmp(6 downto 0) & STD17;
		STD18_tmp := STD18_tmp(6 downto 0) & STD18;
		STD19_tmp := STD19_tmp(6 downto 0) & STD19;
		STD20_tmp := STD20_tmp(6 downto 0) & STD20;
		STD21_tmp := STD21_tmp(6 downto 0) & STD21;
		STD22_tmp := STD22_tmp(6 downto 0) & STD22;
		STD23_tmp := STD23_tmp(6 downto 0) & STD23;
		STD24_tmp := STD24_tmp(6 downto 0) & STD24;
		STD25_tmp := STD25_tmp(6 downto 0) & STD25;
		STD26_tmp := STD26_tmp(6 downto 0) & STD26;
		STD27_tmp := STD27_tmp(6 downto 0) & STD27;
		STD28_tmp := STD28_tmp(6 downto 0) & STD28;
		STD29_tmp := STD29_tmp(6 downto 0) & STD29;
		STD30_tmp := STD30_tmp(6 downto 0) & STD30;
		STD31_tmp := STD31_tmp(6 downto 0) & STD31;
		STD32_tmp := STD32_tmp(6 downto 0) & STD32;
		if incrmt_par='1' then
			par01 <= STD01_tmp;
			par02 <= STD02_tmp;
			par03 <= STD03_tmp;
			par04 <= STD04_tmp;
			par05 <= STD05_tmp;
			par06 <= STD06_tmp;
			par07 <= STD07_tmp;
			par08 <= STD08_tmp;
			par09 <= STD09_tmp;
			par10 <= STD10_tmp;
			par11 <= STD11_tmp;
			par12 <= STD12_tmp;
			par13 <= STD13_tmp;
			par14 <= STD14_tmp;
			par15 <= STD15_tmp;
			par16 <= STD16_tmp;
			par17 <= STD17_tmp;
			par18 <= STD18_tmp;
			par19 <= STD19_tmp;
			par20 <= STD20_tmp;
			par21 <= STD21_tmp;
			par22 <= STD22_tmp;
			par23 <= STD23_tmp;
			par24 <= STD24_tmp;
			par25 <= STD25_tmp;
			par26 <= STD26_tmp;
			par27 <= STD27_tmp;
			par28 <= STD28_tmp;
			par29 <= STD29_tmp;
			par30 <= STD30_tmp;
			par31 <= STD31_tmp;
			par32 <= STD32_tmp;			
		end if;
	end if;
end if;
end process;

--================================================================================================================================
-- MUX :	Put the right 8 bit buffer from the inputs at the right time.
--				It works with MEM_ADDR to feed the memories
--================================================================================================================================
MEM <= 	par01 when count1="00000" else
			par02 when count1="00001" else
			par03 when count1="00010" else
			par04 when count1="00011" else
			par05 when count1="00100" else
			par06 when count1="00101" else
			par07 when count1="00110" else
			par08 when count1="00111" else
			par09 when count1="01000" else
			par10 when count1="01001" else
			par11 when count1="01010" else
			par12 when count1="01011" else
			par13 when count1="01100" else
			par14 when count1="01101" else
			par15 when count1="01110" else
			par16 when count1="01111" else
			par17 when count1="10000" else
			par18 when count1="10001" else
			par19 when count1="10010" else
			par20 when count1="10011" else
			par21 when count1="10100" else
			par22 when count1="10101" else
			par23 when count1="10110" else
			par24 when count1="10111" else
			par25 when count1="11000" else
			par26 when count1="11001" else
			par27 when count1="11010" else
			par28 when count1="11011" else
			par29 when count1="11100" else
			par30 when count1="11101" else
			par31 when count1="11110" else
			par32;

--================================================================================================================================
-- signals :	enable the right memroy for the right clock edge of clkfb.
--						the memories are controlled in double buffering by ENABLE_GAUCHE.
--						MEM_ADDR(9) controls when the first memory is full.
--================================================================================================================================
incrmt_addr_mem <= count32(0) and count32(1) and (count32(2) or DOUBLE_FRQ_AD);
ENA1<=ENABLE_GAUCHE AND (not MEM_ADDR(9)) and incrmt_addr_mem;
ENA2<=ENABLE_GAUCHE AND MEM_ADDR(9) and incrmt_addr_mem;
ENA3<=(not ENABLE_GAUCHE) AND (not MEM_ADDR(9)) and incrmt_addr_mem;
ENA4<=(not ENABLE_GAUCHE) AND MEM_ADDR(9)and incrmt_addr_mem;

--================================================================================================================================
-- memory:	the 4 memories of the double buffering of the data packets of 960 bytes (2 memories per packet)
--================================================================================================================================
----------------------------------------------------------------------left memory
XLXI_1 : RAMB4_S8_S16 PORT MAP (WEA=>'1', ENA=>ENA1, RSTA=>'0', CLKA=>clkfb, ADDRA=>MEM_ADDR(8 downto 0), DIA=>MEM, DOA=>open, 
						  WEB=>'0', ENB=>ENB1, RSTB=>'0', CLKB=>CLK_CAPTURE_MEM, ADDRB=>ADDR_CAPTURE_MEM(7 downto 0), DIB=>"0000000000000000", DOB=>DOB01
						 );

XLXI_2 : RAMB4_S8_S16 PORT MAP (WEA=>'1', ENA=>ENA2, RSTA=>'0', CLKA=>clkfb, ADDRA=>MEM_ADDR(8 downto 0), DIA=>MEM, DOA=>open, 
						  WEB=>'0', ENB=>ENB2, RSTB=>'0', CLKB=>CLK_CAPTURE_MEM, ADDRB=>ADDR_CAPTURE_MEM(7 downto 0), DIB=>"0000000000000000", DOB=>DOB02
						 );
----------------------------------------------------------------------left memory
----------------------------------------------------------------------right memory
XLXI_3 : RAMB4_S8_S16 PORT MAP (WEA=>'1', ENA=>ENA3, RSTA=>'0', CLKA=>clkfb, ADDRA=>MEM_ADDR(8 downto 0), DIA=>MEM, DOA=>open, 
						  WEB=>'0', ENB=>ENB3, RSTB=>'0', CLKB=>CLK_CAPTURE_MEM, ADDRB=>ADDR_CAPTURE_MEM(7 downto 0), DIB=>"0000000000000000", DOB=>DOB03
						 );

XLXI_4 : RAMB4_S8_S16 PORT MAP (WEA=>'1', ENA=>ENA4, RSTA=>'0', CLKA=>clkfb, ADDRA=>MEM_ADDR(8 downto 0), DIA=>MEM, DOA=>open, 
						  WEB=>'0', ENB=>ENB4, RSTB=>'0', CLKB=>CLK_CAPTURE_MEM, ADDRB=>ADDR_CAPTURE_MEM(7 downto 0), DIB=>"0000000000000000", DOB=>DOB04
						 );
----------------------------------------------------------------------right memory

--================================================================================================================================
-- signal :	this enable the double buffering of the memory interfacing with the module SRAM_INTERFACE.
--					each of the 4 memories is activatied at the right time and in opposition to ENAx
--================================================================================================================================
ENB1<=(not ENABLE_GAUCHE) AND (not ADDR_CAPTURE_MEM(8)) and ENABLE_CAPTURE_MEM;
ENB2<=(not ENABLE_GAUCHE) AND (ADDR_CAPTURE_MEM(8) or (ADDR_CAPTURE_MEM(7) and ADDR_CAPTURE_MEM(6) and ADDR_CAPTURE_MEM(5) and ADDR_CAPTURE_MEM(4) and ADDR_CAPTURE_MEM(3) and ADDR_CAPTURE_MEM(2) and ADDR_CAPTURE_MEM(1) and ADDR_CAPTURE_MEM(0))) and ENABLE_CAPTURE_MEM;
ENB3<=ENABLE_GAUCHE AND (not ADDR_CAPTURE_MEM(8)) and ENABLE_CAPTURE_MEM;
ENB4<=ENABLE_GAUCHE AND (ADDR_CAPTURE_MEM(8) or (ADDR_CAPTURE_MEM(7) and ADDR_CAPTURE_MEM(6) and ADDR_CAPTURE_MEM(5) and ADDR_CAPTURE_MEM(4) and ADDR_CAPTURE_MEM(3) and ADDR_CAPTURE_MEM(2) and ADDR_CAPTURE_MEM(1) and ADDR_CAPTURE_MEM(0))) and ENABLE_CAPTURE_MEM;

--================================================================================================================================
-- signal : delay of ENBx of one clock edge otherwise "0000" appears on DATA_CAPTURE_MEM
--================================================================================================================================
process(CLK_CAPTURE_MEM)
begin
	if rising_edge(CLK_CAPTURE_MEM) then
		ENB1_i <= ENB1;
		ENB2_i <= ENB2;
		ENB3_i <= ENB3;
	end if;	
end process;

--================================================================================================================================
-- signal : muliplexing of the 4 memories to 1 data bus
--================================================================================================================================
DATA_CAPTURE_MEM <=	DOB01 when ENB1_i='1' else
										DOB02 when ENB2_i='1' else
										DOB03 when ENB3_i='1' else
										DOB04;
end RTL;
