#!/usr/bin/perl

# Parse a text format ANSI/NIST file given as an argument, like those
# produced by an2k2txt and read by txt2an2k, and produce an XML file
# on the standard output.  This program was written by Joseph Konczal,
# joe.konczal@nist.gov, in order to accomplish the assigned task of
# generating sample data files, not as a product in and of itself.

# The XML encoding of an ANSI/NIST file includes more nested structure
# than a "traditional" file, and the fields are all rearranged.
# Therefore, a translation method based on simple substitution is not
# sufficient.  Furthermmore, the 2011 version of the standard includes
# more complicated structures that required major rewriting of this
# program.

# The ANSI/NIST txt input file is first parsed into a multidimensional
# array indexed by record, field, subfield, and item numbers.  This
# structure is traversed by a function that examines each item in
# sequence and creates an internal tree representation of the
# corresponding XML structures.  Other elements of the
# multidimentional input data array are sometimes examined to
# determine the details of how to construct the XML structures.
# Additional information is included in the tree to help with the
# construction of it and to use in XML comments.  Finally, the
# internal XML data structure tree is converted into the standard
# external XML representation. 

# The translation from multidimentional array to XML tree is directed
# by a series of specifications, indexed by record, field, subfield,
# and item numbers.  Functions can be specified in place of tags to
# perform more complicated processing, like choosing what tag to use
# when the tag is not uniquely determined by the structure of the input
# but also depends on the contents, or when a single item needs to be
# split into multiple xml tags.

# It consists of a hash table with record numbers and names of default
# structures as the keys.  Each record level structure is another hash
# table with field numbers as the keys.  Many fields contain a single
# item, so they are represented as arrays of:
#
#     0: mnemonic, if defined, e.g., FGP
#
#     1: short name of parent node, e.g., 'record', 'body'
#
#     2: numerical position within parent node
#
#     3: tag, or list of tags, or 'array', 
#        or a function to do more complicated processing
#
#     4: optional function to process contents, 
#        or an argument to the complicated processing function
#
#     5: optional argument to content processing function
# 
# Some special field keys used to represent other things.
#
#     tag: The top level tag for a record structure.
#
#     fallback: The tag of the next record level specification to look
#               at to find field and special specifications not in the
#               current record level specification.
#
#     parts: A hash of specifications for required nodes between the
#            record and the leaves, keyed by the short node name.  The
#            contents are arrays similar to above, but with some
#            different alternatives at the end.

# The internal representation an XML tree is designed to facilitate
# the conversion from a four-dimentional array of records, fields,
# subfields, and items to a set of record tags with arbitrarily
# complex internal structures.  The special tag value of 'array' marks
# a data structure for accumulating multiple tags in sequence without
# any additional enclosing XML tags.

# Dynamic XML tree data structure--multiple levels of hashes and
# arrays representing the XML tree, with hash keys:
#
#    key        type      description
#    ===        ====      ===========
#    tag	string	  The XML tag for the structure
#
#    content	string    The tag contents, either text or nested tags
#               array
#
#    comment    string    A comment to add to the XML before the contents.
#
#    nodes	hash	  Used in a record-level structure to hold
#			  references to the nodes within that record,
#			  keyed by the short name.  The subfield index
#			  is appended to the names of nodes that occur
#			  once per subfield.
#
#    repeats	string    Indicates whether child node repeats occur
#                         'one_per_record', 'one_per_subfield',
#                         'accumulate_pairs', 'accumulate_items', etc.
#
#    increment  integer	  The number of items in a field, used to skip
#                         to the next open position when adding
#                         repeating subfields that all go within the
#                         same parent node.

# Without these features it is too easy to make and overlook small
# typographic errors that cause big trouble.
use warnings;
use strict;

# for debugging
use Data::Dumper;

# ANSI/NIST-ITL 2-2008, p. 170, says base-64 is identical to the encoding
# used in PEM, except there is no embedded cleartext.
use MIME::Base64;

# for prettyprinting the XML
my $indent_factor = 3;
my $line_length = 90;

# Map input lines to easily indexed data structures, using four levels
# of array references.  Since index zero is not used, it is available
# for specifing the record or field number, which are required in
# order to know how to handle the data inside.  The record number is
# in field 0, and the field number in subfield 0.
sub parse_txt_file {
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 0;
    my $result;
    my ($ri, $fi, $si, $ii, $rn, $fn, $val, $delimiter, $base_64_value);

    while (<>) {
	# Read, parse, and convert each line of the AN2K text format
	# input file.  All the lines, except continuation lines of
	# base-64 data, have the same components and are terminated by
	# an item separator character  (This is a special character
	# displayed as carat underscore, but entered, as
	# control-underscore.  On a keyboard where underscore is
	# shift-minus, the full keystroke is control-shift-minus.)

	if (/^(\d+)\.(\d+)\.(\d+)\.(\d+) \[(\d+)\.(\d+)\]=([^]*)(?)$/) {
	    # The usual case, a structured line of data
	    ($ri, $fi, $si, $ii, $rn, $fn, $val, $delimiter) = 
		($1, $2, $3, $4, 0+$5, 0+$6, $7, $8);
	    if (!$delimiter) {
		$base_64_value = "\n" . $val;
		next;
	    }
	} elsif ($base_64_value) {
	    # append base-64 data line to the aggregated value
	    $base_64_value .= $_;
	    if (/$/) {		# end of base-64 data value
		($val = $base_64_value) =~ s/$//;
		chomp $val;
		# proceed to process the completed item...
	    } else {
		next;
	    }
	} else {
	    warn "Cannot parse input line: $_";
	    next;
	}
	
	# When we get to here, there is an item to store.
	$result->[$ri]->[0] = $rn if 1 == $fi; # remember the record type
	$result->[$ri]->[$fi]->[0] = $fn if 1 == $si; # remember the field type
	$result->[$ri]->[$fi]->[$si]->[$ii] = $val;
    }
    return $result;
}

# Top level data strucures.
# tag definitions
my $imp_tag =  'biom:FingerprintImageImpressionCaptureCategoryCode';
my $hll_tag =  'biom:ImageHorizontalLineLengthPixelQuantity';
my $vll_tag =  'biom:ImageVerticalLineLengthPixelQuantity';
my $slc_tag =  'biom:ImageScaleUnitsCode';
my $hps_tag =  'biom:ImageHorizontalPixelDensityValue';
my $vps_tag =  'biom:ImageVerticalPixelDensityValue';
my $cga_tag =  'biom:ImageCompressionAlgorithmText';
my $bpx_tag =  'biom:ImageBitsPerPixelQuantity';
my $csp_tag =  'biom:ImageColorSpaceCode';
my $fgp_tag =  'biom:FingerPositionCode'; 
my $plp_tag =  'biom:PalmPositionCode';
my $shps_tag = 'biom:CaptureHorizontalPixelDensityValue';
my $svps_tag = 'biom:CaptureVerticalPixelDensityValue';
my $dmm_tag =  'biom:CaptureDeviceMonitoringModeCode';
my $com_img_tag = 'biom:ImageCommentText';

# common specifications
my $com_img_spec =  ['COM',  'body',        6, $com_img_tag, \&xml_escape];

sub cap_date_spec {
    [shift, 'cap_detail', 3, ['biom:CaptureDate', 'nc:Date'], \&hyphenate_date];
}
my $cap_dui_spec = ['DUI', 'cap_detail', 4, ['biom:CaptureDeviceIdentification',
					     'nc:IdentificationID']];
my $cap_mms_spec = {
    a => ['MAK', 'cap_detail', 5, 'biom:CaptureDeviceMakeText',
	  \&omit_empty],
    b => ['MOD', 'cap_detail', 6, 'biom:CaptureDeviceModelText',
	  \&omit_empty],
    c => ['SER', 'cap_detail', 7, 'biom:CaptureDeviceSerialNumberText',
	  \&omit_empty], 
};
my $cap_shps_spec = ['SHPS', 'cap_detail',  8, $shps_tag];
my $cap_svps_spec = ['SVPS', 'cap_detail', 10, $svps_tag];
my $cap_dmm_spec =  ['DMM',  'cap_detail', 11, $dmm_tag];

my $maj_ppd_spec = {
    dfp => ['', 'major_case', 1, 'array'],
    a => ['DFP', 'dfp', 1, $fgp_tag],
    fic => ['', 'major_case', 2, 'array'],
    b => ['FIC', 'fic', 1, 'biom:MajorCasePrintCode'],
};
my $maj_spd_spec =  {
    pdf => ['', 'major_case', 1, 'array'],
    a => ['PDF', 'pdf', 1, $fgp_tag],
    fic => ['', 'major_case', 2, 'array'],
    b => ['FIC', 'fic', 1, 'biom:MajorCasePrintCode'],
};
my $maj_ppc_spec = {	# used in record types 13 and 14
    ppc => ['PPC', 'major_case', 3, 'biom:MajorCasePrintSegmentOffset'],
    a => ['FVC', 'ppc', 3, 'biom:SegmentFingerViewCode'],
    b => ['LOS', 'ppc', 2, 'biom:SegmentLocationCode'],
    c => ['LHC', 'ppc', 4, 'biom:SegmentLeftHorizontalCoordinateValue'],
    d => ['RHC', 'ppc', 5, 'biom:SegmentRightHorizontalCoordinateValue'],
    e => ['TVC', 'ppc', 6, 'biom:SegmentTopVerticalCoordinateValue'],
    f => ['BVC', 'ppc', 1, 'biom:SegmentBottomVerticalCoordinateValue'],
};

# Be careful not to assign the  same postion number to two things used
# at the same time; one  could overwrite the other.  When programming,
# check whether a slot is assigned already before assigning to it.
my $default_specs = {
    0 => {	
	2 => ['IDC', 'record', 2, ['biom:ImageReferenceIdentification',
				   'nc:IdentificationID'], \&dezero],
    },
    tagged => {
	fallback => 0,
	parts => {
	    cap_detail => ['', 'body', 4, 'biom:ImageCaptureDetail'],
	    cap_org => ['', 'cap_detail', 9, 'biom:CaptureOrganization'],
	    major_case => ['', 'body', 17,
			   'biom:FingerprintImageMajorCasePrint'],
	    udfs => ['', 'record', 3, 'array'],
	},
	
	900 => ['UDF', 'udfs', 1, 'ext:ExampleUserDefinedFields',
		\&make_xml_comment],

	902 => {
	    anns => ['', 'record', 4, 'array', 'one_per_record'],
	    ann => ['ANN', 'anns', 4, 'biom:ProcessAnnotation'],
	    a => ['GMT', 'ann', 1, ['biom:ProcessUTCDate', 'nc:DateTime'],
		  \&punctuate_utc],
	    b => ['NAV', 'ann', 2, 'biom:ProcessName'],
	    c => ['OWN', 'ann', 3, 'biom:ProcessOwnerText'],
	    d => ['PRO', 'ann', 4, 'biom:ProcessDescriptionText'],
	},
	995 =>  {
	    ascs => ['', 'record', 5, 'array', 'one_per_record'],
	    asc => ['ASC', 'ascs', 1, 'biom:AssociatedContext'],
	    a => ['ACN', 'asc', 1, ['biom:ContextIdentification', 
		  	   	     'nc:IdentificationID']],
	    b => ['ASP', 'asc', 2, ['biom:ImageSegmentIdentification',
				    'nc:IdentificationID'], \&omit_empty_esc],
	},
	996 => ['HAS', 'record', 6, 'biom:ImageHashValue'],
	997 =>  {
	    sors => ['', 'record', 7, 'array', 'one_per_record'],
	    sor => ['SOR', 'sors', 1, 'biom:SourceRepresentation'],
	    a => ['SRN', 'sor', 1, 'biom:SourceIdentification'],
	    b => ['RSP', 'sor', 2, ['biom:ImageSegmentIdentification',
				    'nc:IdentificationID']],
	},
	999 => ['DATA', 'body', 1, 'nc:BinaryBase64Object',
		\&read_and_encode_image],
	12 => ['BPX', 'body',  2, $bpx_tag],
	998 => {
	    geo => ['GEO', 'cap_detail', 1, 'biom:CaptureLocation'],
	    tdcoord => ['', 'geo', 3, 
			'biom:LocationTwoDimensionalGeographicCoordinate'],
	    lat => ['', 'tdcoord', 1, 'nc:GeographicCoordinateLatitude'],
	    long => ['', 'tdcoord', 2, 'nc:GeographicCoordinateLongitude'],
	    utm => ['', 'geo', 4, 'nc:LocationUTMCoordinate'],
	    ags => ['', 'geo', 5,
		    'biom:LocationAlternateGeographicSystemValue'],
	    a => ['UTE', 'cap_detail', 2, ['biom:CaptureUTCDateTime',
					   'nc:DateTime'], \&punctuate_utc],
	    b => ['LTD', 'lat', 1, 'nc:LatitudeDegreeValue', \&omit_empty],
	    c => ['LTM', 'lat', 2, 'nc:LatitudeMinuteValue', \&omit_empty],
	    d => ['LTS', 'lat', 3, 'nc:LatitudeSecondValue', \&omit_empty],
	    e => ['LGD', 'long', 1, 'nc:LongitudeDegreeValue', \&omit_empty],
	    f => ['LGM', 'long', 2, 'nc:LongitudeMinuteValue', \&omit_empty],
	    g => ['LGS', 'long', 3, 'nc:LongitudeSecondValue', \&omit_empty],
	    h => ['ELE', 'geo', 2, ['nc:LocationGeographicElevation',
				    'nc:MeasurePointValue'], \&omit_empty],
	    i => ['GDC', 'tdcoord', 3,
		  'biom:GeodeticDatumCoordinateSystemCode', \&omit_empty],
	    j => ['GCM', 'utm', 2, 'nc:UTMGridZoneID', \&omit_empty],
	    k => ['GCE', 'utm', 1, 'nc:UTMEastingValue', \&omit_empty],
	    l => ['GCN', 'utm', 3, 'nc:UTMNorthingValue', \&omit_empty],
	    'm' => ['GRT', 'geo', 1, 'nc:LocationDescriptionText', 
		    \&omit_empty_esc],
	    n => ['OSI', 'ags', 1, 'biom:GeographicLocationSystemName', \&xml_escape],
	    o => ['OCV', 'ags', 2, 'biom:GeographicLocationText', \&xml_escape],
	},
	5 => cap_date_spec('DAT'),
	903 => $cap_dui_spec,
	904 => $cap_mms_spec,	# cap_detail 5-7
	16 => $cap_shps_spec,	# cap_detail 8
	4 => ['SRC', 'cap_org', 1, ['nc:OrganizationIdentification',
				    'nc:IdentificationID']],
	993 => ['SAN', 'src', 2, 'nc:OrganizationName'],
	17 => $cap_svps_spec,	# cap_detail 10
	30 => $cap_dmm_spec,	# cap_detail 11
	
	11 => ['CGA', 'body',  7, $cga_tag],
	6 => ['HLL',  'body',  8, $hll_tag],
	9 => ['HPS',  'body',  9, $hps_tag],
	8 => ['SLC',  'body', 11, $slc_tag],
	7 => ['VLL',  'body', 13, $vll_tag],
	10 => ['VPS', 'body', 14, $vps_tag],
	3 => ['IMP',  'body', 15, $imp_tag],
	13 => {
	    fgps => ['', 'body', 16, 'array', 'one_per_record'],
	    rest => ['FGP', 'fgps', 1, $fgp_tag],
	},
	14 => $maj_ppd_spec,
	15 => $maj_ppc_spec,
	
	20 => $com_img_spec,
    },
    
    1 => {
	tag => 'itl:PackageInformationRecord',
	parts => {
	    body => ['', 'record', 10, 'biom:Transaction'],
	    res =>  ['', 'body',  9, 'biom:TransactionImageResolutionDetails'],
	    dest => ['', 'body',  2, 'biom:TransactionDestinationOrganization'],
	    src =>  ['', 'body',  3, 'biom:TransactionOriginatingOrganization'],
	},
	2 => ['VER', 'body', 10, \&set_type1_ver],
	3 => {
	    cnt =>  ['CNT', 'body', 14, 'biom:TransactionContentSummary',
		     'one_per_record'],
	    '1a' => ['FRC', 'cnt', 1, 'biom:ContentFirstRecordCategoryCode'],
	    '1b' => ['CRC', 'cnt', 2, 'biom:ContentRecordQuantity'],
	    crsum => ['', 'cnt', 3, 'biom:ContentRecordSummary'],
	    a => ['REC', 'crsum', 2, 'biom:RecordCategoryCode', \&dezero],
	    b => ['IDC', 'crsum', 1, ['biom:ImageReferenceIdentification',
				      'nc:IdentificationID'], \&dezero],
	},
	4 =>  ['TOT', 'body', 13, 'biom:TransactionCategoryCode'],
	5 =>  ['DAT', 'body',  1, ['biom:TransactionDate', 'nc:Date'], 
	       \&hyphenate_date],
	6 =>  ['PRY', 'body', 12, 'biom:TransactionPriorityValue'],
	7 =>  ['DAI', 'dest',  1, ['nc:OrganizationIdentification',
				   'nc:IdentificationID']],
	8 =>  ['ORI', 'src',   1, ['nc:OrganizationIdentification',
				   'nc:IdentificationID']],
	9 =>  ['TCN', 'body',  5, ['biom:TransactionControlIdentification',
				   'nc:IdentificationID'], \&xml_escape],
	10 => ['TCR', 'body',  6, 
	       ['biom:TransactionControlReferenceIdentification',
		'nc:IdentificationID'], \&xml_escape],
	11 => ['NSR', 'res',   1, 'biom:NativeScanningResolutionValue'],
	12 => ['NTR', 'res',   2, 'biom:NominalTransmittingResolutionValue'],
	13 => {
	    dom => ['DOM', 'body', 7, 'biom:TransactionDomain'],
	    a => ['DNM', 'dom', 2, 'biom:TransactionDomainName'],
	    b => ['DVN', 'dom', 1, ['biom:DomainVersionNumberIdentification',
				    'nc:IdentificationID'], \&omit_empty],
	},
	14 => ['UTC', 'body', 4, ['biom:TransactionUTCDate',
				  'nc:DateTime'], \&punctuate_utc],
	15 => {
	    dcs => ['DCS', 'body', 15, 'biom:TransactionCharacterSetDirectory'],
	    a => ['CSI', 'dcs', 2, 'biom:CharacterSetIndexCode'],
	    b => ['CSN', 'dcs', 1, 'biom:CharacterSetCommonNameCode'],
	    c => ['CSV', 'dcs', 3, ['biom:CharacterSetVersionIdentification',
				    'nc:IdentificationID']],
	},
	16 => {
	    apss => ['', 'body',  8, 'array', 'one_per_record'],
	    aps => ['APS', 'apss', 1, 'biom:TransactionApplicationProfile'],
	    a => ['APO', 'aps', 1, 'biom:ApplicationProfileOrganizationName'],
	    b => ['APN', 'aps', 2, 'biom:ApplicationProfileName'],
	    c => ['APV', 'aps', 3, 
		  ['biom:ApplicationProfileVersionIdentification',
		   'nc:IdentificationID']],
	},
	17 => {
	    a => ['DAN', 'dest', 2, 'nc:OrganizationName'],
	    b => ['OAN', 'src',  2, 'nc:OrganizationName'],
	},
    },

    2 => {
	tag => 'itl:PackageDescriptiveTextRecord',
	fallback => 0,
	parts => {
	    body => ['', 'record', 10, 'itl:UserDefinedDescriptiveDetail'],
	},
	
	3 => ['udf3', 'body', 1, 'ext:ExampleDomainDefinedDescriptiveDetail',
	      \&make_xml_comment],
	4 => ['udf4', 'body', 2, 'ext:ExampleOtherDescriptiveDetail',
	      \&make_xml_comment],

    },

    # type 3 is deprecated

    4 => {
	tag => 'itl:PackageHighResolutionGrayscaleImageRecord',
	fallback => 0,
	parts => {
	    body => ['', 'record', 10, 'biom:FingerprintImage'],
	},
	3 => ['IMP',  'body', 7, $imp_tag],
	4 => {
	    fgp => ['FGP', 'body',  6, 'biom:FingerprintImagePosition',
		    'accumulate_items'],
	    rest => ['FGP',  'fgp',  1, $fgp_tag, \&fgp_value_func],
	},
	5 => ['ISR',  'body', 2, ['biom:ImageCaptureDetail',
				  'biom:CaptureResolutionCode']],
	6 => ['HLL',  'body', 4, $hll_tag],
	7 => ['VLL',  'body', 5, $vll_tag],
	8 => ['CGA',  'body', 3, 'biom:ImageCompressionAlgorithmCode'],
	9 => ['DATA', 'body', 1, 'nc:BinaryBase64Object',
	      \&read_and_encode_image],
    },

    # types 5 and 6 are deprecated

    7 => {
	tag => 'itl:PackageUserDefinedImageRecord',
	fallback => 4,
    },

    8 => {
	tag => 'itl:PackageSignatureImageRecord',
	fallback => 0,
	parts => {
	    body => ['', 'record', 10, 'biom:SignatureImage'],
	    vecrep => ['DATA', 'body', 5,
		       'biom:SignatureImageVectorRepresentation'],
	},
	3 => ['SIG', 'body', 7, 'biom:SignatureCategoryCode'],
	4 => ['SRT', 'body', 6, 'biom:SignatureRepresentationCode'],
	5 => ['ISR', 'body', 2, ['biom:ImageCaptureDetail',
				 'biom:CaptureResolutionCode']],
	6 => ['HLL', 'body', 3, $hll_tag],
	7 => ['VLL', 'body', 4, $vll_tag],
	8 => {
	    '1a' => ['DATA', 'body', 1, \&set_type8_data],
	    vecs => ['', 'vecrep', 1, 'array', 'one_per_record'],
	    vec => ['', 'vecs', 1, 'biom:SignatureImageVector'],
	    a => ['', 'vec', 3, 'biom:VectorPositionHorizontalCoordinateValue'],
	    b => ['', 'vec', 2, 'biom:VectorPositionVerticalCoordinateValue'],
	    c => ['', 'vec', 1, 'biom:VectorPenPressureValue'],	       
	}, 
    },

    9 => {
	tag => 'itl:PackageMinutiaeRecord',
	fallback => 0,
	parts => {
	    body => ['', 'record', 10, \&set_type9_body_tag],

	    # NIST (deprecated)  no offset
	    nist => ['', 'body', 1, 'itl:MinutiaeNISTStandard'],
	    
	    # FBI IAFIS  offset +10 to avoid static tag collisions
	    fbi => ['', 'body', 12, 'ebts:MinutiaeFBIStandard'],
	    rov => ['ROV', 'body', 15, 'ebts:MinutiaePolygonalVerticesPositions'],
	    # INCITS M1  no offset
	    fiimg => ['', 'body', 5, 'biom:FingerImpressionImage'],
	    dins => ['', 'body', 14, 'array'],
    	    din => ['', 'dins', 1, 'biom:FingerprintPatternDeltaLocation',
		'one_per_subfield'],
	},
	3 => ['IMP', 'record', 3, 'biom:MinutiaeImpressionCaptureCategoryCode'],
	4 => ['FMT', 'record', 4, 'biom:MinutiaeFormatNISTStandardIndicator',
	      \&{sub {(shift eq 'S') ? 'true' : 'false';}}],

	# fields 5--12 NIST Standard Format Features
	5 => {
	    ofr => ['OFR', 'nist', 3, 'biom:MinutiaeReadingSystem'],	    
	    a => ['', 'ofr', 2, 'biom:ReadingSystemName'],
	    b => ['', 'ofr', 1, 'biom:ReadingSystemCodingMethodCode'],
	    c => ['', 'ofr', 3, ['biom:ReadingSystemSubsustemIdentification',
				 'nc:IdentificationID']],
	},
	6 => ['FGP', 'body', 5, \&set_pos_code],
	7 => {		 # doesn't handle multiple items within a subfield 
	    pats => ['', 'body', 4, 'array', 'one_per_subfield'],
	    pat => ['', 'pats', 1, 'itl:MinutiaeFingerPatternDetail'],
	    a => ['FPC', 'pat', 1, 'itl:FingerPatternCodeSourceCode'],
	    b => ['', 'pat', 2, 'biom:FingerPatternCode'],
	    c => ['', 'pat', 3, 'biom:FingerPatternText'],
	},
	8 => {
	    crps => ['', 'body', 2, 'array', 'one_per_record'],
	    crp => ['', 'crps', 1, 'biom:MinutiaeFingerCorePosition'],
	    a => ['CRP', 'crp', 1, \&set_xy_pos, 1],
	},
	9 => {
	    dlts => ['', 'body', 3, 'array', 'one_per_record'],
	    dlt => ['', 'dlts', 1, 'biom:MinutiaeFingerDeltaPosition'],
	    a => ['DLT', 'dlt', 1, \&set_xy_pos, 1],
	},
	10 => ['MIN', 'nist', 2, 'biom:MinutiaeQuantity'],
	11 => ['RDG', 'nist', 4, 'biom:MinutiaeRidgeCountIndicator'],
	12 => {
	    mrcs => ['', 'nist', 1, 'array', 'one_per_record'],
	    mrc => ['MRC', 'mrcs', 1, 'itl:MinutiaDetail'],
	    a => ['', 'mrc', 3, ['biom:MinutiaIdentification',
				 'nc:IdentificationID']],
	    b => ['', 'mrc', 1, \&set_xytheta_values, [1,3]],
	    c => ['', 'mrc', 5, 'biom:MinutiaQualityValue'],
	    d => ['', 'mrc', 6, 'biom:MinutiaCategoryCode'],
	    rest => ['', 'mrc', 7, \&set_ridge_count_values],
	},
	
	# fields 13--30 FBI IAFIS Features: body tags offset +20 to
	# avoid collisions with static tags for other feature types
	## 13 AFIS Feature Vector -- could replace the rest of the IAFIS fields
	14 => ['FGN', 'body', 11, 'biom:MinutiaeFingerPositionCode'],
	15 => ['NMN', 'fbi', 2, 'biom:MinutiaeQuantity'],
	16 => {
	    fcp => ['FCP', 'fbi', 3, 'ebts:MinutiaeReadingSystem'],
	    a => ['VEN', 'fcp', 1, 'biom:ReadingSystemName'],
	    b => ['VID', 'fcp', 2, ['biom:ReadingSystemSubsystemIdentification',
		  		     'nc:IdentificationID']],
	    c => ['MET', 'fcp', 3, 'ebts:ReadingSystemCodingMethodCode'],
	},
	17 => {
	    apcs => ['', 'body', 18, 'array', 'one_per_record'],
	    apc => ['APC', 'apcs', 1, 'ebts:MinutiaeFingerPattern'],
	    a => ['APAT', 'apc', 1,
		  'ebts:FingerprintPatternClassificationCode'],
	    b => ['RCN1', 'apc', 2, 'biom:RidgeCountValue'],
	    c => ['RCN2', 'apc', 3, 'biom:RidgeCountValue'],
	},
	18 => {
	    xyp => ['', 'rov', 1, 'biom:PositionPolygonVertex'],
	    a => ['XYP', 'xyp', 1, \&set_xy_pos, 1],
	},
	19 => {
	    cof => ['COF', 'body', 14, 'ebts:MinutiaCoordinateOffsets'],
	    xypa => ['', 'cof', 1, 'ebts:OffsetUpperLeftCoordinates'],
	    a => ['XYP', 'xypa', 1, \&set_xy_pos, 1],
	    xypb => ['', 'cof', 2, 'ebts:OffsetCenterOfRotation'],
	    b => ['XYP', 'xypb', 1, \&set_xy_pos, 1],
	    c => ['THET', 'cof', 3, 'biom:PositionThetaAngleMeasure'],
	    xypd => ['', 'cof', 4, 'ebts:OffsetTranslatedCenterOfRotation'],
	    d => ['XYP', 'xypd', 1, \&set_xy_pos, 1],
	    xype => ['', 'cof', 5, 'ebts:OffsetTranslatedUpperLeftCoordinates'],
	    e => ['XYP', 'xype', 1, \&set_xy_pos, 1],
	    
	},
	20 => ['ORN', 'body', 21, 'biom:PositionUncertaintyValue'],
	21 => {
	    cras => ['', 'body', 16, 'array', 'one_per_record'],
	    cra => ['CRA', 'cras', 1,
		    'ebts:MinutiaeFingerCoreAttributePosition'],
	    a => ['XYM', 'cra', 2, \&set_xy_pos, 2],
	    b => ['DID', 'cra', 1, 'biom:PositionDirectionDegreeValue'],
	    c => ['PUM', 'cra', 3, 'biom:PositionUncertaintyValue'],
	},
	22 => {
	    dlas => ['', 'body', 17, 'array', 'one_per_record'],
	    dla => ['DLA', 'dlas', 1,
		    'ebts:MinutiaeFingerDeltaAttributePosition'],
	    a => ['XYM', 'dla', 4, \&set_xy_pos, 2],
	    b => ['DID', 'dla', 1, 'biom:PositionDirectionDegreeValue'],
	    c => ['DID', 'dla', 2, 'biom:PositionDirectionDegreeValue'],
	    d => ['DID', 'dla', 3, 'biom:PositionDirectionDegreeValue'],
	    e => ['PUM', 'dla', 5, 'biom:PositionUncertaintyValue'],
	},
	23 => {
	    mats => ['', 'fbi', 1, 'array', 'one_per_record'],
	    mat => ['MAT', 'mats', 1, 'ebts:MinutiaDetail'],
	    a => ['MDX', 'mat',  1,  ['biom:MinutiaIdentification',
		  	    	     'nc:IdentificationID'], \&dezero],
	    b => ['XYT', 'mat',  2, \&set_xytheta_values, [1,3]],
	    c => ['QMS', 'mat',  6, 'biom:MinutiaQualityValue'],
	    d => ['MNT', 'mat', 15, 'ebts:MinutiaTypeCode'],
	    e => ['MRO', 'mat',  7, \&set_iafis_ridge_count],
	    f => ['MRO', 'mat',  8, \&set_iafis_ridge_count],
	    g => ['MRO', 'mat',  9, \&set_iafis_ridge_count],
	    h => ['MRO', 'mat', 10, \&set_iafis_ridge_count],
	    i => ['MRO', 'mat', 11, \&set_iafis_ridge_count],
	    j => ['MRO', 'mat', 12, \&set_iafis_ridge_count],
	    k => ['MRO', 'mat', 13, \&set_iafis_ridge_count],
	    l => ['MRO', 'mat', 14, \&set_iafis_ridge_count],
	    'm' => ['RSO', 'mat',  7, \&set_iafis_octant_residuals],
	},

	# fields 126--150 INCITS M1-378 Features.
	126 => {
	    a => ['CFO', 'body', 1, ['biom:CBEFFFormatOwnerIdentification',
				     'nc:IdentificationID']],
	    b => ['CFT', 'body', 2, ['biom:CBEFFFormatCategoryIdentification',
				     'nc:IdentificationID']],
	    c => ['CPI', 'body', 3, ['biom:CBEFFProductIdentification',
				     'nc:IdentificationID']],
	},
	127 => {
	    cei => ['CEI', 'body', 4, 'biom:ImageCaptureDetail'],
	    a => ['AFS', 'cei', 2, 'biom:CaptureDeviceCertificationCode'],
	    b => ['CID', 'cei', 1, ['biom:CaptureDeviceIdentification',
				    'nc:IdentificationID']],
	},
	128 => ['HLL', 'fiimg', 1, 'biom:ImageHorizontalLineLengthPixelQuantity'],
	129 => ['VLL',  'fiimg', 4, 'biom:ImageVerticalLineLengthPixelQuantity'],
	130 => ['SLC', 'fiimg', 3, 'biom:ImageScaleUnitsCode'],
	131 => ['THPS', 'fiimg', 2, 'biom:ImageHorizontalPixelDensityValue'],
	132 => ['TVPS', 'fiimg', 5, 'biom:ImageVerticalPixelDensityValue'],
	133 => ['FVW', 'body', 6, 'biom:FingerViewNumeric'],
	134 => ['FGP', 'fiimg', 6, 'biom:FingerPositionCode'],
	135 => {
	    fqds => ['', 'body', 7, 'array', 'one_per_record'],
	    fqd => ['FQD', 'fqds', 1, 'biom:MinutiaeQuality'],
	    a => ['QVU', 'fqd', 2, 'biom:QualityValue'],
	    b => ['QAV', 'fqd', 3, ['biom:QualityMeasureVendorIdentification',
		  	   	     'nc:IdentificationID'], \&omit_empty],
	    c => ['QAP', 'fqd', 1, ['biom:QualityAlgorithmProductIdentification',
				 'nc:IdentificationID'], \&omit_empty],
	},
	136 => ['NOM', 'body', 8, 'biom:MinutiaeQuantity'],
	137 => {
	    fmds => ['', 'body', 9, 'array', 'one_per_record'],
	    fmd =>  ['FMD', 'fmds', 1, 'biom:INCITSMinutia'],
	    imloc => ['', 'fmd', 2, 'biom:INCITSMinutiaLocation'],
	    a => ['MAN', 'fmd', 1, ['biom:MinutiaIdentification',
				    'nc:IdentificationID']],
	    b => ['MXC', 'imloc', 1, 'biom:PositionHorizontalCoordinateValue'],
	    c => ['MYC', 'imloc', 2, 'biom:PositionVerticalCoordinateValue'],
	    d => ['MAV', 'imloc', 3, 'biom:ImageLocationThetaAngleMeasure'],
	    e => ['M1M', 'fmd', 3, 'biom:INCITSMinutiaCategoryCode'],
	    f => ['QOM', 'fmd', 4, 'biom:MinutiaQualityValue'],
	},
	138 => {
	    rci => ['RCI', 'body', 12, 'biom:MinutiaeRidgeCountDetail',
		    'one_per_record'],
	    mrci => ['', 'rci', 2, 'biom:MinutiaeRidgeCountItem'],
	    '1a' => ['REM', 'rci', 1, 'biom:INCITSRidgeCountAlgorithmCode'],
	    '1b' => 'ignore', # FI1
	    '1c' => 'ignore', # FI2
	    a => ['CMI', 'mrci', 1, ['biom:MinutiaIdentification',
		  	    	      'nc:IdentificationID']],
	    b => ['NMN', 'mrci', 2, ['biom:MinutiaReferenceIdentification',
		  	    	      'nc:IdentificationID']],
	    c => ['NRC', 'mrci', 3, 'biom:RidgeCountValue'],
	},
	139 => {
	    cins => ['', 'body', 13, 'array', 'one_per_record'],
	    cin => ['CIN', 'cins', 1, 'biom:FingerprintPatternCoreLocation'],
	    a => ['XCC', 'cin', 1, 'biom:PositionHorizontalCoordinateValue'],
	    b => ['YCC', 'cin', 2, 'biom:PositionVerticalCoordinateValue'],
	    c => ['ANGC', 'cin', 3, 'biom:ImageLocationThetaAngleMeasure'],
	},
	140 => {
	    a => ['XCD', 'din', 1, 'biom:PositionHorizontalCoordinateValue'],
	    b => ['YCD', 'din', 2, 'biom:PositionVerticalCoordinateValue'],
	    c => ['ANG1', 'din', 3, 'biom:ImageLocationThetaAngleMeasure'],
	},
	141 => {
	    a => ['ANG2', 'din', 4, 'biom:ImageLocationThetaAngleMeasure'],
	    b => ['ANG3', 'din', 5, 'biom:ImageLocationThetaAngleMeasure'],
	},

	901 => {
	    ulas => ['', 'record', 7, 'array', 'one_per_record'],
	    a => ['ULA', 'ulas', 1, 
		  'biom:MinutiaeUniversalLatentWorkstationAnnotationText'],
	},
    },

    10 => {
	tag => 'itl:PackageFacialAndSMTImageRecord',
	fallback => 'tagged',
	parts => {
	    body => ['', 'record', 10, \&set_type10_body_tag],
	    pfdds => ['', 'body', 35, 'array'],
	    pfdd => ['', 'pfdds', 1, 'biom:PhysicalFeatureDescriptionDetail'],
	},
	# 2 -- from tagged
	3 => ['IMT', 'body', 12, 'biom:ImageCategoryCode'],
	5 => cap_date_spec('PHD'),
	# 6-8 -- from tagged
	9 =>  ['THPS', 'body', 9, $hps_tag],
	10 => ['TVPS', 'body', 14, $vps_tag],
	# 11 -- from tagged
	12 => ['CSP',  'body', 5, $csp_tag],
	13 => ['SAP',  'body', 17, 'biom:FaceImageAcquisitionProfileCode'],
	14 => {
	    fip => ['FIP', 'body', 27, 'biom:FaceImageBoundingSquare'],
	    a => ['LHC', 'fip', 3, 'biom:SegmentLevtHorizontalCoordinateValue'],
	    b => ['RHC', 'fip', 4, 'biom:SegmentRightHorizontalCoordinateValue'],
	    c => ['TVC', 'fip', 5, 'biom:SegmentTopVerticalCoordinateValue'],
	    d => ['BVC', 'fip', 2, 'biom:SegmentBottomVerticalCoordinateValue'],
	    e => ['BBC', 'fip', 1, 'biom:FaceImageBoundingCategoryCode'],
	},
	15 => {
	    fpfi => ['FPFI', 'body', 28, 'biom:FaceImageBoundary'],
	    a => ['BYC', 'fpfi', 1, 'biom:FaceImageBoundaryShapeCode'],
	    b => ['NOP', 'fpfi', 2, 'biom:PositionPolygonVertexQuantity'],
	    ppv => ['', 'fpfi', 3, 'biom:PositionPolygonVertex',
		    'accumulate_pairs', 1],
	    odd => ['HPO', 'ppv', 1, 'biom:PositionHorizontalCoordinateValue'],
	    even => ['VPO', 'ppv', 2, 'biom:PositionVerticalCoordinateValue'],
	},
	# 16 & 17 -- from tagged
	18 => {
	    dist => ['DIST', 'body', 15, 'biom:ImageDistortion'],
	    a => ['IDK', 'dist', 1, 'biom:ImageDistortionCategoryCode'],
	    b => ['IDM', 'dist', 2, 'biom:ImageDistortionMeasurementCode'],
	    c => ['DSC', 'dist', 3, 'biom:ImageDistortionSeverityCode'],
	},
        19 => {
	    laf => ['', 'body', 29, 'array'],
	    a => ['LAF', 'laf', 1, 'biom:FaceImageLightingArtifactsCode'],
	},
	20 => ['POS', 'body', 25, 'biom:FaceImageSubjectPoseCode'],
	21 => ['POA', 'body', 24, 'biom:FaceImagePoseOffsetAngleMeasure'],
	22 => {
	    pxss => ['', 'body', 18, 'array', 'one_per_record'],
	    pxs => ['PXS', 'pxs', 1, 'biom:FaceImageAttribute'],
	    a => ['', 'pxs', 1, 'biom:FaceImageAttributeCode'],
	    b => ['', 'pxs', 2, 'biom:FaceImageAtrributeText', \&omit_empty],
	},
	23 => {
	    pas => ['PAS', 'body', 26, 'itl:FaceImageAcquisitionSource'],
	    a => ['PAC', 'pas', 1, 'biom:CaptureSourceCode'],
	    b => ['VSD', 'pas', 2, 'biom:CaptureSourceDescriptionText', 
		  \&omit_empty],
	},
	24 => {
	    sqss => ['', 'body', 10, 'array', 'one_per_record'],
	    sqs => ['SQS', 'sqss',  1, 'biom:ImageQuality'],
	    a => ['QAV', 'sqs', 2, 'biom:QualityValue'],
	    b => ['QVU', 'sqs', 3, ['biom:QualityMeasureVendorIdentification',
		         	     'nc:IdentificationID']],
	    c => ['QAP', 'sqs', 1, ['biom:QualityAlgorithmProductIdentification',
			            'nc:IdentificationID']],
	},
	25 => {
	    spa => ['SPA', 'body', 16, 'biom:FaceImage3DPoseAngle'],
	    a => ['', 'spa', 3, 'biom:PoseYawAngleMeasure'],
	    b => ['', 'spa', 1, 'biom:PosePitchAngleMeasure'],
	    c => ['', 'spa', 2, 'biom:PoseRollAngleMeasure'],
	},
	26 => ['SXS', 'body', 19, \&face_image_description],
	27 => ['SEC', 'body', 21, 'biom:FaceImageEyeColorAttributeCode'],
	28 => ['SHC', 'body', 23, 'biom:FaceImageHairColorAttributeCode'],
	29 => {
	    ffps => ['', 'body', 22, 'array', 'one_per_record'],
	    ffp => ['FFP', 'ffps', 1, 'biom:FaceImageFeaturePoint'],
	    # a-d not yet implemented
	},
	# 30 -- from tagged
	# 31-33 -- not yet implemented
	# 34-37 -- reserved
	38 => $com_img_spec,
	# 39 -- not yet implemented
	40 => ['SMT', 'body', 36, 'biom:PhysicalFeatureNCICCode'],
	41 => {
	    sms => ['SMS', 'body', 37, 'biom:PhysicalFeatureSize'],
	    a => ['', 'sms', 1, 'biom:PhysicalFeatureHeightMeasure'],
	    b => ['', 'sms', 2, 'biom:PhysicalFeatureWidthMeasure'],
	},
	42 => {
	    a => ['SMI', 'pfdd', 3, 'biom:PhysicalFeatureCategoryCode'],
	    b => ['TAC', 'pfdd', 4, 'biom:PhysicalFeatureClassCode',
		  \&omit_empty],
	    c => ['TSC', 'pfdd', 6, 'biom:PhysicalFeatureSubClassCode',
		  \&omit_empty],
	    d => ['TDS', 'pfdd', 5, 'biom:PhysicalFeatureDescriptionText',
		  \&omit_empty],
	},
	43 => {
	    col => ['COL', 'pfdd', 1, 'biom:PhysicalFeatureColorDetail'],
	    a => ['TC1', 'col', 1, 'biom:PhysicalFeaturePrimaryColorCode'],
	    b => ['TC2', 'col', 2, 'biom:PhysicalFeatureSecondaryColorCode',
		  \&omit_empty],
	    c => ['TC3', 'col', 3, 'biom:PhysicalFeatureSecondaryColorCode',
		  \&omit_empty],
	    d => ['TC4', 'col', 4, 'biom:PhysicalFeatureSecondaryColorCode',
		  \&omit_empty],
	    e => ['TC5', 'col', 5, 'biom:PhysicalFeatureSecondaryColorCode',
		  \&omit_empty],
	    f => ['TC6', 'col', 6, 'biom:PhysicalFeatureSecondaryColorCode',
		  \&omit_empty],
	},
	# 44-45 -- not yet implemented
	# 46-199 -- reserved
    },

    # types 11 and 12 are reserved

    13 => {
	tag => 'itl:PackageLatentImageRecord',
	fallback => 'tagged',
	parts => {
	    # The body tag might need to be changed, e.g., if FGP is a palm
	    # position instead of a finger, but the contents remain the same.
	    body => ['', 'record', 10, 'itl:FingerprintImage'],
	},
	# 2-4 -- from tagged
	5 => cap_date_spec('LCD'),
	# 6-13 -- from tagged
	14 => $maj_spd_spec,
	# 15-17 -- from tagged
	# 18-19 -- reserved
	# 20 -- from tagged
	# 21-13 -- reserved
	24 => {
	    lqms => ['', 'body', 18, 'array', 'one_per_record'],
	    lqm => ['LQM', 'lqms', 1, \&get_type13_quality_tag],
	    a => ['FRMP', 'lqm', 1, \&set_type13_quality_fgp],
	    b => ['QVU', 'lqm', 4, 'biom:QualityValue'],
	    c => ['QAV', 'lqm', 3,
		  ['biom:QualityAlgorithmVendorIdentification',
		   'nc:IdentificationID']],
	    d => ['QAP', 'lqm', 2,
		  ['biom:QualityAlgorithmProductIdentification',
		   'nc:IdentificationID']],
	},
	# 25-199 -- reserved
	# 900s -- from tagged
    },

    14 => {
	tag => 'itl:PackageFingerprintImageRecord',
	fallback => 'tagged',
	parts => {
	    body => ['', 'record', 10, 'biom:FingerImpressionImage'],
	},
	# 2 -- from tagged
	# 4 -- from tagged
	5 => cap_date_spec('FCD'),
	# 6-17 -- from tagged
	18 => { amp => ['AMP', 'body', 18,
	    'biom:FingerprintImageFingerMissing'], a => ['FRAP',
	    'amp', 1, 'biom:FingerPositionCode'], b => ['ABC', 'amp',
	    2, 'biom:FingerMissingCode'],
	},
	# 19 -- reserved
	# 20 -- from tagged
	21 => {
	    seg => ['SEG', 'body', 19, 
		    'biom:FingerprintImageSegmentPositionSquare'],
	    a => ['FRSP', 'seg', 1, 'biom:FingerPositionCode'],
	    b => ['LHC',  'seg', 3, 'biom:SegmentLeftHorizontalCoordinateValue'],
	    c => ['RHC',  'seg', 4, 'biom:SegmentRightHorizontalCoordinateValue'],
	    d => ['TVC',  'seg', 5, 'biom:SegmentTopVerticalCoordinateValue'],
	    e => ['BVC',  'seg', 2, 'biom:SegmentBottomVerticalCoordinateValue'],
	},
	22 => {
	    nqm => ['NQM', 'body', 20, 'biom:FingerprintImageNISTQuality'],
	    a => ['FRNP', 'nqm', 1, 'biom:FingerPositionCode'],
	    b => ['IQS', 'nqm', 2, 'biom:NISTQualityMeasure'],
	},
	23 => {
	    sqm => ['SQM', 'body', 21,
		    'biom:FingerprintImageSegmentationQuality'],
	    a => ['FRQP', 'sqm', 1, 'biom:FingerPositionCode'],
	    b => ['QVU', 'sqm', 4, 'biom:FingerprintImage'],
	    c => ['QAV', 'sqm', 3, ['biom:QualityAlgorithmVendorIdentification',
				    'nc:IdentificationID']],
	    d => ['QAP', 'sqm', 2, ['biom:QualityAlgorithmProductIdentification',
				    'nc:IdentificationID']],
	},
	24 => {
	    fqm => ['FQM', 'body', 22, 'biom:FingerprintImageQuality'],
	    a => ['FRMP', 'fqm', 1, 'biom:FingerPositionCode'],
	    b => ['QVU', 'fqm', 4, 'biom:QualityValue'],
	    c => ['QAV', 'fqm', 3, ['biom:QualityAlgorithmVendorIdentification',
				    'nc:IdentificationID']],
	    d => ['QAP', 'fqm', 2, ['biom:QualityAlgorithProductIdentification',
				    'nc:IdenticationID']],
	},
	25 => {
	    aseg => ['ASEG', 'body', 23,
		     'biom:FingerprintImageSegmentationPositionPolygon'],
	    a => ['FRAS', 'aseg', 1, 'biom:FingerPositionCode'],
	    b => ['NOP', 'aseg', 2, 'biom:PositionPolygonVertexQuantity'],
	    ppv => ['', 'aseg', 3, 'biom:PositionPolygonVertex',
		    'accumulate_pairs', 1],
	    odd => ['HPO', 'ppv', 1, 'biom:PositionHorizontalCoordinateValue'],
	    even => ['VPO', 'ppv', 2, 'biom:PositionVerticalCoordinateValue'],
	},
	26 => ['SCF', 'body', ],
	27 => ['SIF', ],
	# 28-29 -- reserved
	# 30 -- from tagged
	31 => ['FAP', 'body', 24, ],
	# 32-199 -- reserved
	# 200-900 -- user defined
	# 901 -- reserved
	# 902-999 -- tagged
    },
    
    15 => {
	tag => 'itl:PackagePalmprintImageRecord',
	fallback => 'tagged',
	parts => {
	    body => ['', 'record', 10, 'biom:PalmprintImage'],
	},
	# 2-4 -- from tagged
	5 => cap_date_spec('PCD'),
	# 6-12 -- from tagged
	13 => ['FGP', 'body', 16, $plp_tag],
	# 14-15 -- reserved
	# 16-17 -- from tagged
	18 => {
	    amps => ['', 'body', 18, 'array', 'one_per_record'],
	    amp => ['AMP', 'amps', 1, 'biom:PalmprintImageMissingArea'],
	    a => ['FRAP', 'amp', 1, 'biom:PalmPositionCode'],
	    b => ['ABC', 'amp', 2, 'biom:PalmMissingCode'],
	},
	# 19 -- reserved
	# 20 -- from tagged
	# 21-23 -- reserved
	24 => {
	    pqms => ['', 'body', 22, 'array', 'one_per_record'],
	    pqm => ['PQM', 'pqms', 1, 'biom:PalmprintImageQuality'],
	    a => ['FRMP', 'pqm', 1, 'biom:PalmPositionCode'],
	    b => ['QVU', 'pqm', 3, 'biom:QualityValue'],
	    c => ['QAV', 'pqm', 4, ['biom:QualityAlgorithmVendorIdentification',
				    'nc:IdentificationID']],
	    d => ['QAP', 'pqm', 2, ['biom:QualityAlgorithProductIdentification',
				    'nc:IdenticationID']],
	},
	# 25-29 -- reserved
	# 30 -- from tagged
	# 31-199 -- reserved
	# 200-900 -- user defined
	# 901 -- reserved
	# 902-999 --  from tagged
    },

    16 => {
	tag => 'itl:PackageUserDefinedTestingImageRecord',
	fallback => 'tagged',
	parts => {
	    body => ['', 'record', 10, 'biom:TestImage'],
	},
	24 => {
	    uqs => ['UQS', 'body', 10, 'biom:ImageQuality'],
	    a => ['QVU', 'uqs', 2, 'biom:QualityValue'],
	    b => ['QAV', 'uqs', 3, ['biom:QualityAlgorithmVendorIdentification',
				    'nc:IdentificationID']],
	    c => ['QAP', 'uqs', 1, ['biom:QualityAlgorithmProductIdentification',
				    'nc:IdentificationID']],
	},
    },

    17 => {
	tag => 'itl:PackageIrisImageRecord',
	fallback => 'tagged',
	parts => {
	    body => ['', 'record', 10, 'biom:IrisImage'],
	    aqls => ['', 'body', 20, 'biom:IrisImageAcquisitionLightingSpectrum'],
	},
	# 2 -- from tagged
	3 => ['ELR', 'body', 15, 'biom:IrisEyePositionCode'],
	# 4-12 -- from tagged
	13 => ['CSP', 'body', 5, 'biom:ImageColorSpaceCode'],
	14 => ['RAE', 'body', 16, 'biom:IrisEyeRotationAngleText'],
	15 => ['RAU', 'body', 17, 'biom:IrisEyeRotationUncertaintyValueText'],
	16 => {
	    ipc => ['IPC', 'body', 18, 'biom:IrisImageCapture'],
	    a => ['IHO', 'ipc', 1, 'biom:IrisImageHorizontalOrientationCode'],
	    b => ['IVO', 'ipc', 3, 'biom:IrisImageVerticalOrientationCode'],
	    c => ['IST', 'ipc', 2, 'biom:IrisImageScanCategoryCode'],
	},
	17 => $cap_dui_spec,
	# 18 -- deprecated
	19 => $cap_mms_spec,
	20 => ['ECL', 'body', 19, 'biom:IrisEyeColorAttributeCode'],
	21 => $com_img_spec,
	22 => $cap_shps_spec,
	23 => $cap_svps_spec,
	24 => {
	    iqs => ['IQS', 'body', 10, 'biom:ImageQuality'],
	    a => ['QVU', 'iqs', 2, 'biom:QualityValue'],
	    b => ['QAV', 'iqs', 3, ['biom:QualityAlgorithmVendorIdentification',
				    'nc:IdentificationID']],
	    c => ['QAP', 'iqs', 1, ['biom:QualityAlgorithmProductIdentification',
				    'nc:IdentificationID']],
	},
	25 => ['EAS', 'aqls', 1, 'biom:AcquisitionLightingSpectrumCode'],
	26 => ['IRD', 'body', 21, 'biom:IrisDiameterPixelQuantity'],
	27 => {
	    a => ['LOW', 'aqls', 2,
		  'biom:AcquisitionLightingSpectrumLowerMeasure'],
	    b => ['HIG', 'aqls', 3,
		  'biom:AcquisitionLightingSpectrumUpperMeasure'],
	},
	28 => ['DME', 'body', 22, 'biom:IrisImageMissingReasonCode'],
	
	30 => $cap_dmm_spec,
	31 => ['IAP', 'body', 23, 'biom:IrisImageAcquisitionProfileCode'],
	32 => ['ISF', 'body', 24, 'biom:IrisImageStorageFormatCode'],
	33 => {
	    ipb => ['IPB', 'body', 25, 'biom:IrisImageIrisPupilBoundary'],
	    a => ['BYC', 'ipb', 1, 'biom:IrisBoundaryShapeCode'],
	    b => ['NOP', 'ipb', 2, 'biom:ImageFeatureVertexQuantity'],
	    pifv => ['', 'ipb', 3, 'biom:ImageFeatureVertex',
		     'accumulate_pairs', 1],
	    odd => ['HPO', 'pifv', 1, 'biom:PositionHorizontalCoordinateValue'],
	    even => ['VPO', 'pifv', 2, 'biom:PositionVerticalCoordinateValue'],
	},
	34 => {
	    isb => ['ISB', 'body', 26, 'biom:IrisImageIrisScleraBoundary'],
	    a => ['BYC', 'isb', 1, 'biom:IrisBoundaryShapeCode'],
	    b => ['NOP', 'isb', 2, 'biom:ImageFeatureVertexQuantity'],
	    isifv => ['', 'isb', 3, 'biom:ImageFeatureVertex',
		      'accumulate_pairs', 1],
	    odd => ['HPO', 'isifv', 1, 'biom:PositionHorizontalCoordinateValue'],
	    even => ['VPO', 'isifv', 2, 'biom:PositionVerticalCoordinateValue'],
	},
	35 => {
	    ueb => ['UEB', 'body', 27, 'biom:IrisImageIrisUpperEyelidBoundary'],
	    a => ['BYC', 'ueb', 1, 'biom:IrisBoundaryShapeCode'],
	    b => ['NOP', 'ueb', 2, 'biom:ImageFeatureVertexQuantity'],
	    ueifv => ['', 'ueb', 3, 'biom:ImageFeatureVertex',
		      'accumulate_pairs', 1],
	    odd => ['HPO', 'ueifv', 1, 'biom:PositionHorizontalCoordinateValue'],
	    even => ['VPO', 'ueifv', 2, 'biom:PositionVerticalCoordinateValue'],
	},
	36 => {
	    leb => ['LEB', 'body', 28, 'biom:IrisImageIrisLowerEyelidBoundary'],
	    a => ['BYC', 'leb', 1, 'biom:IrisBoundaryShapeCode'],
	    b => ['NOP', 'leb', 2, 'biom:ImageFeatureVertexQuantity'],
	    leifv => ['', 'leb', 3, 'biom:ImageFeatureVertex',
		      'accumulate_pairs', 1],
	    odd => ['HPO', 'leifv', 1, 'biom:PositionHorizontalCoordinateValue'],
	    even => ['VPO', 'leifv', 2, 'biom:PositionVerticalCoordinateValue'],
	},
	37 => {
	    neos => ['', 'body', 29, 'array', 'one_per_record'],
	    neo => ['NEO', 'neos', 1, 'biom:IrisImageOcclusion'],
	    a => ['OCY', 'neo', 1, 'biom:IrisImageOcclusionOpacityCode'],
	    b => ['OCT', 'neo', 2, 'biom:IrisImageOcclusionCategoryCode'],
	    c => ['NOP', 'neo', 3, 'biom:ImageFeatureVertexQuantity'],
	    neifv => ['', 'neo', 4, 'biom:ImageFeatureVertex', 
		      'accumulate_pairs', 0],
	    even => ['HPO', 'neifv', 1, 'biom:PositionHorizontalCoordinateValue'],
	    odd => ['VPO', 'neifv', 2, 'biom:PositionVerticalCoordinateValue'],
	},
	# 38-39 -- reserved
	40 => ['RAN', 'body', 30, 'biom:IrisImageRangeMeasure'],
	41 => ['GAZ', 'body', 31, 'biom:IrisImageGazeAngleMeasure'],
    },

    18 => {
	tag => 'itl:PackageDNARecord',
	fallback => 'tagged',
	parts => {
	    body => ['', 'record', 10, 'biom:DNASample'],
	    # The capture detail tag is different for type 18
	    cap_detail => ['', 'body', 4, 'biom:BiometricCaptureDetail'],
	},
	
	3 => {
	    dls => ['DLS', 'body', 5, 'biom:DNALaboratory', 'one_per_record'],
	    a => ['UTY', 'dls', 3, 'biom:DNALaboratoryUnitCategoryCode'],
	    b => ['LTY', 'dls', 4, 'biom:DNALaboratoryCategoryCode'],
	    c => ['ACC', 'dls', 5, \&set_accredation],
	    d => ['NOO', 'dls', 1, 'nc:OrganizationName'],
	    e => ['POC', 'dls', 2, ['nc:OrganizationPrimaryContactInformation',
		  	   	    'nc:ContactInformationDescriptionText'],
		  \&xml_escape],
	    f => ['CSC', 'dls', 6, \&set_country_code],
	    g => ['ION', 'dls', 7, 
		  'biom:DNALaboratoryInternationalOrganizationName'],
	},
	5 => ['NAL', 'body', 6, 'biom:DNAAnalysisQuantityCode'],
	6 => {
	    sdi => ['SDI', 'body', 7, 'biom:DNADonor'],
	    a => ['DSD', 'sdi', 4, 'biom:DNADonorCategoryCode'],
	    b => ['GID', 'sdi', 3, 'nc:PersonSexCode'],
	    c => ['DLC', 'sdi', 5, ['biom:DNADonorLastContactDate', 'nc:Date'],
		  \&hyphenate_date],
	    d => ['DOB', 'sdi', 1, ['nc:PersonBirthDate', 'nc:Date'],
		  \&hyphenate_date],
	    e => ['EGP', 'sdi', 2, 'nc:PersonEthnicityText', \&xml_escape],
	    f => ['DRA', 'sdi', 6, 'biom:DNADonorDentalRecordsAvailableCode'],
	    g => ['LLC', 'sdi', 7,
		  'biom:DNADonorCollectionLocationDescriptionText',
		  \&xml_escape],
	    h => ['SDS', 'sdi', 8, 'biom:DNADonorStatusCode'],
	},
	7 => ['COPR', 'body', 8, 'biom:DNAClaimedRelationshipCode'],
	8 => ['VRS', 'body', 9, 'biom:DNAValidatedRelationshipCode'],
	9 => {
	    ped => ['PED', 'body', 10, 'biom:DNAPedigree'], 
	    a => ['PID', 'ped', 1, ['biom:DNAPedigreeIdentification',
				    'nc:IdentificationID']],
	    b => ['PMI', 'ped', 2, ['biom:DNAPedigreeMemberIdentification',
				    'nc:IdentificationID']],
	    c => ['PMS', 'ped', 3, 'biom:DNAPedigreeMemberStatusCode'],
	    d => ['SID', 'ped', 4, ['biom:DNAPedigreeSampleIdentification',
				    'nc:IdentificationID']],
	    e => ['FID', 'ped', 5, ['biom:DNAPedigreeFatherIdentification',
				    'nc:IdentificationID']],
	    f => ['MID', 'ped', 6, ['biom:DNAPedigreeMotherIdentification',
				    'nc:IdentificationID']],
	    g => ['PCM', 'ped', 7, 'biom:DNAPedigreeCommentText', \&xml_escape],
	},
	10 => {
	    sty => ['STY', 'body', 11, 'biom:DNASampleOrigin'],
	    a => ['SCT', 'sty', 1, 'biom:DNACellularCategoryCode'],
	    b => ['SMO', 'sty', 2, 'biom:DNASampleOriginCode'],
	},
	11 => {
	    stis => ['', 'body', 12, 'array', 'one_per_record'],
	    a => ['STI', 'stis', 1, 'biom:DNATypingTechnologyCategoryCode'],
	},
	12 => ['SCM', 'body', 13, 'biom:DNASampleCollectionMethodText',
	       \&xml_escape],
	13 => ['SCD', 'cap_detail', 3, ['biom:CaptureDate', 'nc:DateTime'],
	       \&punctuate_utc],
	14 => ['PSD', 'body', 14, ['biom:DNAProfileStorageDate', 'nc:DateTime'],
	       \&punctuate_utc],
	15 => {
	    dpd => ['DPD', 'body', 15, 'biom:DNAProfile'],
	    a => ['PTP', 'dpd', 1, 'biom:DNAProfileCategoryCode'],
	    b => ['RES', 'dpd', 2, 'biom:DNAProfileResultCode', \&omit_empty],
	    c => ['PRF', 'dpd', 3, ['biom:DNAProfileIdentification',
				    'nc:IdentificationID']],
	    d => ['SUP', 'dpd', 4, 'biom:DNAProfileSupplementalText',
		  \&omit_empty_esc],
	    e => ['DPC', 'dpd', 5, 'biom:DNAProfileCommentText', 
		  \&omit_empty_esc],
	},
	16 => {
	    strs => ['', 'body', 16, 'array', 'one_per_record'],
	    'str' => ['STR', 'strs', 16, 'biom:DNASTRProfile'],
	    a => ['DST', 'str', 1, 'biom:DNASTRProfileCategoryCode'],
	    b => ['DLR', 'str', 2, ['biom:DNALocusIdentification', 
				    'nc:IdentificationID']],
	    c => ['ALL', 'str', 3, 'biom:DNAAlleleIndicator'],
	    d => ['LAI', 'str', 4, 'biom:DNALocusAnalysisIndicator'],
	    e => ['PCDT', 'str', 5, 'biom:DNAPreciseCallIndicator'],
	    f => ['AL1', 'str', 6, 'biom:DNAAlleleCall1Text', \&omit_empty],
	    g => ['AL2', 'str', 7, 'biom:DNAAlleleCall2Text', \&omit_empty],
	    h => ['AL3', 'str', 8, 'biom:DNAAlleleCall3Text', \&omit_empty],
	    i => ['BID', 'str', 9, ['biom:DNABatchIdentification',
				    'nc:IdentificationID'], \&omit_empty],
	    j => ['ECR', 'str', 10, ['biom:DNAElectropherogramIdentification',
				     'nc:IdentificationID'], \&omit_empty],
	    k => ['LCR', 'str', 11, 
		  ['biom:DNAElectropherogramLadderIdentification',
		   'nc:IdentificationID'], \&omit_empty],
	    dnakit => ['', 'str', 12, 'biom:DNAKit'],
	    l => ['KID', 'dnakit', 1, ['biom:DNAKitIdentification',
				       'nc:IdentificationID']],
	    'm' => ['KNM', 'dnakit', 2, 'biom:DNAKitName', \&omit_empty],
	    n => ['KMF', 'dnakit', 3, 'biom:DNAKitManufacturerName',
		  \&omit_empty],
	    o => ['KDS', 'dnakit',4 , 'biom:DNAKitDescriptionText',
		  \&omit_empty],
	},
	17 => {
	    'dmd' => ['DMD', 'body', 17, 'biom:DNAMitochondiralData'],
	    a => ['MT1', 'dmd', 1, 'biom:DNAMitoControRegion1Text',
		  \&xml_escape],
	    b => ['MT2', 'dmd', 2, 'biom:DNAMitoControRegion2Text',
		  \&xml_escape],
	    c => ['BSP', 'dmd', 3, 'biom:DNAMitoBaseStartNumeric'],
	    d => ['BEP', 'dmd', 4, 'biom:DNAMitoBaseEndNumeric'],
	    e => ['BCA', 'dmd', 5, 'biom:DNAMitoBaseAdenineQuantity'],
	    f => ['BCG', 'dmd', 6, 'biom:DNAMitoBaseGuanineQuantity'],
	    g => ['BCC', 'dmd', 7, 'biom:DNAMitoBaseCytosineQuantity'],
	    h => ['BCT', 'dmd', 8, 'biom:DNAMitoBaseThymineQuantity'],
	},
	# 18 user defined profile
	19 => {
	    epds => ['', 'body', 19, 'array', 'one_per_record'],
	    epd => ['EPD', 'epds', 1, 'biom:DNAElectropherogram'],
	    a => ['EIR', 'epd', 1, ['biom:DNAElectropherogramIdentification',
				    'nc:IdentificationID']],
	    b => ['EST', 'epd', 2, 'biom:DNAElectropherogramFileStorageText',
		  \&xml_escape],
	    c => ['IDD', 'epd', 3, 
		  'biom:DNAElectropherogramDataDescriptionText', \&xml_escape],
	    d => ['ELPD', 'epd', 4, 'biom:DNAElectropherogramBinaryObject'],
	    e => ['EPS', 'epd', 5, ['biom:DNAElectropherogramScreenshotImage',
				    'nc:BinaryBase64Object']],
	},
	20 => ['DGD', 'body', 21, 'biom:DNAGenotypeDistributionCode'],
	21 => {
	    gaps => ['', 'body', 22, 'array', 'one_per_record'],
	    gap => ['GAP', 'gaps', 1, 'biom:GenotypeAllelePair'],
	    a => ['GLR', 'gap', 1, ['biom:DNALocusIdentification',
				    'nc:IdentificationID']],
	    b => ['ALP', 'gap', 2, 'biom:DNAGenotypeAllelePairText'],
	    c => ['GNW', 'gap', 3, 'biom:DNAGenotypeWeightNumeric'],
	},
	22 => ['COM', 'body', 23, 'biom:DNACommentText'],
	23 => {
	    epls => ['', 'body', 20, 'array', 'one_per_record'],
	    epl => ['EPL', 'epls', 1, 'biom:DNAElectropherogramLadder'],
	    a => ['LIR', 'epl', 1, ['biom:DNAElectropherogramIdentification',
				    'nc:IdentificationID']],
	    b => ['LST', 'epl', 2, 'biom:DNAElectropherogramFileStorageText'],
	    c => ['LDD', 'epl', 3, 'biom:DNAElectropherogramDataDescriptionText'],
	    d => ['LEPD', 'epl', 4, 'biom:DNAElectropherogramBinaryObject'],
	    e => ['LES', 'epl', 5, ['biom:DNAElectropherogramScreenshotImage',
				    'nc:BinaryBase64Object'], \&omit_empty],
	},
    },
		
    19 => {
	tag => 'itl:PackagePlantarImageRecord',
	fallback => 'tagged',
	parts => {
	    body => ['', 'record', 10, 'biom:PlantarImage'],
	},
	13 => ['FGP', 'body', 16, 'biom:PlanterPositionCode'],
	18 => {
	    amps => ['', 'body', 17, 'array', 'one_per_record'],
	    amp => ['AMP', 'amps', 1, 'biom:PlantarImageMissingArea'],
	    a => ['FRAP', 'amp', 1, 'biom:PlantarPositionCode'],
	    b => ['ABC', 'amp', 2, 'biom:PlantarMissingAreaReasonCode'],
	},
	19 => {
	    fsps => ['', 'body', 19, 'array', 'one_per_record'],
	    fsp => ['FSP', 'fsps', 1, 'biom:PlantarImageSegmentPositionPloygon'],
	    a => ['FRSP', 'fsp', 1, 'biom:PlantarPositionCode'],
	    b => ['NOP', 'fsp', 2, 'biom:PositionPolygonVertexQuantity'],
	    ppv => ['', 'fsp', 3, 'biom:PositionPolygonVertex',
		    'accumulate_pairs', 1],
	    odd => ['HPO', 'ppv', 1, 'biom:PositionHorizontalCoordinateValue'],
	    even => ['VPO', 'ppv', 2, 'biom:PositionVerticalCoordinateValue'],
	},
	24 => {
	    fqms => ['', 'body', 18, 'array', 'one_per_record'],
	    fqm => ['FQM', 'fqms', 1, 'biom:PlantarImageQuality'],
	    a => ['FRMP', 'fqm', 1, 'biom:PlantarPositionCode'],
	    b => ['QVU', 'fqm', 4, 'biom:QualityValue'],
	    c => ['QAV', 'fqm', 3, 
		  ['biom:QualityAlgorithmVendorIdentification',
		   'nc:IdentificationID']],
	    d => ['QAP', 'fqm', 2,
		  ['biom:QualityAlgorithmProductIdentification',
		   'nc:IdentificationID']],
	},
    },

    20 => {
	tag => 'itl:PackageSourceRepresentationRecord',
	fallback => 'tagged',
	parts => {
	    body => ['', 'record', 13, \&set_type20_body_tag],
	},
	# 902, 995, 996 -- from tagged
	3 => ['CAR', 'record', 7, 'biom:SourceRecordCardinalityCode'],
	14 => {
	    aqss => ['', 'record', 8, 'array', 'one_per_record'],
	    aqs => ['AQS', 'aqss', 1, 'biom:SourceAcquisition'],
	    a => ['AQT', 'aqs', 1, 'biom:AcquisitionSourceCode'],
	    b => ['A2D', 'aqs', 2, 
		  'biom:AcquisitionDigitalConversionDescriptionText'],
	    c => ['FDN', 'aqs', 3, 'biom:AcquisitionFormatDescriptionText'],
	    d => ['AQSC', 'aqs', 4, 
		  'biom:AcquisitionSpecialCharacteristicsText'],
	},
	15 => {
	    sft => ['SFT', 'record', 9, 'biom:SourceFileFormat'],
	    a => ['FTY', 'sft', 1, 'biom:SourceFileCategoryText'],
	    b => ['DEI', 'sft', 2, 'biom:SourceFileDecodingInstructionsText'],
	},
	20 => ['COM', 'record', 10, 'biom:SourceCommentText'],
	21 => ['SRN', 'record', 11, ['biom:SourceIdentification',
				     'nc:IdentificationID']],
	994 => ['EFR', 'record', 12, 'biom:SourceExternalFileReferenceText'],
	# 999, 12, 998 -- from tagged
	5 => cap_date_spec('DAT'),
        # 903, and 904 -- from tagged
	17 => $cap_shps_spec,
	# 4, 993 -- from tagged
	18 => $cap_svps_spec,
	13 => ['CSP',  'body', 5, $csp_tag],
	# 11, 6, 9, 8, 7, 10 -- from tagged
	16 => {
	    segs => ['', 'body', 15, 'array', 'one_per_record'],
	    seg => ['SEG', 'segs', 1, 'biom:ImageSegment'],
	    a => ['RSP', 'seg', 1, 'biom:ImageSegmentIdentification'],
	    b => ['IPT', 'seg', 2, 'biom:ImageSegmentInternalIdentification'],
	    c => ['NOP', 'seg', 3, 'biom:PositionPolygonVertexQuantity'],
	    ppv => ['', 'seg', 4, 'biom:PositionPolygonVertex',
		    'accumulate_pairs', 0],
	    even => ['HPO', 'ppv', 1, 'biom:PositionHorizontalCoordinateValue'],
	    odd => ['VPO', 'ppv', 2, 'biom:PositionVerticalCoordinateValue'],
	},
	19 => {
	    tixs => ['', 'body', 16, 'array', 'one_per_segment'],
	    tix => ['TIX', 'tixs', 1, 'biom:TimeSegment'],
	    a => ['TIS', 'tix', 1, 'biom:TimeSegmentStartTimeValue'],
	    b => ['TIE', 'tix', 2, 'biom:TimeSetmentEndTimeValue'],
	},
    },
    
    21 => {
	tag => 'itl:PackageAssociatedContextRecord',
	fallback => 'tagged',
	parts => {
	    # There are three other possible types for the body, which
	    # could be determined automatically by examining the data
	    # to determine whether it is a still image, video, audio,
	    # or external file.
	    body => ['', 'record', 11, 'biom:ContextImage'],
	},
	# 902, 996 -- from tagged
	15 => {
	    acf => ['ACF', 'record', 7, 'biom:ContextFileFormat'],
	    a => ['FTY', 'acf', 1, 'biom:ContextFileCategoryText'],
	    b => ['DEI', 'acf', 2, 'biom:ContextFileDecodingInstructionsText'],
	},
	20 => ['COM', 'record', 8, 'biom:ContextCommentText'],
	21 => ['ACN', 'record', 9, 'biom:ContextIdentification'],
	994 => ['EFR', 'record', 12, 'biom:ContextExternalFileReferenceText'],
	# 999, 998 -- from tagged
	5 => cap_date_spec('ACD'),
	# 4, 993 -- from tagged
	16 => {
	    seg => ['SEG', 'body', 5, 'biom:ImageSegment'],
	    ver => ['', 'seg', 4, 'biom:PositionPolygonVertex'],
	    a => ['ASP', 'seg', 1, 'biom:ImageSegmentIdentification'],
	    b => ['IPT', 'seg', 2, 'biom:ImageSegmentInternalIdentification'],
	    c => ['NOP', 'seg', 3, 'biom:PositionPolygonVertexQuantity'],
	    d => ['HPO', 'ver', 1, 'biom:PositionHorizontalCoordinateValue'],
	    e => ['VPO', 'ver', 2, 'biom:PositionVerticalCoordinageValue'],
	},
	19 => {
	    tix => ['TIX', 'body', 6, 'biom:TimeSegment'],
	    a => ['TIS', 'tix', 1, 'biom:TimeSegmentStartTime'],
	    b => ['TIE', 'tix', 2, 'biom:TimeSegmentEndTime'],
	},
	21 => ['ACN', 'record', 9, ['biom:ContextIdentification', 
				    'nc:IdentificationID']],
    },

    # record types 22 through 97 are reserved

    98 => {
	tag => 'itl:PackageInformationAssuranceRecord',
	fallback => 0,
    },

    99 => {
	tag => 'itl:PackageCBEFFBiometricDataRecord',
	fallback => 0,
    },
};


# Lookup a specification for converting a particular item, starting
# from the most particular and working towards more general matches.
sub lookup_item_spec {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 5;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 6;
    my ($specs, $rn, $fn, $si, $ii, $quiet) = @_;
    my $ia = item_letter($ii);
    my $spec;

#    warn "lookup item spec \[$rn.$fn\]";
    # Look for subfield item and default item spec.
    if (defined $specs->{$rn}->{$fn}) {
	my $fsh = $specs->{$rn}->{$fn};
	if ('HASH' eq ref $fsh && defined $si && defined $ii) {
	    if (exists $fsh->{"$si$ia"}) { # subfield-number, item-letter
		$spec = $fsh->{"$si$ia"};
	    } elsif (exists $fsh->{$ia}) { # item-letter alone
		$spec = $fsh->{$ia};
	    } elsif ($ii%2 == 1 && exists $fsh->{odd}) {
		$spec = $fsh->{odd};
	    } elsif ($ii%2 == 0 && exists $fsh->{even}) {
		$spec = $fsh->{even};
	    } elsif (exists $fsh->{rest}) {
		$spec = $fsh->{rest};
	    }
	} else {
	    $spec = $fsh;
	}
    }
    
    if (!$spec && exists $specs->{$rn}->{fallback}){
	$spec = lookup_item_spec($specs, $specs->{$rn}->{fallback},
				 $fn, $si, $ii, 1);
    }
    
    if (!$quiet && !$spec) {
	warn "No item spec found for $rn.$fn-$si$ia";
    }
    return $spec;
}

# Lookup a specification for creating a particular node, starting
# from the most particular and working towards more general matches.
sub lookup_node_spec {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 4;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 5;
    my ($specs, $rn, $fn, $name, $quiet) = @_;
    my $spec;
    my @augmented_spec;
    
#    warn "lookup node spec \[$rn.$fn\] $name";
    # Look for subfield node and default node spec.
    my $rsh = $specs->{$rn};
    if (exists $rsh->{$fn}) {
	my $fsh = $rsh->{$fn};
	if ('HASH' eq ref $fsh && exists $fsh->{$name}) {
	    $spec = $fsh->{$name};
	    if (@$spec > 4) {
		return $spec;
	    } else {
		@augmented_spec = (@$spec, 'one_per_subfield');
		return \@augmented_spec;
	    }
	}
    }
    if (exists $rsh->{parts} &&
	exists $rsh->{parts}->{$name}) {
	$spec = $rsh->{parts}->{$name};
	if (@$spec > 4) {
	    return $spec;
	} else {
	    @augmented_spec = (@$spec, 'one_per_record');
	    return \@augmented_spec;
	}
    }
    if (exists $specs->{$rn}->{fallback}) {
	$spec = lookup_node_spec($specs, $specs->{$rn}->{fallback},
				 $fn, $name, 1);
	return $spec if $spec;
    }

    warn "No node spec found for $rn.$fn-$name or $rn.parts.$name" unless $quiet;
    return undef;
}

sub lookup_field_spec {
    my ($specs, $rn, $fn) = @_;

#    warn "lookup field spec \[$rn.$fn\]";    
    if (exists $specs->{$rn}->{$fn}) {
	return $specs->{$rn}->{$fn};
    } elsif (exists $specs->{$rn}->{fallback}) {
	return lookup_field_spec($specs, $specs->{$rn}->{fallback}, $fn);
    } else {
	return undef;
    }
}

# Count the possible number of items in a field.
sub count_items {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 3;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 3;
    my ($specs, $rn, $fn) = @_;

    my $spec_hash = lookup_field_spec($specs, $rn, $fn);
    if (! defined $spec_hash) {
	warn "No specs found for $rn.$fn";
	return undef;
    }
    my $count;
    if (ref $spec_hash eq 'HASH') {
	$count = grep {/^[a-z]$/} keys %$spec_hash;
    } else {
	$count = 1;
    }
    return $count;
}

# Record a reference to a node structure in an XML record so it can be
# looked up with lookup_node.
sub record_node {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 6;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 7;
    my ($xmlrec, $loc, $node, $si, $ii, $mode, $mode_arg) = @_;
    # insert in data structures
    my $key = $loc;
    if ($mode eq 'one_per_subfield') {
	$key .= ('-s' . $si);
    } elsif ($mode eq 'accumulate_pairs') {
	# Nodes to accumulate pairs of items get two entries, so you
	# don't need to know whether a node is the first or second in
	# order to look up its parent node using the item number.
	my $ikey = "$key-s$si-i$ii";
	$xmlrec->{nodes}->{$ikey} = {node => $node};
	$key .= ("-s$si-i" . ($ii + 1));
    }
    $xmlrec->{nodes}->{$key} = {node => $node};
}

# Lookup the location of a node reference in an XML record that was
# recorded by record_node.
sub lookup_node {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 4;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 4;
    my ($record, $loc, $si, $ii) = @_;
    # retrieve from data structure
    my $node;

    if ($loc eq 'record') {
	$node = $record;
    } elsif (defined $record->{nodes}->{$loc}) {
	$node = $record->{nodes}->{$loc}->{node};
    } elsif (defined $record->{nodes}->{"$loc-s$si"}) {
	$node = $record->{nodes}->{"$loc-s$si"}->{node};
    } elsif (defined $record->{nodes}->{"$loc-s$si-i$ii"}) {
	$node = $record->{nodes}->{"$loc-s$si-i$ii"}->{node};
    }
    return $node;
}

# Create the specified node, and any ancester nodes that have not yet
# been created.  This is not used to create leaf nodes.
sub create_ancestor_nodes {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 10;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 10;
    my ($specs, $txtrec, $xmlrec, $rn, $fn, $ri, $fi, $si, $ii, $name) = @_;
    my $spec = lookup_node_spec($specs, $rn, $fn, $name);
    return undef unless $spec;
    
    # The specification of the current node is used to determine its
    # parent node.
    my ($mnm, $loc, $pos, $tag, $mode, $mode_arg) = @$spec; 
    
    # Find the parent node if it exists, otherwise recurse to create it too.
    my $parent_node = lookup_node($xmlrec, $loc, $si, $ii);
    if (!$parent_node) {
	$parent_node = 
	    create_ancestor_nodes($specs, $txtrec, $xmlrec,
				  $rn, $fn, $ri, $fi, $si, $ii, $loc);
    }
    return undef unless $parent_node;

    # If a function is provided to determine the tag, instead of a tag
    # name or list of tag names, run it.
    if (ref $tag eq 'CODE') {
	$tag = &$tag($ri, $fi, $si, $ii, $rn, $fn, $specs, $xmlrec, $txtrec);
    }
    
    # Create the node.
    my $new_node = {tag => $tag, content => [], repeats => $mode};
    if ($mnm) {
	$new_node->{comment} = field_comment($rn, $fn, $mnm, 0, 0);
    }
    if ($tag eq 'array' || $mode eq 'one_per_record') {
	$new_node->{increment} = count_items($specs, $rn, $fn);
    }
    
    if ($parent_node->{tag} eq 'array' || 
	(defined $parent_node->{repeats} &&
	 $parent_node->{repeats} eq 'one_per_record')) {
	$pos += (($si - 1) * $parent_node->{increment});
    }
    if ($mode eq 'accumulate_pairs') {
	$pos += pair_index($ii, $mode_arg);
    }
    if (defined $parent_node->{content}->[$pos]) {
	warn "overwriting $loc\[$pos\]: ", 
	Dumper($parent_node->{content}->[$pos]), 
	" with $ri.$fi.$si.$ii \[$rn.$fn\] $name: ", Dumper($new_node);
    }
    $parent_node->{content}->[$pos] = $new_node;
    record_node($xmlrec, $name, $new_node, $si, $ii, $mode, $mode_arg);
    return $new_node;
}

sub pair_index {
    my $ii = shift;
    my $arg = shift;

    if ($ii % 2 == $arg) {
	return int($ii / 2);
    } else {
	return int(($ii - 1) / 2);
    }
}

# Recieve the item index in the argument and return the corresponding
# item letter.
sub itlet {
    my $i = shift;

    if ($i < 26) {
	return chr(ord('a') + $i);
    } else {
	return itlet($i / 26) . itlet($i % 26);
    }
}

sub item_letter {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 1;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 1;

    my $ii = shift;
    return itlet($ii - 1);
}

# Recieve information about a field and return a comment string
# pertaining to that field.  Required arguments are: indent_level,
# record type number, and field type number.  The remaining arguments
# are optional: mnemonic, item letter, very brief comment.
sub field_comment {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 2;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 6;
    my ($rn, $fn, $mnm, $si, $ii, $comment) = @_;
    $mnm = '' if !defined $mnm;
    my $item_letter = (defined $ii) ? item_letter($ii) : '';
    $comment = '' if !defined $comment;
    my $sfit_str;
    if ($si != 0 && $ii != 0) {
	$sfit_str = "-$item_letter$si";
    } else {
	$sfit_str = '';
    }

    return sprintf("fieldID: %d.%03d%s%s%s", $rn, $fn, $sfit_str,
		   $mnm ? " $mnm" : '', $comment ? " ($comment)" : '');
}

# Recieve either one tag string or list of tag strings, and a value,
# and return a single level or nested hash of tags and contents.
sub entag_hash {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 2;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 2;
    my ($tags, $val) = @_;
    my $result = {};
    my $tags_type = ref $tags;
    my $node = $result;

    if ($tags_type eq '') {
	$node->{tag} = $tags;
	$node->{content} = $val;
    } elsif ($tags_type eq 'ARRAY') {
	for (my $i = 0; $i < $#$tags; $i++) {
	    $node->{tag} = $tags->[$i];
	    $node->{content} = [{}];
	    $node = $node->{content}->[0];
	}
	$node->{tag} = $tags->[-1];
	$node->{content} = $val;
    } else {
	warn "ignoring unknown tags type: $tags_type";
	return undef;
    }
    return $result;
}

# Process the 4-D txt array to produce an internal representation of
# the corresponding XML.
sub generate_xml_tree {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 2;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 2;
    my ($specs, $txt) = @_;

    # An array of data structures each corresponding to a record.
    # Each record has its own direct subtree of the XML transaction
    # document.
    my $xmltree = [];

    for (my $ri = 1; $ri < @$txt; $ri++) {
	my $txtrec = $txt->[$ri];
	my $rn = $txtrec->[0];
	my $xmlrec = {tag => $specs->{$rn}->{tag},
		      content => [],
		      nodes => {},
	              comment => ["=",
				  sprintf("   RECORD TYPE %02d", $rn) . 
				  (' ' x ($line_length - 30)),
				  "="]};
	$xmltree->[$ri] = $xmlrec;
	$xmlrec->{content}->[0] = {
	    tag => 'biom:RecordCategoryCode',
	    content => $rn,
	};

	# Field 1 is the record size, which is not used in the XML format.
	for (my $fi = 2; $fi < @$txtrec; $fi++) {
	    my $fld = $txtrec->[$fi];
	    my $fn = $fld->[0];

	    for (my $si = 1; $si < @$fld; $si++) {
		my $subf = $fld->[$si];

		for (my $ii = 1; $ii < @$subf; $ii++) {
		    my $item = $subf->[$ii];

		    my $spec = lookup_item_spec($specs, $rn, $fn, $si, $ii);
		    next unless $spec;
		    next if $spec eq 'ignore';
		    my ($mnm, $loc, $pos, $tags, $val_func, $vf_arg) = @$spec;
		    my ($set_func, $sf_arg) = ($tags, $val_func);

 
		    if (ref $val_func eq 'CODE') {
			$item = &$val_func($item, $vf_arg);
			# Zero is a value we want to keep, by itself it's
			# false, but it is defined.  We do, however, want
			# to reject an empty string.
			next unless (defined $item && $item ne '');
		    }

		    my $node = lookup_node($xmlrec, $loc, $si, $ii);
		    if (!$node) {
			$node = 
			    create_ancestor_nodes($specs, $txtrec, $xmlrec,
						  $rn, $fn, $ri, $fi, $si, $ii,
						  $loc);
		    }
		    if (!$node) {
			warn "cannot make node for field: ", 
			sprintf("%d.%d.%d.%d \[%d.%03d\]: '%.10s%s'",
				$ri, $fi, $si, $ii, $rn, $fn, $item,
				(length($item) > 10) ? '...' : '');
			next;
		    }
		    if (exists $node->{repeats} &&
			$node->{repeats} eq 'accumulate_items') {
			$pos += $ii;
		    }

		    my $tags_type = ref $tags;
		    if ('CODE' eq $tags_type) {
			&$set_func($ri, $fi, $si, $ii, $rn, $fn, $mnm, 
				   $item, $node, $pos, $specs, $xmlrec,
				   $txtrec, $sf_arg);
		    } elsif ('' eq $tags_type || 'ARRAY' eq $tags_type) {
			if ($node->{tag} eq 'array' || 
			    (exists $node->{repeats} && 
			     $node->{repeats} eq 'one_per_record')) {
			    $pos += (($si - 1) * $node->{increment});
			}
			if (defined $node->{content}->[$pos]) {
			    warn "overwriting $loc\[$pos\] ",
			    Dumper($node->{content}->[$pos]), " with ",
			    sprintf("%d.%d.%d.%d \[%d.%03d\]: '%.10s%s'",
				    $ri, $fi, $si, $ii, $rn, $fn, $item,
				    (length($item) > 10) ? '...' : '');
			}
			$node->{content}->[$pos] = entag_hash($tags, $item);
			$node->{content}->[$pos]->{comment} =
			    field_comment($rn, $fn, $mnm, $si, $ii);
		    } else {
			warn "'$tags_type' not implemented for tags: $tags";
		    }
		}
	    }
	}
    }
    return $xmltree;
}

# Convert the internal XML data structure to standard XML document
# format.  However, the outer tag with all the namespace attributes
# still needs to be added.
sub data_to_xml {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 2;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 2;
    my ($content, $indent_level) = @_;
    my $content_type = ref $content;
    my $result = "";
    $indent_level = 0 unless defined $indent_level;
    my $indent_len = $indent_level * $indent_factor;
    my $indent_str = ' ' x $indent_len;

    if ($content_type eq '') {
	if (length($content) > 100) {
	    $result = $content;
 	} else {
	    $result = "$indent_str$content\n";
	}
    } elsif ($content_type eq 'ARRAY') {
	for my $node (@$content) {
	    if (!defined $node) {
		next;
	    }
	    my $node_type = ref $node;
	    if ('HASH' ne $node_type) {
		print "skip node type '$node_type': ", Dumper($node), "\n";
		next;
	    }
	    if (exists $node->{comment}) {
		my $comments;
		if (ref $node->{comment} eq 'ARRAY') {
		    $comments = $node->{comment};
		} else {
		    $comments = [$node->{comment}];
	        }
	        for my $comment (@$comments) {
		    if ($comment eq '=') {			
			$result .= ("$indent_str<!-- " . 
				    ('=' x ($line_length - $indent_len - 10)) .
				    " -->\n");
		    } else {
#			my $eqs = '=' x ($line_length - 14 - $indent_len -
#					 length($comment));
#			$result .= "$indent_str<!-- $eqs $comment == -->\n";
			$result .= "$indent_str<!-- $comment -->\n";
		    }
		}
	    }

	    if ($node->{tag} eq 'array') {
		# anonymous array of nodes
		$result .= data_to_xml($node->{content}, $indent_level);
	    } else {
		my $incremental_str = (
		    "$indent_str<$node->{tag}>\n" .
		    data_to_xml($node->{content}, $indent_level+1) .
		    "$indent_str</$node->{tag}>\n"
		    );
		my $shortened_str = $incremental_str;
		$shortened_str =~ s/\s*\n\s*//g;
		$shortened_str .= "\n";
		if (length($shortened_str) < $line_length ||
		    $node->{tag} =~ /HashValue/) {
		    $result .= $shortened_str;
		} else {
		    $result .= $incremental_str;
		}
	    }
	}
    }
    return $result;
}

# The following are data formatting and conversion functions, which
# are used at the end of a field spec.  These take one required
# argument, the value, and return one result, the new value.  One
# optional argument can be specified after the conversion function in
# the field spec.

sub xml_escape {
    my $val = shift;
    
    $val =~ s/&/&amp;/;
    return $val;
}

sub omit_empty_esc {
    return omit_empty(xml_escape(shift));
}

# Delete any leading zeros from an integer value.
sub dezero {
    my $value = shift;
    if ($value !~ /^\d+$/) {
	die "non-numeric argument '$value'";
    }
    return int($value);
}

# Convert a compact numerical date value to the ISO standard format with
# hyphens.
sub hyphenate_date {
    my $date = shift;
    if ($date !~ /^\d{8}$/) {
	die "not a valid numeric date '$date'";
    }
    return substr($date, 0, 4) .'-'. substr($date, 4, 2) .'-'. substr($date, 6, 2);
}

# Convert a compact numerical GMT date value to the ISO standard format
# with hyphens, colons, and the letters T and Z.
sub punctuate_utc {
    my $date = shift;
    if ($date !~ /^\d{14}Z$/) {
	die "not a valid compact numerical GMT date '$date'";
    }
    return hyphenate_date(substr($date, 0, 8)) .'T'. substr($date, 8, 2) .':'.
	substr($date, 10, 2) .':'. substr($date, 12, 2) .'Z';
}

# Read in an image data file and convert it to base64 as required to embed
# it into XML.  
sub read_and_encode_image {
    # Whitespace characters are ignored, so there is no need to wrap the
    # tags tighly around the encoded data returned.

    my $filename = shift;

    open IMG, '<', $filename or die "cannot open image file: $filename";
    my $image_data = do { local $/; <IMG> };
    close IMG;
    return encode_base64($image_data);
}

# Recieve a text string and return the same string enclosed in XML
# comment delimiters.
sub make_xml_comment {
    return "<!-- " . (shift) . " -->";
}

# Skip finger positions designated unused with code 255.
sub fgp_value_func {
    my $fgp = shift;
    return (255 == $fgp) ? undef : $fgp;
}

# Skip if the value is empty.
sub omit_empty {
    my $arg = shift;
    return $arg;
}

# Field conversion functions for special cases, used instead of a tag and
# optional data conversion function in a field spec.

# The VER field is split between the major and minor version number tags.
sub set_type1_ver {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos,
	$specs, $xmlrec, $txtrec, $arg) = @_;
    
    $node->{content}->[$pos] = {
	comment => field_comment($rn, $fn, $mnm, $si, $ii, 'first two digits'),
	tag => 'biom:TransactionMajorVersionValue',
	content => substr($val, 0, 2),
    };
    $node->{content}->[$pos+1] = {
	comment => field_comment($rn, $fn, $mnm, $si, $ii, 'last two digits'),
	tag => 'biom:TransactionMinorVersionValue',
	content => substr($val, 2, 2),
    };
}

# Type 8 data appears in field 8 of the traditional format regardless
# of whether it is an image or vector, but the two types of data
# appear in different positions in the XML representation.  The binary
# data position is specified in the arguments, and the vector data
# position relative to the other position is hard-coded into the function.
sub set_type8_data {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos,
	$specs, $xmlrec, $txtrec, $arg) = @_;
    
    my $srt;
    for (my $i = 1; $i < @$txtrec; $i++) {
	my $txtfld = $txtrec->[$i];
	if (defined $txtfld && $txtfld->[0] == 4) { # SRT
	    $srt = $txtfld->[1]->[1];
	    last;
	}
    }
    if (0 == $srt || 1 == $srt) {	# image
	my $comment = (0 == $srt) ? 'raw' : 'fax';
	$node->{content}->[$pos] = {
	    comment => field_comment($rn, $fn, $mnm, $si, $ii, $comment),
	    tag => 'nc:BinaryBase64Object',
	    content => read_and_encode_image($val),
	};
    } elsif (2 == $srt) {			# vector
	# The location and position must match those specified for
	# item 8.008a in subsequent subfields.
	my $vspec = lookup_item_spec($specs, $rn, $fn, 2, $ii);
	my ($vmnm, $vloc, $vpos, $vtag, $vmode, $vmode_arg) = @$vspec;
	my $vnode = create_ancestor_nodes($specs, $txtrec, $xmlrec, $rn, $fn,
					  $ri, $fi, $si, $ii, $vloc);
	$vnode->{content}->[$vpos] = { 
	    comment => field_comment($rn, $fn, $mnm, $si, $ii),
	    tag => $vtag,
	    content => $val,
	};
    } else {
	warn "unknown signature representation code (SRT): $srt";
    }
}

sub face_image_description {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos, 
	$specs, $xmlrec, $txtrec, $arg) = @_;
    
    # Besides these codes from table 46, the codes from NCIC are also included.
    my @codes = (
	'UNKNOWN',		'NEUTRAL',		'SMILE',
	'MOUTH OPEN',		'TEETH VISIBLE',	'RAISED BROWS',
	'FROWNING',		'EYES AWAY',		'SQUINTING',
	'LEFT EYE PATCH',	'RIGHT EYE PATCH',	'CLEAR GLASSES',
	'DARK GLASSED',		'HAT',			'SCARF',
	'MOUSTACHE',		'BEARD',		'NO EAR',
	'BLINK',		'DISTORTING CONDITION');
    my ($tag, $comment);
    if (grep {/$val/} @codes) {
	$tag = 'biom:FaceImageDescriptionCode';
	$comment = 'Table 46';
    } elsif ($val =~ /^[A-Z ]{3,10}$/) {
	$tag = 'biom:FaceImageDescriptionCode';
	$comment = 'looks like NCIC';
    } else {
	$tag = 'biom:FaceImageDescriptionText';
    }
    $node->{content}->[$pos] = {
	comment => field_comment($rn, $fn, $mnm, $si, $ii, $comment),
	tag => $tag,
	content => &xml_escape($val),
    };
}

# Set the position code appropriately for finger or palm positions.
sub set_pos_code {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos, 
	$specs, $xmlrec, $txtrec, $arg) = @_;
    my $tag = ($val < 20) ?
	'biom:MinutiaeFingerPositionCode' : 'biom:MinutiaePalmPositionCode';
    
    $node->{content}->[$pos] = { tag => $tag, content => dezero($val) };
}

# NIST minutia core and delta positions include both X and Y coordinates in
# the same item: XXXXYYYY.  This function splits them apart and creates the
# corresponding coordinate value elements within the specified container
# element.
sub set_xy_pos {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos,
	$specs, $xmlrec, $txtrec, $offset) = @_;
    
    $node->{content}->[$pos] = {
	comment => field_comment($rn, $fn, $mnm, $si, $ii, 'first four digits'),
	tag => 'biom:PositionHorizontalCoordinateValue',
	content => substr($val, 0, 4),
    };
    $node->{content}->[$pos + $offset] = {
	comment => field_comment($rn, $fn, $mnm, $si, $ii, 'second four digits'),
	tag => 'biom:PositionVerticalCoordinateValue',
	content => substr($val, 4, 4),
    };
}

# NIST minutia details include x, y, and theta values combined into single
# items: XXXXYYYYTTT.  This function separates them and creates their
# corresponding tags within the parent element.
sub set_xytheta_values {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos,
	$specs, $xmlrec, $txtrec, $offsets) = @_;
    
    set_xy_pos($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos,
	       $specs, $xmlrec, $txtrec, $offsets->[0]);
    $node->{content}->[$pos + $offsets->[1]] = {
	comment => field_comment($rn, $fn, $mnm, $si, $ii, 'last three digits'),
	tag => 'biom:PositionThetaAngleMeasure',
	content => dezero(substr($val, 8, 3)),
    };
}

# Split the comma-separated minutia number and ridge count and create the
# corresponding set of nested elements.
sub set_ridge_count_values {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos,
	$specs, $xmlrec, $txtrec, $arg) = @_;
    
    $val =~ /([0-9]+),([0-9]+)/;
    my ($id, $count) = ($1, $2);

    $node->{content}->[$pos + $ii - 5] = {
	comment => field_comment($rn, $fn, $mnm, $si, $ii, 'id,count'),
	tag => 'biom:MinutiaRidgeCount',
	content => [
	    entag_hash(['biom:RidgeCountReferenceIdentification',
			'nc:IdentificationID'], $id),
	    {
		tag => 'biom:RidgeCountValue',
		content => $count,
	    },
	    ],
    }
}

# Distribute the individual octant residuals to their corresponding
# ridge counts, from $pos to $pos + 7.
sub set_iafis_octant_residuals {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos,
	$specs, $xmlrec, $txtrec, $arg) = @_;

    return unless $val;
    my @octant_residuals = split(//, $val);    
    for (my $i = 0; $i < 8; $i++) {
	my $new_leaf = {
	    tag => 'ebts:MinutiaOctantResidualNumeric',
	    content => $octant_residuals[$i],
	};
	push(@{$node->{content}->[$pos+$i]->{content}}, $new_leaf)
    }
}

sub set_iafis_ridge_count {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos,
	$specs, $xmlrec, $txtrec, $arg) = @_;
    my $octant = $ii - 5;
    
    $node->{content}->[$pos] = {
	tag => 'ebts:MinutiaRidgeCount',
	comment => field_comment($rn, $fn, $mnm, $si, $ii, 'rrrcc'),
	content => [
	    entag_hash(['biom:RidgeCountReferenceIdentification',
			'nc:IdentificationID'], dezero(substr($val, 0, 3))),
	    {
		tag => 'biom:RidgeCountValue',
		content => dezero(substr($val, 3, 2)),
	    },
	    {
		tag => 'ebts:MinutiaOctantNumeric',
		content => $octant,
	    },
	    ],
    };
}

# The single data item from the binary format needs to be broken down
# into components that are tagged separately in the XML format.
sub set_accredation {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos, 
	$specs, $xmlrec, $txtrec, $arg) = @_;
    my $result_array = [];
    
    $node->{content}->[$pos] = {
	comment => field_comment($rn, $fn, $mnm, $si, $ii),
	tag => 'biom:DNALaboratoryAccreditation',
	content => $result_array,
    };
    if (0 == $val || 255 == $val) {
	push(@$result_array, {
	    tag => 'biom:DNALaboratoryAccreditationLevelCode',
	    content => $val,
	     });
    } else {
	my @pairs = split(/,/, $val);
	for my $pair (sort @pairs) {
	    $pair =~ /^(\d)([NMDO]+)$/;
	    my ($org_code, $scopes) = ($1, $2);
	    push(@$result_array, {
		tag => 'biom:DNALaboratoryAccreditationLevelCode',
		content => $org_code,
		 });
	    for my $scope_code (split(//, $scopes)) {
		push(@$result_array, {
		    tag => 'biom:DNALaboratoryAccreditationScopeCode',
		    content => $scope_code,
		     });
	    }
	}
    }
}

sub set_country_code {
    my ($ri, $fi, $si, $ii, $rn, $fn, $mnm, $val, $node, $pos,
	$specs, $xmlrec, $txtrec, $arg) = @_;
    my $tag;
    
    if ($val =~ /^\d+$/) {
	$tag = 'biom:DNALaboratoryProcessingCountryISO3166NumericCode';
    } elsif (length($val) == 2) {
	$tag = 'biom:DNALaboratoryProcessingCountryISO3166Alpha2Code';
    } elsif (length($val) == 3) {
	$tag = 'biom:DNALaboratoryProcessingCountryISO3166Alpha3Code';
    } else {
	my $errmsg = "invalid country code: $val";
	warn $errmsg;
	return;
    }
    $node->{content}->[$pos] = {
	comment => field_comment($rn, $fn, $mnm, $si, $ii),
	tag => $tag,
	content => $val,
    };
}

# The following functions are used to set tags on record nodes in the
# parts list.  The arguments are different than those for functions
# that set leaf nodes.  The return value is used as the tag.
sub set_type9_body_tag {
    die ((caller(0))[3], ": too few arguments: ", Dumper(@_)) if @_ < 9;
    die ((caller(0))[3], ": too many arguments: ", Dumper(@_)) if @_ > 10;
    my ($ri, $fi, $si, $ii, $rn, $fn, $specs, $xmlrec, $txtrec, $arg) = @_;
    
    for (my $i = 1; $i < @$txtrec; $i++) {
	next unless exists $txtrec->[$i];
	my $fld = $txtrec->[$i];
	my $n = $fld->[0];
	if (4 == $n && $fld->[1]->[1] eq 'S') {
	    return 'itl:Minutiae';
	} elsif (14 == $n) {
	    return 'ebts:Minutiae';
	} elsif (126 == $n) {
	    return 'biom:INCITSMinutiae';
	}
    }
    warn "unrecognized minutiae format";
    return 'unrecognized minutiae format';
}

# Find out whether 10.003 contains 'FACE' or something else and set
# the tag accordingly.
sub set_type10_body_tag {
    my ($ri, $fi, $si, $ii, $rn, $fn, $specs, $xmlrec, $txtrec, $arg) = @_;    

    for (my $i = 1; $i < @$txtrec; $i++) {
	next unless exists $txtrec->[$i];
	my $fld = $txtrec->[$i];
	my $n = $fld->[0];
	if (3 == $n) {
	    # It took a while to see why this always returned
	    # 'biom:PhysicalFeatureImage', even when 10.003 was
	    # clearly 'FACE', when the first expression in the
	    # comparison was "$fld->[1]-[1]". (The second '>' was
	    # missing, but apparently it was valid Perl syntax.)
	    return ($fld->[1]->[1] eq 'FACE') ? 
		'itl:FaceImage' : 'biom:PhysicalFeatureImage';
	}
    }
    warn "could not determine type-10 contents type";
}

# Create a complete standard XML document by wrapping the converted
# record data structures in a top-level ANSI/NIST tag with namespace
# declarations.
sub make_xml {
    my ($xml_ds) = @_;
    my $tag = 'itl:NISTBiometricInformationExchangePackage';
    my $header = <<"EOT";
<$tag
 xmlns:biom="http://niem.gov/niem/biometrics/1.0"
 xmlns:itl="http://biometrics.nist.gov/standard/2011"
 xmlns:ext="http://example.org/extension"
 xmlns:s="http://niem.gov/niem/structures/2.0"
 xmlns:nc="http://niem.gov/niem/niem-core/2.0"
 xmlns:ebts="http://cjis.fbi.gov/fbi_ebts/3.0"
 xmlns:j="http://niem.gov/niem/domains/jxdm/4.1"
 xmlns:xsd="http://www.w3.org/2001/XMLSchema">
EOT
    return $header, data_to_xml($xml_ds, 1), "</$tag>\n";
}


# Main executable section
my $schema = 'nist';
for (my $i = 0; $i < @ARGV; $i++) {
    if ($ARGV[$i] eq 'fbi' || $ARGV[$i] eq 'nist') {
	$schema = $ARGV[$i];
	splice(@ARGV, $i, 1);
	last;
    }
}
if ($schema eq 'fbi') {
    $default_specs->{2}->{3} = 
	['udf3', 'body', 1, 'ebts:DomainDefinedDescriptiveFields',
	 \&make_xml_comment];
    $default_specs->{2}->{4} = 
	['udf4', 'body', 2, 'ebts:DoDDefinedDescriptiveDetail',
	 \&make_xml_comment];    
}

my $parsed_text = parse_txt_file();
#warn "parsed txt: ", Dumper($parsed_text);

my $xml_tree = generate_xml_tree($default_specs, $parsed_text);
#warn "xml tree: ", Dumper($xml_tree);

print make_xml($xml_tree);
